/*
 * Copyright 2013-2018 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.cloudwatch.model;

import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.Consumer;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 */
@Generated("software.amazon.awssdk:codegen")
public final class GetMetricDataRequest extends CloudWatchRequest implements
        ToCopyableBuilder<GetMetricDataRequest.Builder, GetMetricDataRequest> {
    private final List<MetricDataQuery> metricDataQueries;

    private final Instant startTime;

    private final Instant endTime;

    private final String nextToken;

    private final String scanBy;

    private final Integer maxDatapoints;

    private GetMetricDataRequest(BuilderImpl builder) {
        super(builder);
        this.metricDataQueries = builder.metricDataQueries;
        this.startTime = builder.startTime;
        this.endTime = builder.endTime;
        this.nextToken = builder.nextToken;
        this.scanBy = builder.scanBy;
        this.maxDatapoints = builder.maxDatapoints;
    }

    /**
     * <p>
     * The metric queries to be returned. A single <code>GetMetricData</code> call can include as many as 100
     * <code>MetricDataQuery</code> structures. Each of these structures can specify either a metric to retrieve, or a
     * math expression to perform on retrieved data.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return The metric queries to be returned. A single <code>GetMetricData</code> call can include as many as 100
     *         <code>MetricDataQuery</code> structures. Each of these structures can specify either a metric to
     *         retrieve, or a math expression to perform on retrieved data.
     */
    public List<MetricDataQuery> metricDataQueries() {
        return metricDataQueries;
    }

    /**
     * <p>
     * The time stamp indicating the earliest data to be returned.
     * </p>
     * 
     * @return The time stamp indicating the earliest data to be returned.
     */
    public Instant startTime() {
        return startTime;
    }

    /**
     * <p>
     * The time stamp indicating the latest data to be returned.
     * </p>
     * 
     * @return The time stamp indicating the latest data to be returned.
     */
    public Instant endTime() {
        return endTime;
    }

    /**
     * <p>
     * Include this value, if it was returned by the previous call, to get the next set of data points.
     * </p>
     * 
     * @return Include this value, if it was returned by the previous call, to get the next set of data points.
     */
    public String nextToken() {
        return nextToken;
    }

    /**
     * <p>
     * The order in which data points should be returned. <code>TimestampDescending</code> returns the newest data first
     * and paginates when the <code>MaxDatapoints</code> limit is reached. <code>TimestampAscending</code> returns the
     * oldest data first and paginates when the <code>MaxDatapoints</code> limit is reached.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #scanBy} will
     * return {@link ScanBy#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #scanByAsString}.
     * </p>
     * 
     * @return The order in which data points should be returned. <code>TimestampDescending</code> returns the newest
     *         data first and paginates when the <code>MaxDatapoints</code> limit is reached.
     *         <code>TimestampAscending</code> returns the oldest data first and paginates when the
     *         <code>MaxDatapoints</code> limit is reached.
     * @see ScanBy
     */
    public ScanBy scanBy() {
        return ScanBy.fromValue(scanBy);
    }

    /**
     * <p>
     * The order in which data points should be returned. <code>TimestampDescending</code> returns the newest data first
     * and paginates when the <code>MaxDatapoints</code> limit is reached. <code>TimestampAscending</code> returns the
     * oldest data first and paginates when the <code>MaxDatapoints</code> limit is reached.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #scanBy} will
     * return {@link ScanBy#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #scanByAsString}.
     * </p>
     * 
     * @return The order in which data points should be returned. <code>TimestampDescending</code> returns the newest
     *         data first and paginates when the <code>MaxDatapoints</code> limit is reached.
     *         <code>TimestampAscending</code> returns the oldest data first and paginates when the
     *         <code>MaxDatapoints</code> limit is reached.
     * @see ScanBy
     */
    public String scanByAsString() {
        return scanBy;
    }

    /**
     * <p>
     * The maximum number of data points the request should return before paginating. If you omit this, the default of
     * 100,800 is used.
     * </p>
     * 
     * @return The maximum number of data points the request should return before paginating. If you omit this, the
     *         default of 100,800 is used.
     */
    public Integer maxDatapoints() {
        return maxDatapoints;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(metricDataQueries());
        hashCode = 31 * hashCode + Objects.hashCode(startTime());
        hashCode = 31 * hashCode + Objects.hashCode(endTime());
        hashCode = 31 * hashCode + Objects.hashCode(nextToken());
        hashCode = 31 * hashCode + Objects.hashCode(scanByAsString());
        hashCode = 31 * hashCode + Objects.hashCode(maxDatapoints());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof GetMetricDataRequest)) {
            return false;
        }
        GetMetricDataRequest other = (GetMetricDataRequest) obj;
        return Objects.equals(metricDataQueries(), other.metricDataQueries()) && Objects.equals(startTime(), other.startTime())
                && Objects.equals(endTime(), other.endTime()) && Objects.equals(nextToken(), other.nextToken())
                && Objects.equals(scanByAsString(), other.scanByAsString())
                && Objects.equals(maxDatapoints(), other.maxDatapoints());
    }

    @Override
    public String toString() {
        return ToString.builder("GetMetricDataRequest").add("MetricDataQueries", metricDataQueries())
                .add("StartTime", startTime()).add("EndTime", endTime()).add("NextToken", nextToken())
                .add("ScanBy", scanByAsString()).add("MaxDatapoints", maxDatapoints()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "MetricDataQueries":
            return Optional.ofNullable(clazz.cast(metricDataQueries()));
        case "StartTime":
            return Optional.ofNullable(clazz.cast(startTime()));
        case "EndTime":
            return Optional.ofNullable(clazz.cast(endTime()));
        case "NextToken":
            return Optional.ofNullable(clazz.cast(nextToken()));
        case "ScanBy":
            return Optional.ofNullable(clazz.cast(scanByAsString()));
        case "MaxDatapoints":
            return Optional.ofNullable(clazz.cast(maxDatapoints()));
        default:
            return Optional.empty();
        }
    }

    public interface Builder extends CloudWatchRequest.Builder, CopyableBuilder<Builder, GetMetricDataRequest> {
        /**
         * <p>
         * The metric queries to be returned. A single <code>GetMetricData</code> call can include as many as 100
         * <code>MetricDataQuery</code> structures. Each of these structures can specify either a metric to retrieve, or
         * a math expression to perform on retrieved data.
         * </p>
         * 
         * @param metricDataQueries
         *        The metric queries to be returned. A single <code>GetMetricData</code> call can include as many as 100
         *        <code>MetricDataQuery</code> structures. Each of these structures can specify either a metric to
         *        retrieve, or a math expression to perform on retrieved data.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder metricDataQueries(Collection<MetricDataQuery> metricDataQueries);

        /**
         * <p>
         * The metric queries to be returned. A single <code>GetMetricData</code> call can include as many as 100
         * <code>MetricDataQuery</code> structures. Each of these structures can specify either a metric to retrieve, or
         * a math expression to perform on retrieved data.
         * </p>
         * 
         * @param metricDataQueries
         *        The metric queries to be returned. A single <code>GetMetricData</code> call can include as many as 100
         *        <code>MetricDataQuery</code> structures. Each of these structures can specify either a metric to
         *        retrieve, or a math expression to perform on retrieved data.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder metricDataQueries(MetricDataQuery... metricDataQueries);

        /**
         * <p>
         * The metric queries to be returned. A single <code>GetMetricData</code> call can include as many as 100
         * <code>MetricDataQuery</code> structures. Each of these structures can specify either a metric to retrieve, or
         * a math expression to perform on retrieved data.
         * </p>
         * This is a convenience that creates an instance of the {@link List<MetricDataQuery>.Builder} avoiding the need
         * to create one manually via {@link List<MetricDataQuery>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<MetricDataQuery>.Builder#build()} is called immediately and
         * its result is passed to {@link #metricDataQueries(List<MetricDataQuery>)}.
         * 
         * @param metricDataQueries
         *        a consumer that will call methods on {@link List<MetricDataQuery>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #metricDataQueries(List<MetricDataQuery>)
         */
        Builder metricDataQueries(Consumer<MetricDataQuery.Builder>... metricDataQueries);

        /**
         * <p>
         * The time stamp indicating the earliest data to be returned.
         * </p>
         * 
         * @param startTime
         *        The time stamp indicating the earliest data to be returned.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder startTime(Instant startTime);

        /**
         * <p>
         * The time stamp indicating the latest data to be returned.
         * </p>
         * 
         * @param endTime
         *        The time stamp indicating the latest data to be returned.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder endTime(Instant endTime);

        /**
         * <p>
         * Include this value, if it was returned by the previous call, to get the next set of data points.
         * </p>
         * 
         * @param nextToken
         *        Include this value, if it was returned by the previous call, to get the next set of data points.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder nextToken(String nextToken);

        /**
         * <p>
         * The order in which data points should be returned. <code>TimestampDescending</code> returns the newest data
         * first and paginates when the <code>MaxDatapoints</code> limit is reached. <code>TimestampAscending</code>
         * returns the oldest data first and paginates when the <code>MaxDatapoints</code> limit is reached.
         * </p>
         * 
         * @param scanBy
         *        The order in which data points should be returned. <code>TimestampDescending</code> returns the newest
         *        data first and paginates when the <code>MaxDatapoints</code> limit is reached.
         *        <code>TimestampAscending</code> returns the oldest data first and paginates when the
         *        <code>MaxDatapoints</code> limit is reached.
         * @see ScanBy
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ScanBy
         */
        Builder scanBy(String scanBy);

        /**
         * <p>
         * The order in which data points should be returned. <code>TimestampDescending</code> returns the newest data
         * first and paginates when the <code>MaxDatapoints</code> limit is reached. <code>TimestampAscending</code>
         * returns the oldest data first and paginates when the <code>MaxDatapoints</code> limit is reached.
         * </p>
         * 
         * @param scanBy
         *        The order in which data points should be returned. <code>TimestampDescending</code> returns the newest
         *        data first and paginates when the <code>MaxDatapoints</code> limit is reached.
         *        <code>TimestampAscending</code> returns the oldest data first and paginates when the
         *        <code>MaxDatapoints</code> limit is reached.
         * @see ScanBy
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ScanBy
         */
        Builder scanBy(ScanBy scanBy);

        /**
         * <p>
         * The maximum number of data points the request should return before paginating. If you omit this, the default
         * of 100,800 is used.
         * </p>
         * 
         * @param maxDatapoints
         *        The maximum number of data points the request should return before paginating. If you omit this, the
         *        default of 100,800 is used.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder maxDatapoints(Integer maxDatapoints);

        @Override
        Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration);

        @Override
        Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer);
    }

    static final class BuilderImpl extends CloudWatchRequest.BuilderImpl implements Builder {
        private List<MetricDataQuery> metricDataQueries = DefaultSdkAutoConstructList.getInstance();

        private Instant startTime;

        private Instant endTime;

        private String nextToken;

        private String scanBy;

        private Integer maxDatapoints;

        private BuilderImpl() {
        }

        private BuilderImpl(GetMetricDataRequest model) {
            super(model);
            metricDataQueries(model.metricDataQueries);
            startTime(model.startTime);
            endTime(model.endTime);
            nextToken(model.nextToken);
            scanBy(model.scanBy);
            maxDatapoints(model.maxDatapoints);
        }

        public final Collection<MetricDataQuery.Builder> getMetricDataQueries() {
            return metricDataQueries != null ? metricDataQueries.stream().map(MetricDataQuery::toBuilder)
                    .collect(Collectors.toList()) : null;
        }

        @Override
        public final Builder metricDataQueries(Collection<MetricDataQuery> metricDataQueries) {
            this.metricDataQueries = MetricDataQueriesCopier.copy(metricDataQueries);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder metricDataQueries(MetricDataQuery... metricDataQueries) {
            metricDataQueries(Arrays.asList(metricDataQueries));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder metricDataQueries(Consumer<MetricDataQuery.Builder>... metricDataQueries) {
            metricDataQueries(Stream.of(metricDataQueries).map(c -> MetricDataQuery.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final void setMetricDataQueries(Collection<MetricDataQuery.BuilderImpl> metricDataQueries) {
            this.metricDataQueries = MetricDataQueriesCopier.copyFromBuilder(metricDataQueries);
        }

        public final Instant getStartTime() {
            return startTime;
        }

        @Override
        public final Builder startTime(Instant startTime) {
            this.startTime = startTime;
            return this;
        }

        public final void setStartTime(Instant startTime) {
            this.startTime = startTime;
        }

        public final Instant getEndTime() {
            return endTime;
        }

        @Override
        public final Builder endTime(Instant endTime) {
            this.endTime = endTime;
            return this;
        }

        public final void setEndTime(Instant endTime) {
            this.endTime = endTime;
        }

        public final String getNextToken() {
            return nextToken;
        }

        @Override
        public final Builder nextToken(String nextToken) {
            this.nextToken = nextToken;
            return this;
        }

        public final void setNextToken(String nextToken) {
            this.nextToken = nextToken;
        }

        public final String getScanBy() {
            return scanBy;
        }

        @Override
        public final Builder scanBy(String scanBy) {
            this.scanBy = scanBy;
            return this;
        }

        @Override
        public final Builder scanBy(ScanBy scanBy) {
            this.scanBy(scanBy.toString());
            return this;
        }

        public final void setScanBy(String scanBy) {
            this.scanBy = scanBy;
        }

        public final Integer getMaxDatapoints() {
            return maxDatapoints;
        }

        @Override
        public final Builder maxDatapoints(Integer maxDatapoints) {
            this.maxDatapoints = maxDatapoints;
            return this;
        }

        public final void setMaxDatapoints(Integer maxDatapoints) {
            this.maxDatapoints = maxDatapoints;
        }

        @Override
        public Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration) {
            super.overrideConfiguration(overrideConfiguration);
            return this;
        }

        @Override
        public Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer) {
            super.overrideConfiguration(builderConsumer);
            return this;
        }

        @Override
        public GetMetricDataRequest build() {
            return new GetMetricDataRequest(this);
        }
    }
}
