/*
 * Copyright 2013-2018 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.cloudwatch.model;

import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.Consumer;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import javax.annotation.Generated;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Represents an alarm.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public class MetricAlarm implements ToCopyableBuilder<MetricAlarm.Builder, MetricAlarm> {
    private final String alarmName;

    private final String alarmArn;

    private final String alarmDescription;

    private final Instant alarmConfigurationUpdatedTimestamp;

    private final Boolean actionsEnabled;

    private final List<String> okActions;

    private final List<String> alarmActions;

    private final List<String> insufficientDataActions;

    private final String stateValue;

    private final String stateReason;

    private final String stateReasonData;

    private final Instant stateUpdatedTimestamp;

    private final String metricName;

    private final String namespace;

    private final String statistic;

    private final String extendedStatistic;

    private final List<Dimension> dimensions;

    private final Integer period;

    private final String unit;

    private final Integer evaluationPeriods;

    private final Double threshold;

    private final String comparisonOperator;

    private final String treatMissingData;

    private final String evaluateLowSampleCountPercentile;

    private MetricAlarm(BuilderImpl builder) {
        this.alarmName = builder.alarmName;
        this.alarmArn = builder.alarmArn;
        this.alarmDescription = builder.alarmDescription;
        this.alarmConfigurationUpdatedTimestamp = builder.alarmConfigurationUpdatedTimestamp;
        this.actionsEnabled = builder.actionsEnabled;
        this.okActions = builder.okActions;
        this.alarmActions = builder.alarmActions;
        this.insufficientDataActions = builder.insufficientDataActions;
        this.stateValue = builder.stateValue;
        this.stateReason = builder.stateReason;
        this.stateReasonData = builder.stateReasonData;
        this.stateUpdatedTimestamp = builder.stateUpdatedTimestamp;
        this.metricName = builder.metricName;
        this.namespace = builder.namespace;
        this.statistic = builder.statistic;
        this.extendedStatistic = builder.extendedStatistic;
        this.dimensions = builder.dimensions;
        this.period = builder.period;
        this.unit = builder.unit;
        this.evaluationPeriods = builder.evaluationPeriods;
        this.threshold = builder.threshold;
        this.comparisonOperator = builder.comparisonOperator;
        this.treatMissingData = builder.treatMissingData;
        this.evaluateLowSampleCountPercentile = builder.evaluateLowSampleCountPercentile;
    }

    /**
     * <p>
     * The name of the alarm.
     * </p>
     * 
     * @return The name of the alarm.
     */
    public String alarmName() {
        return alarmName;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) of the alarm.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) of the alarm.
     */
    public String alarmArn() {
        return alarmArn;
    }

    /**
     * <p>
     * The description of the alarm.
     * </p>
     * 
     * @return The description of the alarm.
     */
    public String alarmDescription() {
        return alarmDescription;
    }

    /**
     * <p>
     * The time stamp of the last update to the alarm configuration.
     * </p>
     * 
     * @return The time stamp of the last update to the alarm configuration.
     */
    public Instant alarmConfigurationUpdatedTimestamp() {
        return alarmConfigurationUpdatedTimestamp;
    }

    /**
     * <p>
     * Indicates whether actions should be executed during any changes to the alarm state.
     * </p>
     * 
     * @return Indicates whether actions should be executed during any changes to the alarm state.
     */
    public Boolean actionsEnabled() {
        return actionsEnabled;
    }

    /**
     * <p>
     * The actions to execute when this alarm transitions to the <code>OK</code> state from any other state. Each action
     * is specified as an Amazon Resource Name (ARN).
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return The actions to execute when this alarm transitions to the <code>OK</code> state from any other state.
     *         Each action is specified as an Amazon Resource Name (ARN).
     */
    public List<String> okActions() {
        return okActions;
    }

    /**
     * <p>
     * The actions to execute when this alarm transitions to the <code>ALARM</code> state from any other state. Each
     * action is specified as an Amazon Resource Name (ARN).
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return The actions to execute when this alarm transitions to the <code>ALARM</code> state from any other state.
     *         Each action is specified as an Amazon Resource Name (ARN).
     */
    public List<String> alarmActions() {
        return alarmActions;
    }

    /**
     * <p>
     * The actions to execute when this alarm transitions to the <code>INSUFFICIENT_DATA</code> state from any other
     * state. Each action is specified as an Amazon Resource Name (ARN).
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return The actions to execute when this alarm transitions to the <code>INSUFFICIENT_DATA</code> state from any
     *         other state. Each action is specified as an Amazon Resource Name (ARN).
     */
    public List<String> insufficientDataActions() {
        return insufficientDataActions;
    }

    /**
     * <p>
     * The state value for the alarm.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #stateValue} will
     * return {@link StateValue#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #stateValueAsString}.
     * </p>
     * 
     * @return The state value for the alarm.
     * @see StateValue
     */
    public StateValue stateValue() {
        return StateValue.fromValue(stateValue);
    }

    /**
     * <p>
     * The state value for the alarm.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #stateValue} will
     * return {@link StateValue#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #stateValueAsString}.
     * </p>
     * 
     * @return The state value for the alarm.
     * @see StateValue
     */
    public String stateValueAsString() {
        return stateValue;
    }

    /**
     * <p>
     * An explanation for the alarm state, in text format.
     * </p>
     * 
     * @return An explanation for the alarm state, in text format.
     */
    public String stateReason() {
        return stateReason;
    }

    /**
     * <p>
     * An explanation for the alarm state, in JSON format.
     * </p>
     * 
     * @return An explanation for the alarm state, in JSON format.
     */
    public String stateReasonData() {
        return stateReasonData;
    }

    /**
     * <p>
     * The time stamp of the last update to the alarm state.
     * </p>
     * 
     * @return The time stamp of the last update to the alarm state.
     */
    public Instant stateUpdatedTimestamp() {
        return stateUpdatedTimestamp;
    }

    /**
     * <p>
     * The name of the metric associated with the alarm.
     * </p>
     * 
     * @return The name of the metric associated with the alarm.
     */
    public String metricName() {
        return metricName;
    }

    /**
     * <p>
     * The namespace of the metric associated with the alarm.
     * </p>
     * 
     * @return The namespace of the metric associated with the alarm.
     */
    public String namespace() {
        return namespace;
    }

    /**
     * <p>
     * The statistic for the metric associated with the alarm, other than percentile. For percentile statistics, use
     * <code>ExtendedStatistic</code>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #statistic} will
     * return {@link Statistic#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #statisticAsString}.
     * </p>
     * 
     * @return The statistic for the metric associated with the alarm, other than percentile. For percentile statistics,
     *         use <code>ExtendedStatistic</code>.
     * @see Statistic
     */
    public Statistic statistic() {
        return Statistic.fromValue(statistic);
    }

    /**
     * <p>
     * The statistic for the metric associated with the alarm, other than percentile. For percentile statistics, use
     * <code>ExtendedStatistic</code>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #statistic} will
     * return {@link Statistic#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #statisticAsString}.
     * </p>
     * 
     * @return The statistic for the metric associated with the alarm, other than percentile. For percentile statistics,
     *         use <code>ExtendedStatistic</code>.
     * @see Statistic
     */
    public String statisticAsString() {
        return statistic;
    }

    /**
     * <p>
     * The percentile statistic for the metric associated with the alarm. Specify a value between p0.0 and p100.
     * </p>
     * 
     * @return The percentile statistic for the metric associated with the alarm. Specify a value between p0.0 and p100.
     */
    public String extendedStatistic() {
        return extendedStatistic;
    }

    /**
     * <p>
     * The dimensions for the metric associated with the alarm.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return The dimensions for the metric associated with the alarm.
     */
    public List<Dimension> dimensions() {
        return dimensions;
    }

    /**
     * <p>
     * The period, in seconds, over which the statistic is applied.
     * </p>
     * 
     * @return The period, in seconds, over which the statistic is applied.
     */
    public Integer period() {
        return period;
    }

    /**
     * <p>
     * The unit of the metric associated with the alarm.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #unit} will return
     * {@link StandardUnit#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #unitAsString}.
     * </p>
     * 
     * @return The unit of the metric associated with the alarm.
     * @see StandardUnit
     */
    public StandardUnit unit() {
        return StandardUnit.fromValue(unit);
    }

    /**
     * <p>
     * The unit of the metric associated with the alarm.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #unit} will return
     * {@link StandardUnit#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #unitAsString}.
     * </p>
     * 
     * @return The unit of the metric associated with the alarm.
     * @see StandardUnit
     */
    public String unitAsString() {
        return unit;
    }

    /**
     * <p>
     * The number of periods over which data is compared to the specified threshold.
     * </p>
     * 
     * @return The number of periods over which data is compared to the specified threshold.
     */
    public Integer evaluationPeriods() {
        return evaluationPeriods;
    }

    /**
     * <p>
     * The value to compare with the specified statistic.
     * </p>
     * 
     * @return The value to compare with the specified statistic.
     */
    public Double threshold() {
        return threshold;
    }

    /**
     * <p>
     * The arithmetic operation to use when comparing the specified statistic and threshold. The specified statistic
     * value is used as the first operand.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #comparisonOperator} will return {@link ComparisonOperator#UNKNOWN_TO_SDK_VERSION}. The raw value returned
     * by the service is available from {@link #comparisonOperatorAsString}.
     * </p>
     * 
     * @return The arithmetic operation to use when comparing the specified statistic and threshold. The specified
     *         statistic value is used as the first operand.
     * @see ComparisonOperator
     */
    public ComparisonOperator comparisonOperator() {
        return ComparisonOperator.fromValue(comparisonOperator);
    }

    /**
     * <p>
     * The arithmetic operation to use when comparing the specified statistic and threshold. The specified statistic
     * value is used as the first operand.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #comparisonOperator} will return {@link ComparisonOperator#UNKNOWN_TO_SDK_VERSION}. The raw value returned
     * by the service is available from {@link #comparisonOperatorAsString}.
     * </p>
     * 
     * @return The arithmetic operation to use when comparing the specified statistic and threshold. The specified
     *         statistic value is used as the first operand.
     * @see ComparisonOperator
     */
    public String comparisonOperatorAsString() {
        return comparisonOperator;
    }

    /**
     * <p>
     * Sets how this alarm is to handle missing data points. If this parameter is omitted, the default behavior of
     * <code>missing</code> is used.
     * </p>
     * 
     * @return Sets how this alarm is to handle missing data points. If this parameter is omitted, the default behavior
     *         of <code>missing</code> is used.
     */
    public String treatMissingData() {
        return treatMissingData;
    }

    /**
     * <p>
     * Used only for alarms based on percentiles. If <code>ignore</code>, the alarm state does not change during periods
     * with too few data points to be statistically significant. If <code>evaluate</code> or this parameter is not used,
     * the alarm is always evaluated and possibly changes state no matter how many data points are available.
     * </p>
     * 
     * @return Used only for alarms based on percentiles. If <code>ignore</code>, the alarm state does not change during
     *         periods with too few data points to be statistically significant. If <code>evaluate</code> or this
     *         parameter is not used, the alarm is always evaluated and possibly changes state no matter how many data
     *         points are available.
     */
    public String evaluateLowSampleCountPercentile() {
        return evaluateLowSampleCountPercentile;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(alarmName());
        hashCode = 31 * hashCode + Objects.hashCode(alarmArn());
        hashCode = 31 * hashCode + Objects.hashCode(alarmDescription());
        hashCode = 31 * hashCode + Objects.hashCode(alarmConfigurationUpdatedTimestamp());
        hashCode = 31 * hashCode + Objects.hashCode(actionsEnabled());
        hashCode = 31 * hashCode + Objects.hashCode(okActions());
        hashCode = 31 * hashCode + Objects.hashCode(alarmActions());
        hashCode = 31 * hashCode + Objects.hashCode(insufficientDataActions());
        hashCode = 31 * hashCode + Objects.hashCode(stateValueAsString());
        hashCode = 31 * hashCode + Objects.hashCode(stateReason());
        hashCode = 31 * hashCode + Objects.hashCode(stateReasonData());
        hashCode = 31 * hashCode + Objects.hashCode(stateUpdatedTimestamp());
        hashCode = 31 * hashCode + Objects.hashCode(metricName());
        hashCode = 31 * hashCode + Objects.hashCode(namespace());
        hashCode = 31 * hashCode + Objects.hashCode(statisticAsString());
        hashCode = 31 * hashCode + Objects.hashCode(extendedStatistic());
        hashCode = 31 * hashCode + Objects.hashCode(dimensions());
        hashCode = 31 * hashCode + Objects.hashCode(period());
        hashCode = 31 * hashCode + Objects.hashCode(unitAsString());
        hashCode = 31 * hashCode + Objects.hashCode(evaluationPeriods());
        hashCode = 31 * hashCode + Objects.hashCode(threshold());
        hashCode = 31 * hashCode + Objects.hashCode(comparisonOperatorAsString());
        hashCode = 31 * hashCode + Objects.hashCode(treatMissingData());
        hashCode = 31 * hashCode + Objects.hashCode(evaluateLowSampleCountPercentile());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof MetricAlarm)) {
            return false;
        }
        MetricAlarm other = (MetricAlarm) obj;
        return Objects.equals(alarmName(), other.alarmName()) && Objects.equals(alarmArn(), other.alarmArn())
                && Objects.equals(alarmDescription(), other.alarmDescription())
                && Objects.equals(alarmConfigurationUpdatedTimestamp(), other.alarmConfigurationUpdatedTimestamp())
                && Objects.equals(actionsEnabled(), other.actionsEnabled()) && Objects.equals(okActions(), other.okActions())
                && Objects.equals(alarmActions(), other.alarmActions())
                && Objects.equals(insufficientDataActions(), other.insufficientDataActions())
                && Objects.equals(stateValueAsString(), other.stateValueAsString())
                && Objects.equals(stateReason(), other.stateReason())
                && Objects.equals(stateReasonData(), other.stateReasonData())
                && Objects.equals(stateUpdatedTimestamp(), other.stateUpdatedTimestamp())
                && Objects.equals(metricName(), other.metricName()) && Objects.equals(namespace(), other.namespace())
                && Objects.equals(statisticAsString(), other.statisticAsString())
                && Objects.equals(extendedStatistic(), other.extendedStatistic())
                && Objects.equals(dimensions(), other.dimensions()) && Objects.equals(period(), other.period())
                && Objects.equals(unitAsString(), other.unitAsString())
                && Objects.equals(evaluationPeriods(), other.evaluationPeriods())
                && Objects.equals(threshold(), other.threshold())
                && Objects.equals(comparisonOperatorAsString(), other.comparisonOperatorAsString())
                && Objects.equals(treatMissingData(), other.treatMissingData())
                && Objects.equals(evaluateLowSampleCountPercentile(), other.evaluateLowSampleCountPercentile());
    }

    @Override
    public String toString() {
        return ToString.builder("MetricAlarm").add("AlarmName", alarmName()).add("AlarmArn", alarmArn())
                .add("AlarmDescription", alarmDescription())
                .add("AlarmConfigurationUpdatedTimestamp", alarmConfigurationUpdatedTimestamp())
                .add("ActionsEnabled", actionsEnabled()).add("OKActions", okActions()).add("AlarmActions", alarmActions())
                .add("InsufficientDataActions", insufficientDataActions()).add("StateValue", stateValueAsString())
                .add("StateReason", stateReason()).add("StateReasonData", stateReasonData())
                .add("StateUpdatedTimestamp", stateUpdatedTimestamp()).add("MetricName", metricName())
                .add("Namespace", namespace()).add("Statistic", statisticAsString())
                .add("ExtendedStatistic", extendedStatistic()).add("Dimensions", dimensions()).add("Period", period())
                .add("Unit", unitAsString()).add("EvaluationPeriods", evaluationPeriods()).add("Threshold", threshold())
                .add("ComparisonOperator", comparisonOperatorAsString()).add("TreatMissingData", treatMissingData())
                .add("EvaluateLowSampleCountPercentile", evaluateLowSampleCountPercentile()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "AlarmName":
            return Optional.of(clazz.cast(alarmName()));
        case "AlarmArn":
            return Optional.of(clazz.cast(alarmArn()));
        case "AlarmDescription":
            return Optional.of(clazz.cast(alarmDescription()));
        case "AlarmConfigurationUpdatedTimestamp":
            return Optional.of(clazz.cast(alarmConfigurationUpdatedTimestamp()));
        case "ActionsEnabled":
            return Optional.of(clazz.cast(actionsEnabled()));
        case "OKActions":
            return Optional.of(clazz.cast(okActions()));
        case "AlarmActions":
            return Optional.of(clazz.cast(alarmActions()));
        case "InsufficientDataActions":
            return Optional.of(clazz.cast(insufficientDataActions()));
        case "StateValue":
            return Optional.of(clazz.cast(stateValueAsString()));
        case "StateReason":
            return Optional.of(clazz.cast(stateReason()));
        case "StateReasonData":
            return Optional.of(clazz.cast(stateReasonData()));
        case "StateUpdatedTimestamp":
            return Optional.of(clazz.cast(stateUpdatedTimestamp()));
        case "MetricName":
            return Optional.of(clazz.cast(metricName()));
        case "Namespace":
            return Optional.of(clazz.cast(namespace()));
        case "Statistic":
            return Optional.of(clazz.cast(statisticAsString()));
        case "ExtendedStatistic":
            return Optional.of(clazz.cast(extendedStatistic()));
        case "Dimensions":
            return Optional.of(clazz.cast(dimensions()));
        case "Period":
            return Optional.of(clazz.cast(period()));
        case "Unit":
            return Optional.of(clazz.cast(unitAsString()));
        case "EvaluationPeriods":
            return Optional.of(clazz.cast(evaluationPeriods()));
        case "Threshold":
            return Optional.of(clazz.cast(threshold()));
        case "ComparisonOperator":
            return Optional.of(clazz.cast(comparisonOperatorAsString()));
        case "TreatMissingData":
            return Optional.of(clazz.cast(treatMissingData()));
        case "EvaluateLowSampleCountPercentile":
            return Optional.of(clazz.cast(evaluateLowSampleCountPercentile()));
        default:
            return Optional.empty();
        }
    }

    public interface Builder extends CopyableBuilder<Builder, MetricAlarm> {
        /**
         * <p>
         * The name of the alarm.
         * </p>
         * 
         * @param alarmName
         *        The name of the alarm.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder alarmName(String alarmName);

        /**
         * <p>
         * The Amazon Resource Name (ARN) of the alarm.
         * </p>
         * 
         * @param alarmArn
         *        The Amazon Resource Name (ARN) of the alarm.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder alarmArn(String alarmArn);

        /**
         * <p>
         * The description of the alarm.
         * </p>
         * 
         * @param alarmDescription
         *        The description of the alarm.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder alarmDescription(String alarmDescription);

        /**
         * <p>
         * The time stamp of the last update to the alarm configuration.
         * </p>
         * 
         * @param alarmConfigurationUpdatedTimestamp
         *        The time stamp of the last update to the alarm configuration.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder alarmConfigurationUpdatedTimestamp(Instant alarmConfigurationUpdatedTimestamp);

        /**
         * <p>
         * Indicates whether actions should be executed during any changes to the alarm state.
         * </p>
         * 
         * @param actionsEnabled
         *        Indicates whether actions should be executed during any changes to the alarm state.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder actionsEnabled(Boolean actionsEnabled);

        /**
         * <p>
         * The actions to execute when this alarm transitions to the <code>OK</code> state from any other state. Each
         * action is specified as an Amazon Resource Name (ARN).
         * </p>
         * 
         * @param okActions
         *        The actions to execute when this alarm transitions to the <code>OK</code> state from any other state.
         *        Each action is specified as an Amazon Resource Name (ARN).
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder okActions(Collection<String> okActions);

        /**
         * <p>
         * The actions to execute when this alarm transitions to the <code>OK</code> state from any other state. Each
         * action is specified as an Amazon Resource Name (ARN).
         * </p>
         * 
         * @param okActions
         *        The actions to execute when this alarm transitions to the <code>OK</code> state from any other state.
         *        Each action is specified as an Amazon Resource Name (ARN).
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder okActions(String... okActions);

        /**
         * <p>
         * The actions to execute when this alarm transitions to the <code>ALARM</code> state from any other state. Each
         * action is specified as an Amazon Resource Name (ARN).
         * </p>
         * 
         * @param alarmActions
         *        The actions to execute when this alarm transitions to the <code>ALARM</code> state from any other
         *        state. Each action is specified as an Amazon Resource Name (ARN).
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder alarmActions(Collection<String> alarmActions);

        /**
         * <p>
         * The actions to execute when this alarm transitions to the <code>ALARM</code> state from any other state. Each
         * action is specified as an Amazon Resource Name (ARN).
         * </p>
         * 
         * @param alarmActions
         *        The actions to execute when this alarm transitions to the <code>ALARM</code> state from any other
         *        state. Each action is specified as an Amazon Resource Name (ARN).
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder alarmActions(String... alarmActions);

        /**
         * <p>
         * The actions to execute when this alarm transitions to the <code>INSUFFICIENT_DATA</code> state from any other
         * state. Each action is specified as an Amazon Resource Name (ARN).
         * </p>
         * 
         * @param insufficientDataActions
         *        The actions to execute when this alarm transitions to the <code>INSUFFICIENT_DATA</code> state from
         *        any other state. Each action is specified as an Amazon Resource Name (ARN).
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder insufficientDataActions(Collection<String> insufficientDataActions);

        /**
         * <p>
         * The actions to execute when this alarm transitions to the <code>INSUFFICIENT_DATA</code> state from any other
         * state. Each action is specified as an Amazon Resource Name (ARN).
         * </p>
         * 
         * @param insufficientDataActions
         *        The actions to execute when this alarm transitions to the <code>INSUFFICIENT_DATA</code> state from
         *        any other state. Each action is specified as an Amazon Resource Name (ARN).
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder insufficientDataActions(String... insufficientDataActions);

        /**
         * <p>
         * The state value for the alarm.
         * </p>
         * 
         * @param stateValue
         *        The state value for the alarm.
         * @see StateValue
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see StateValue
         */
        Builder stateValue(String stateValue);

        /**
         * <p>
         * The state value for the alarm.
         * </p>
         * 
         * @param stateValue
         *        The state value for the alarm.
         * @see StateValue
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see StateValue
         */
        Builder stateValue(StateValue stateValue);

        /**
         * <p>
         * An explanation for the alarm state, in text format.
         * </p>
         * 
         * @param stateReason
         *        An explanation for the alarm state, in text format.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder stateReason(String stateReason);

        /**
         * <p>
         * An explanation for the alarm state, in JSON format.
         * </p>
         * 
         * @param stateReasonData
         *        An explanation for the alarm state, in JSON format.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder stateReasonData(String stateReasonData);

        /**
         * <p>
         * The time stamp of the last update to the alarm state.
         * </p>
         * 
         * @param stateUpdatedTimestamp
         *        The time stamp of the last update to the alarm state.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder stateUpdatedTimestamp(Instant stateUpdatedTimestamp);

        /**
         * <p>
         * The name of the metric associated with the alarm.
         * </p>
         * 
         * @param metricName
         *        The name of the metric associated with the alarm.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder metricName(String metricName);

        /**
         * <p>
         * The namespace of the metric associated with the alarm.
         * </p>
         * 
         * @param namespace
         *        The namespace of the metric associated with the alarm.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder namespace(String namespace);

        /**
         * <p>
         * The statistic for the metric associated with the alarm, other than percentile. For percentile statistics, use
         * <code>ExtendedStatistic</code>.
         * </p>
         * 
         * @param statistic
         *        The statistic for the metric associated with the alarm, other than percentile. For percentile
         *        statistics, use <code>ExtendedStatistic</code>.
         * @see Statistic
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see Statistic
         */
        Builder statistic(String statistic);

        /**
         * <p>
         * The statistic for the metric associated with the alarm, other than percentile. For percentile statistics, use
         * <code>ExtendedStatistic</code>.
         * </p>
         * 
         * @param statistic
         *        The statistic for the metric associated with the alarm, other than percentile. For percentile
         *        statistics, use <code>ExtendedStatistic</code>.
         * @see Statistic
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see Statistic
         */
        Builder statistic(Statistic statistic);

        /**
         * <p>
         * The percentile statistic for the metric associated with the alarm. Specify a value between p0.0 and p100.
         * </p>
         * 
         * @param extendedStatistic
         *        The percentile statistic for the metric associated with the alarm. Specify a value between p0.0 and
         *        p100.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder extendedStatistic(String extendedStatistic);

        /**
         * <p>
         * The dimensions for the metric associated with the alarm.
         * </p>
         * 
         * @param dimensions
         *        The dimensions for the metric associated with the alarm.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder dimensions(Collection<Dimension> dimensions);

        /**
         * <p>
         * The dimensions for the metric associated with the alarm.
         * </p>
         * 
         * @param dimensions
         *        The dimensions for the metric associated with the alarm.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder dimensions(Dimension... dimensions);

        /**
         * <p>
         * The dimensions for the metric associated with the alarm.
         * </p>
         * This is a convenience that creates an instance of the {@link List<Dimension>.Builder} avoiding the need to
         * create one manually via {@link List<Dimension>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<Dimension>.Builder#build()} is called immediately and its
         * result is passed to {@link #dimensions(List<Dimension>)}.
         * 
         * @param dimensions
         *        a consumer that will call methods on {@link List<Dimension>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #dimensions(List<Dimension>)
         */
        Builder dimensions(Consumer<Dimension.Builder>... dimensions);

        /**
         * <p>
         * The period, in seconds, over which the statistic is applied.
         * </p>
         * 
         * @param period
         *        The period, in seconds, over which the statistic is applied.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder period(Integer period);

        /**
         * <p>
         * The unit of the metric associated with the alarm.
         * </p>
         * 
         * @param unit
         *        The unit of the metric associated with the alarm.
         * @see StandardUnit
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see StandardUnit
         */
        Builder unit(String unit);

        /**
         * <p>
         * The unit of the metric associated with the alarm.
         * </p>
         * 
         * @param unit
         *        The unit of the metric associated with the alarm.
         * @see StandardUnit
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see StandardUnit
         */
        Builder unit(StandardUnit unit);

        /**
         * <p>
         * The number of periods over which data is compared to the specified threshold.
         * </p>
         * 
         * @param evaluationPeriods
         *        The number of periods over which data is compared to the specified threshold.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder evaluationPeriods(Integer evaluationPeriods);

        /**
         * <p>
         * The value to compare with the specified statistic.
         * </p>
         * 
         * @param threshold
         *        The value to compare with the specified statistic.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder threshold(Double threshold);

        /**
         * <p>
         * The arithmetic operation to use when comparing the specified statistic and threshold. The specified statistic
         * value is used as the first operand.
         * </p>
         * 
         * @param comparisonOperator
         *        The arithmetic operation to use when comparing the specified statistic and threshold. The specified
         *        statistic value is used as the first operand.
         * @see ComparisonOperator
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ComparisonOperator
         */
        Builder comparisonOperator(String comparisonOperator);

        /**
         * <p>
         * The arithmetic operation to use when comparing the specified statistic and threshold. The specified statistic
         * value is used as the first operand.
         * </p>
         * 
         * @param comparisonOperator
         *        The arithmetic operation to use when comparing the specified statistic and threshold. The specified
         *        statistic value is used as the first operand.
         * @see ComparisonOperator
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ComparisonOperator
         */
        Builder comparisonOperator(ComparisonOperator comparisonOperator);

        /**
         * <p>
         * Sets how this alarm is to handle missing data points. If this parameter is omitted, the default behavior of
         * <code>missing</code> is used.
         * </p>
         * 
         * @param treatMissingData
         *        Sets how this alarm is to handle missing data points. If this parameter is omitted, the default
         *        behavior of <code>missing</code> is used.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder treatMissingData(String treatMissingData);

        /**
         * <p>
         * Used only for alarms based on percentiles. If <code>ignore</code>, the alarm state does not change during
         * periods with too few data points to be statistically significant. If <code>evaluate</code> or this parameter
         * is not used, the alarm is always evaluated and possibly changes state no matter how many data points are
         * available.
         * </p>
         * 
         * @param evaluateLowSampleCountPercentile
         *        Used only for alarms based on percentiles. If <code>ignore</code>, the alarm state does not change
         *        during periods with too few data points to be statistically significant. If <code>evaluate</code> or
         *        this parameter is not used, the alarm is always evaluated and possibly changes state no matter how
         *        many data points are available.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder evaluateLowSampleCountPercentile(String evaluateLowSampleCountPercentile);
    }

    static final class BuilderImpl implements Builder {
        private String alarmName;

        private String alarmArn;

        private String alarmDescription;

        private Instant alarmConfigurationUpdatedTimestamp;

        private Boolean actionsEnabled;

        private List<String> okActions;

        private List<String> alarmActions;

        private List<String> insufficientDataActions;

        private String stateValue;

        private String stateReason;

        private String stateReasonData;

        private Instant stateUpdatedTimestamp;

        private String metricName;

        private String namespace;

        private String statistic;

        private String extendedStatistic;

        private List<Dimension> dimensions;

        private Integer period;

        private String unit;

        private Integer evaluationPeriods;

        private Double threshold;

        private String comparisonOperator;

        private String treatMissingData;

        private String evaluateLowSampleCountPercentile;

        private BuilderImpl() {
        }

        private BuilderImpl(MetricAlarm model) {
            alarmName(model.alarmName);
            alarmArn(model.alarmArn);
            alarmDescription(model.alarmDescription);
            alarmConfigurationUpdatedTimestamp(model.alarmConfigurationUpdatedTimestamp);
            actionsEnabled(model.actionsEnabled);
            okActions(model.okActions);
            alarmActions(model.alarmActions);
            insufficientDataActions(model.insufficientDataActions);
            stateValue(model.stateValue);
            stateReason(model.stateReason);
            stateReasonData(model.stateReasonData);
            stateUpdatedTimestamp(model.stateUpdatedTimestamp);
            metricName(model.metricName);
            namespace(model.namespace);
            statistic(model.statistic);
            extendedStatistic(model.extendedStatistic);
            dimensions(model.dimensions);
            period(model.period);
            unit(model.unit);
            evaluationPeriods(model.evaluationPeriods);
            threshold(model.threshold);
            comparisonOperator(model.comparisonOperator);
            treatMissingData(model.treatMissingData);
            evaluateLowSampleCountPercentile(model.evaluateLowSampleCountPercentile);
        }

        public final String getAlarmName() {
            return alarmName;
        }

        @Override
        public final Builder alarmName(String alarmName) {
            this.alarmName = alarmName;
            return this;
        }

        public final void setAlarmName(String alarmName) {
            this.alarmName = alarmName;
        }

        public final String getAlarmArn() {
            return alarmArn;
        }

        @Override
        public final Builder alarmArn(String alarmArn) {
            this.alarmArn = alarmArn;
            return this;
        }

        public final void setAlarmArn(String alarmArn) {
            this.alarmArn = alarmArn;
        }

        public final String getAlarmDescription() {
            return alarmDescription;
        }

        @Override
        public final Builder alarmDescription(String alarmDescription) {
            this.alarmDescription = alarmDescription;
            return this;
        }

        public final void setAlarmDescription(String alarmDescription) {
            this.alarmDescription = alarmDescription;
        }

        public final Instant getAlarmConfigurationUpdatedTimestamp() {
            return alarmConfigurationUpdatedTimestamp;
        }

        @Override
        public final Builder alarmConfigurationUpdatedTimestamp(Instant alarmConfigurationUpdatedTimestamp) {
            this.alarmConfigurationUpdatedTimestamp = alarmConfigurationUpdatedTimestamp;
            return this;
        }

        public final void setAlarmConfigurationUpdatedTimestamp(Instant alarmConfigurationUpdatedTimestamp) {
            this.alarmConfigurationUpdatedTimestamp = alarmConfigurationUpdatedTimestamp;
        }

        public final Boolean getActionsEnabled() {
            return actionsEnabled;
        }

        @Override
        public final Builder actionsEnabled(Boolean actionsEnabled) {
            this.actionsEnabled = actionsEnabled;
            return this;
        }

        public final void setActionsEnabled(Boolean actionsEnabled) {
            this.actionsEnabled = actionsEnabled;
        }

        public final Collection<String> getOKActions() {
            return okActions;
        }

        @Override
        public final Builder okActions(Collection<String> okActions) {
            this.okActions = ResourceListCopier.copy(okActions);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder okActions(String... okActions) {
            okActions(Arrays.asList(okActions));
            return this;
        }

        public final void setOKActions(Collection<String> okActions) {
            this.okActions = ResourceListCopier.copy(okActions);
        }

        public final Collection<String> getAlarmActions() {
            return alarmActions;
        }

        @Override
        public final Builder alarmActions(Collection<String> alarmActions) {
            this.alarmActions = ResourceListCopier.copy(alarmActions);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder alarmActions(String... alarmActions) {
            alarmActions(Arrays.asList(alarmActions));
            return this;
        }

        public final void setAlarmActions(Collection<String> alarmActions) {
            this.alarmActions = ResourceListCopier.copy(alarmActions);
        }

        public final Collection<String> getInsufficientDataActions() {
            return insufficientDataActions;
        }

        @Override
        public final Builder insufficientDataActions(Collection<String> insufficientDataActions) {
            this.insufficientDataActions = ResourceListCopier.copy(insufficientDataActions);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder insufficientDataActions(String... insufficientDataActions) {
            insufficientDataActions(Arrays.asList(insufficientDataActions));
            return this;
        }

        public final void setInsufficientDataActions(Collection<String> insufficientDataActions) {
            this.insufficientDataActions = ResourceListCopier.copy(insufficientDataActions);
        }

        public final String getStateValue() {
            return stateValue;
        }

        @Override
        public final Builder stateValue(String stateValue) {
            this.stateValue = stateValue;
            return this;
        }

        @Override
        public final Builder stateValue(StateValue stateValue) {
            this.stateValue(stateValue.toString());
            return this;
        }

        public final void setStateValue(String stateValue) {
            this.stateValue = stateValue;
        }

        public final String getStateReason() {
            return stateReason;
        }

        @Override
        public final Builder stateReason(String stateReason) {
            this.stateReason = stateReason;
            return this;
        }

        public final void setStateReason(String stateReason) {
            this.stateReason = stateReason;
        }

        public final String getStateReasonData() {
            return stateReasonData;
        }

        @Override
        public final Builder stateReasonData(String stateReasonData) {
            this.stateReasonData = stateReasonData;
            return this;
        }

        public final void setStateReasonData(String stateReasonData) {
            this.stateReasonData = stateReasonData;
        }

        public final Instant getStateUpdatedTimestamp() {
            return stateUpdatedTimestamp;
        }

        @Override
        public final Builder stateUpdatedTimestamp(Instant stateUpdatedTimestamp) {
            this.stateUpdatedTimestamp = stateUpdatedTimestamp;
            return this;
        }

        public final void setStateUpdatedTimestamp(Instant stateUpdatedTimestamp) {
            this.stateUpdatedTimestamp = stateUpdatedTimestamp;
        }

        public final String getMetricName() {
            return metricName;
        }

        @Override
        public final Builder metricName(String metricName) {
            this.metricName = metricName;
            return this;
        }

        public final void setMetricName(String metricName) {
            this.metricName = metricName;
        }

        public final String getNamespace() {
            return namespace;
        }

        @Override
        public final Builder namespace(String namespace) {
            this.namespace = namespace;
            return this;
        }

        public final void setNamespace(String namespace) {
            this.namespace = namespace;
        }

        public final String getStatistic() {
            return statistic;
        }

        @Override
        public final Builder statistic(String statistic) {
            this.statistic = statistic;
            return this;
        }

        @Override
        public final Builder statistic(Statistic statistic) {
            this.statistic(statistic.toString());
            return this;
        }

        public final void setStatistic(String statistic) {
            this.statistic = statistic;
        }

        public final String getExtendedStatistic() {
            return extendedStatistic;
        }

        @Override
        public final Builder extendedStatistic(String extendedStatistic) {
            this.extendedStatistic = extendedStatistic;
            return this;
        }

        public final void setExtendedStatistic(String extendedStatistic) {
            this.extendedStatistic = extendedStatistic;
        }

        public final Collection<Dimension.Builder> getDimensions() {
            return dimensions != null ? dimensions.stream().map(Dimension::toBuilder).collect(Collectors.toList()) : null;
        }

        @Override
        public final Builder dimensions(Collection<Dimension> dimensions) {
            this.dimensions = DimensionsCopier.copy(dimensions);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder dimensions(Dimension... dimensions) {
            dimensions(Arrays.asList(dimensions));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder dimensions(Consumer<Dimension.Builder>... dimensions) {
            dimensions(Stream.of(dimensions).map(c -> Dimension.builder().apply(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final void setDimensions(Collection<Dimension.BuilderImpl> dimensions) {
            this.dimensions = DimensionsCopier.copyFromBuilder(dimensions);
        }

        public final Integer getPeriod() {
            return period;
        }

        @Override
        public final Builder period(Integer period) {
            this.period = period;
            return this;
        }

        public final void setPeriod(Integer period) {
            this.period = period;
        }

        public final String getUnit() {
            return unit;
        }

        @Override
        public final Builder unit(String unit) {
            this.unit = unit;
            return this;
        }

        @Override
        public final Builder unit(StandardUnit unit) {
            this.unit(unit.toString());
            return this;
        }

        public final void setUnit(String unit) {
            this.unit = unit;
        }

        public final Integer getEvaluationPeriods() {
            return evaluationPeriods;
        }

        @Override
        public final Builder evaluationPeriods(Integer evaluationPeriods) {
            this.evaluationPeriods = evaluationPeriods;
            return this;
        }

        public final void setEvaluationPeriods(Integer evaluationPeriods) {
            this.evaluationPeriods = evaluationPeriods;
        }

        public final Double getThreshold() {
            return threshold;
        }

        @Override
        public final Builder threshold(Double threshold) {
            this.threshold = threshold;
            return this;
        }

        public final void setThreshold(Double threshold) {
            this.threshold = threshold;
        }

        public final String getComparisonOperator() {
            return comparisonOperator;
        }

        @Override
        public final Builder comparisonOperator(String comparisonOperator) {
            this.comparisonOperator = comparisonOperator;
            return this;
        }

        @Override
        public final Builder comparisonOperator(ComparisonOperator comparisonOperator) {
            this.comparisonOperator(comparisonOperator.toString());
            return this;
        }

        public final void setComparisonOperator(String comparisonOperator) {
            this.comparisonOperator = comparisonOperator;
        }

        public final String getTreatMissingData() {
            return treatMissingData;
        }

        @Override
        public final Builder treatMissingData(String treatMissingData) {
            this.treatMissingData = treatMissingData;
            return this;
        }

        public final void setTreatMissingData(String treatMissingData) {
            this.treatMissingData = treatMissingData;
        }

        public final String getEvaluateLowSampleCountPercentile() {
            return evaluateLowSampleCountPercentile;
        }

        @Override
        public final Builder evaluateLowSampleCountPercentile(String evaluateLowSampleCountPercentile) {
            this.evaluateLowSampleCountPercentile = evaluateLowSampleCountPercentile;
            return this;
        }

        public final void setEvaluateLowSampleCountPercentile(String evaluateLowSampleCountPercentile) {
            this.evaluateLowSampleCountPercentile = evaluateLowSampleCountPercentile;
        }

        @Override
        public MetricAlarm build() {
            return new MetricAlarm(this);
        }
    }
}
