/*
 * Copyright 2014-2019 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.cloudfront.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * A complex type that controls whether CloudFront caches the response to requests using the specified HTTP methods.
 * There are two choices:
 * </p>
 * <ul>
 * <li>
 * <p>
 * CloudFront caches responses to <code>GET</code> and <code>HEAD</code> requests.
 * </p>
 * </li>
 * <li>
 * <p>
 * CloudFront caches responses to <code>GET</code>, <code>HEAD</code>, and <code>OPTIONS</code> requests.
 * </p>
 * </li>
 * </ul>
 * <p>
 * If you pick the second choice for your Amazon S3 Origin, you may need to forward Access-Control-Request-Method,
 * Access-Control-Request-Headers, and Origin headers for the responses to be cached correctly.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class CachedMethods implements SdkPojo, Serializable, ToCopyableBuilder<CachedMethods.Builder, CachedMethods> {
    private static final SdkField<Integer> QUANTITY_FIELD = SdkField
            .<Integer> builder(MarshallingType.INTEGER)
            .getter(getter(CachedMethods::quantity))
            .setter(setter(Builder::quantity))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Quantity")
                    .unmarshallLocationName("Quantity").build()).build();

    private static final SdkField<List<String>> ITEMS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .getter(getter(CachedMethods::itemsAsStrings))
            .setter(setter(Builder::itemsWithStrings))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Items")
                    .unmarshallLocationName("Items").build(),
                    ListTrait
                            .builder()
                            .memberLocationName("Method")
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("Method").unmarshallLocationName("Method").build()).build())
                            .build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(QUANTITY_FIELD, ITEMS_FIELD));

    private static final long serialVersionUID = 1L;

    private final Integer quantity;

    private final List<String> items;

    private CachedMethods(BuilderImpl builder) {
        this.quantity = builder.quantity;
        this.items = builder.items;
    }

    /**
     * <p>
     * The number of HTTP methods for which you want CloudFront to cache responses. Valid values are <code>2</code> (for
     * caching responses to <code>GET</code> and <code>HEAD</code> requests) and <code>3</code> (for caching responses
     * to <code>GET</code>, <code>HEAD</code>, and <code>OPTIONS</code> requests).
     * </p>
     * 
     * @return The number of HTTP methods for which you want CloudFront to cache responses. Valid values are
     *         <code>2</code> (for caching responses to <code>GET</code> and <code>HEAD</code> requests) and
     *         <code>3</code> (for caching responses to <code>GET</code>, <code>HEAD</code>, and <code>OPTIONS</code>
     *         requests).
     */
    public Integer quantity() {
        return quantity;
    }

    /**
     * <p>
     * A complex type that contains the HTTP methods that you want CloudFront to cache responses to.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return A complex type that contains the HTTP methods that you want CloudFront to cache responses to.
     */
    public List<Method> items() {
        return MethodsListCopier.copyStringToEnum(items);
    }

    /**
     * <p>
     * A complex type that contains the HTTP methods that you want CloudFront to cache responses to.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return A complex type that contains the HTTP methods that you want CloudFront to cache responses to.
     */
    public List<String> itemsAsStrings() {
        return items;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(quantity());
        hashCode = 31 * hashCode + Objects.hashCode(itemsAsStrings());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof CachedMethods)) {
            return false;
        }
        CachedMethods other = (CachedMethods) obj;
        return Objects.equals(quantity(), other.quantity()) && Objects.equals(itemsAsStrings(), other.itemsAsStrings());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("CachedMethods").add("Quantity", quantity()).add("Items", itemsAsStrings()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Quantity":
            return Optional.ofNullable(clazz.cast(quantity()));
        case "Items":
            return Optional.ofNullable(clazz.cast(itemsAsStrings()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<CachedMethods, T> g) {
        return obj -> g.apply((CachedMethods) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, CachedMethods> {
        /**
         * <p>
         * The number of HTTP methods for which you want CloudFront to cache responses. Valid values are <code>2</code>
         * (for caching responses to <code>GET</code> and <code>HEAD</code> requests) and <code>3</code> (for caching
         * responses to <code>GET</code>, <code>HEAD</code>, and <code>OPTIONS</code> requests).
         * </p>
         * 
         * @param quantity
         *        The number of HTTP methods for which you want CloudFront to cache responses. Valid values are
         *        <code>2</code> (for caching responses to <code>GET</code> and <code>HEAD</code> requests) and
         *        <code>3</code> (for caching responses to <code>GET</code>, <code>HEAD</code>, and <code>OPTIONS</code>
         *        requests).
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder quantity(Integer quantity);

        /**
         * <p>
         * A complex type that contains the HTTP methods that you want CloudFront to cache responses to.
         * </p>
         * 
         * @param items
         *        A complex type that contains the HTTP methods that you want CloudFront to cache responses to.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder itemsWithStrings(Collection<String> items);

        /**
         * <p>
         * A complex type that contains the HTTP methods that you want CloudFront to cache responses to.
         * </p>
         * 
         * @param items
         *        A complex type that contains the HTTP methods that you want CloudFront to cache responses to.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder itemsWithStrings(String... items);

        /**
         * <p>
         * A complex type that contains the HTTP methods that you want CloudFront to cache responses to.
         * </p>
         * 
         * @param items
         *        A complex type that contains the HTTP methods that you want CloudFront to cache responses to.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder items(Collection<Method> items);

        /**
         * <p>
         * A complex type that contains the HTTP methods that you want CloudFront to cache responses to.
         * </p>
         * 
         * @param items
         *        A complex type that contains the HTTP methods that you want CloudFront to cache responses to.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder items(Method... items);
    }

    static final class BuilderImpl implements Builder {
        private Integer quantity;

        private List<String> items = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(CachedMethods model) {
            quantity(model.quantity);
            itemsWithStrings(model.items);
        }

        public final Integer getQuantity() {
            return quantity;
        }

        @Override
        public final Builder quantity(Integer quantity) {
            this.quantity = quantity;
            return this;
        }

        public final void setQuantity(Integer quantity) {
            this.quantity = quantity;
        }

        public final Collection<String> getItemsAsStrings() {
            return items;
        }

        @Override
        public final Builder itemsWithStrings(Collection<String> items) {
            this.items = MethodsListCopier.copy(items);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder itemsWithStrings(String... items) {
            itemsWithStrings(Arrays.asList(items));
            return this;
        }

        @Override
        public final Builder items(Collection<Method> items) {
            this.items = MethodsListCopier.copyEnumToString(items);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder items(Method... items) {
            items(Arrays.asList(items));
            return this;
        }

        public final void setItemsWithStrings(Collection<String> items) {
            this.items = MethodsListCopier.copy(items);
        }

        @Override
        public CachedMethods build() {
            return new CachedMethods(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
