/*
 * Copyright 2014-2019 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.cloudfront.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * A complex type that describes the Amazon S3 bucket, HTTP server (for example, a web server), Amazon MediaStore, or
 * other server from which CloudFront gets your files. This can also be an origin group, if you've created an origin
 * group. You must specify at least one origin or origin group.
 * </p>
 * <p>
 * For the current limit on the number of origins or origin groups that you can specify for a distribution, see <a
 * href="https://docs.aws.amazon.com/general/latest/gr/aws_service_limits.html#limits_cloudfront">Amazon CloudFront
 * Limits</a> in the <i>AWS General Reference</i>.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class Origin implements SdkPojo, Serializable, ToCopyableBuilder<Origin.Builder, Origin> {
    private static final SdkField<String> ID_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .getter(getter(Origin::id))
            .setter(setter(Builder::id))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Id").unmarshallLocationName("Id")
                    .build()).build();

    private static final SdkField<String> DOMAIN_NAME_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .getter(getter(Origin::domainName))
            .setter(setter(Builder::domainName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DomainName")
                    .unmarshallLocationName("DomainName").build()).build();

    private static final SdkField<String> ORIGIN_PATH_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .getter(getter(Origin::originPath))
            .setter(setter(Builder::originPath))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("OriginPath")
                    .unmarshallLocationName("OriginPath").build()).build();

    private static final SdkField<CustomHeaders> CUSTOM_HEADERS_FIELD = SdkField
            .<CustomHeaders> builder(MarshallingType.SDK_POJO)
            .getter(getter(Origin::customHeaders))
            .setter(setter(Builder::customHeaders))
            .constructor(CustomHeaders::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CustomHeaders")
                    .unmarshallLocationName("CustomHeaders").build()).build();

    private static final SdkField<S3OriginConfig> S3_ORIGIN_CONFIG_FIELD = SdkField
            .<S3OriginConfig> builder(MarshallingType.SDK_POJO)
            .getter(getter(Origin::s3OriginConfig))
            .setter(setter(Builder::s3OriginConfig))
            .constructor(S3OriginConfig::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("S3OriginConfig")
                    .unmarshallLocationName("S3OriginConfig").build()).build();

    private static final SdkField<CustomOriginConfig> CUSTOM_ORIGIN_CONFIG_FIELD = SdkField
            .<CustomOriginConfig> builder(MarshallingType.SDK_POJO)
            .getter(getter(Origin::customOriginConfig))
            .setter(setter(Builder::customOriginConfig))
            .constructor(CustomOriginConfig::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CustomOriginConfig")
                    .unmarshallLocationName("CustomOriginConfig").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(ID_FIELD, DOMAIN_NAME_FIELD,
            ORIGIN_PATH_FIELD, CUSTOM_HEADERS_FIELD, S3_ORIGIN_CONFIG_FIELD, CUSTOM_ORIGIN_CONFIG_FIELD));

    private static final long serialVersionUID = 1L;

    private final String id;

    private final String domainName;

    private final String originPath;

    private final CustomHeaders customHeaders;

    private final S3OriginConfig s3OriginConfig;

    private final CustomOriginConfig customOriginConfig;

    private Origin(BuilderImpl builder) {
        this.id = builder.id;
        this.domainName = builder.domainName;
        this.originPath = builder.originPath;
        this.customHeaders = builder.customHeaders;
        this.s3OriginConfig = builder.s3OriginConfig;
        this.customOriginConfig = builder.customOriginConfig;
    }

    /**
     * <p>
     * A unique identifier for the origin or origin group. The value of <code>Id</code> must be unique within the
     * distribution.
     * </p>
     * <p>
     * When you specify the value of <code>TargetOriginId</code> for the default cache behavior or for another cache
     * behavior, you indicate the origin to which you want the cache behavior to route requests by specifying the value
     * of the <code>Id</code> element for that origin. When a request matches the path pattern for that cache behavior,
     * CloudFront routes the request to the specified origin. For more information, see <a href=
     * "https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/distribution-web-values-specify.html#DownloadDistValuesCacheBehavior"
     * >Cache Behavior Settings</a> in the <i>Amazon CloudFront Developer Guide</i>.
     * </p>
     * 
     * @return A unique identifier for the origin or origin group. The value of <code>Id</code> must be unique within
     *         the distribution.</p>
     *         <p>
     *         When you specify the value of <code>TargetOriginId</code> for the default cache behavior or for another
     *         cache behavior, you indicate the origin to which you want the cache behavior to route requests by
     *         specifying the value of the <code>Id</code> element for that origin. When a request matches the path
     *         pattern for that cache behavior, CloudFront routes the request to the specified origin. For more
     *         information, see <a href=
     *         "https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/distribution-web-values-specify.html#DownloadDistValuesCacheBehavior"
     *         >Cache Behavior Settings</a> in the <i>Amazon CloudFront Developer Guide</i>.
     */
    public String id() {
        return id;
    }

    /**
     * <p>
     * <b>Amazon S3 origins</b>: The DNS name of the Amazon S3 bucket from which you want CloudFront to get objects for
     * this origin, for example, <code>myawsbucket.s3.amazonaws.com</code>. If you set up your bucket to be configured
     * as a website endpoint, enter the Amazon S3 static website hosting endpoint for the bucket.
     * </p>
     * <p>
     * For more information about specifying this value for different types of origins, see <a href=
     * "https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/distribution-web-values-specify.html#DownloadDistValuesDomainName"
     * >Origin Domain Name</a> in the <i>Amazon CloudFront Developer Guide</i>.
     * </p>
     * <p>
     * Constraints for Amazon S3 origins:
     * </p>
     * <ul>
     * <li>
     * <p>
     * If you configured Amazon S3 Transfer Acceleration for your bucket, don't specify the <code>s3-accelerate</code>
     * endpoint for <code>DomainName</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * The bucket name must be between 3 and 63 characters long (inclusive).
     * </p>
     * </li>
     * <li>
     * <p>
     * The bucket name must contain only lowercase characters, numbers, periods, underscores, and dashes.
     * </p>
     * </li>
     * <li>
     * <p>
     * The bucket name must not contain adjacent periods.
     * </p>
     * </li>
     * </ul>
     * <p>
     * <b>Custom Origins</b>: The DNS domain name for the HTTP server from which you want CloudFront to get objects for
     * this origin, for example, <code>www.example.com</code>.
     * </p>
     * <p>
     * Constraints for custom origins:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>DomainName</code> must be a valid DNS name that contains only a-z, A-Z, 0-9, dot (.), hyphen (-), or
     * underscore (_) characters.
     * </p>
     * </li>
     * <li>
     * <p>
     * The name cannot exceed 128 characters.
     * </p>
     * </li>
     * </ul>
     * 
     * @return <b>Amazon S3 origins</b>: The DNS name of the Amazon S3 bucket from which you want CloudFront to get
     *         objects for this origin, for example, <code>myawsbucket.s3.amazonaws.com</code>. If you set up your
     *         bucket to be configured as a website endpoint, enter the Amazon S3 static website hosting endpoint for
     *         the bucket.</p>
     *         <p>
     *         For more information about specifying this value for different types of origins, see <a href=
     *         "https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/distribution-web-values-specify.html#DownloadDistValuesDomainName"
     *         >Origin Domain Name</a> in the <i>Amazon CloudFront Developer Guide</i>.
     *         </p>
     *         <p>
     *         Constraints for Amazon S3 origins:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         If you configured Amazon S3 Transfer Acceleration for your bucket, don't specify the
     *         <code>s3-accelerate</code> endpoint for <code>DomainName</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         The bucket name must be between 3 and 63 characters long (inclusive).
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         The bucket name must contain only lowercase characters, numbers, periods, underscores, and dashes.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         The bucket name must not contain adjacent periods.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         <b>Custom Origins</b>: The DNS domain name for the HTTP server from which you want CloudFront to get
     *         objects for this origin, for example, <code>www.example.com</code>.
     *         </p>
     *         <p>
     *         Constraints for custom origins:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>DomainName</code> must be a valid DNS name that contains only a-z, A-Z, 0-9, dot (.), hyphen (-),
     *         or underscore (_) characters.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         The name cannot exceed 128 characters.
     *         </p>
     *         </li>
     */
    public String domainName() {
        return domainName;
    }

    /**
     * <p>
     * An optional element that causes CloudFront to request your content from a directory in your Amazon S3 bucket or
     * your custom origin. When you include the <code>OriginPath</code> element, specify the directory name, beginning
     * with a <code>/</code>. CloudFront appends the directory name to the value of <code>DomainName</code>, for
     * example, <code>example.com/production</code>. Do not include a <code>/</code> at the end of the directory name.
     * </p>
     * <p>
     * For example, suppose you've specified the following values for your distribution:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>DomainName</code>: An Amazon S3 bucket named <code>myawsbucket</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>OriginPath</code>: <code>/production</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>CNAME</code>: <code>example.com</code>
     * </p>
     * </li>
     * </ul>
     * <p>
     * When a user enters <code>example.com/index.html</code> in a browser, CloudFront sends a request to Amazon S3 for
     * <code>myawsbucket/production/index.html</code>.
     * </p>
     * <p>
     * When a user enters <code>example.com/acme/index.html</code> in a browser, CloudFront sends a request to Amazon S3
     * for <code>myawsbucket/production/acme/index.html</code>.
     * </p>
     * 
     * @return An optional element that causes CloudFront to request your content from a directory in your Amazon S3
     *         bucket or your custom origin. When you include the <code>OriginPath</code> element, specify the directory
     *         name, beginning with a <code>/</code>. CloudFront appends the directory name to the value of
     *         <code>DomainName</code>, for example, <code>example.com/production</code>. Do not include a
     *         <code>/</code> at the end of the directory name.</p>
     *         <p>
     *         For example, suppose you've specified the following values for your distribution:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>DomainName</code>: An Amazon S3 bucket named <code>myawsbucket</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>OriginPath</code>: <code>/production</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>CNAME</code>: <code>example.com</code>
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         When a user enters <code>example.com/index.html</code> in a browser, CloudFront sends a request to Amazon
     *         S3 for <code>myawsbucket/production/index.html</code>.
     *         </p>
     *         <p>
     *         When a user enters <code>example.com/acme/index.html</code> in a browser, CloudFront sends a request to
     *         Amazon S3 for <code>myawsbucket/production/acme/index.html</code>.
     */
    public String originPath() {
        return originPath;
    }

    /**
     * <p>
     * A complex type that contains names and values for the custom headers that you want.
     * </p>
     * 
     * @return A complex type that contains names and values for the custom headers that you want.
     */
    public CustomHeaders customHeaders() {
        return customHeaders;
    }

    /**
     * <p>
     * A complex type that contains information about the Amazon S3 origin. If the origin is a custom origin, use the
     * <code>CustomOriginConfig</code> element instead.
     * </p>
     * 
     * @return A complex type that contains information about the Amazon S3 origin. If the origin is a custom origin,
     *         use the <code>CustomOriginConfig</code> element instead.
     */
    public S3OriginConfig s3OriginConfig() {
        return s3OriginConfig;
    }

    /**
     * <p>
     * A complex type that contains information about a custom origin. If the origin is an Amazon S3 bucket, use the
     * <code>S3OriginConfig</code> element instead.
     * </p>
     * 
     * @return A complex type that contains information about a custom origin. If the origin is an Amazon S3 bucket, use
     *         the <code>S3OriginConfig</code> element instead.
     */
    public CustomOriginConfig customOriginConfig() {
        return customOriginConfig;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(id());
        hashCode = 31 * hashCode + Objects.hashCode(domainName());
        hashCode = 31 * hashCode + Objects.hashCode(originPath());
        hashCode = 31 * hashCode + Objects.hashCode(customHeaders());
        hashCode = 31 * hashCode + Objects.hashCode(s3OriginConfig());
        hashCode = 31 * hashCode + Objects.hashCode(customOriginConfig());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof Origin)) {
            return false;
        }
        Origin other = (Origin) obj;
        return Objects.equals(id(), other.id()) && Objects.equals(domainName(), other.domainName())
                && Objects.equals(originPath(), other.originPath()) && Objects.equals(customHeaders(), other.customHeaders())
                && Objects.equals(s3OriginConfig(), other.s3OriginConfig())
                && Objects.equals(customOriginConfig(), other.customOriginConfig());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("Origin").add("Id", id()).add("DomainName", domainName()).add("OriginPath", originPath())
                .add("CustomHeaders", customHeaders()).add("S3OriginConfig", s3OriginConfig())
                .add("CustomOriginConfig", customOriginConfig()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Id":
            return Optional.ofNullable(clazz.cast(id()));
        case "DomainName":
            return Optional.ofNullable(clazz.cast(domainName()));
        case "OriginPath":
            return Optional.ofNullable(clazz.cast(originPath()));
        case "CustomHeaders":
            return Optional.ofNullable(clazz.cast(customHeaders()));
        case "S3OriginConfig":
            return Optional.ofNullable(clazz.cast(s3OriginConfig()));
        case "CustomOriginConfig":
            return Optional.ofNullable(clazz.cast(customOriginConfig()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<Origin, T> g) {
        return obj -> g.apply((Origin) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, Origin> {
        /**
         * <p>
         * A unique identifier for the origin or origin group. The value of <code>Id</code> must be unique within the
         * distribution.
         * </p>
         * <p>
         * When you specify the value of <code>TargetOriginId</code> for the default cache behavior or for another cache
         * behavior, you indicate the origin to which you want the cache behavior to route requests by specifying the
         * value of the <code>Id</code> element for that origin. When a request matches the path pattern for that cache
         * behavior, CloudFront routes the request to the specified origin. For more information, see <a href=
         * "https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/distribution-web-values-specify.html#DownloadDistValuesCacheBehavior"
         * >Cache Behavior Settings</a> in the <i>Amazon CloudFront Developer Guide</i>.
         * </p>
         * 
         * @param id
         *        A unique identifier for the origin or origin group. The value of <code>Id</code> must be unique within
         *        the distribution.</p>
         *        <p>
         *        When you specify the value of <code>TargetOriginId</code> for the default cache behavior or for
         *        another cache behavior, you indicate the origin to which you want the cache behavior to route requests
         *        by specifying the value of the <code>Id</code> element for that origin. When a request matches the
         *        path pattern for that cache behavior, CloudFront routes the request to the specified origin. For more
         *        information, see <a href=
         *        "https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/distribution-web-values-specify.html#DownloadDistValuesCacheBehavior"
         *        >Cache Behavior Settings</a> in the <i>Amazon CloudFront Developer Guide</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder id(String id);

        /**
         * <p>
         * <b>Amazon S3 origins</b>: The DNS name of the Amazon S3 bucket from which you want CloudFront to get objects
         * for this origin, for example, <code>myawsbucket.s3.amazonaws.com</code>. If you set up your bucket to be
         * configured as a website endpoint, enter the Amazon S3 static website hosting endpoint for the bucket.
         * </p>
         * <p>
         * For more information about specifying this value for different types of origins, see <a href=
         * "https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/distribution-web-values-specify.html#DownloadDistValuesDomainName"
         * >Origin Domain Name</a> in the <i>Amazon CloudFront Developer Guide</i>.
         * </p>
         * <p>
         * Constraints for Amazon S3 origins:
         * </p>
         * <ul>
         * <li>
         * <p>
         * If you configured Amazon S3 Transfer Acceleration for your bucket, don't specify the
         * <code>s3-accelerate</code> endpoint for <code>DomainName</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * The bucket name must be between 3 and 63 characters long (inclusive).
         * </p>
         * </li>
         * <li>
         * <p>
         * The bucket name must contain only lowercase characters, numbers, periods, underscores, and dashes.
         * </p>
         * </li>
         * <li>
         * <p>
         * The bucket name must not contain adjacent periods.
         * </p>
         * </li>
         * </ul>
         * <p>
         * <b>Custom Origins</b>: The DNS domain name for the HTTP server from which you want CloudFront to get objects
         * for this origin, for example, <code>www.example.com</code>.
         * </p>
         * <p>
         * Constraints for custom origins:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>DomainName</code> must be a valid DNS name that contains only a-z, A-Z, 0-9, dot (.), hyphen (-), or
         * underscore (_) characters.
         * </p>
         * </li>
         * <li>
         * <p>
         * The name cannot exceed 128 characters.
         * </p>
         * </li>
         * </ul>
         * 
         * @param domainName
         *        <b>Amazon S3 origins</b>: The DNS name of the Amazon S3 bucket from which you want CloudFront to get
         *        objects for this origin, for example, <code>myawsbucket.s3.amazonaws.com</code>. If you set up your
         *        bucket to be configured as a website endpoint, enter the Amazon S3 static website hosting endpoint for
         *        the bucket.</p>
         *        <p>
         *        For more information about specifying this value for different types of origins, see <a href=
         *        "https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/distribution-web-values-specify.html#DownloadDistValuesDomainName"
         *        >Origin Domain Name</a> in the <i>Amazon CloudFront Developer Guide</i>.
         *        </p>
         *        <p>
         *        Constraints for Amazon S3 origins:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        If you configured Amazon S3 Transfer Acceleration for your bucket, don't specify the
         *        <code>s3-accelerate</code> endpoint for <code>DomainName</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        The bucket name must be between 3 and 63 characters long (inclusive).
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        The bucket name must contain only lowercase characters, numbers, periods, underscores, and dashes.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        The bucket name must not contain adjacent periods.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        <b>Custom Origins</b>: The DNS domain name for the HTTP server from which you want CloudFront to get
         *        objects for this origin, for example, <code>www.example.com</code>.
         *        </p>
         *        <p>
         *        Constraints for custom origins:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>DomainName</code> must be a valid DNS name that contains only a-z, A-Z, 0-9, dot (.), hyphen
         *        (-), or underscore (_) characters.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        The name cannot exceed 128 characters.
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder domainName(String domainName);

        /**
         * <p>
         * An optional element that causes CloudFront to request your content from a directory in your Amazon S3 bucket
         * or your custom origin. When you include the <code>OriginPath</code> element, specify the directory name,
         * beginning with a <code>/</code>. CloudFront appends the directory name to the value of
         * <code>DomainName</code>, for example, <code>example.com/production</code>. Do not include a <code>/</code> at
         * the end of the directory name.
         * </p>
         * <p>
         * For example, suppose you've specified the following values for your distribution:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>DomainName</code>: An Amazon S3 bucket named <code>myawsbucket</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>OriginPath</code>: <code>/production</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>CNAME</code>: <code>example.com</code>
         * </p>
         * </li>
         * </ul>
         * <p>
         * When a user enters <code>example.com/index.html</code> in a browser, CloudFront sends a request to Amazon S3
         * for <code>myawsbucket/production/index.html</code>.
         * </p>
         * <p>
         * When a user enters <code>example.com/acme/index.html</code> in a browser, CloudFront sends a request to
         * Amazon S3 for <code>myawsbucket/production/acme/index.html</code>.
         * </p>
         * 
         * @param originPath
         *        An optional element that causes CloudFront to request your content from a directory in your Amazon S3
         *        bucket or your custom origin. When you include the <code>OriginPath</code> element, specify the
         *        directory name, beginning with a <code>/</code>. CloudFront appends the directory name to the value of
         *        <code>DomainName</code>, for example, <code>example.com/production</code>. Do not include a
         *        <code>/</code> at the end of the directory name.</p>
         *        <p>
         *        For example, suppose you've specified the following values for your distribution:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>DomainName</code>: An Amazon S3 bucket named <code>myawsbucket</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>OriginPath</code>: <code>/production</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>CNAME</code>: <code>example.com</code>
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        When a user enters <code>example.com/index.html</code> in a browser, CloudFront sends a request to
         *        Amazon S3 for <code>myawsbucket/production/index.html</code>.
         *        </p>
         *        <p>
         *        When a user enters <code>example.com/acme/index.html</code> in a browser, CloudFront sends a request
         *        to Amazon S3 for <code>myawsbucket/production/acme/index.html</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder originPath(String originPath);

        /**
         * <p>
         * A complex type that contains names and values for the custom headers that you want.
         * </p>
         * 
         * @param customHeaders
         *        A complex type that contains names and values for the custom headers that you want.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder customHeaders(CustomHeaders customHeaders);

        /**
         * <p>
         * A complex type that contains names and values for the custom headers that you want.
         * </p>
         * This is a convenience that creates an instance of the {@link CustomHeaders.Builder} avoiding the need to
         * create one manually via {@link CustomHeaders#builder()}.
         *
         * When the {@link Consumer} completes, {@link CustomHeaders.Builder#build()} is called immediately and its
         * result is passed to {@link #customHeaders(CustomHeaders)}.
         * 
         * @param customHeaders
         *        a consumer that will call methods on {@link CustomHeaders.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #customHeaders(CustomHeaders)
         */
        default Builder customHeaders(Consumer<CustomHeaders.Builder> customHeaders) {
            return customHeaders(CustomHeaders.builder().applyMutation(customHeaders).build());
        }

        /**
         * <p>
         * A complex type that contains information about the Amazon S3 origin. If the origin is a custom origin, use
         * the <code>CustomOriginConfig</code> element instead.
         * </p>
         * 
         * @param s3OriginConfig
         *        A complex type that contains information about the Amazon S3 origin. If the origin is a custom origin,
         *        use the <code>CustomOriginConfig</code> element instead.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder s3OriginConfig(S3OriginConfig s3OriginConfig);

        /**
         * <p>
         * A complex type that contains information about the Amazon S3 origin. If the origin is a custom origin, use
         * the <code>CustomOriginConfig</code> element instead.
         * </p>
         * This is a convenience that creates an instance of the {@link S3OriginConfig.Builder} avoiding the need to
         * create one manually via {@link S3OriginConfig#builder()}.
         *
         * When the {@link Consumer} completes, {@link S3OriginConfig.Builder#build()} is called immediately and its
         * result is passed to {@link #s3OriginConfig(S3OriginConfig)}.
         * 
         * @param s3OriginConfig
         *        a consumer that will call methods on {@link S3OriginConfig.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #s3OriginConfig(S3OriginConfig)
         */
        default Builder s3OriginConfig(Consumer<S3OriginConfig.Builder> s3OriginConfig) {
            return s3OriginConfig(S3OriginConfig.builder().applyMutation(s3OriginConfig).build());
        }

        /**
         * <p>
         * A complex type that contains information about a custom origin. If the origin is an Amazon S3 bucket, use the
         * <code>S3OriginConfig</code> element instead.
         * </p>
         * 
         * @param customOriginConfig
         *        A complex type that contains information about a custom origin. If the origin is an Amazon S3 bucket,
         *        use the <code>S3OriginConfig</code> element instead.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder customOriginConfig(CustomOriginConfig customOriginConfig);

        /**
         * <p>
         * A complex type that contains information about a custom origin. If the origin is an Amazon S3 bucket, use the
         * <code>S3OriginConfig</code> element instead.
         * </p>
         * This is a convenience that creates an instance of the {@link CustomOriginConfig.Builder} avoiding the need to
         * create one manually via {@link CustomOriginConfig#builder()}.
         *
         * When the {@link Consumer} completes, {@link CustomOriginConfig.Builder#build()} is called immediately and its
         * result is passed to {@link #customOriginConfig(CustomOriginConfig)}.
         * 
         * @param customOriginConfig
         *        a consumer that will call methods on {@link CustomOriginConfig.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #customOriginConfig(CustomOriginConfig)
         */
        default Builder customOriginConfig(Consumer<CustomOriginConfig.Builder> customOriginConfig) {
            return customOriginConfig(CustomOriginConfig.builder().applyMutation(customOriginConfig).build());
        }
    }

    static final class BuilderImpl implements Builder {
        private String id;

        private String domainName;

        private String originPath;

        private CustomHeaders customHeaders;

        private S3OriginConfig s3OriginConfig;

        private CustomOriginConfig customOriginConfig;

        private BuilderImpl() {
        }

        private BuilderImpl(Origin model) {
            id(model.id);
            domainName(model.domainName);
            originPath(model.originPath);
            customHeaders(model.customHeaders);
            s3OriginConfig(model.s3OriginConfig);
            customOriginConfig(model.customOriginConfig);
        }

        public final String getId() {
            return id;
        }

        @Override
        public final Builder id(String id) {
            this.id = id;
            return this;
        }

        public final void setId(String id) {
            this.id = id;
        }

        public final String getDomainName() {
            return domainName;
        }

        @Override
        public final Builder domainName(String domainName) {
            this.domainName = domainName;
            return this;
        }

        public final void setDomainName(String domainName) {
            this.domainName = domainName;
        }

        public final String getOriginPath() {
            return originPath;
        }

        @Override
        public final Builder originPath(String originPath) {
            this.originPath = originPath;
            return this;
        }

        public final void setOriginPath(String originPath) {
            this.originPath = originPath;
        }

        public final CustomHeaders.Builder getCustomHeaders() {
            return customHeaders != null ? customHeaders.toBuilder() : null;
        }

        @Override
        public final Builder customHeaders(CustomHeaders customHeaders) {
            this.customHeaders = customHeaders;
            return this;
        }

        public final void setCustomHeaders(CustomHeaders.BuilderImpl customHeaders) {
            this.customHeaders = customHeaders != null ? customHeaders.build() : null;
        }

        public final S3OriginConfig.Builder getS3OriginConfig() {
            return s3OriginConfig != null ? s3OriginConfig.toBuilder() : null;
        }

        @Override
        public final Builder s3OriginConfig(S3OriginConfig s3OriginConfig) {
            this.s3OriginConfig = s3OriginConfig;
            return this;
        }

        public final void setS3OriginConfig(S3OriginConfig.BuilderImpl s3OriginConfig) {
            this.s3OriginConfig = s3OriginConfig != null ? s3OriginConfig.build() : null;
        }

        public final CustomOriginConfig.Builder getCustomOriginConfig() {
            return customOriginConfig != null ? customOriginConfig.toBuilder() : null;
        }

        @Override
        public final Builder customOriginConfig(CustomOriginConfig customOriginConfig) {
            this.customOriginConfig = customOriginConfig;
            return this;
        }

        public final void setCustomOriginConfig(CustomOriginConfig.BuilderImpl customOriginConfig) {
            this.customOriginConfig = customOriginConfig != null ? customOriginConfig.build() : null;
        }

        @Override
        public Origin build() {
            return new Origin(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
