/*
 * Copyright 2014-2019 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.cloudfront;

import java.util.concurrent.CompletableFuture;
import java.util.function.Consumer;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.awscore.client.handler.AwsAsyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.ApiName;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.handler.AsyncClientHandler;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.util.VersionInfo;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.xml.AwsXmlProtocolFactory;
import software.amazon.awssdk.protocols.xml.XmlOperationMetadata;
import software.amazon.awssdk.services.cloudfront.model.AccessDeniedException;
import software.amazon.awssdk.services.cloudfront.model.BatchTooLargeException;
import software.amazon.awssdk.services.cloudfront.model.CannotChangeImmutablePublicKeyFieldsException;
import software.amazon.awssdk.services.cloudfront.model.CloudFrontException;
import software.amazon.awssdk.services.cloudfront.model.CloudFrontOriginAccessIdentityAlreadyExistsException;
import software.amazon.awssdk.services.cloudfront.model.CloudFrontOriginAccessIdentityInUseException;
import software.amazon.awssdk.services.cloudfront.model.CloudFrontRequest;
import software.amazon.awssdk.services.cloudfront.model.CnameAlreadyExistsException;
import software.amazon.awssdk.services.cloudfront.model.CreateCloudFrontOriginAccessIdentityRequest;
import software.amazon.awssdk.services.cloudfront.model.CreateCloudFrontOriginAccessIdentityResponse;
import software.amazon.awssdk.services.cloudfront.model.CreateDistributionRequest;
import software.amazon.awssdk.services.cloudfront.model.CreateDistributionResponse;
import software.amazon.awssdk.services.cloudfront.model.CreateDistributionWithTagsRequest;
import software.amazon.awssdk.services.cloudfront.model.CreateDistributionWithTagsResponse;
import software.amazon.awssdk.services.cloudfront.model.CreateFieldLevelEncryptionConfigRequest;
import software.amazon.awssdk.services.cloudfront.model.CreateFieldLevelEncryptionConfigResponse;
import software.amazon.awssdk.services.cloudfront.model.CreateFieldLevelEncryptionProfileRequest;
import software.amazon.awssdk.services.cloudfront.model.CreateFieldLevelEncryptionProfileResponse;
import software.amazon.awssdk.services.cloudfront.model.CreateInvalidationRequest;
import software.amazon.awssdk.services.cloudfront.model.CreateInvalidationResponse;
import software.amazon.awssdk.services.cloudfront.model.CreatePublicKeyRequest;
import software.amazon.awssdk.services.cloudfront.model.CreatePublicKeyResponse;
import software.amazon.awssdk.services.cloudfront.model.CreateStreamingDistributionRequest;
import software.amazon.awssdk.services.cloudfront.model.CreateStreamingDistributionResponse;
import software.amazon.awssdk.services.cloudfront.model.CreateStreamingDistributionWithTagsRequest;
import software.amazon.awssdk.services.cloudfront.model.CreateStreamingDistributionWithTagsResponse;
import software.amazon.awssdk.services.cloudfront.model.DeleteCloudFrontOriginAccessIdentityRequest;
import software.amazon.awssdk.services.cloudfront.model.DeleteCloudFrontOriginAccessIdentityResponse;
import software.amazon.awssdk.services.cloudfront.model.DeleteDistributionRequest;
import software.amazon.awssdk.services.cloudfront.model.DeleteDistributionResponse;
import software.amazon.awssdk.services.cloudfront.model.DeleteFieldLevelEncryptionConfigRequest;
import software.amazon.awssdk.services.cloudfront.model.DeleteFieldLevelEncryptionConfigResponse;
import software.amazon.awssdk.services.cloudfront.model.DeleteFieldLevelEncryptionProfileRequest;
import software.amazon.awssdk.services.cloudfront.model.DeleteFieldLevelEncryptionProfileResponse;
import software.amazon.awssdk.services.cloudfront.model.DeletePublicKeyRequest;
import software.amazon.awssdk.services.cloudfront.model.DeletePublicKeyResponse;
import software.amazon.awssdk.services.cloudfront.model.DeleteStreamingDistributionRequest;
import software.amazon.awssdk.services.cloudfront.model.DeleteStreamingDistributionResponse;
import software.amazon.awssdk.services.cloudfront.model.DistributionAlreadyExistsException;
import software.amazon.awssdk.services.cloudfront.model.DistributionNotDisabledException;
import software.amazon.awssdk.services.cloudfront.model.FieldLevelEncryptionConfigAlreadyExistsException;
import software.amazon.awssdk.services.cloudfront.model.FieldLevelEncryptionConfigInUseException;
import software.amazon.awssdk.services.cloudfront.model.FieldLevelEncryptionProfileAlreadyExistsException;
import software.amazon.awssdk.services.cloudfront.model.FieldLevelEncryptionProfileInUseException;
import software.amazon.awssdk.services.cloudfront.model.FieldLevelEncryptionProfileSizeExceededException;
import software.amazon.awssdk.services.cloudfront.model.GetCloudFrontOriginAccessIdentityConfigRequest;
import software.amazon.awssdk.services.cloudfront.model.GetCloudFrontOriginAccessIdentityConfigResponse;
import software.amazon.awssdk.services.cloudfront.model.GetCloudFrontOriginAccessIdentityRequest;
import software.amazon.awssdk.services.cloudfront.model.GetCloudFrontOriginAccessIdentityResponse;
import software.amazon.awssdk.services.cloudfront.model.GetDistributionConfigRequest;
import software.amazon.awssdk.services.cloudfront.model.GetDistributionConfigResponse;
import software.amazon.awssdk.services.cloudfront.model.GetDistributionRequest;
import software.amazon.awssdk.services.cloudfront.model.GetDistributionResponse;
import software.amazon.awssdk.services.cloudfront.model.GetFieldLevelEncryptionConfigRequest;
import software.amazon.awssdk.services.cloudfront.model.GetFieldLevelEncryptionConfigResponse;
import software.amazon.awssdk.services.cloudfront.model.GetFieldLevelEncryptionProfileConfigRequest;
import software.amazon.awssdk.services.cloudfront.model.GetFieldLevelEncryptionProfileConfigResponse;
import software.amazon.awssdk.services.cloudfront.model.GetFieldLevelEncryptionProfileRequest;
import software.amazon.awssdk.services.cloudfront.model.GetFieldLevelEncryptionProfileResponse;
import software.amazon.awssdk.services.cloudfront.model.GetFieldLevelEncryptionRequest;
import software.amazon.awssdk.services.cloudfront.model.GetFieldLevelEncryptionResponse;
import software.amazon.awssdk.services.cloudfront.model.GetInvalidationRequest;
import software.amazon.awssdk.services.cloudfront.model.GetInvalidationResponse;
import software.amazon.awssdk.services.cloudfront.model.GetPublicKeyConfigRequest;
import software.amazon.awssdk.services.cloudfront.model.GetPublicKeyConfigResponse;
import software.amazon.awssdk.services.cloudfront.model.GetPublicKeyRequest;
import software.amazon.awssdk.services.cloudfront.model.GetPublicKeyResponse;
import software.amazon.awssdk.services.cloudfront.model.GetStreamingDistributionConfigRequest;
import software.amazon.awssdk.services.cloudfront.model.GetStreamingDistributionConfigResponse;
import software.amazon.awssdk.services.cloudfront.model.GetStreamingDistributionRequest;
import software.amazon.awssdk.services.cloudfront.model.GetStreamingDistributionResponse;
import software.amazon.awssdk.services.cloudfront.model.IllegalFieldLevelEncryptionConfigAssociationWithCacheBehaviorException;
import software.amazon.awssdk.services.cloudfront.model.IllegalUpdateException;
import software.amazon.awssdk.services.cloudfront.model.InconsistentQuantitiesException;
import software.amazon.awssdk.services.cloudfront.model.InvalidArgumentException;
import software.amazon.awssdk.services.cloudfront.model.InvalidDefaultRootObjectException;
import software.amazon.awssdk.services.cloudfront.model.InvalidErrorCodeException;
import software.amazon.awssdk.services.cloudfront.model.InvalidForwardCookiesException;
import software.amazon.awssdk.services.cloudfront.model.InvalidGeoRestrictionParameterException;
import software.amazon.awssdk.services.cloudfront.model.InvalidHeadersForS3OriginException;
import software.amazon.awssdk.services.cloudfront.model.InvalidIfMatchVersionException;
import software.amazon.awssdk.services.cloudfront.model.InvalidLambdaFunctionAssociationException;
import software.amazon.awssdk.services.cloudfront.model.InvalidLocationCodeException;
import software.amazon.awssdk.services.cloudfront.model.InvalidMinimumProtocolVersionException;
import software.amazon.awssdk.services.cloudfront.model.InvalidOriginAccessIdentityException;
import software.amazon.awssdk.services.cloudfront.model.InvalidOriginException;
import software.amazon.awssdk.services.cloudfront.model.InvalidOriginKeepaliveTimeoutException;
import software.amazon.awssdk.services.cloudfront.model.InvalidOriginReadTimeoutException;
import software.amazon.awssdk.services.cloudfront.model.InvalidProtocolSettingsException;
import software.amazon.awssdk.services.cloudfront.model.InvalidQueryStringParametersException;
import software.amazon.awssdk.services.cloudfront.model.InvalidRelativePathException;
import software.amazon.awssdk.services.cloudfront.model.InvalidRequiredProtocolException;
import software.amazon.awssdk.services.cloudfront.model.InvalidResponseCodeException;
import software.amazon.awssdk.services.cloudfront.model.InvalidTaggingException;
import software.amazon.awssdk.services.cloudfront.model.InvalidTtlOrderException;
import software.amazon.awssdk.services.cloudfront.model.InvalidViewerCertificateException;
import software.amazon.awssdk.services.cloudfront.model.InvalidWebAclIdException;
import software.amazon.awssdk.services.cloudfront.model.ListCloudFrontOriginAccessIdentitiesRequest;
import software.amazon.awssdk.services.cloudfront.model.ListCloudFrontOriginAccessIdentitiesResponse;
import software.amazon.awssdk.services.cloudfront.model.ListDistributionsByWebAclIdRequest;
import software.amazon.awssdk.services.cloudfront.model.ListDistributionsByWebAclIdResponse;
import software.amazon.awssdk.services.cloudfront.model.ListDistributionsRequest;
import software.amazon.awssdk.services.cloudfront.model.ListDistributionsResponse;
import software.amazon.awssdk.services.cloudfront.model.ListFieldLevelEncryptionConfigsRequest;
import software.amazon.awssdk.services.cloudfront.model.ListFieldLevelEncryptionConfigsResponse;
import software.amazon.awssdk.services.cloudfront.model.ListFieldLevelEncryptionProfilesRequest;
import software.amazon.awssdk.services.cloudfront.model.ListFieldLevelEncryptionProfilesResponse;
import software.amazon.awssdk.services.cloudfront.model.ListInvalidationsRequest;
import software.amazon.awssdk.services.cloudfront.model.ListInvalidationsResponse;
import software.amazon.awssdk.services.cloudfront.model.ListPublicKeysRequest;
import software.amazon.awssdk.services.cloudfront.model.ListPublicKeysResponse;
import software.amazon.awssdk.services.cloudfront.model.ListStreamingDistributionsRequest;
import software.amazon.awssdk.services.cloudfront.model.ListStreamingDistributionsResponse;
import software.amazon.awssdk.services.cloudfront.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.cloudfront.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.cloudfront.model.MissingBodyException;
import software.amazon.awssdk.services.cloudfront.model.NoSuchCloudFrontOriginAccessIdentityException;
import software.amazon.awssdk.services.cloudfront.model.NoSuchDistributionException;
import software.amazon.awssdk.services.cloudfront.model.NoSuchFieldLevelEncryptionConfigException;
import software.amazon.awssdk.services.cloudfront.model.NoSuchFieldLevelEncryptionProfileException;
import software.amazon.awssdk.services.cloudfront.model.NoSuchInvalidationException;
import software.amazon.awssdk.services.cloudfront.model.NoSuchOriginException;
import software.amazon.awssdk.services.cloudfront.model.NoSuchPublicKeyException;
import software.amazon.awssdk.services.cloudfront.model.NoSuchResourceException;
import software.amazon.awssdk.services.cloudfront.model.NoSuchStreamingDistributionException;
import software.amazon.awssdk.services.cloudfront.model.PreconditionFailedException;
import software.amazon.awssdk.services.cloudfront.model.PublicKeyAlreadyExistsException;
import software.amazon.awssdk.services.cloudfront.model.PublicKeyInUseException;
import software.amazon.awssdk.services.cloudfront.model.QueryArgProfileEmptyException;
import software.amazon.awssdk.services.cloudfront.model.StreamingDistributionAlreadyExistsException;
import software.amazon.awssdk.services.cloudfront.model.StreamingDistributionNotDisabledException;
import software.amazon.awssdk.services.cloudfront.model.TagResourceRequest;
import software.amazon.awssdk.services.cloudfront.model.TagResourceResponse;
import software.amazon.awssdk.services.cloudfront.model.TooManyCacheBehaviorsException;
import software.amazon.awssdk.services.cloudfront.model.TooManyCertificatesException;
import software.amazon.awssdk.services.cloudfront.model.TooManyCloudFrontOriginAccessIdentitiesException;
import software.amazon.awssdk.services.cloudfront.model.TooManyCookieNamesInWhiteListException;
import software.amazon.awssdk.services.cloudfront.model.TooManyDistributionCnamEsException;
import software.amazon.awssdk.services.cloudfront.model.TooManyDistributionsAssociatedToFieldLevelEncryptionConfigException;
import software.amazon.awssdk.services.cloudfront.model.TooManyDistributionsException;
import software.amazon.awssdk.services.cloudfront.model.TooManyDistributionsWithLambdaAssociationsException;
import software.amazon.awssdk.services.cloudfront.model.TooManyFieldLevelEncryptionConfigsException;
import software.amazon.awssdk.services.cloudfront.model.TooManyFieldLevelEncryptionContentTypeProfilesException;
import software.amazon.awssdk.services.cloudfront.model.TooManyFieldLevelEncryptionEncryptionEntitiesException;
import software.amazon.awssdk.services.cloudfront.model.TooManyFieldLevelEncryptionFieldPatternsException;
import software.amazon.awssdk.services.cloudfront.model.TooManyFieldLevelEncryptionProfilesException;
import software.amazon.awssdk.services.cloudfront.model.TooManyFieldLevelEncryptionQueryArgProfilesException;
import software.amazon.awssdk.services.cloudfront.model.TooManyHeadersInForwardedValuesException;
import software.amazon.awssdk.services.cloudfront.model.TooManyInvalidationsInProgressException;
import software.amazon.awssdk.services.cloudfront.model.TooManyLambdaFunctionAssociationsException;
import software.amazon.awssdk.services.cloudfront.model.TooManyOriginCustomHeadersException;
import software.amazon.awssdk.services.cloudfront.model.TooManyOriginGroupsPerDistributionException;
import software.amazon.awssdk.services.cloudfront.model.TooManyOriginsException;
import software.amazon.awssdk.services.cloudfront.model.TooManyPublicKeysException;
import software.amazon.awssdk.services.cloudfront.model.TooManyQueryStringParametersException;
import software.amazon.awssdk.services.cloudfront.model.TooManyStreamingDistributionCnamEsException;
import software.amazon.awssdk.services.cloudfront.model.TooManyStreamingDistributionsException;
import software.amazon.awssdk.services.cloudfront.model.TooManyTrustedSignersException;
import software.amazon.awssdk.services.cloudfront.model.TrustedSignerDoesNotExistException;
import software.amazon.awssdk.services.cloudfront.model.UntagResourceRequest;
import software.amazon.awssdk.services.cloudfront.model.UntagResourceResponse;
import software.amazon.awssdk.services.cloudfront.model.UpdateCloudFrontOriginAccessIdentityRequest;
import software.amazon.awssdk.services.cloudfront.model.UpdateCloudFrontOriginAccessIdentityResponse;
import software.amazon.awssdk.services.cloudfront.model.UpdateDistributionRequest;
import software.amazon.awssdk.services.cloudfront.model.UpdateDistributionResponse;
import software.amazon.awssdk.services.cloudfront.model.UpdateFieldLevelEncryptionConfigRequest;
import software.amazon.awssdk.services.cloudfront.model.UpdateFieldLevelEncryptionConfigResponse;
import software.amazon.awssdk.services.cloudfront.model.UpdateFieldLevelEncryptionProfileRequest;
import software.amazon.awssdk.services.cloudfront.model.UpdateFieldLevelEncryptionProfileResponse;
import software.amazon.awssdk.services.cloudfront.model.UpdatePublicKeyRequest;
import software.amazon.awssdk.services.cloudfront.model.UpdatePublicKeyResponse;
import software.amazon.awssdk.services.cloudfront.model.UpdateStreamingDistributionRequest;
import software.amazon.awssdk.services.cloudfront.model.UpdateStreamingDistributionResponse;
import software.amazon.awssdk.services.cloudfront.transform.CreateCloudFrontOriginAccessIdentityRequestMarshaller;
import software.amazon.awssdk.services.cloudfront.transform.CreateDistributionRequestMarshaller;
import software.amazon.awssdk.services.cloudfront.transform.CreateDistributionWithTagsRequestMarshaller;
import software.amazon.awssdk.services.cloudfront.transform.CreateFieldLevelEncryptionConfigRequestMarshaller;
import software.amazon.awssdk.services.cloudfront.transform.CreateFieldLevelEncryptionProfileRequestMarshaller;
import software.amazon.awssdk.services.cloudfront.transform.CreateInvalidationRequestMarshaller;
import software.amazon.awssdk.services.cloudfront.transform.CreatePublicKeyRequestMarshaller;
import software.amazon.awssdk.services.cloudfront.transform.CreateStreamingDistributionRequestMarshaller;
import software.amazon.awssdk.services.cloudfront.transform.CreateStreamingDistributionWithTagsRequestMarshaller;
import software.amazon.awssdk.services.cloudfront.transform.DeleteCloudFrontOriginAccessIdentityRequestMarshaller;
import software.amazon.awssdk.services.cloudfront.transform.DeleteDistributionRequestMarshaller;
import software.amazon.awssdk.services.cloudfront.transform.DeleteFieldLevelEncryptionConfigRequestMarshaller;
import software.amazon.awssdk.services.cloudfront.transform.DeleteFieldLevelEncryptionProfileRequestMarshaller;
import software.amazon.awssdk.services.cloudfront.transform.DeletePublicKeyRequestMarshaller;
import software.amazon.awssdk.services.cloudfront.transform.DeleteStreamingDistributionRequestMarshaller;
import software.amazon.awssdk.services.cloudfront.transform.GetCloudFrontOriginAccessIdentityConfigRequestMarshaller;
import software.amazon.awssdk.services.cloudfront.transform.GetCloudFrontOriginAccessIdentityRequestMarshaller;
import software.amazon.awssdk.services.cloudfront.transform.GetDistributionConfigRequestMarshaller;
import software.amazon.awssdk.services.cloudfront.transform.GetDistributionRequestMarshaller;
import software.amazon.awssdk.services.cloudfront.transform.GetFieldLevelEncryptionConfigRequestMarshaller;
import software.amazon.awssdk.services.cloudfront.transform.GetFieldLevelEncryptionProfileConfigRequestMarshaller;
import software.amazon.awssdk.services.cloudfront.transform.GetFieldLevelEncryptionProfileRequestMarshaller;
import software.amazon.awssdk.services.cloudfront.transform.GetFieldLevelEncryptionRequestMarshaller;
import software.amazon.awssdk.services.cloudfront.transform.GetInvalidationRequestMarshaller;
import software.amazon.awssdk.services.cloudfront.transform.GetPublicKeyConfigRequestMarshaller;
import software.amazon.awssdk.services.cloudfront.transform.GetPublicKeyRequestMarshaller;
import software.amazon.awssdk.services.cloudfront.transform.GetStreamingDistributionConfigRequestMarshaller;
import software.amazon.awssdk.services.cloudfront.transform.GetStreamingDistributionRequestMarshaller;
import software.amazon.awssdk.services.cloudfront.transform.ListCloudFrontOriginAccessIdentitiesRequestMarshaller;
import software.amazon.awssdk.services.cloudfront.transform.ListDistributionsByWebAclIdRequestMarshaller;
import software.amazon.awssdk.services.cloudfront.transform.ListDistributionsRequestMarshaller;
import software.amazon.awssdk.services.cloudfront.transform.ListFieldLevelEncryptionConfigsRequestMarshaller;
import software.amazon.awssdk.services.cloudfront.transform.ListFieldLevelEncryptionProfilesRequestMarshaller;
import software.amazon.awssdk.services.cloudfront.transform.ListInvalidationsRequestMarshaller;
import software.amazon.awssdk.services.cloudfront.transform.ListPublicKeysRequestMarshaller;
import software.amazon.awssdk.services.cloudfront.transform.ListStreamingDistributionsRequestMarshaller;
import software.amazon.awssdk.services.cloudfront.transform.ListTagsForResourceRequestMarshaller;
import software.amazon.awssdk.services.cloudfront.transform.TagResourceRequestMarshaller;
import software.amazon.awssdk.services.cloudfront.transform.UntagResourceRequestMarshaller;
import software.amazon.awssdk.services.cloudfront.transform.UpdateCloudFrontOriginAccessIdentityRequestMarshaller;
import software.amazon.awssdk.services.cloudfront.transform.UpdateDistributionRequestMarshaller;
import software.amazon.awssdk.services.cloudfront.transform.UpdateFieldLevelEncryptionConfigRequestMarshaller;
import software.amazon.awssdk.services.cloudfront.transform.UpdateFieldLevelEncryptionProfileRequestMarshaller;
import software.amazon.awssdk.services.cloudfront.transform.UpdatePublicKeyRequestMarshaller;
import software.amazon.awssdk.services.cloudfront.transform.UpdateStreamingDistributionRequestMarshaller;
import software.amazon.awssdk.utils.CompletableFutureUtils;

/**
 * Internal implementation of {@link CloudFrontAsyncClient}.
 *
 * @see CloudFrontAsyncClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultCloudFrontAsyncClient implements CloudFrontAsyncClient {
    private static final Logger log = LoggerFactory.getLogger(DefaultCloudFrontAsyncClient.class);

    private final AsyncClientHandler clientHandler;

    private final AwsXmlProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultCloudFrontAsyncClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsAsyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.protocolFactory = init();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    /**
     * <p>
     * Creates a new origin access identity. If you're using Amazon S3 for your origin, you can use an origin access
     * identity to require users to access your content using a CloudFront URL instead of the Amazon S3 URL. For more
     * information about how to use origin access identities, see <a
     * href="https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/PrivateContent.html">Serving Private
     * Content through CloudFront</a> in the <i>Amazon CloudFront Developer Guide</i>.
     * </p>
     *
     * @param createCloudFrontOriginAccessIdentityRequest
     *        The request to create a new origin access identity (OAI). An origin access identity is a special
     *        CloudFront user that you can associate with Amazon S3 origins, so that you can secure all or just some of
     *        your Amazon S3 content. For more information, see <a href=
     *        "https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/private-content-restricting-access-to-s3.html"
     *        > Restricting Access to Amazon S3 Content by Using an Origin Access Identity</a> in the <i>Amazon
     *        CloudFront Developer Guide</i>.
     * @return A Java Future containing the result of the CreateCloudFrontOriginAccessIdentity operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>CloudFrontOriginAccessIdentityAlreadyExistsException If the <code>CallerReference</code> is a value
     *         you already sent in a previous request to create an identity but the content of the
     *         <code>CloudFrontOriginAccessIdentityConfig</code> is different from the original request, CloudFront
     *         returns a <code>CloudFrontOriginAccessIdentityAlreadyExists</code> error.</li>
     *         <li>MissingBodyException This operation requires a body. Ensure that the body is present and the
     *         <code>Content-Type</code> header is set.</li>
     *         <li>TooManyCloudFrontOriginAccessIdentitiesException Processing your request would cause you to exceed
     *         the maximum number of origin access identities allowed.</li>
     *         <li>InvalidArgumentException The argument is invalid.</li>
     *         <li>InconsistentQuantitiesException The value of <code>Quantity</code> and the size of <code>Items</code>
     *         don't match.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CloudFrontException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CloudFrontAsyncClient.CreateCloudFrontOriginAccessIdentity
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/cloudfront-2019-03-26/CreateCloudFrontOriginAccessIdentity"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateCloudFrontOriginAccessIdentityResponse> createCloudFrontOriginAccessIdentity(
            CreateCloudFrontOriginAccessIdentityRequest createCloudFrontOriginAccessIdentityRequest) {
        try {

            HttpResponseHandler<CreateCloudFrontOriginAccessIdentityResponse> responseHandler = protocolFactory
                    .createResponseHandler(CreateCloudFrontOriginAccessIdentityResponse::builder,
                            new XmlOperationMetadata().withHasStreamingSuccessResponse(false));

            HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();

            CompletableFuture<CreateCloudFrontOriginAccessIdentityResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateCloudFrontOriginAccessIdentityRequest, CreateCloudFrontOriginAccessIdentityResponse>()
                            .withOperationName("CreateCloudFrontOriginAccessIdentity")
                            .withMarshaller(new CreateCloudFrontOriginAccessIdentityRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(createCloudFrontOriginAccessIdentityRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a new web distribution. You create a CloudFront distribution to tell CloudFront where you want content to
     * be delivered from, and the details about how to track and manage content delivery. Send a <code>POST</code>
     * request to the <code>/<i>CloudFront API version</i>/distribution</code>/<code>distribution ID</code> resource.
     * </p>
     * <important>
     * <p>
     * When you update a distribution, there are more required fields than when you create a distribution. When you
     * update your distribution by using <a
     * href="https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_UpdateDistribution.html"
     * >UpdateDistribution</a>, follow the steps included in the documentation to get the current configuration and then
     * make your updates. This helps to make sure that you include all of the required fields. To view a summary, see <a
     * href
     * ="https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/distribution-overview-required-fields.html"
     * >Required Fields for Create Distribution and Update Distribution</a> in the <i>Amazon CloudFront Developer
     * Guide</i>.
     * </p>
     * </important>
     *
     * @param createDistributionRequest
     *        The request to create a new distribution.
     * @return A Java Future containing the result of the CreateDistribution operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>CnameAlreadyExistsException The CNAME specified is already defined for CloudFront.</li>
     *         <li>DistributionAlreadyExistsException The caller reference you attempted to create the distribution with
     *         is associated with another distribution.</li>
     *         <li>InvalidOriginException The Amazon S3 origin server specified does not refer to a valid Amazon S3
     *         bucket.</li>
     *         <li>InvalidOriginAccessIdentityException The origin access identity is not valid or doesn't exist.</li>
     *         <li>AccessDeniedException Access denied.</li>
     *         <li>TooManyTrustedSignersException Your request contains more trusted signers than are allowed per
     *         distribution.</li>
     *         <li>TrustedSignerDoesNotExistException One or more of your trusted signers don't exist.</li>
     *         <li>InvalidViewerCertificateException A viewer certificate specified in the response body is not valid.</li>
     *         <li>InvalidMinimumProtocolVersionException The minimum protocol version specified is not valid.</li>
     *         <li>MissingBodyException This operation requires a body. Ensure that the body is present and the
     *         <code>Content-Type</code> header is set.</li>
     *         <li>TooManyDistributionCnamEsException Your request contains more CNAMEs than are allowed per
     *         distribution.</li>
     *         <li>TooManyDistributionsException Processing your request would cause you to exceed the maximum number of
     *         distributions allowed.</li>
     *         <li>InvalidDefaultRootObjectException The default root object file name is too big or contains an invalid
     *         character.</li>
     *         <li>InvalidRelativePathException The relative path is too big, is not URL-encoded, or does not begin with
     *         a slash (/).</li>
     *         <li>InvalidErrorCodeException An invalid error code was specified.</li>
     *         <li>InvalidResponseCodeException A response code specified in the response body is not valid.</li>
     *         <li>InvalidArgumentException The argument is invalid.</li>
     *         <li>InvalidRequiredProtocolException This operation requires the HTTPS protocol. Ensure that you specify
     *         the HTTPS protocol in your request, or omit the <code>RequiredProtocols</code> element from your
     *         distribution configuration.</li>
     *         <li>NoSuchOriginException No origin exists with the specified <code>Origin Id</code>.</li>
     *         <li>TooManyOriginsException You cannot create more origins for the distribution.</li>
     *         <li>TooManyOriginGroupsPerDistributionException Processing your request would cause you to exceed the
     *         maximum number of origin groups allowed.</li>
     *         <li>TooManyCacheBehaviorsException You cannot create more cache behaviors for the distribution.</li>
     *         <li>TooManyCookieNamesInWhiteListException Your request contains more cookie names in the whitelist than
     *         are allowed per cache behavior.</li>
     *         <li>InvalidForwardCookiesException Your request contains forward cookies option which doesn't match with
     *         the expectation for the <code>whitelisted</code> list of cookie names. Either list of cookie names has
     *         been specified when not allowed or list of cookie names is missing when expected.</li>
     *         <li>TooManyHeadersInForwardedValuesException Your request contains too many headers in forwarded values.</li>
     *         <li>InvalidHeadersForS3OriginException The headers specified are not valid for an Amazon S3 origin.</li>
     *         <li>InconsistentQuantitiesException The value of <code>Quantity</code> and the size of <code>Items</code>
     *         don't match.</li>
     *         <li>TooManyCertificatesException You cannot create anymore custom SSL/TLS certificates.</li>
     *         <li>InvalidLocationCodeException The location code specified is not valid.</li>
     *         <li>InvalidGeoRestrictionParameterException The specified geo restriction parameter is not valid.</li>
     *         <li>InvalidProtocolSettingsException You cannot specify SSLv3 as the minimum protocol version if you only
     *         want to support only clients that support Server Name Indication (SNI).</li>
     *         <li>InvalidTtlOrderException TTL order specified in the response body is not valid.</li>
     *         <li>InvalidWebAclIdException A web ACL id specified in the response body is not valid.</li>
     *         <li>TooManyOriginCustomHeadersException Your request contains too many origin custom headers.</li>
     *         <li>TooManyQueryStringParametersException Your request contains too many query string parameters.</li>
     *         <li>InvalidQueryStringParametersException Query string parameters specified in the response body are not
     *         valid.</li>
     *         <li>TooManyDistributionsWithLambdaAssociationsException Processing your request would cause the maximum
     *         number of distributions with Lambda function associations per owner to be exceeded.</li>
     *         <li>TooManyLambdaFunctionAssociationsException Your request contains more Lambda function associations
     *         than are allowed per distribution.</li>
     *         <li>InvalidLambdaFunctionAssociationException The specified Lambda function association is invalid.</li>
     *         <li>InvalidOriginReadTimeoutException The read timeout specified for the origin is not valid.</li>
     *         <li>InvalidOriginKeepaliveTimeoutException The keep alive timeout specified for the origin is not valid.</li>
     *         <li>NoSuchFieldLevelEncryptionConfigException The specified configuration for field-level encryption
     *         doesn't exist.</li>
     *         <li>IllegalFieldLevelEncryptionConfigAssociationWithCacheBehaviorException The specified configuration
     *         for field-level encryption can't be associated with the specified cache behavior.</li>
     *         <li>TooManyDistributionsAssociatedToFieldLevelEncryptionConfigException The maximum number of
     *         distributions have been associated with the specified configuration for field-level encryption.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CloudFrontException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CloudFrontAsyncClient.CreateDistribution
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/cloudfront-2019-03-26/CreateDistribution" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateDistributionResponse> createDistribution(CreateDistributionRequest createDistributionRequest) {
        try {

            HttpResponseHandler<CreateDistributionResponse> responseHandler = protocolFactory.createResponseHandler(
                    CreateDistributionResponse::builder, new XmlOperationMetadata().withHasStreamingSuccessResponse(false));

            HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();

            CompletableFuture<CreateDistributionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateDistributionRequest, CreateDistributionResponse>()
                            .withOperationName("CreateDistribution")
                            .withMarshaller(new CreateDistributionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(createDistributionRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Create a new distribution with tags.
     * </p>
     *
     * @param createDistributionWithTagsRequest
     *        The request to create a new distribution with tags.
     * @return A Java Future containing the result of the CreateDistributionWithTags operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>CnameAlreadyExistsException The CNAME specified is already defined for CloudFront.</li>
     *         <li>DistributionAlreadyExistsException The caller reference you attempted to create the distribution with
     *         is associated with another distribution.</li>
     *         <li>InvalidOriginException The Amazon S3 origin server specified does not refer to a valid Amazon S3
     *         bucket.</li>
     *         <li>InvalidOriginAccessIdentityException The origin access identity is not valid or doesn't exist.</li>
     *         <li>AccessDeniedException Access denied.</li>
     *         <li>TooManyTrustedSignersException Your request contains more trusted signers than are allowed per
     *         distribution.</li>
     *         <li>TrustedSignerDoesNotExistException One or more of your trusted signers don't exist.</li>
     *         <li>InvalidViewerCertificateException A viewer certificate specified in the response body is not valid.</li>
     *         <li>InvalidMinimumProtocolVersionException The minimum protocol version specified is not valid.</li>
     *         <li>MissingBodyException This operation requires a body. Ensure that the body is present and the
     *         <code>Content-Type</code> header is set.</li>
     *         <li>TooManyDistributionCnamEsException Your request contains more CNAMEs than are allowed per
     *         distribution.</li>
     *         <li>TooManyDistributionsException Processing your request would cause you to exceed the maximum number of
     *         distributions allowed.</li>
     *         <li>InvalidDefaultRootObjectException The default root object file name is too big or contains an invalid
     *         character.</li>
     *         <li>InvalidRelativePathException The relative path is too big, is not URL-encoded, or does not begin with
     *         a slash (/).</li>
     *         <li>InvalidErrorCodeException An invalid error code was specified.</li>
     *         <li>InvalidResponseCodeException A response code specified in the response body is not valid.</li>
     *         <li>InvalidArgumentException The argument is invalid.</li>
     *         <li>InvalidRequiredProtocolException This operation requires the HTTPS protocol. Ensure that you specify
     *         the HTTPS protocol in your request, or omit the <code>RequiredProtocols</code> element from your
     *         distribution configuration.</li>
     *         <li>NoSuchOriginException No origin exists with the specified <code>Origin Id</code>.</li>
     *         <li>TooManyOriginsException You cannot create more origins for the distribution.</li>
     *         <li>TooManyOriginGroupsPerDistributionException Processing your request would cause you to exceed the
     *         maximum number of origin groups allowed.</li>
     *         <li>TooManyCacheBehaviorsException You cannot create more cache behaviors for the distribution.</li>
     *         <li>TooManyCookieNamesInWhiteListException Your request contains more cookie names in the whitelist than
     *         are allowed per cache behavior.</li>
     *         <li>InvalidForwardCookiesException Your request contains forward cookies option which doesn't match with
     *         the expectation for the <code>whitelisted</code> list of cookie names. Either list of cookie names has
     *         been specified when not allowed or list of cookie names is missing when expected.</li>
     *         <li>TooManyHeadersInForwardedValuesException Your request contains too many headers in forwarded values.</li>
     *         <li>InvalidHeadersForS3OriginException The headers specified are not valid for an Amazon S3 origin.</li>
     *         <li>InconsistentQuantitiesException The value of <code>Quantity</code> and the size of <code>Items</code>
     *         don't match.</li>
     *         <li>TooManyCertificatesException You cannot create anymore custom SSL/TLS certificates.</li>
     *         <li>InvalidLocationCodeException The location code specified is not valid.</li>
     *         <li>InvalidGeoRestrictionParameterException The specified geo restriction parameter is not valid.</li>
     *         <li>InvalidProtocolSettingsException You cannot specify SSLv3 as the minimum protocol version if you only
     *         want to support only clients that support Server Name Indication (SNI).</li>
     *         <li>InvalidTtlOrderException TTL order specified in the response body is not valid.</li>
     *         <li>InvalidWebAclIdException A web ACL id specified in the response body is not valid.</li>
     *         <li>TooManyOriginCustomHeadersException Your request contains too many origin custom headers.</li>
     *         <li>InvalidTaggingException Tagging specified in the response body is not valid.</li>
     *         <li>TooManyQueryStringParametersException Your request contains too many query string parameters.</li>
     *         <li>InvalidQueryStringParametersException Query string parameters specified in the response body are not
     *         valid.</li>
     *         <li>TooManyDistributionsWithLambdaAssociationsException Processing your request would cause the maximum
     *         number of distributions with Lambda function associations per owner to be exceeded.</li>
     *         <li>TooManyLambdaFunctionAssociationsException Your request contains more Lambda function associations
     *         than are allowed per distribution.</li>
     *         <li>InvalidLambdaFunctionAssociationException The specified Lambda function association is invalid.</li>
     *         <li>InvalidOriginReadTimeoutException The read timeout specified for the origin is not valid.</li>
     *         <li>InvalidOriginKeepaliveTimeoutException The keep alive timeout specified for the origin is not valid.</li>
     *         <li>NoSuchFieldLevelEncryptionConfigException The specified configuration for field-level encryption
     *         doesn't exist.</li>
     *         <li>IllegalFieldLevelEncryptionConfigAssociationWithCacheBehaviorException The specified configuration
     *         for field-level encryption can't be associated with the specified cache behavior.</li>
     *         <li>TooManyDistributionsAssociatedToFieldLevelEncryptionConfigException The maximum number of
     *         distributions have been associated with the specified configuration for field-level encryption.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CloudFrontException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CloudFrontAsyncClient.CreateDistributionWithTags
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/cloudfront-2019-03-26/CreateDistributionWithTags"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateDistributionWithTagsResponse> createDistributionWithTags(
            CreateDistributionWithTagsRequest createDistributionWithTagsRequest) {
        try {

            HttpResponseHandler<CreateDistributionWithTagsResponse> responseHandler = protocolFactory.createResponseHandler(
                    CreateDistributionWithTagsResponse::builder,
                    new XmlOperationMetadata().withHasStreamingSuccessResponse(false));

            HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();

            CompletableFuture<CreateDistributionWithTagsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateDistributionWithTagsRequest, CreateDistributionWithTagsResponse>()
                            .withOperationName("CreateDistributionWithTags")
                            .withMarshaller(new CreateDistributionWithTagsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(createDistributionWithTagsRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Create a new field-level encryption configuration.
     * </p>
     *
     * @param createFieldLevelEncryptionConfigRequest
     * @return A Java Future containing the result of the CreateFieldLevelEncryptionConfig operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InconsistentQuantitiesException The value of <code>Quantity</code> and the size of <code>Items</code>
     *         don't match.</li>
     *         <li>InvalidArgumentException The argument is invalid.</li>
     *         <li>NoSuchFieldLevelEncryptionProfileException The specified profile for field-level encryption doesn't
     *         exist.</li>
     *         <li>FieldLevelEncryptionConfigAlreadyExistsException The specified configuration for field-level
     *         encryption already exists.</li>
     *         <li>TooManyFieldLevelEncryptionConfigsException The maximum number of configurations for field-level
     *         encryption have been created.</li>
     *         <li>TooManyFieldLevelEncryptionQueryArgProfilesException The maximum number of query arg profiles for
     *         field-level encryption have been created.</li>
     *         <li>TooManyFieldLevelEncryptionContentTypeProfilesException The maximum number of content type profiles
     *         for field-level encryption have been created.</li>
     *         <li>QueryArgProfileEmptyException No profile specified for the field-level encryption query argument.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CloudFrontException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CloudFrontAsyncClient.CreateFieldLevelEncryptionConfig
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/cloudfront-2019-03-26/CreateFieldLevelEncryptionConfig"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateFieldLevelEncryptionConfigResponse> createFieldLevelEncryptionConfig(
            CreateFieldLevelEncryptionConfigRequest createFieldLevelEncryptionConfigRequest) {
        try {

            HttpResponseHandler<CreateFieldLevelEncryptionConfigResponse> responseHandler = protocolFactory
                    .createResponseHandler(CreateFieldLevelEncryptionConfigResponse::builder,
                            new XmlOperationMetadata().withHasStreamingSuccessResponse(false));

            HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();

            CompletableFuture<CreateFieldLevelEncryptionConfigResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateFieldLevelEncryptionConfigRequest, CreateFieldLevelEncryptionConfigResponse>()
                            .withOperationName("CreateFieldLevelEncryptionConfig")
                            .withMarshaller(new CreateFieldLevelEncryptionConfigRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(createFieldLevelEncryptionConfigRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Create a field-level encryption profile.
     * </p>
     *
     * @param createFieldLevelEncryptionProfileRequest
     * @return A Java Future containing the result of the CreateFieldLevelEncryptionProfile operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InconsistentQuantitiesException The value of <code>Quantity</code> and the size of <code>Items</code>
     *         don't match.</li>
     *         <li>InvalidArgumentException The argument is invalid.</li>
     *         <li>NoSuchPublicKeyException The specified public key doesn't exist.</li>
     *         <li>FieldLevelEncryptionProfileAlreadyExistsException The specified profile for field-level encryption
     *         already exists.</li>
     *         <li>FieldLevelEncryptionProfileSizeExceededException The maximum size of a profile for field-level
     *         encryption was exceeded.</li>
     *         <li>TooManyFieldLevelEncryptionProfilesException The maximum number of profiles for field-level
     *         encryption have been created.</li>
     *         <li>TooManyFieldLevelEncryptionEncryptionEntitiesException The maximum number of encryption entities for
     *         field-level encryption have been created.</li>
     *         <li>TooManyFieldLevelEncryptionFieldPatternsException The maximum number of field patterns for
     *         field-level encryption have been created.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CloudFrontException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CloudFrontAsyncClient.CreateFieldLevelEncryptionProfile
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/cloudfront-2019-03-26/CreateFieldLevelEncryptionProfile"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateFieldLevelEncryptionProfileResponse> createFieldLevelEncryptionProfile(
            CreateFieldLevelEncryptionProfileRequest createFieldLevelEncryptionProfileRequest) {
        try {

            HttpResponseHandler<CreateFieldLevelEncryptionProfileResponse> responseHandler = protocolFactory
                    .createResponseHandler(CreateFieldLevelEncryptionProfileResponse::builder,
                            new XmlOperationMetadata().withHasStreamingSuccessResponse(false));

            HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();

            CompletableFuture<CreateFieldLevelEncryptionProfileResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateFieldLevelEncryptionProfileRequest, CreateFieldLevelEncryptionProfileResponse>()
                            .withOperationName("CreateFieldLevelEncryptionProfile")
                            .withMarshaller(new CreateFieldLevelEncryptionProfileRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(createFieldLevelEncryptionProfileRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Create a new invalidation.
     * </p>
     *
     * @param createInvalidationRequest
     *        The request to create an invalidation.
     * @return A Java Future containing the result of the CreateInvalidation operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException Access denied.</li>
     *         <li>MissingBodyException This operation requires a body. Ensure that the body is present and the
     *         <code>Content-Type</code> header is set.</li>
     *         <li>InvalidArgumentException The argument is invalid.</li>
     *         <li>NoSuchDistributionException The specified distribution does not exist.</li>
     *         <li>BatchTooLargeException Invalidation batch specified is too large.</li>
     *         <li>TooManyInvalidationsInProgressException You have exceeded the maximum number of allowable InProgress
     *         invalidation batch requests, or invalidation objects.</li>
     *         <li>InconsistentQuantitiesException The value of <code>Quantity</code> and the size of <code>Items</code>
     *         don't match.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CloudFrontException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CloudFrontAsyncClient.CreateInvalidation
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/cloudfront-2019-03-26/CreateInvalidation" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateInvalidationResponse> createInvalidation(CreateInvalidationRequest createInvalidationRequest) {
        try {

            HttpResponseHandler<CreateInvalidationResponse> responseHandler = protocolFactory.createResponseHandler(
                    CreateInvalidationResponse::builder, new XmlOperationMetadata().withHasStreamingSuccessResponse(false));

            HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();

            CompletableFuture<CreateInvalidationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateInvalidationRequest, CreateInvalidationResponse>()
                            .withOperationName("CreateInvalidation")
                            .withMarshaller(new CreateInvalidationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(createInvalidationRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Add a new public key to CloudFront to use, for example, for field-level encryption. You can add a maximum of 10
     * public keys with one AWS account.
     * </p>
     *
     * @param createPublicKeyRequest
     * @return A Java Future containing the result of the CreatePublicKey operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>PublicKeyAlreadyExistsException The specified public key already exists.</li>
     *         <li>InvalidArgumentException The argument is invalid.</li>
     *         <li>TooManyPublicKeysException The maximum number of public keys for field-level encryption have been
     *         created. To create a new public key, delete one of the existing keys.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CloudFrontException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CloudFrontAsyncClient.CreatePublicKey
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/cloudfront-2019-03-26/CreatePublicKey" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<CreatePublicKeyResponse> createPublicKey(CreatePublicKeyRequest createPublicKeyRequest) {
        try {

            HttpResponseHandler<CreatePublicKeyResponse> responseHandler = protocolFactory.createResponseHandler(
                    CreatePublicKeyResponse::builder, new XmlOperationMetadata().withHasStreamingSuccessResponse(false));

            HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();

            CompletableFuture<CreatePublicKeyResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreatePublicKeyRequest, CreatePublicKeyResponse>()
                            .withOperationName("CreatePublicKey")
                            .withMarshaller(new CreatePublicKeyRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(createPublicKeyRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a new RTMP distribution. An RTMP distribution is similar to a web distribution, but an RTMP distribution
     * streams media files using the Adobe Real-Time Messaging Protocol (RTMP) instead of serving files using HTTP.
     * </p>
     * <p>
     * To create a new distribution, submit a <code>POST</code> request to the <i>CloudFront API
     * version</i>/distribution resource. The request body must include a document with a
     * <i>StreamingDistributionConfig</i> element. The response echoes the <code>StreamingDistributionConfig</code>
     * element and returns other information about the RTMP distribution.
     * </p>
     * <p>
     * To get the status of your request, use the <i>GET StreamingDistribution</i> API action. When the value of
     * <code>Enabled</code> is <code>true</code> and the value of <code>Status</code> is <code>Deployed</code>, your
     * distribution is ready. A distribution usually deploys in less than 15 minutes.
     * </p>
     * <p>
     * For more information about web distributions, see <a
     * href="https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/distribution-rtmp.html">Working with
     * RTMP Distributions</a> in the <i>Amazon CloudFront Developer Guide</i>.
     * </p>
     * <important>
     * <p>
     * Beginning with the 2012-05-05 version of the CloudFront API, we made substantial changes to the format of the XML
     * document that you include in the request body when you create or update a web distribution or an RTMP
     * distribution, and when you invalidate objects. With previous versions of the API, we discovered that it was too
     * easy to accidentally delete one or more values for an element that accepts multiple values, for example, CNAMEs
     * and trusted signers. Our changes for the 2012-05-05 release are intended to prevent these accidental deletions
     * and to notify you when there's a mismatch between the number of values you say you're specifying in the
     * <code>Quantity</code> element and the number of values specified.
     * </p>
     * </important>
     *
     * @param createStreamingDistributionRequest
     *        The request to create a new streaming distribution.
     * @return A Java Future containing the result of the CreateStreamingDistribution operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>CnameAlreadyExistsException The CNAME specified is already defined for CloudFront.</li>
     *         <li>StreamingDistributionAlreadyExistsException The caller reference you attempted to create the
     *         streaming distribution with is associated with another distribution</li>
     *         <li>InvalidOriginException The Amazon S3 origin server specified does not refer to a valid Amazon S3
     *         bucket.</li>
     *         <li>InvalidOriginAccessIdentityException The origin access identity is not valid or doesn't exist.</li>
     *         <li>AccessDeniedException Access denied.</li>
     *         <li>TooManyTrustedSignersException Your request contains more trusted signers than are allowed per
     *         distribution.</li>
     *         <li>TrustedSignerDoesNotExistException One or more of your trusted signers don't exist.</li>
     *         <li>MissingBodyException This operation requires a body. Ensure that the body is present and the
     *         <code>Content-Type</code> header is set.</li>
     *         <li>TooManyStreamingDistributionCnamEsException Your request contains more CNAMEs than are allowed per
     *         distribution.</li>
     *         <li>TooManyStreamingDistributionsException Processing your request would cause you to exceed the maximum
     *         number of streaming distributions allowed.</li>
     *         <li>InvalidArgumentException The argument is invalid.</li>
     *         <li>InconsistentQuantitiesException The value of <code>Quantity</code> and the size of <code>Items</code>
     *         don't match.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CloudFrontException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CloudFrontAsyncClient.CreateStreamingDistribution
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/cloudfront-2019-03-26/CreateStreamingDistribution"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateStreamingDistributionResponse> createStreamingDistribution(
            CreateStreamingDistributionRequest createStreamingDistributionRequest) {
        try {

            HttpResponseHandler<CreateStreamingDistributionResponse> responseHandler = protocolFactory.createResponseHandler(
                    CreateStreamingDistributionResponse::builder,
                    new XmlOperationMetadata().withHasStreamingSuccessResponse(false));

            HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();

            CompletableFuture<CreateStreamingDistributionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateStreamingDistributionRequest, CreateStreamingDistributionResponse>()
                            .withOperationName("CreateStreamingDistribution")
                            .withMarshaller(new CreateStreamingDistributionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(createStreamingDistributionRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Create a new streaming distribution with tags.
     * </p>
     *
     * @param createStreamingDistributionWithTagsRequest
     *        The request to create a new streaming distribution with tags.
     * @return A Java Future containing the result of the CreateStreamingDistributionWithTags operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>CnameAlreadyExistsException The CNAME specified is already defined for CloudFront.</li>
     *         <li>StreamingDistributionAlreadyExistsException The caller reference you attempted to create the
     *         streaming distribution with is associated with another distribution</li>
     *         <li>InvalidOriginException The Amazon S3 origin server specified does not refer to a valid Amazon S3
     *         bucket.</li>
     *         <li>InvalidOriginAccessIdentityException The origin access identity is not valid or doesn't exist.</li>
     *         <li>AccessDeniedException Access denied.</li>
     *         <li>TooManyTrustedSignersException Your request contains more trusted signers than are allowed per
     *         distribution.</li>
     *         <li>TrustedSignerDoesNotExistException One or more of your trusted signers don't exist.</li>
     *         <li>MissingBodyException This operation requires a body. Ensure that the body is present and the
     *         <code>Content-Type</code> header is set.</li>
     *         <li>TooManyStreamingDistributionCnamEsException Your request contains more CNAMEs than are allowed per
     *         distribution.</li>
     *         <li>TooManyStreamingDistributionsException Processing your request would cause you to exceed the maximum
     *         number of streaming distributions allowed.</li>
     *         <li>InvalidArgumentException The argument is invalid.</li>
     *         <li>InconsistentQuantitiesException The value of <code>Quantity</code> and the size of <code>Items</code>
     *         don't match.</li>
     *         <li>InvalidTaggingException Tagging specified in the response body is not valid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CloudFrontException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CloudFrontAsyncClient.CreateStreamingDistributionWithTags
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/cloudfront-2019-03-26/CreateStreamingDistributionWithTags"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateStreamingDistributionWithTagsResponse> createStreamingDistributionWithTags(
            CreateStreamingDistributionWithTagsRequest createStreamingDistributionWithTagsRequest) {
        try {

            HttpResponseHandler<CreateStreamingDistributionWithTagsResponse> responseHandler = protocolFactory
                    .createResponseHandler(CreateStreamingDistributionWithTagsResponse::builder,
                            new XmlOperationMetadata().withHasStreamingSuccessResponse(false));

            HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();

            CompletableFuture<CreateStreamingDistributionWithTagsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateStreamingDistributionWithTagsRequest, CreateStreamingDistributionWithTagsResponse>()
                            .withOperationName("CreateStreamingDistributionWithTags")
                            .withMarshaller(new CreateStreamingDistributionWithTagsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(createStreamingDistributionWithTagsRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Delete an origin access identity.
     * </p>
     *
     * @param deleteCloudFrontOriginAccessIdentityRequest
     *        Deletes a origin access identity.
     * @return A Java Future containing the result of the DeleteCloudFrontOriginAccessIdentity operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException Access denied.</li>
     *         <li>InvalidIfMatchVersionException The <code>If-Match</code> version is missing or not valid for the
     *         distribution.</li>
     *         <li>NoSuchCloudFrontOriginAccessIdentityException The specified origin access identity does not exist.</li>
     *         <li>PreconditionFailedException The precondition given in one or more of the request-header fields
     *         evaluated to <code>false</code>.</li>
     *         <li>CloudFrontOriginAccessIdentityInUseException The Origin Access Identity specified is already in use.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CloudFrontException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CloudFrontAsyncClient.DeleteCloudFrontOriginAccessIdentity
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/cloudfront-2019-03-26/DeleteCloudFrontOriginAccessIdentity"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteCloudFrontOriginAccessIdentityResponse> deleteCloudFrontOriginAccessIdentity(
            DeleteCloudFrontOriginAccessIdentityRequest deleteCloudFrontOriginAccessIdentityRequest) {
        try {

            HttpResponseHandler<DeleteCloudFrontOriginAccessIdentityResponse> responseHandler = protocolFactory
                    .createResponseHandler(DeleteCloudFrontOriginAccessIdentityResponse::builder,
                            new XmlOperationMetadata().withHasStreamingSuccessResponse(false));

            HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();

            CompletableFuture<DeleteCloudFrontOriginAccessIdentityResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteCloudFrontOriginAccessIdentityRequest, DeleteCloudFrontOriginAccessIdentityResponse>()
                            .withOperationName("DeleteCloudFrontOriginAccessIdentity")
                            .withMarshaller(new DeleteCloudFrontOriginAccessIdentityRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(deleteCloudFrontOriginAccessIdentityRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Delete a distribution.
     * </p>
     *
     * @param deleteDistributionRequest
     *        This action deletes a web distribution. To delete a web distribution using the CloudFront API, perform the
     *        following steps.</p>
     *        <p>
     *        <b>To delete a web distribution using the CloudFront API:</b>
     *        </p>
     *        <ol>
     *        <li>
     *        <p>
     *        Disable the web distribution
     *        </p>
     *        </li>
     *        <li>
     *        <p>
     *        Submit a <code>GET Distribution Config</code> request to get the current configuration and the
     *        <code>Etag</code> header for the distribution.
     *        </p>
     *        </li>
     *        <li>
     *        <p>
     *        Update the XML document that was returned in the response to your <code>GET Distribution Config</code>
     *        request to change the value of <code>Enabled</code> to <code>false</code>.
     *        </p>
     *        </li>
     *        <li>
     *        <p>
     *        Submit a <code>PUT Distribution Config</code> request to update the configuration for your distribution.
     *        In the request body, include the XML document that you updated in Step 3. Set the value of the HTTP
     *        <code>If-Match</code> header to the value of the <code>ETag</code> header that CloudFront returned when
     *        you submitted the <code>GET Distribution Config</code> request in Step 2.
     *        </p>
     *        </li>
     *        <li>
     *        <p>
     *        Review the response to the <code>PUT Distribution Config</code> request to confirm that the distribution
     *        was successfully disabled.
     *        </p>
     *        </li>
     *        <li>
     *        <p>
     *        Submit a <code>GET Distribution</code> request to confirm that your changes have propagated. When
     *        propagation is complete, the value of <code>Status</code> is <code>Deployed</code>.
     *        </p>
     *        </li>
     *        <li>
     *        <p>
     *        Submit a <code>DELETE Distribution</code> request. Set the value of the HTTP <code>If-Match</code> header
     *        to the value of the <code>ETag</code> header that CloudFront returned when you submitted the
     *        <code>GET Distribution Config</code> request in Step 6.
     *        </p>
     *        </li>
     *        <li>
     *        <p>
     *        Review the response to your <code>DELETE Distribution</code> request to confirm that the distribution was
     *        successfully deleted.
     *        </p>
     *        </li>
     *        </ol>
     *        <p>
     *        For information about deleting a distribution using the CloudFront console, see <a
     *        href="https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/HowToDeleteDistribution.html"
     *        >Deleting a Distribution</a> in the <i>Amazon CloudFront Developer Guide</i>.
     * @return A Java Future containing the result of the DeleteDistribution operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException Access denied.</li> <li>DistributionNotDisabledException The specified
     *         CloudFront distribution is not disabled. You must disable the distribution before you can delete it.</li>
     *         <li>InvalidIfMatchVersionException The <code>If-Match</code> version is missing or not valid for the
     *         distribution.</li> <li>NoSuchDistributionException The specified distribution does not exist.</li> <li>
     *         PreconditionFailedException The precondition given in one or more of the request-header fields evaluated
     *         to <code>false</code>.</li> <li>SdkException Base class for all exceptions that can be thrown by the SDK
     *         (both service and client). Can be used for catch all scenarios.</li> <li>SdkClientException If any client
     *         side error occurs such as an IO related failure, failure to get credentials, etc.</li> <li>
     *         CloudFrontException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CloudFrontAsyncClient.DeleteDistribution
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/cloudfront-2019-03-26/DeleteDistribution" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteDistributionResponse> deleteDistribution(DeleteDistributionRequest deleteDistributionRequest) {
        try {

            HttpResponseHandler<DeleteDistributionResponse> responseHandler = protocolFactory.createResponseHandler(
                    DeleteDistributionResponse::builder, new XmlOperationMetadata().withHasStreamingSuccessResponse(false));

            HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();

            CompletableFuture<DeleteDistributionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteDistributionRequest, DeleteDistributionResponse>()
                            .withOperationName("DeleteDistribution")
                            .withMarshaller(new DeleteDistributionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(deleteDistributionRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Remove a field-level encryption configuration.
     * </p>
     *
     * @param deleteFieldLevelEncryptionConfigRequest
     * @return A Java Future containing the result of the DeleteFieldLevelEncryptionConfig operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException Access denied.</li>
     *         <li>InvalidIfMatchVersionException The <code>If-Match</code> version is missing or not valid for the
     *         distribution.</li>
     *         <li>NoSuchFieldLevelEncryptionConfigException The specified configuration for field-level encryption
     *         doesn't exist.</li>
     *         <li>PreconditionFailedException The precondition given in one or more of the request-header fields
     *         evaluated to <code>false</code>.</li>
     *         <li>FieldLevelEncryptionConfigInUseException The specified configuration for field-level encryption is in
     *         use.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CloudFrontException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CloudFrontAsyncClient.DeleteFieldLevelEncryptionConfig
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/cloudfront-2019-03-26/DeleteFieldLevelEncryptionConfig"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteFieldLevelEncryptionConfigResponse> deleteFieldLevelEncryptionConfig(
            DeleteFieldLevelEncryptionConfigRequest deleteFieldLevelEncryptionConfigRequest) {
        try {

            HttpResponseHandler<DeleteFieldLevelEncryptionConfigResponse> responseHandler = protocolFactory
                    .createResponseHandler(DeleteFieldLevelEncryptionConfigResponse::builder,
                            new XmlOperationMetadata().withHasStreamingSuccessResponse(false));

            HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();

            CompletableFuture<DeleteFieldLevelEncryptionConfigResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteFieldLevelEncryptionConfigRequest, DeleteFieldLevelEncryptionConfigResponse>()
                            .withOperationName("DeleteFieldLevelEncryptionConfig")
                            .withMarshaller(new DeleteFieldLevelEncryptionConfigRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(deleteFieldLevelEncryptionConfigRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Remove a field-level encryption profile.
     * </p>
     *
     * @param deleteFieldLevelEncryptionProfileRequest
     * @return A Java Future containing the result of the DeleteFieldLevelEncryptionProfile operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException Access denied.</li>
     *         <li>InvalidIfMatchVersionException The <code>If-Match</code> version is missing or not valid for the
     *         distribution.</li>
     *         <li>NoSuchFieldLevelEncryptionProfileException The specified profile for field-level encryption doesn't
     *         exist.</li>
     *         <li>PreconditionFailedException The precondition given in one or more of the request-header fields
     *         evaluated to <code>false</code>.</li>
     *         <li>FieldLevelEncryptionProfileInUseException The specified profile for field-level encryption is in use.
     *         </li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CloudFrontException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CloudFrontAsyncClient.DeleteFieldLevelEncryptionProfile
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/cloudfront-2019-03-26/DeleteFieldLevelEncryptionProfile"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteFieldLevelEncryptionProfileResponse> deleteFieldLevelEncryptionProfile(
            DeleteFieldLevelEncryptionProfileRequest deleteFieldLevelEncryptionProfileRequest) {
        try {

            HttpResponseHandler<DeleteFieldLevelEncryptionProfileResponse> responseHandler = protocolFactory
                    .createResponseHandler(DeleteFieldLevelEncryptionProfileResponse::builder,
                            new XmlOperationMetadata().withHasStreamingSuccessResponse(false));

            HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();

            CompletableFuture<DeleteFieldLevelEncryptionProfileResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteFieldLevelEncryptionProfileRequest, DeleteFieldLevelEncryptionProfileResponse>()
                            .withOperationName("DeleteFieldLevelEncryptionProfile")
                            .withMarshaller(new DeleteFieldLevelEncryptionProfileRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(deleteFieldLevelEncryptionProfileRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Remove a public key you previously added to CloudFront.
     * </p>
     *
     * @param deletePublicKeyRequest
     * @return A Java Future containing the result of the DeletePublicKey operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException Access denied.</li>
     *         <li>PublicKeyInUseException The specified public key is in use.</li>
     *         <li>InvalidIfMatchVersionException The <code>If-Match</code> version is missing or not valid for the
     *         distribution.</li>
     *         <li>NoSuchPublicKeyException The specified public key doesn't exist.</li>
     *         <li>PreconditionFailedException The precondition given in one or more of the request-header fields
     *         evaluated to <code>false</code>.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CloudFrontException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CloudFrontAsyncClient.DeletePublicKey
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/cloudfront-2019-03-26/DeletePublicKey" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DeletePublicKeyResponse> deletePublicKey(DeletePublicKeyRequest deletePublicKeyRequest) {
        try {

            HttpResponseHandler<DeletePublicKeyResponse> responseHandler = protocolFactory.createResponseHandler(
                    DeletePublicKeyResponse::builder, new XmlOperationMetadata().withHasStreamingSuccessResponse(false));

            HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();

            CompletableFuture<DeletePublicKeyResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeletePublicKeyRequest, DeletePublicKeyResponse>()
                            .withOperationName("DeletePublicKey")
                            .withMarshaller(new DeletePublicKeyRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(deletePublicKeyRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Delete a streaming distribution. To delete an RTMP distribution using the CloudFront API, perform the following
     * steps.
     * </p>
     * <p>
     * <b>To delete an RTMP distribution using the CloudFront API</b>:
     * </p>
     * <ol>
     * <li>
     * <p>
     * Disable the RTMP distribution.
     * </p>
     * </li>
     * <li>
     * <p>
     * Submit a <code>GET Streaming Distribution Config</code> request to get the current configuration and the
     * <code>Etag</code> header for the distribution.
     * </p>
     * </li>
     * <li>
     * <p>
     * Update the XML document that was returned in the response to your <code>GET Streaming Distribution Config</code>
     * request to change the value of <code>Enabled</code> to <code>false</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * Submit a <code>PUT Streaming Distribution Config</code> request to update the configuration for your
     * distribution. In the request body, include the XML document that you updated in Step 3. Then set the value of the
     * HTTP <code>If-Match</code> header to the value of the <code>ETag</code> header that CloudFront returned when you
     * submitted the <code>GET Streaming Distribution Config</code> request in Step 2.
     * </p>
     * </li>
     * <li>
     * <p>
     * Review the response to the <code>PUT Streaming Distribution Config</code> request to confirm that the
     * distribution was successfully disabled.
     * </p>
     * </li>
     * <li>
     * <p>
     * Submit a <code>GET Streaming Distribution Config</code> request to confirm that your changes have propagated.
     * When propagation is complete, the value of <code>Status</code> is <code>Deployed</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * Submit a <code>DELETE Streaming Distribution</code> request. Set the value of the HTTP <code>If-Match</code>
     * header to the value of the <code>ETag</code> header that CloudFront returned when you submitted the
     * <code>GET Streaming Distribution Config</code> request in Step 2.
     * </p>
     * </li>
     * <li>
     * <p>
     * Review the response to your <code>DELETE Streaming Distribution</code> request to confirm that the distribution
     * was successfully deleted.
     * </p>
     * </li>
     * </ol>
     * <p>
     * For information about deleting a distribution using the CloudFront console, see <a
     * href="https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/HowToDeleteDistribution.html">Deleting a
     * Distribution</a> in the <i>Amazon CloudFront Developer Guide</i>.
     * </p>
     *
     * @param deleteStreamingDistributionRequest
     *        The request to delete a streaming distribution.
     * @return A Java Future containing the result of the DeleteStreamingDistribution operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException Access denied.</li>
     *         <li>StreamingDistributionNotDisabledException The specified CloudFront distribution is not disabled. You
     *         must disable the distribution before you can delete it.</li>
     *         <li>InvalidIfMatchVersionException The <code>If-Match</code> version is missing or not valid for the
     *         distribution.</li>
     *         <li>NoSuchStreamingDistributionException The specified streaming distribution does not exist.</li>
     *         <li>PreconditionFailedException The precondition given in one or more of the request-header fields
     *         evaluated to <code>false</code>.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CloudFrontException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CloudFrontAsyncClient.DeleteStreamingDistribution
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/cloudfront-2019-03-26/DeleteStreamingDistribution"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteStreamingDistributionResponse> deleteStreamingDistribution(
            DeleteStreamingDistributionRequest deleteStreamingDistributionRequest) {
        try {

            HttpResponseHandler<DeleteStreamingDistributionResponse> responseHandler = protocolFactory.createResponseHandler(
                    DeleteStreamingDistributionResponse::builder,
                    new XmlOperationMetadata().withHasStreamingSuccessResponse(false));

            HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();

            CompletableFuture<DeleteStreamingDistributionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteStreamingDistributionRequest, DeleteStreamingDistributionResponse>()
                            .withOperationName("DeleteStreamingDistribution")
                            .withMarshaller(new DeleteStreamingDistributionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(deleteStreamingDistributionRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Get the information about an origin access identity.
     * </p>
     *
     * @param getCloudFrontOriginAccessIdentityRequest
     *        The request to get an origin access identity's information.
     * @return A Java Future containing the result of the GetCloudFrontOriginAccessIdentity operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>NoSuchCloudFrontOriginAccessIdentityException The specified origin access identity does not exist.</li>
     *         <li>AccessDeniedException Access denied.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CloudFrontException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CloudFrontAsyncClient.GetCloudFrontOriginAccessIdentity
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/cloudfront-2019-03-26/GetCloudFrontOriginAccessIdentity"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetCloudFrontOriginAccessIdentityResponse> getCloudFrontOriginAccessIdentity(
            GetCloudFrontOriginAccessIdentityRequest getCloudFrontOriginAccessIdentityRequest) {
        try {

            HttpResponseHandler<GetCloudFrontOriginAccessIdentityResponse> responseHandler = protocolFactory
                    .createResponseHandler(GetCloudFrontOriginAccessIdentityResponse::builder,
                            new XmlOperationMetadata().withHasStreamingSuccessResponse(false));

            HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();

            CompletableFuture<GetCloudFrontOriginAccessIdentityResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetCloudFrontOriginAccessIdentityRequest, GetCloudFrontOriginAccessIdentityResponse>()
                            .withOperationName("GetCloudFrontOriginAccessIdentity")
                            .withMarshaller(new GetCloudFrontOriginAccessIdentityRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(getCloudFrontOriginAccessIdentityRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Get the configuration information about an origin access identity.
     * </p>
     *
     * @param getCloudFrontOriginAccessIdentityConfigRequest
     *        The origin access identity's configuration information. For more information, see <a href=
     *        "https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_CloudFrontOriginAccessIdentityConfig.html"
     *        >CloudFrontOriginAccessIdentityConfig</a>.
     * @return A Java Future containing the result of the GetCloudFrontOriginAccessIdentityConfig operation returned by
     *         the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>NoSuchCloudFrontOriginAccessIdentityException The specified origin access identity does not exist.</li>
     *         <li>AccessDeniedException Access denied.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CloudFrontException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CloudFrontAsyncClient.GetCloudFrontOriginAccessIdentityConfig
     * @see <a
     *      href="http://docs.aws.amazon.com/goto/WebAPI/cloudfront-2019-03-26/GetCloudFrontOriginAccessIdentityConfig"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetCloudFrontOriginAccessIdentityConfigResponse> getCloudFrontOriginAccessIdentityConfig(
            GetCloudFrontOriginAccessIdentityConfigRequest getCloudFrontOriginAccessIdentityConfigRequest) {
        try {

            HttpResponseHandler<GetCloudFrontOriginAccessIdentityConfigResponse> responseHandler = protocolFactory
                    .createResponseHandler(GetCloudFrontOriginAccessIdentityConfigResponse::builder,
                            new XmlOperationMetadata().withHasStreamingSuccessResponse(false));

            HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();

            CompletableFuture<GetCloudFrontOriginAccessIdentityConfigResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetCloudFrontOriginAccessIdentityConfigRequest, GetCloudFrontOriginAccessIdentityConfigResponse>()
                            .withOperationName("GetCloudFrontOriginAccessIdentityConfig")
                            .withMarshaller(new GetCloudFrontOriginAccessIdentityConfigRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(getCloudFrontOriginAccessIdentityConfigRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Get the information about a distribution.
     * </p>
     *
     * @param getDistributionRequest
     *        The request to get a distribution's information.
     * @return A Java Future containing the result of the GetDistribution operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>NoSuchDistributionException The specified distribution does not exist.</li>
     *         <li>AccessDeniedException Access denied.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CloudFrontException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CloudFrontAsyncClient.GetDistribution
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/cloudfront-2019-03-26/GetDistribution" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetDistributionResponse> getDistribution(GetDistributionRequest getDistributionRequest) {
        try {

            HttpResponseHandler<GetDistributionResponse> responseHandler = protocolFactory.createResponseHandler(
                    GetDistributionResponse::builder, new XmlOperationMetadata().withHasStreamingSuccessResponse(false));

            HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();

            CompletableFuture<GetDistributionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetDistributionRequest, GetDistributionResponse>()
                            .withOperationName("GetDistribution")
                            .withMarshaller(new GetDistributionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(getDistributionRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Get the configuration information about a distribution.
     * </p>
     *
     * @param getDistributionConfigRequest
     *        The request to get a distribution configuration.
     * @return A Java Future containing the result of the GetDistributionConfig operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>NoSuchDistributionException The specified distribution does not exist.</li>
     *         <li>AccessDeniedException Access denied.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CloudFrontException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CloudFrontAsyncClient.GetDistributionConfig
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/cloudfront-2019-03-26/GetDistributionConfig"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetDistributionConfigResponse> getDistributionConfig(
            GetDistributionConfigRequest getDistributionConfigRequest) {
        try {

            HttpResponseHandler<GetDistributionConfigResponse> responseHandler = protocolFactory.createResponseHandler(
                    GetDistributionConfigResponse::builder, new XmlOperationMetadata().withHasStreamingSuccessResponse(false));

            HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();

            CompletableFuture<GetDistributionConfigResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetDistributionConfigRequest, GetDistributionConfigResponse>()
                            .withOperationName("GetDistributionConfig")
                            .withMarshaller(new GetDistributionConfigRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(getDistributionConfigRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Get the field-level encryption configuration information.
     * </p>
     *
     * @param getFieldLevelEncryptionRequest
     * @return A Java Future containing the result of the GetFieldLevelEncryption operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException Access denied.</li>
     *         <li>NoSuchFieldLevelEncryptionConfigException The specified configuration for field-level encryption
     *         doesn't exist.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CloudFrontException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CloudFrontAsyncClient.GetFieldLevelEncryption
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/cloudfront-2019-03-26/GetFieldLevelEncryption"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetFieldLevelEncryptionResponse> getFieldLevelEncryption(
            GetFieldLevelEncryptionRequest getFieldLevelEncryptionRequest) {
        try {

            HttpResponseHandler<GetFieldLevelEncryptionResponse> responseHandler = protocolFactory.createResponseHandler(
                    GetFieldLevelEncryptionResponse::builder, new XmlOperationMetadata().withHasStreamingSuccessResponse(false));

            HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();

            CompletableFuture<GetFieldLevelEncryptionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetFieldLevelEncryptionRequest, GetFieldLevelEncryptionResponse>()
                            .withOperationName("GetFieldLevelEncryption")
                            .withMarshaller(new GetFieldLevelEncryptionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(getFieldLevelEncryptionRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Get the field-level encryption configuration information.
     * </p>
     *
     * @param getFieldLevelEncryptionConfigRequest
     * @return A Java Future containing the result of the GetFieldLevelEncryptionConfig operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException Access denied.</li>
     *         <li>NoSuchFieldLevelEncryptionConfigException The specified configuration for field-level encryption
     *         doesn't exist.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CloudFrontException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CloudFrontAsyncClient.GetFieldLevelEncryptionConfig
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/cloudfront-2019-03-26/GetFieldLevelEncryptionConfig"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetFieldLevelEncryptionConfigResponse> getFieldLevelEncryptionConfig(
            GetFieldLevelEncryptionConfigRequest getFieldLevelEncryptionConfigRequest) {
        try {

            HttpResponseHandler<GetFieldLevelEncryptionConfigResponse> responseHandler = protocolFactory.createResponseHandler(
                    GetFieldLevelEncryptionConfigResponse::builder,
                    new XmlOperationMetadata().withHasStreamingSuccessResponse(false));

            HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();

            CompletableFuture<GetFieldLevelEncryptionConfigResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetFieldLevelEncryptionConfigRequest, GetFieldLevelEncryptionConfigResponse>()
                            .withOperationName("GetFieldLevelEncryptionConfig")
                            .withMarshaller(new GetFieldLevelEncryptionConfigRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(getFieldLevelEncryptionConfigRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Get the field-level encryption profile information.
     * </p>
     *
     * @param getFieldLevelEncryptionProfileRequest
     * @return A Java Future containing the result of the GetFieldLevelEncryptionProfile operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException Access denied.</li>
     *         <li>NoSuchFieldLevelEncryptionProfileException The specified profile for field-level encryption doesn't
     *         exist.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CloudFrontException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CloudFrontAsyncClient.GetFieldLevelEncryptionProfile
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/cloudfront-2019-03-26/GetFieldLevelEncryptionProfile"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetFieldLevelEncryptionProfileResponse> getFieldLevelEncryptionProfile(
            GetFieldLevelEncryptionProfileRequest getFieldLevelEncryptionProfileRequest) {
        try {

            HttpResponseHandler<GetFieldLevelEncryptionProfileResponse> responseHandler = protocolFactory.createResponseHandler(
                    GetFieldLevelEncryptionProfileResponse::builder,
                    new XmlOperationMetadata().withHasStreamingSuccessResponse(false));

            HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();

            CompletableFuture<GetFieldLevelEncryptionProfileResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetFieldLevelEncryptionProfileRequest, GetFieldLevelEncryptionProfileResponse>()
                            .withOperationName("GetFieldLevelEncryptionProfile")
                            .withMarshaller(new GetFieldLevelEncryptionProfileRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(getFieldLevelEncryptionProfileRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Get the field-level encryption profile configuration information.
     * </p>
     *
     * @param getFieldLevelEncryptionProfileConfigRequest
     * @return A Java Future containing the result of the GetFieldLevelEncryptionProfileConfig operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException Access denied.</li>
     *         <li>NoSuchFieldLevelEncryptionProfileException The specified profile for field-level encryption doesn't
     *         exist.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CloudFrontException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CloudFrontAsyncClient.GetFieldLevelEncryptionProfileConfig
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/cloudfront-2019-03-26/GetFieldLevelEncryptionProfileConfig"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetFieldLevelEncryptionProfileConfigResponse> getFieldLevelEncryptionProfileConfig(
            GetFieldLevelEncryptionProfileConfigRequest getFieldLevelEncryptionProfileConfigRequest) {
        try {

            HttpResponseHandler<GetFieldLevelEncryptionProfileConfigResponse> responseHandler = protocolFactory
                    .createResponseHandler(GetFieldLevelEncryptionProfileConfigResponse::builder,
                            new XmlOperationMetadata().withHasStreamingSuccessResponse(false));

            HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();

            CompletableFuture<GetFieldLevelEncryptionProfileConfigResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetFieldLevelEncryptionProfileConfigRequest, GetFieldLevelEncryptionProfileConfigResponse>()
                            .withOperationName("GetFieldLevelEncryptionProfileConfig")
                            .withMarshaller(new GetFieldLevelEncryptionProfileConfigRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(getFieldLevelEncryptionProfileConfigRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Get the information about an invalidation.
     * </p>
     *
     * @param getInvalidationRequest
     *        The request to get an invalidation's information.
     * @return A Java Future containing the result of the GetInvalidation operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>NoSuchInvalidationException The specified invalidation does not exist.</li>
     *         <li>NoSuchDistributionException The specified distribution does not exist.</li>
     *         <li>AccessDeniedException Access denied.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CloudFrontException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CloudFrontAsyncClient.GetInvalidation
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/cloudfront-2019-03-26/GetInvalidation" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetInvalidationResponse> getInvalidation(GetInvalidationRequest getInvalidationRequest) {
        try {

            HttpResponseHandler<GetInvalidationResponse> responseHandler = protocolFactory.createResponseHandler(
                    GetInvalidationResponse::builder, new XmlOperationMetadata().withHasStreamingSuccessResponse(false));

            HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();

            CompletableFuture<GetInvalidationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetInvalidationRequest, GetInvalidationResponse>()
                            .withOperationName("GetInvalidation")
                            .withMarshaller(new GetInvalidationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(getInvalidationRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Get the public key information.
     * </p>
     *
     * @param getPublicKeyRequest
     * @return A Java Future containing the result of the GetPublicKey operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException Access denied.</li>
     *         <li>NoSuchPublicKeyException The specified public key doesn't exist.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CloudFrontException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CloudFrontAsyncClient.GetPublicKey
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/cloudfront-2019-03-26/GetPublicKey" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetPublicKeyResponse> getPublicKey(GetPublicKeyRequest getPublicKeyRequest) {
        try {

            HttpResponseHandler<GetPublicKeyResponse> responseHandler = protocolFactory.createResponseHandler(
                    GetPublicKeyResponse::builder, new XmlOperationMetadata().withHasStreamingSuccessResponse(false));

            HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();

            CompletableFuture<GetPublicKeyResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetPublicKeyRequest, GetPublicKeyResponse>()
                            .withOperationName("GetPublicKey").withMarshaller(new GetPublicKeyRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(getPublicKeyRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Return public key configuration informaation
     * </p>
     *
     * @param getPublicKeyConfigRequest
     * @return A Java Future containing the result of the GetPublicKeyConfig operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException Access denied.</li>
     *         <li>NoSuchPublicKeyException The specified public key doesn't exist.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CloudFrontException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CloudFrontAsyncClient.GetPublicKeyConfig
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/cloudfront-2019-03-26/GetPublicKeyConfig" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<GetPublicKeyConfigResponse> getPublicKeyConfig(GetPublicKeyConfigRequest getPublicKeyConfigRequest) {
        try {

            HttpResponseHandler<GetPublicKeyConfigResponse> responseHandler = protocolFactory.createResponseHandler(
                    GetPublicKeyConfigResponse::builder, new XmlOperationMetadata().withHasStreamingSuccessResponse(false));

            HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();

            CompletableFuture<GetPublicKeyConfigResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetPublicKeyConfigRequest, GetPublicKeyConfigResponse>()
                            .withOperationName("GetPublicKeyConfig")
                            .withMarshaller(new GetPublicKeyConfigRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(getPublicKeyConfigRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Gets information about a specified RTMP distribution, including the distribution configuration.
     * </p>
     *
     * @param getStreamingDistributionRequest
     *        The request to get a streaming distribution's information.
     * @return A Java Future containing the result of the GetStreamingDistribution operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>NoSuchStreamingDistributionException The specified streaming distribution does not exist.</li>
     *         <li>AccessDeniedException Access denied.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CloudFrontException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CloudFrontAsyncClient.GetStreamingDistribution
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/cloudfront-2019-03-26/GetStreamingDistribution"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetStreamingDistributionResponse> getStreamingDistribution(
            GetStreamingDistributionRequest getStreamingDistributionRequest) {
        try {

            HttpResponseHandler<GetStreamingDistributionResponse> responseHandler = protocolFactory.createResponseHandler(
                    GetStreamingDistributionResponse::builder, new XmlOperationMetadata().withHasStreamingSuccessResponse(false));

            HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();

            CompletableFuture<GetStreamingDistributionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetStreamingDistributionRequest, GetStreamingDistributionResponse>()
                            .withOperationName("GetStreamingDistribution")
                            .withMarshaller(new GetStreamingDistributionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(getStreamingDistributionRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Get the configuration information about a streaming distribution.
     * </p>
     *
     * @param getStreamingDistributionConfigRequest
     *        To request to get a streaming distribution configuration.
     * @return A Java Future containing the result of the GetStreamingDistributionConfig operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>NoSuchStreamingDistributionException The specified streaming distribution does not exist.</li>
     *         <li>AccessDeniedException Access denied.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CloudFrontException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CloudFrontAsyncClient.GetStreamingDistributionConfig
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/cloudfront-2019-03-26/GetStreamingDistributionConfig"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetStreamingDistributionConfigResponse> getStreamingDistributionConfig(
            GetStreamingDistributionConfigRequest getStreamingDistributionConfigRequest) {
        try {

            HttpResponseHandler<GetStreamingDistributionConfigResponse> responseHandler = protocolFactory.createResponseHandler(
                    GetStreamingDistributionConfigResponse::builder,
                    new XmlOperationMetadata().withHasStreamingSuccessResponse(false));

            HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();

            CompletableFuture<GetStreamingDistributionConfigResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetStreamingDistributionConfigRequest, GetStreamingDistributionConfigResponse>()
                            .withOperationName("GetStreamingDistributionConfig")
                            .withMarshaller(new GetStreamingDistributionConfigRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(getStreamingDistributionConfigRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists origin access identities.
     * </p>
     *
     * @param listCloudFrontOriginAccessIdentitiesRequest
     *        The request to list origin access identities.
     * @return A Java Future containing the result of the ListCloudFrontOriginAccessIdentities operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidArgumentException The argument is invalid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CloudFrontException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CloudFrontAsyncClient.ListCloudFrontOriginAccessIdentities
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/cloudfront-2019-03-26/ListCloudFrontOriginAccessIdentities"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListCloudFrontOriginAccessIdentitiesResponse> listCloudFrontOriginAccessIdentities(
            ListCloudFrontOriginAccessIdentitiesRequest listCloudFrontOriginAccessIdentitiesRequest) {
        try {

            HttpResponseHandler<ListCloudFrontOriginAccessIdentitiesResponse> responseHandler = protocolFactory
                    .createResponseHandler(ListCloudFrontOriginAccessIdentitiesResponse::builder,
                            new XmlOperationMetadata().withHasStreamingSuccessResponse(false));

            HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();

            CompletableFuture<ListCloudFrontOriginAccessIdentitiesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListCloudFrontOriginAccessIdentitiesRequest, ListCloudFrontOriginAccessIdentitiesResponse>()
                            .withOperationName("ListCloudFrontOriginAccessIdentities")
                            .withMarshaller(new ListCloudFrontOriginAccessIdentitiesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(listCloudFrontOriginAccessIdentitiesRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * List CloudFront distributions.
     * </p>
     *
     * @param listDistributionsRequest
     *        The request to list your distributions.
     * @return A Java Future containing the result of the ListDistributions operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidArgumentException The argument is invalid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CloudFrontException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CloudFrontAsyncClient.ListDistributions
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/cloudfront-2019-03-26/ListDistributions" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<ListDistributionsResponse> listDistributions(ListDistributionsRequest listDistributionsRequest) {
        try {

            HttpResponseHandler<ListDistributionsResponse> responseHandler = protocolFactory.createResponseHandler(
                    ListDistributionsResponse::builder, new XmlOperationMetadata().withHasStreamingSuccessResponse(false));

            HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();

            CompletableFuture<ListDistributionsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListDistributionsRequest, ListDistributionsResponse>()
                            .withOperationName("ListDistributions")
                            .withMarshaller(new ListDistributionsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(listDistributionsRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * List the distributions that are associated with a specified AWS WAF web ACL.
     * </p>
     *
     * @param listDistributionsByWebAclIdRequest
     *        The request to list distributions that are associated with a specified AWS WAF web ACL.
     * @return A Java Future containing the result of the ListDistributionsByWebACLId operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidArgumentException The argument is invalid.</li>
     *         <li>InvalidWebAclIdException A web ACL id specified in the response body is not valid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CloudFrontException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CloudFrontAsyncClient.ListDistributionsByWebACLId
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/cloudfront-2019-03-26/ListDistributionsByWebACLId"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListDistributionsByWebAclIdResponse> listDistributionsByWebACLId(
            ListDistributionsByWebAclIdRequest listDistributionsByWebAclIdRequest) {
        try {

            HttpResponseHandler<ListDistributionsByWebAclIdResponse> responseHandler = protocolFactory.createResponseHandler(
                    ListDistributionsByWebAclIdResponse::builder,
                    new XmlOperationMetadata().withHasStreamingSuccessResponse(false));

            HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();

            CompletableFuture<ListDistributionsByWebAclIdResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListDistributionsByWebAclIdRequest, ListDistributionsByWebAclIdResponse>()
                            .withOperationName("ListDistributionsByWebACLId")
                            .withMarshaller(new ListDistributionsByWebAclIdRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(listDistributionsByWebAclIdRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * List all field-level encryption configurations that have been created in CloudFront for this account.
     * </p>
     *
     * @param listFieldLevelEncryptionConfigsRequest
     * @return A Java Future containing the result of the ListFieldLevelEncryptionConfigs operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidArgumentException The argument is invalid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CloudFrontException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CloudFrontAsyncClient.ListFieldLevelEncryptionConfigs
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/cloudfront-2019-03-26/ListFieldLevelEncryptionConfigs"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListFieldLevelEncryptionConfigsResponse> listFieldLevelEncryptionConfigs(
            ListFieldLevelEncryptionConfigsRequest listFieldLevelEncryptionConfigsRequest) {
        try {

            HttpResponseHandler<ListFieldLevelEncryptionConfigsResponse> responseHandler = protocolFactory.createResponseHandler(
                    ListFieldLevelEncryptionConfigsResponse::builder,
                    new XmlOperationMetadata().withHasStreamingSuccessResponse(false));

            HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();

            CompletableFuture<ListFieldLevelEncryptionConfigsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListFieldLevelEncryptionConfigsRequest, ListFieldLevelEncryptionConfigsResponse>()
                            .withOperationName("ListFieldLevelEncryptionConfigs")
                            .withMarshaller(new ListFieldLevelEncryptionConfigsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(listFieldLevelEncryptionConfigsRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Request a list of field-level encryption profiles that have been created in CloudFront for this account.
     * </p>
     *
     * @param listFieldLevelEncryptionProfilesRequest
     * @return A Java Future containing the result of the ListFieldLevelEncryptionProfiles operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidArgumentException The argument is invalid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CloudFrontException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CloudFrontAsyncClient.ListFieldLevelEncryptionProfiles
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/cloudfront-2019-03-26/ListFieldLevelEncryptionProfiles"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListFieldLevelEncryptionProfilesResponse> listFieldLevelEncryptionProfiles(
            ListFieldLevelEncryptionProfilesRequest listFieldLevelEncryptionProfilesRequest) {
        try {

            HttpResponseHandler<ListFieldLevelEncryptionProfilesResponse> responseHandler = protocolFactory
                    .createResponseHandler(ListFieldLevelEncryptionProfilesResponse::builder,
                            new XmlOperationMetadata().withHasStreamingSuccessResponse(false));

            HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();

            CompletableFuture<ListFieldLevelEncryptionProfilesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListFieldLevelEncryptionProfilesRequest, ListFieldLevelEncryptionProfilesResponse>()
                            .withOperationName("ListFieldLevelEncryptionProfiles")
                            .withMarshaller(new ListFieldLevelEncryptionProfilesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(listFieldLevelEncryptionProfilesRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists invalidation batches.
     * </p>
     *
     * @param listInvalidationsRequest
     *        The request to list invalidations.
     * @return A Java Future containing the result of the ListInvalidations operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidArgumentException The argument is invalid.</li>
     *         <li>NoSuchDistributionException The specified distribution does not exist.</li>
     *         <li>AccessDeniedException Access denied.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CloudFrontException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CloudFrontAsyncClient.ListInvalidations
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/cloudfront-2019-03-26/ListInvalidations" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<ListInvalidationsResponse> listInvalidations(ListInvalidationsRequest listInvalidationsRequest) {
        try {

            HttpResponseHandler<ListInvalidationsResponse> responseHandler = protocolFactory.createResponseHandler(
                    ListInvalidationsResponse::builder, new XmlOperationMetadata().withHasStreamingSuccessResponse(false));

            HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();

            CompletableFuture<ListInvalidationsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListInvalidationsRequest, ListInvalidationsResponse>()
                            .withOperationName("ListInvalidations")
                            .withMarshaller(new ListInvalidationsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(listInvalidationsRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * List all public keys that have been added to CloudFront for this account.
     * </p>
     *
     * @param listPublicKeysRequest
     * @return A Java Future containing the result of the ListPublicKeys operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidArgumentException The argument is invalid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CloudFrontException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CloudFrontAsyncClient.ListPublicKeys
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/cloudfront-2019-03-26/ListPublicKeys" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListPublicKeysResponse> listPublicKeys(ListPublicKeysRequest listPublicKeysRequest) {
        try {

            HttpResponseHandler<ListPublicKeysResponse> responseHandler = protocolFactory.createResponseHandler(
                    ListPublicKeysResponse::builder, new XmlOperationMetadata().withHasStreamingSuccessResponse(false));

            HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();

            CompletableFuture<ListPublicKeysResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListPublicKeysRequest, ListPublicKeysResponse>()
                            .withOperationName("ListPublicKeys")
                            .withMarshaller(new ListPublicKeysRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(listPublicKeysRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * List streaming distributions.
     * </p>
     *
     * @param listStreamingDistributionsRequest
     *        The request to list your streaming distributions.
     * @return A Java Future containing the result of the ListStreamingDistributions operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidArgumentException The argument is invalid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CloudFrontException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CloudFrontAsyncClient.ListStreamingDistributions
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/cloudfront-2019-03-26/ListStreamingDistributions"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListStreamingDistributionsResponse> listStreamingDistributions(
            ListStreamingDistributionsRequest listStreamingDistributionsRequest) {
        try {

            HttpResponseHandler<ListStreamingDistributionsResponse> responseHandler = protocolFactory.createResponseHandler(
                    ListStreamingDistributionsResponse::builder,
                    new XmlOperationMetadata().withHasStreamingSuccessResponse(false));

            HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();

            CompletableFuture<ListStreamingDistributionsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListStreamingDistributionsRequest, ListStreamingDistributionsResponse>()
                            .withOperationName("ListStreamingDistributions")
                            .withMarshaller(new ListStreamingDistributionsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(listStreamingDistributionsRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * List tags for a CloudFront resource.
     * </p>
     *
     * @param listTagsForResourceRequest
     *        The request to list tags for a CloudFront resource.
     * @return A Java Future containing the result of the ListTagsForResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException Access denied.</li>
     *         <li>InvalidArgumentException The argument is invalid.</li>
     *         <li>InvalidTaggingException Tagging specified in the response body is not valid.</li>
     *         <li>NoSuchResourceException A resource that was specified is not valid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CloudFrontException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CloudFrontAsyncClient.ListTagsForResource
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/cloudfront-2019-03-26/ListTagsForResource" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<ListTagsForResourceResponse> listTagsForResource(
            ListTagsForResourceRequest listTagsForResourceRequest) {
        try {

            HttpResponseHandler<ListTagsForResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                    ListTagsForResourceResponse::builder, new XmlOperationMetadata().withHasStreamingSuccessResponse(false));

            HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();

            CompletableFuture<ListTagsForResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListTagsForResourceRequest, ListTagsForResourceResponse>()
                            .withOperationName("ListTagsForResource")
                            .withMarshaller(new ListTagsForResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(listTagsForResourceRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Add tags to a CloudFront resource.
     * </p>
     *
     * @param tagResourceRequest
     *        The request to add tags to a CloudFront resource.
     * @return A Java Future containing the result of the TagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException Access denied.</li>
     *         <li>InvalidArgumentException The argument is invalid.</li>
     *         <li>InvalidTaggingException Tagging specified in the response body is not valid.</li>
     *         <li>NoSuchResourceException A resource that was specified is not valid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CloudFrontException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CloudFrontAsyncClient.TagResource
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/cloudfront-2019-03-26/TagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<TagResourceResponse> tagResource(TagResourceRequest tagResourceRequest) {
        try {

            HttpResponseHandler<TagResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                    TagResourceResponse::builder, new XmlOperationMetadata().withHasStreamingSuccessResponse(false));

            HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();

            CompletableFuture<TagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<TagResourceRequest, TagResourceResponse>()
                            .withOperationName("TagResource").withMarshaller(new TagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(tagResourceRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Remove tags from a CloudFront resource.
     * </p>
     *
     * @param untagResourceRequest
     *        The request to remove tags from a CloudFront resource.
     * @return A Java Future containing the result of the UntagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException Access denied.</li>
     *         <li>InvalidArgumentException The argument is invalid.</li>
     *         <li>InvalidTaggingException Tagging specified in the response body is not valid.</li>
     *         <li>NoSuchResourceException A resource that was specified is not valid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CloudFrontException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CloudFrontAsyncClient.UntagResource
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/cloudfront-2019-03-26/UntagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<UntagResourceResponse> untagResource(UntagResourceRequest untagResourceRequest) {
        try {

            HttpResponseHandler<UntagResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                    UntagResourceResponse::builder, new XmlOperationMetadata().withHasStreamingSuccessResponse(false));

            HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();

            CompletableFuture<UntagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UntagResourceRequest, UntagResourceResponse>()
                            .withOperationName("UntagResource")
                            .withMarshaller(new UntagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(untagResourceRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Update an origin access identity.
     * </p>
     *
     * @param updateCloudFrontOriginAccessIdentityRequest
     *        The request to update an origin access identity.
     * @return A Java Future containing the result of the UpdateCloudFrontOriginAccessIdentity operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException Access denied.</li>
     *         <li>IllegalUpdateException Origin and <code>CallerReference</code> cannot be updated.</li>
     *         <li>InvalidIfMatchVersionException The <code>If-Match</code> version is missing or not valid for the
     *         distribution.</li>
     *         <li>MissingBodyException This operation requires a body. Ensure that the body is present and the
     *         <code>Content-Type</code> header is set.</li>
     *         <li>NoSuchCloudFrontOriginAccessIdentityException The specified origin access identity does not exist.</li>
     *         <li>PreconditionFailedException The precondition given in one or more of the request-header fields
     *         evaluated to <code>false</code>.</li>
     *         <li>InvalidArgumentException The argument is invalid.</li>
     *         <li>InconsistentQuantitiesException The value of <code>Quantity</code> and the size of <code>Items</code>
     *         don't match.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CloudFrontException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CloudFrontAsyncClient.UpdateCloudFrontOriginAccessIdentity
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/cloudfront-2019-03-26/UpdateCloudFrontOriginAccessIdentity"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateCloudFrontOriginAccessIdentityResponse> updateCloudFrontOriginAccessIdentity(
            UpdateCloudFrontOriginAccessIdentityRequest updateCloudFrontOriginAccessIdentityRequest) {
        try {

            HttpResponseHandler<UpdateCloudFrontOriginAccessIdentityResponse> responseHandler = protocolFactory
                    .createResponseHandler(UpdateCloudFrontOriginAccessIdentityResponse::builder,
                            new XmlOperationMetadata().withHasStreamingSuccessResponse(false));

            HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();

            CompletableFuture<UpdateCloudFrontOriginAccessIdentityResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateCloudFrontOriginAccessIdentityRequest, UpdateCloudFrontOriginAccessIdentityResponse>()
                            .withOperationName("UpdateCloudFrontOriginAccessIdentity")
                            .withMarshaller(new UpdateCloudFrontOriginAccessIdentityRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(updateCloudFrontOriginAccessIdentityRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates the configuration for a web distribution.
     * </p>
     * <important>
     * <p>
     * When you update a distribution, there are more required fields than when you create a distribution. When you
     * update your distribution by using this API action, follow the steps here to get the current configuration and
     * then make your updates, to make sure that you include all of the required fields. To view a summary, see <a href=
     * "https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/distribution-overview-required-fields.html"
     * >Required Fields for Create Distribution and Update Distribution</a> in the <i>Amazon CloudFront Developer
     * Guide</i>.
     * </p>
     * </important>
     * <p>
     * The update process includes getting the current distribution configuration, updating the XML document that is
     * returned to make your changes, and then submitting an <code>UpdateDistribution</code> request to make the
     * updates.
     * </p>
     * <p>
     * For information about updating a distribution using the CloudFront console instead, see <a href=
     * "https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/distribution-web-creating-console.html"
     * >Creating a Distribution</a> in the <i>Amazon CloudFront Developer Guide</i>.
     * </p>
     * <p>
     * <b>To update a web distribution using the CloudFront API</b>
     * </p>
     * <ol>
     * <li>
     * <p>
     * Submit a <a href="https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_GetDistributionConfig.html">
     * GetDistributionConfig</a> request to get the current configuration and an <code>Etag</code> header for the
     * distribution.
     * </p>
     * <note>
     * <p>
     * If you update the distribution again, you must get a new <code>Etag</code> header.
     * </p>
     * </note></li>
     * <li>
     * <p>
     * Update the XML document that was returned in the response to your <code>GetDistributionConfig</code> request to
     * include your changes.
     * </p>
     * <important>
     * <p>
     * When you edit the XML file, be aware of the following:
     * </p>
     * <ul>
     * <li>
     * <p>
     * You must strip out the ETag parameter that is returned.
     * </p>
     * </li>
     * <li>
     * <p>
     * Additional fields are required when you update a distribution. There may be fields included in the XML file for
     * features that you haven't configured for your distribution. This is expected and required to successfully update
     * the distribution.
     * </p>
     * </li>
     * <li>
     * <p>
     * You can't change the value of <code>CallerReference</code>. If you try to change this value, CloudFront returns
     * an <code>IllegalUpdate</code> error.
     * </p>
     * </li>
     * <li>
     * <p>
     * The new configuration replaces the existing configuration; the values that you specify in an
     * <code>UpdateDistribution</code> request are not merged into your existing configuration. When you add, delete, or
     * replace values in an element that allows multiple values (for example, <code>CNAME</code>), you must specify all
     * of the values that you want to appear in the updated distribution. In addition, you must update the corresponding
     * <code>Quantity</code> element.
     * </p>
     * </li>
     * </ul>
     * </important></li>
     * <li>
     * <p>
     * Submit an <code>UpdateDistribution</code> request to update the configuration for your distribution:
     * </p>
     * <ul>
     * <li>
     * <p>
     * In the request body, include the XML document that you updated in Step 2. The request body must include an XML
     * document with a <code>DistributionConfig</code> element.
     * </p>
     * </li>
     * <li>
     * <p>
     * Set the value of the HTTP <code>If-Match</code> header to the value of the <code>ETag</code> header that
     * CloudFront returned when you submitted the <code>GetDistributionConfig</code> request in Step 1.
     * </p>
     * </li>
     * </ul>
     * </li>
     * <li>
     * <p>
     * Review the response to the <code>UpdateDistribution</code> request to confirm that the configuration was
     * successfully updated.
     * </p>
     * </li>
     * <li>
     * <p>
     * Optional: Submit a <a
     * href="https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_GetDistribution.html">GetDistribution</a>
     * request to confirm that your changes have propagated. When propagation is complete, the value of
     * <code>Status</code> is <code>Deployed</code>.
     * </p>
     * </li>
     * </ol>
     *
     * @param updateDistributionRequest
     *        The request to update a distribution.
     * @return A Java Future containing the result of the UpdateDistribution operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException Access denied.</li>
     *         <li>CnameAlreadyExistsException The CNAME specified is already defined for CloudFront.</li>
     *         <li>IllegalUpdateException Origin and <code>CallerReference</code> cannot be updated.</li>
     *         <li>InvalidIfMatchVersionException The <code>If-Match</code> version is missing or not valid for the
     *         distribution.</li>
     *         <li>MissingBodyException This operation requires a body. Ensure that the body is present and the
     *         <code>Content-Type</code> header is set.</li>
     *         <li>NoSuchDistributionException The specified distribution does not exist.</li>
     *         <li>PreconditionFailedException The precondition given in one or more of the request-header fields
     *         evaluated to <code>false</code>.</li>
     *         <li>TooManyDistributionCnamEsException Your request contains more CNAMEs than are allowed per
     *         distribution.</li>
     *         <li>InvalidDefaultRootObjectException The default root object file name is too big or contains an invalid
     *         character.</li>
     *         <li>InvalidRelativePathException The relative path is too big, is not URL-encoded, or does not begin with
     *         a slash (/).</li>
     *         <li>InvalidErrorCodeException An invalid error code was specified.</li>
     *         <li>InvalidResponseCodeException A response code specified in the response body is not valid.</li>
     *         <li>InvalidArgumentException The argument is invalid.</li>
     *         <li>InvalidOriginAccessIdentityException The origin access identity is not valid or doesn't exist.</li>
     *         <li>TooManyTrustedSignersException Your request contains more trusted signers than are allowed per
     *         distribution.</li>
     *         <li>TrustedSignerDoesNotExistException One or more of your trusted signers don't exist.</li>
     *         <li>InvalidViewerCertificateException A viewer certificate specified in the response body is not valid.</li>
     *         <li>InvalidMinimumProtocolVersionException The minimum protocol version specified is not valid.</li>
     *         <li>InvalidRequiredProtocolException This operation requires the HTTPS protocol. Ensure that you specify
     *         the HTTPS protocol in your request, or omit the <code>RequiredProtocols</code> element from your
     *         distribution configuration.</li>
     *         <li>NoSuchOriginException No origin exists with the specified <code>Origin Id</code>.</li>
     *         <li>TooManyOriginsException You cannot create more origins for the distribution.</li>
     *         <li>TooManyOriginGroupsPerDistributionException Processing your request would cause you to exceed the
     *         maximum number of origin groups allowed.</li>
     *         <li>TooManyCacheBehaviorsException You cannot create more cache behaviors for the distribution.</li>
     *         <li>TooManyCookieNamesInWhiteListException Your request contains more cookie names in the whitelist than
     *         are allowed per cache behavior.</li>
     *         <li>InvalidForwardCookiesException Your request contains forward cookies option which doesn't match with
     *         the expectation for the <code>whitelisted</code> list of cookie names. Either list of cookie names has
     *         been specified when not allowed or list of cookie names is missing when expected.</li>
     *         <li>TooManyHeadersInForwardedValuesException Your request contains too many headers in forwarded values.</li>
     *         <li>InvalidHeadersForS3OriginException The headers specified are not valid for an Amazon S3 origin.</li>
     *         <li>InconsistentQuantitiesException The value of <code>Quantity</code> and the size of <code>Items</code>
     *         don't match.</li>
     *         <li>TooManyCertificatesException You cannot create anymore custom SSL/TLS certificates.</li>
     *         <li>InvalidLocationCodeException The location code specified is not valid.</li>
     *         <li>InvalidGeoRestrictionParameterException The specified geo restriction parameter is not valid.</li>
     *         <li>InvalidTtlOrderException TTL order specified in the response body is not valid.</li>
     *         <li>InvalidWebAclIdException A web ACL id specified in the response body is not valid.</li>
     *         <li>TooManyOriginCustomHeadersException Your request contains too many origin custom headers.</li>
     *         <li>TooManyQueryStringParametersException Your request contains too many query string parameters.</li>
     *         <li>InvalidQueryStringParametersException Query string parameters specified in the response body are not
     *         valid.</li>
     *         <li>TooManyDistributionsWithLambdaAssociationsException Processing your request would cause the maximum
     *         number of distributions with Lambda function associations per owner to be exceeded.</li>
     *         <li>TooManyLambdaFunctionAssociationsException Your request contains more Lambda function associations
     *         than are allowed per distribution.</li>
     *         <li>InvalidLambdaFunctionAssociationException The specified Lambda function association is invalid.</li>
     *         <li>InvalidOriginReadTimeoutException The read timeout specified for the origin is not valid.</li>
     *         <li>InvalidOriginKeepaliveTimeoutException The keep alive timeout specified for the origin is not valid.</li>
     *         <li>NoSuchFieldLevelEncryptionConfigException The specified configuration for field-level encryption
     *         doesn't exist.</li>
     *         <li>IllegalFieldLevelEncryptionConfigAssociationWithCacheBehaviorException The specified configuration
     *         for field-level encryption can't be associated with the specified cache behavior.</li>
     *         <li>TooManyDistributionsAssociatedToFieldLevelEncryptionConfigException The maximum number of
     *         distributions have been associated with the specified configuration for field-level encryption.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CloudFrontException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CloudFrontAsyncClient.UpdateDistribution
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/cloudfront-2019-03-26/UpdateDistribution" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateDistributionResponse> updateDistribution(UpdateDistributionRequest updateDistributionRequest) {
        try {

            HttpResponseHandler<UpdateDistributionResponse> responseHandler = protocolFactory.createResponseHandler(
                    UpdateDistributionResponse::builder, new XmlOperationMetadata().withHasStreamingSuccessResponse(false));

            HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();

            CompletableFuture<UpdateDistributionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateDistributionRequest, UpdateDistributionResponse>()
                            .withOperationName("UpdateDistribution")
                            .withMarshaller(new UpdateDistributionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(updateDistributionRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Update a field-level encryption configuration.
     * </p>
     *
     * @param updateFieldLevelEncryptionConfigRequest
     * @return A Java Future containing the result of the UpdateFieldLevelEncryptionConfig operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException Access denied.</li>
     *         <li>IllegalUpdateException Origin and <code>CallerReference</code> cannot be updated.</li>
     *         <li>InconsistentQuantitiesException The value of <code>Quantity</code> and the size of <code>Items</code>
     *         don't match.</li>
     *         <li>InvalidArgumentException The argument is invalid.</li>
     *         <li>InvalidIfMatchVersionException The <code>If-Match</code> version is missing or not valid for the
     *         distribution.</li>
     *         <li>NoSuchFieldLevelEncryptionProfileException The specified profile for field-level encryption doesn't
     *         exist.</li>
     *         <li>NoSuchFieldLevelEncryptionConfigException The specified configuration for field-level encryption
     *         doesn't exist.</li>
     *         <li>PreconditionFailedException The precondition given in one or more of the request-header fields
     *         evaluated to <code>false</code>.</li>
     *         <li>TooManyFieldLevelEncryptionQueryArgProfilesException The maximum number of query arg profiles for
     *         field-level encryption have been created.</li>
     *         <li>TooManyFieldLevelEncryptionContentTypeProfilesException The maximum number of content type profiles
     *         for field-level encryption have been created.</li>
     *         <li>QueryArgProfileEmptyException No profile specified for the field-level encryption query argument.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CloudFrontException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CloudFrontAsyncClient.UpdateFieldLevelEncryptionConfig
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/cloudfront-2019-03-26/UpdateFieldLevelEncryptionConfig"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateFieldLevelEncryptionConfigResponse> updateFieldLevelEncryptionConfig(
            UpdateFieldLevelEncryptionConfigRequest updateFieldLevelEncryptionConfigRequest) {
        try {

            HttpResponseHandler<UpdateFieldLevelEncryptionConfigResponse> responseHandler = protocolFactory
                    .createResponseHandler(UpdateFieldLevelEncryptionConfigResponse::builder,
                            new XmlOperationMetadata().withHasStreamingSuccessResponse(false));

            HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();

            CompletableFuture<UpdateFieldLevelEncryptionConfigResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateFieldLevelEncryptionConfigRequest, UpdateFieldLevelEncryptionConfigResponse>()
                            .withOperationName("UpdateFieldLevelEncryptionConfig")
                            .withMarshaller(new UpdateFieldLevelEncryptionConfigRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(updateFieldLevelEncryptionConfigRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Update a field-level encryption profile.
     * </p>
     *
     * @param updateFieldLevelEncryptionProfileRequest
     * @return A Java Future containing the result of the UpdateFieldLevelEncryptionProfile operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException Access denied.</li>
     *         <li>FieldLevelEncryptionProfileAlreadyExistsException The specified profile for field-level encryption
     *         already exists.</li>
     *         <li>IllegalUpdateException Origin and <code>CallerReference</code> cannot be updated.</li>
     *         <li>InconsistentQuantitiesException The value of <code>Quantity</code> and the size of <code>Items</code>
     *         don't match.</li>
     *         <li>InvalidArgumentException The argument is invalid.</li>
     *         <li>InvalidIfMatchVersionException The <code>If-Match</code> version is missing or not valid for the
     *         distribution.</li>
     *         <li>NoSuchPublicKeyException The specified public key doesn't exist.</li>
     *         <li>NoSuchFieldLevelEncryptionProfileException The specified profile for field-level encryption doesn't
     *         exist.</li>
     *         <li>PreconditionFailedException The precondition given in one or more of the request-header fields
     *         evaluated to <code>false</code>.</li>
     *         <li>FieldLevelEncryptionProfileSizeExceededException The maximum size of a profile for field-level
     *         encryption was exceeded.</li>
     *         <li>TooManyFieldLevelEncryptionEncryptionEntitiesException The maximum number of encryption entities for
     *         field-level encryption have been created.</li>
     *         <li>TooManyFieldLevelEncryptionFieldPatternsException The maximum number of field patterns for
     *         field-level encryption have been created.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CloudFrontException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CloudFrontAsyncClient.UpdateFieldLevelEncryptionProfile
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/cloudfront-2019-03-26/UpdateFieldLevelEncryptionProfile"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateFieldLevelEncryptionProfileResponse> updateFieldLevelEncryptionProfile(
            UpdateFieldLevelEncryptionProfileRequest updateFieldLevelEncryptionProfileRequest) {
        try {

            HttpResponseHandler<UpdateFieldLevelEncryptionProfileResponse> responseHandler = protocolFactory
                    .createResponseHandler(UpdateFieldLevelEncryptionProfileResponse::builder,
                            new XmlOperationMetadata().withHasStreamingSuccessResponse(false));

            HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();

            CompletableFuture<UpdateFieldLevelEncryptionProfileResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateFieldLevelEncryptionProfileRequest, UpdateFieldLevelEncryptionProfileResponse>()
                            .withOperationName("UpdateFieldLevelEncryptionProfile")
                            .withMarshaller(new UpdateFieldLevelEncryptionProfileRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(updateFieldLevelEncryptionProfileRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Update public key information. Note that the only value you can change is the comment.
     * </p>
     *
     * @param updatePublicKeyRequest
     * @return A Java Future containing the result of the UpdatePublicKey operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException Access denied.</li>
     *         <li>CannotChangeImmutablePublicKeyFieldsException You can't change the value of a public key.</li>
     *         <li>InvalidArgumentException The argument is invalid.</li>
     *         <li>InvalidIfMatchVersionException The <code>If-Match</code> version is missing or not valid for the
     *         distribution.</li>
     *         <li>IllegalUpdateException Origin and <code>CallerReference</code> cannot be updated.</li>
     *         <li>NoSuchPublicKeyException The specified public key doesn't exist.</li>
     *         <li>PreconditionFailedException The precondition given in one or more of the request-header fields
     *         evaluated to <code>false</code>.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CloudFrontException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CloudFrontAsyncClient.UpdatePublicKey
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/cloudfront-2019-03-26/UpdatePublicKey" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<UpdatePublicKeyResponse> updatePublicKey(UpdatePublicKeyRequest updatePublicKeyRequest) {
        try {

            HttpResponseHandler<UpdatePublicKeyResponse> responseHandler = protocolFactory.createResponseHandler(
                    UpdatePublicKeyResponse::builder, new XmlOperationMetadata().withHasStreamingSuccessResponse(false));

            HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();

            CompletableFuture<UpdatePublicKeyResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdatePublicKeyRequest, UpdatePublicKeyResponse>()
                            .withOperationName("UpdatePublicKey")
                            .withMarshaller(new UpdatePublicKeyRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(updatePublicKeyRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Update a streaming distribution.
     * </p>
     *
     * @param updateStreamingDistributionRequest
     *        The request to update a streaming distribution.
     * @return A Java Future containing the result of the UpdateStreamingDistribution operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException Access denied.</li>
     *         <li>CnameAlreadyExistsException The CNAME specified is already defined for CloudFront.</li>
     *         <li>IllegalUpdateException Origin and <code>CallerReference</code> cannot be updated.</li>
     *         <li>InvalidIfMatchVersionException The <code>If-Match</code> version is missing or not valid for the
     *         distribution.</li>
     *         <li>MissingBodyException This operation requires a body. Ensure that the body is present and the
     *         <code>Content-Type</code> header is set.</li>
     *         <li>NoSuchStreamingDistributionException The specified streaming distribution does not exist.</li>
     *         <li>PreconditionFailedException The precondition given in one or more of the request-header fields
     *         evaluated to <code>false</code>.</li>
     *         <li>TooManyStreamingDistributionCnamEsException Your request contains more CNAMEs than are allowed per
     *         distribution.</li>
     *         <li>InvalidArgumentException The argument is invalid.</li>
     *         <li>InvalidOriginAccessIdentityException The origin access identity is not valid or doesn't exist.</li>
     *         <li>TooManyTrustedSignersException Your request contains more trusted signers than are allowed per
     *         distribution.</li>
     *         <li>TrustedSignerDoesNotExistException One or more of your trusted signers don't exist.</li>
     *         <li>InconsistentQuantitiesException The value of <code>Quantity</code> and the size of <code>Items</code>
     *         don't match.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CloudFrontException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CloudFrontAsyncClient.UpdateStreamingDistribution
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/cloudfront-2019-03-26/UpdateStreamingDistribution"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateStreamingDistributionResponse> updateStreamingDistribution(
            UpdateStreamingDistributionRequest updateStreamingDistributionRequest) {
        try {

            HttpResponseHandler<UpdateStreamingDistributionResponse> responseHandler = protocolFactory.createResponseHandler(
                    UpdateStreamingDistributionResponse::builder,
                    new XmlOperationMetadata().withHasStreamingSuccessResponse(false));

            HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();

            CompletableFuture<UpdateStreamingDistributionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateStreamingDistributionRequest, UpdateStreamingDistributionResponse>()
                            .withOperationName("UpdateStreamingDistribution")
                            .withMarshaller(new UpdateStreamingDistributionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(updateStreamingDistributionRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    @Override
    public void close() {
        clientHandler.close();
    }

    private AwsXmlProtocolFactory init() {
        return AwsXmlProtocolFactory
                .builder()
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("StreamingDistributionAlreadyExists")
                                .exceptionBuilderSupplier(StreamingDistributionAlreadyExistsException::builder)
                                .httpStatusCode(409).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("TrustedSignerDoesNotExist")
                                .exceptionBuilderSupplier(TrustedSignerDoesNotExistException::builder).httpStatusCode(400)
                                .build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("NoSuchInvalidation")
                                .exceptionBuilderSupplier(NoSuchInvalidationException::builder).httpStatusCode(404).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("NoSuchFieldLevelEncryptionConfig")
                                .exceptionBuilderSupplier(NoSuchFieldLevelEncryptionConfigException::builder).httpStatusCode(404)
                                .build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("TooManyHeadersInForwardedValues")
                                .exceptionBuilderSupplier(TooManyHeadersInForwardedValuesException::builder).httpStatusCode(400)
                                .build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidOriginReadTimeout")
                                .exceptionBuilderSupplier(InvalidOriginReadTimeoutException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidMinimumProtocolVersion")
                                .exceptionBuilderSupplier(InvalidMinimumProtocolVersionException::builder).httpStatusCode(400)
                                .build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("TooManyInvalidationsInProgress")
                                .exceptionBuilderSupplier(TooManyInvalidationsInProgressException::builder).httpStatusCode(400)
                                .build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("TooManyFieldLevelEncryptionConfigs")
                                .exceptionBuilderSupplier(TooManyFieldLevelEncryptionConfigsException::builder)
                                .httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("TooManyPublicKeys")
                                .exceptionBuilderSupplier(TooManyPublicKeysException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidErrorCode")
                                .exceptionBuilderSupplier(InvalidErrorCodeException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("TooManyCacheBehaviors")
                                .exceptionBuilderSupplier(TooManyCacheBehaviorsException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("CloudFrontOriginAccessIdentityInUse")
                                .exceptionBuilderSupplier(CloudFrontOriginAccessIdentityInUseException::builder)
                                .httpStatusCode(409).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("FieldLevelEncryptionProfileAlreadyExists")
                                .exceptionBuilderSupplier(FieldLevelEncryptionProfileAlreadyExistsException::builder)
                                .httpStatusCode(409).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("TooManyFieldLevelEncryptionFieldPatterns")
                                .exceptionBuilderSupplier(TooManyFieldLevelEncryptionFieldPatternsException::builder)
                                .httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("MissingBody")
                                .exceptionBuilderSupplier(MissingBodyException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("TooManyOrigins")
                                .exceptionBuilderSupplier(TooManyOriginsException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata
                                .builder()
                                .errorCode("IllegalFieldLevelEncryptionConfigAssociationWithCacheBehavior")
                                .exceptionBuilderSupplier(
                                        IllegalFieldLevelEncryptionConfigAssociationWithCacheBehaviorException::builder)
                                .httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidRequiredProtocol")
                                .exceptionBuilderSupplier(InvalidRequiredProtocolException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("NoSuchOrigin")
                                .exceptionBuilderSupplier(NoSuchOriginException::builder).httpStatusCode(404).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("TooManyTrustedSigners")
                                .exceptionBuilderSupplier(TooManyTrustedSignersException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("BatchTooLarge")
                                .exceptionBuilderSupplier(BatchTooLargeException::builder).httpStatusCode(413).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidResponseCode")
                                .exceptionBuilderSupplier(InvalidResponseCodeException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("FieldLevelEncryptionConfigAlreadyExists")
                                .exceptionBuilderSupplier(FieldLevelEncryptionConfigAlreadyExistsException::builder)
                                .httpStatusCode(409).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidOriginKeepaliveTimeout")
                                .exceptionBuilderSupplier(InvalidOriginKeepaliveTimeoutException::builder).httpStatusCode(400)
                                .build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("TooManyStreamingDistributionCNAMEs")
                                .exceptionBuilderSupplier(TooManyStreamingDistributionCnamEsException::builder)
                                .httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("TooManyStreamingDistributions")
                                .exceptionBuilderSupplier(TooManyStreamingDistributionsException::builder).httpStatusCode(400)
                                .build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("PublicKeyInUse")
                                .exceptionBuilderSupplier(PublicKeyInUseException::builder).httpStatusCode(409).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("StreamingDistributionNotDisabled")
                                .exceptionBuilderSupplier(StreamingDistributionNotDisabledException::builder).httpStatusCode(409)
                                .build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("TooManyDistributionCNAMEs")
                                .exceptionBuilderSupplier(TooManyDistributionCnamEsException::builder).httpStatusCode(400)
                                .build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("PreconditionFailed")
                                .exceptionBuilderSupplier(PreconditionFailedException::builder).httpStatusCode(412).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("TooManyQueryStringParameters")
                                .exceptionBuilderSupplier(TooManyQueryStringParametersException::builder).httpStatusCode(400)
                                .build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("TooManyFieldLevelEncryptionEncryptionEntities")
                                .exceptionBuilderSupplier(TooManyFieldLevelEncryptionEncryptionEntitiesException::builder)
                                .httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("FieldLevelEncryptionConfigInUse")
                                .exceptionBuilderSupplier(FieldLevelEncryptionConfigInUseException::builder).httpStatusCode(409)
                                .build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("DistributionAlreadyExists")
                                .exceptionBuilderSupplier(DistributionAlreadyExistsException::builder).httpStatusCode(409)
                                .build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("CloudFrontOriginAccessIdentityAlreadyExists")
                                .exceptionBuilderSupplier(CloudFrontOriginAccessIdentityAlreadyExistsException::builder)
                                .httpStatusCode(409).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidProtocolSettings")
                                .exceptionBuilderSupplier(InvalidProtocolSettingsException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("TooManyFieldLevelEncryptionProfiles")
                                .exceptionBuilderSupplier(TooManyFieldLevelEncryptionProfilesException::builder)
                                .httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("TooManyOriginCustomHeaders")
                                .exceptionBuilderSupplier(TooManyOriginCustomHeadersException::builder).httpStatusCode(400)
                                .build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidLocationCode")
                                .exceptionBuilderSupplier(InvalidLocationCodeException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidForwardCookies")
                                .exceptionBuilderSupplier(InvalidForwardCookiesException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("FieldLevelEncryptionProfileSizeExceeded")
                                .exceptionBuilderSupplier(FieldLevelEncryptionProfileSizeExceededException::builder)
                                .httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("TooManyCertificates")
                                .exceptionBuilderSupplier(TooManyCertificatesException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("QueryArgProfileEmpty")
                                .exceptionBuilderSupplier(QueryArgProfileEmptyException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidOriginAccessIdentity")
                                .exceptionBuilderSupplier(InvalidOriginAccessIdentityException::builder).httpStatusCode(400)
                                .build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidQueryStringParameters")
                                .exceptionBuilderSupplier(InvalidQueryStringParametersException::builder).httpStatusCode(400)
                                .build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("IllegalUpdate")
                                .exceptionBuilderSupplier(IllegalUpdateException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("TooManyCookieNamesInWhiteList")
                                .exceptionBuilderSupplier(TooManyCookieNamesInWhiteListException::builder).httpStatusCode(400)
                                .build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("NoSuchDistribution")
                                .exceptionBuilderSupplier(NoSuchDistributionException::builder).httpStatusCode(404).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("CNAMEAlreadyExists")
                                .exceptionBuilderSupplier(CnameAlreadyExistsException::builder).httpStatusCode(409).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("NoSuchFieldLevelEncryptionProfile")
                                .exceptionBuilderSupplier(NoSuchFieldLevelEncryptionProfileException::builder)
                                .httpStatusCode(404).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("NoSuchResource")
                                .exceptionBuilderSupplier(NoSuchResourceException::builder).httpStatusCode(404).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidTTLOrder")
                                .exceptionBuilderSupplier(InvalidTtlOrderException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InconsistentQuantities")
                                .exceptionBuilderSupplier(InconsistentQuantitiesException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidLambdaFunctionAssociation")
                                .exceptionBuilderSupplier(InvalidLambdaFunctionAssociationException::builder).httpStatusCode(400)
                                .build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("TooManyLambdaFunctionAssociations")
                                .exceptionBuilderSupplier(TooManyLambdaFunctionAssociationsException::builder)
                                .httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidViewerCertificate")
                                .exceptionBuilderSupplier(InvalidViewerCertificateException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("DistributionNotDisabled")
                                .exceptionBuilderSupplier(DistributionNotDisabledException::builder).httpStatusCode(409).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("TooManyOriginGroupsPerDistribution")
                                .exceptionBuilderSupplier(TooManyOriginGroupsPerDistributionException::builder)
                                .httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidDefaultRootObject")
                                .exceptionBuilderSupplier(InvalidDefaultRootObjectException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("AccessDenied")
                                .exceptionBuilderSupplier(AccessDeniedException::builder).httpStatusCode(403).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidArgument")
                                .exceptionBuilderSupplier(InvalidArgumentException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("NoSuchStreamingDistribution")
                                .exceptionBuilderSupplier(NoSuchStreamingDistributionException::builder).httpStatusCode(404)
                                .build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("TooManyCloudFrontOriginAccessIdentities")
                                .exceptionBuilderSupplier(TooManyCloudFrontOriginAccessIdentitiesException::builder)
                                .httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("NoSuchPublicKey")
                                .exceptionBuilderSupplier(NoSuchPublicKeyException::builder).httpStatusCode(404).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("TooManyFieldLevelEncryptionContentTypeProfiles")
                                .exceptionBuilderSupplier(TooManyFieldLevelEncryptionContentTypeProfilesException::builder)
                                .httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidRelativePath")
                                .exceptionBuilderSupplier(InvalidRelativePathException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidWebACLId")
                                .exceptionBuilderSupplier(InvalidWebAclIdException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata
                                .builder()
                                .errorCode("TooManyDistributionsAssociatedToFieldLevelEncryptionConfig")
                                .exceptionBuilderSupplier(
                                        TooManyDistributionsAssociatedToFieldLevelEncryptionConfigException::builder)
                                .httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("TooManyDistributionsWithLambdaAssociations")
                                .exceptionBuilderSupplier(TooManyDistributionsWithLambdaAssociationsException::builder)
                                .httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidHeadersForS3Origin")
                                .exceptionBuilderSupplier(InvalidHeadersForS3OriginException::builder).httpStatusCode(400)
                                .build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("TooManyDistributions")
                                .exceptionBuilderSupplier(TooManyDistributionsException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("CannotChangeImmutablePublicKeyFields")
                                .exceptionBuilderSupplier(CannotChangeImmutablePublicKeyFieldsException::builder)
                                .httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("TooManyFieldLevelEncryptionQueryArgProfiles")
                                .exceptionBuilderSupplier(TooManyFieldLevelEncryptionQueryArgProfilesException::builder)
                                .httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("PublicKeyAlreadyExists")
                                .exceptionBuilderSupplier(PublicKeyAlreadyExistsException::builder).httpStatusCode(409).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("FieldLevelEncryptionProfileInUse")
                                .exceptionBuilderSupplier(FieldLevelEncryptionProfileInUseException::builder).httpStatusCode(409)
                                .build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidIfMatchVersion")
                                .exceptionBuilderSupplier(InvalidIfMatchVersionException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidTagging")
                                .exceptionBuilderSupplier(InvalidTaggingException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidOrigin")
                                .exceptionBuilderSupplier(InvalidOriginException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("NoSuchCloudFrontOriginAccessIdentity")
                                .exceptionBuilderSupplier(NoSuchCloudFrontOriginAccessIdentityException::builder)
                                .httpStatusCode(404).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidGeoRestrictionParameter")
                                .exceptionBuilderSupplier(InvalidGeoRestrictionParameterException::builder).httpStatusCode(400)
                                .build()).clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(CloudFrontException::builder).build();
    }

    private <T extends CloudFrontRequest> T applyPaginatorUserAgent(T request) {
        Consumer<AwsRequestOverrideConfiguration.Builder> userAgentApplier = b -> b.addApiName(ApiName.builder()
                .version(VersionInfo.SDK_VERSION).name("PAGINATED").build());
        AwsRequestOverrideConfiguration overrideConfiguration = request.overrideConfiguration()
                .map(c -> c.toBuilder().applyMutation(userAgentApplier).build())
                .orElse((AwsRequestOverrideConfiguration.builder().applyMutation(userAgentApplier).build()));
        return (T) request.toBuilder().overrideConfiguration(overrideConfiguration).build();
    }
}
