/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.cloudfront.model;

import java.beans.Transient;
import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * A distribution tells CloudFront where you want content to be delivered from, and the details about how to track and
 * manage content delivery.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class Distribution implements SdkPojo, Serializable, ToCopyableBuilder<Distribution.Builder, Distribution> {
    private static final SdkField<String> ID_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("Id")
            .getter(getter(Distribution::id))
            .setter(setter(Builder::id))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Id").unmarshallLocationName("Id")
                    .build()).build();

    private static final SdkField<String> ARN_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("ARN")
            .getter(getter(Distribution::arn))
            .setter(setter(Builder::arn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ARN").unmarshallLocationName("ARN")
                    .build()).build();

    private static final SdkField<String> STATUS_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("Status")
            .getter(getter(Distribution::status))
            .setter(setter(Builder::status))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Status")
                    .unmarshallLocationName("Status").build()).build();

    private static final SdkField<Instant> LAST_MODIFIED_TIME_FIELD = SdkField
            .<Instant> builder(MarshallingType.INSTANT)
            .memberName("LastModifiedTime")
            .getter(getter(Distribution::lastModifiedTime))
            .setter(setter(Builder::lastModifiedTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("LastModifiedTime")
                    .unmarshallLocationName("LastModifiedTime").build()).build();

    private static final SdkField<Integer> IN_PROGRESS_INVALIDATION_BATCHES_FIELD = SdkField
            .<Integer> builder(MarshallingType.INTEGER)
            .memberName("InProgressInvalidationBatches")
            .getter(getter(Distribution::inProgressInvalidationBatches))
            .setter(setter(Builder::inProgressInvalidationBatches))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("InProgressInvalidationBatches")
                    .unmarshallLocationName("InProgressInvalidationBatches").build()).build();

    private static final SdkField<String> DOMAIN_NAME_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("DomainName")
            .getter(getter(Distribution::domainName))
            .setter(setter(Builder::domainName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DomainName")
                    .unmarshallLocationName("DomainName").build()).build();

    private static final SdkField<ActiveTrustedSigners> ACTIVE_TRUSTED_SIGNERS_FIELD = SdkField
            .<ActiveTrustedSigners> builder(MarshallingType.SDK_POJO)
            .memberName("ActiveTrustedSigners")
            .getter(getter(Distribution::activeTrustedSigners))
            .setter(setter(Builder::activeTrustedSigners))
            .constructor(ActiveTrustedSigners::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ActiveTrustedSigners")
                    .unmarshallLocationName("ActiveTrustedSigners").build()).build();

    private static final SdkField<ActiveTrustedKeyGroups> ACTIVE_TRUSTED_KEY_GROUPS_FIELD = SdkField
            .<ActiveTrustedKeyGroups> builder(MarshallingType.SDK_POJO)
            .memberName("ActiveTrustedKeyGroups")
            .getter(getter(Distribution::activeTrustedKeyGroups))
            .setter(setter(Builder::activeTrustedKeyGroups))
            .constructor(ActiveTrustedKeyGroups::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ActiveTrustedKeyGroups")
                    .unmarshallLocationName("ActiveTrustedKeyGroups").build()).build();

    private static final SdkField<DistributionConfig> DISTRIBUTION_CONFIG_FIELD = SdkField
            .<DistributionConfig> builder(MarshallingType.SDK_POJO)
            .memberName("DistributionConfig")
            .getter(getter(Distribution::distributionConfig))
            .setter(setter(Builder::distributionConfig))
            .constructor(DistributionConfig::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DistributionConfig")
                    .unmarshallLocationName("DistributionConfig").build()).build();

    private static final SdkField<List<AliasICPRecordal>> ALIAS_ICP_RECORDALS_FIELD = SdkField
            .<List<AliasICPRecordal>> builder(MarshallingType.LIST)
            .memberName("AliasICPRecordals")
            .getter(getter(Distribution::aliasICPRecordals))
            .setter(setter(Builder::aliasICPRecordals))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AliasICPRecordals")
                    .unmarshallLocationName("AliasICPRecordals").build(),
                    ListTrait
                            .builder()
                            .memberLocationName("AliasICPRecordal")
                            .memberFieldInfo(
                                    SdkField.<AliasICPRecordal> builder(MarshallingType.SDK_POJO)
                                            .constructor(AliasICPRecordal::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("AliasICPRecordal").unmarshallLocationName("AliasICPRecordal")
                                                    .build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(ID_FIELD, ARN_FIELD,
            STATUS_FIELD, LAST_MODIFIED_TIME_FIELD, IN_PROGRESS_INVALIDATION_BATCHES_FIELD, DOMAIN_NAME_FIELD,
            ACTIVE_TRUSTED_SIGNERS_FIELD, ACTIVE_TRUSTED_KEY_GROUPS_FIELD, DISTRIBUTION_CONFIG_FIELD, ALIAS_ICP_RECORDALS_FIELD));

    private static final long serialVersionUID = 1L;

    private final String id;

    private final String arn;

    private final String status;

    private final Instant lastModifiedTime;

    private final Integer inProgressInvalidationBatches;

    private final String domainName;

    private final ActiveTrustedSigners activeTrustedSigners;

    private final ActiveTrustedKeyGroups activeTrustedKeyGroups;

    private final DistributionConfig distributionConfig;

    private final List<AliasICPRecordal> aliasICPRecordals;

    private Distribution(BuilderImpl builder) {
        this.id = builder.id;
        this.arn = builder.arn;
        this.status = builder.status;
        this.lastModifiedTime = builder.lastModifiedTime;
        this.inProgressInvalidationBatches = builder.inProgressInvalidationBatches;
        this.domainName = builder.domainName;
        this.activeTrustedSigners = builder.activeTrustedSigners;
        this.activeTrustedKeyGroups = builder.activeTrustedKeyGroups;
        this.distributionConfig = builder.distributionConfig;
        this.aliasICPRecordals = builder.aliasICPRecordals;
    }

    /**
     * <p>
     * The identifier for the distribution. For example: <code>EDFDVBD632BHDS5</code>.
     * </p>
     * 
     * @return The identifier for the distribution. For example: <code>EDFDVBD632BHDS5</code>.
     */
    public final String id() {
        return id;
    }

    /**
     * <p>
     * The ARN (Amazon Resource Name) for the distribution. For example:
     * <code>arn:aws:cloudfront::123456789012:distribution/EDFDVBD632BHDS5</code>, where <code>123456789012</code> is
     * your account ID.
     * </p>
     * 
     * @return The ARN (Amazon Resource Name) for the distribution. For example:
     *         <code>arn:aws:cloudfront::123456789012:distribution/EDFDVBD632BHDS5</code>, where
     *         <code>123456789012</code> is your account ID.
     */
    public final String arn() {
        return arn;
    }

    /**
     * <p>
     * This response element indicates the current status of the distribution. When the status is <code>Deployed</code>,
     * the distribution's information is fully propagated to all CloudFront edge locations.
     * </p>
     * 
     * @return This response element indicates the current status of the distribution. When the status is
     *         <code>Deployed</code>, the distribution's information is fully propagated to all CloudFront edge
     *         locations.
     */
    public final String status() {
        return status;
    }

    /**
     * <p>
     * The date and time the distribution was last modified.
     * </p>
     * 
     * @return The date and time the distribution was last modified.
     */
    public final Instant lastModifiedTime() {
        return lastModifiedTime;
    }

    /**
     * <p>
     * The number of invalidation batches currently in progress.
     * </p>
     * 
     * @return The number of invalidation batches currently in progress.
     */
    public final Integer inProgressInvalidationBatches() {
        return inProgressInvalidationBatches;
    }

    /**
     * <p>
     * The domain name corresponding to the distribution, for example, <code>d111111abcdef8.cloudfront.net</code>.
     * </p>
     * 
     * @return The domain name corresponding to the distribution, for example,
     *         <code>d111111abcdef8.cloudfront.net</code>.
     */
    public final String domainName() {
        return domainName;
    }

    /**
     * <important>
     * <p>
     * We recommend using <code>TrustedKeyGroups</code> instead of <code>TrustedSigners</code>.
     * </p>
     * </important>
     * <p>
     * CloudFront automatically adds this field to the response if you’ve configured a cache behavior in this
     * distribution to serve private content using trusted signers. This field contains a list of account IDs and the
     * active CloudFront key pairs in each account that CloudFront can use to verify the signatures of signed URLs or
     * signed cookies.
     * </p>
     * 
     * @return <p>
     *         We recommend using <code>TrustedKeyGroups</code> instead of <code>TrustedSigners</code>.
     *         </p>
     *         </important>
     *         <p>
     *         CloudFront automatically adds this field to the response if you’ve configured a cache behavior in this
     *         distribution to serve private content using trusted signers. This field contains a list of account IDs
     *         and the active CloudFront key pairs in each account that CloudFront can use to verify the signatures of
     *         signed URLs or signed cookies.
     */
    public final ActiveTrustedSigners activeTrustedSigners() {
        return activeTrustedSigners;
    }

    /**
     * <p>
     * CloudFront automatically adds this field to the response if you’ve configured a cache behavior in this
     * distribution to serve private content using key groups. This field contains a list of key groups and the public
     * keys in each key group that CloudFront can use to verify the signatures of signed URLs or signed cookies.
     * </p>
     * 
     * @return CloudFront automatically adds this field to the response if you’ve configured a cache behavior in this
     *         distribution to serve private content using key groups. This field contains a list of key groups and the
     *         public keys in each key group that CloudFront can use to verify the signatures of signed URLs or signed
     *         cookies.
     */
    public final ActiveTrustedKeyGroups activeTrustedKeyGroups() {
        return activeTrustedKeyGroups;
    }

    /**
     * <p>
     * The current configuration information for the distribution. Send a <code>GET</code> request to the
     * <code>/<i>CloudFront API version</i>/distribution ID/config</code> resource.
     * </p>
     * 
     * @return The current configuration information for the distribution. Send a <code>GET</code> request to the
     *         <code>/<i>CloudFront API version</i>/distribution ID/config</code> resource.
     */
    public final DistributionConfig distributionConfig() {
        return distributionConfig;
    }

    /**
     * For responses, this returns true if the service returned a value for the AliasICPRecordals property. This DOES
     * NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasAliasICPRecordals() {
        return aliasICPRecordals != null && !(aliasICPRecordals instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * Amazon Web Services services in China customers must file for an Internet Content Provider (ICP) recordal if they
     * want to serve content publicly on an alternate domain name, also known as a CNAME, that they've added to
     * CloudFront. AliasICPRecordal provides the ICP recordal status for CNAMEs associated with distributions.
     * </p>
     * <p>
     * For more information about ICP recordals, see <a
     * href="https://docs.amazonaws.cn/en_us/aws/latest/userguide/accounts-and-credentials.html"> Signup, Accounts, and
     * Credentials</a> in <i>Getting Started with Amazon Web Services services in China</i>.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasAliasICPRecordals} method.
     * </p>
     * 
     * @return Amazon Web Services services in China customers must file for an Internet Content Provider (ICP) recordal
     *         if they want to serve content publicly on an alternate domain name, also known as a CNAME, that they've
     *         added to CloudFront. AliasICPRecordal provides the ICP recordal status for CNAMEs associated with
     *         distributions.</p>
     *         <p>
     *         For more information about ICP recordals, see <a
     *         href="https://docs.amazonaws.cn/en_us/aws/latest/userguide/accounts-and-credentials.html"> Signup,
     *         Accounts, and Credentials</a> in <i>Getting Started with Amazon Web Services services in China</i>.
     */
    public final List<AliasICPRecordal> aliasICPRecordals() {
        return aliasICPRecordals;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(id());
        hashCode = 31 * hashCode + Objects.hashCode(arn());
        hashCode = 31 * hashCode + Objects.hashCode(status());
        hashCode = 31 * hashCode + Objects.hashCode(lastModifiedTime());
        hashCode = 31 * hashCode + Objects.hashCode(inProgressInvalidationBatches());
        hashCode = 31 * hashCode + Objects.hashCode(domainName());
        hashCode = 31 * hashCode + Objects.hashCode(activeTrustedSigners());
        hashCode = 31 * hashCode + Objects.hashCode(activeTrustedKeyGroups());
        hashCode = 31 * hashCode + Objects.hashCode(distributionConfig());
        hashCode = 31 * hashCode + Objects.hashCode(hasAliasICPRecordals() ? aliasICPRecordals() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof Distribution)) {
            return false;
        }
        Distribution other = (Distribution) obj;
        return Objects.equals(id(), other.id()) && Objects.equals(arn(), other.arn()) && Objects.equals(status(), other.status())
                && Objects.equals(lastModifiedTime(), other.lastModifiedTime())
                && Objects.equals(inProgressInvalidationBatches(), other.inProgressInvalidationBatches())
                && Objects.equals(domainName(), other.domainName())
                && Objects.equals(activeTrustedSigners(), other.activeTrustedSigners())
                && Objects.equals(activeTrustedKeyGroups(), other.activeTrustedKeyGroups())
                && Objects.equals(distributionConfig(), other.distributionConfig())
                && hasAliasICPRecordals() == other.hasAliasICPRecordals()
                && Objects.equals(aliasICPRecordals(), other.aliasICPRecordals());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("Distribution").add("Id", id()).add("ARN", arn()).add("Status", status())
                .add("LastModifiedTime", lastModifiedTime())
                .add("InProgressInvalidationBatches", inProgressInvalidationBatches()).add("DomainName", domainName())
                .add("ActiveTrustedSigners", activeTrustedSigners()).add("ActiveTrustedKeyGroups", activeTrustedKeyGroups())
                .add("DistributionConfig", distributionConfig())
                .add("AliasICPRecordals", hasAliasICPRecordals() ? aliasICPRecordals() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Id":
            return Optional.ofNullable(clazz.cast(id()));
        case "ARN":
            return Optional.ofNullable(clazz.cast(arn()));
        case "Status":
            return Optional.ofNullable(clazz.cast(status()));
        case "LastModifiedTime":
            return Optional.ofNullable(clazz.cast(lastModifiedTime()));
        case "InProgressInvalidationBatches":
            return Optional.ofNullable(clazz.cast(inProgressInvalidationBatches()));
        case "DomainName":
            return Optional.ofNullable(clazz.cast(domainName()));
        case "ActiveTrustedSigners":
            return Optional.ofNullable(clazz.cast(activeTrustedSigners()));
        case "ActiveTrustedKeyGroups":
            return Optional.ofNullable(clazz.cast(activeTrustedKeyGroups()));
        case "DistributionConfig":
            return Optional.ofNullable(clazz.cast(distributionConfig()));
        case "AliasICPRecordals":
            return Optional.ofNullable(clazz.cast(aliasICPRecordals()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<Distribution, T> g) {
        return obj -> g.apply((Distribution) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, Distribution> {
        /**
         * <p>
         * The identifier for the distribution. For example: <code>EDFDVBD632BHDS5</code>.
         * </p>
         * 
         * @param id
         *        The identifier for the distribution. For example: <code>EDFDVBD632BHDS5</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder id(String id);

        /**
         * <p>
         * The ARN (Amazon Resource Name) for the distribution. For example:
         * <code>arn:aws:cloudfront::123456789012:distribution/EDFDVBD632BHDS5</code>, where <code>123456789012</code>
         * is your account ID.
         * </p>
         * 
         * @param arn
         *        The ARN (Amazon Resource Name) for the distribution. For example:
         *        <code>arn:aws:cloudfront::123456789012:distribution/EDFDVBD632BHDS5</code>, where
         *        <code>123456789012</code> is your account ID.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder arn(String arn);

        /**
         * <p>
         * This response element indicates the current status of the distribution. When the status is
         * <code>Deployed</code>, the distribution's information is fully propagated to all CloudFront edge locations.
         * </p>
         * 
         * @param status
         *        This response element indicates the current status of the distribution. When the status is
         *        <code>Deployed</code>, the distribution's information is fully propagated to all CloudFront edge
         *        locations.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder status(String status);

        /**
         * <p>
         * The date and time the distribution was last modified.
         * </p>
         * 
         * @param lastModifiedTime
         *        The date and time the distribution was last modified.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder lastModifiedTime(Instant lastModifiedTime);

        /**
         * <p>
         * The number of invalidation batches currently in progress.
         * </p>
         * 
         * @param inProgressInvalidationBatches
         *        The number of invalidation batches currently in progress.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder inProgressInvalidationBatches(Integer inProgressInvalidationBatches);

        /**
         * <p>
         * The domain name corresponding to the distribution, for example, <code>d111111abcdef8.cloudfront.net</code>.
         * </p>
         * 
         * @param domainName
         *        The domain name corresponding to the distribution, for example,
         *        <code>d111111abcdef8.cloudfront.net</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder domainName(String domainName);

        /**
         * <important>
         * <p>
         * We recommend using <code>TrustedKeyGroups</code> instead of <code>TrustedSigners</code>.
         * </p>
         * </important>
         * <p>
         * CloudFront automatically adds this field to the response if you’ve configured a cache behavior in this
         * distribution to serve private content using trusted signers. This field contains a list of account IDs and
         * the active CloudFront key pairs in each account that CloudFront can use to verify the signatures of signed
         * URLs or signed cookies.
         * </p>
         * 
         * @param activeTrustedSigners
         *        <p>
         *        We recommend using <code>TrustedKeyGroups</code> instead of <code>TrustedSigners</code>.
         *        </p>
         *        </important>
         *        <p>
         *        CloudFront automatically adds this field to the response if you’ve configured a cache behavior in this
         *        distribution to serve private content using trusted signers. This field contains a list of account IDs
         *        and the active CloudFront key pairs in each account that CloudFront can use to verify the signatures
         *        of signed URLs or signed cookies.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder activeTrustedSigners(ActiveTrustedSigners activeTrustedSigners);

        /**
         * <important>
         * <p>
         * We recommend using <code>TrustedKeyGroups</code> instead of <code>TrustedSigners</code>.
         * </p>
         * </important>
         * <p>
         * CloudFront automatically adds this field to the response if you’ve configured a cache behavior in this
         * distribution to serve private content using trusted signers. This field contains a list of account IDs and
         * the active CloudFront key pairs in each account that CloudFront can use to verify the signatures of signed
         * URLs or signed cookies.
         * </p>
         * This is a convenience that creates an instance of the {@link ActiveTrustedSigners.Builder} avoiding the need
         * to create one manually via {@link ActiveTrustedSigners#builder()}.
         *
         * When the {@link Consumer} completes, {@link ActiveTrustedSigners.Builder#build()} is called immediately and
         * its result is passed to {@link #activeTrustedSigners(ActiveTrustedSigners)}.
         * 
         * @param activeTrustedSigners
         *        a consumer that will call methods on {@link ActiveTrustedSigners.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #activeTrustedSigners(ActiveTrustedSigners)
         */
        default Builder activeTrustedSigners(Consumer<ActiveTrustedSigners.Builder> activeTrustedSigners) {
            return activeTrustedSigners(ActiveTrustedSigners.builder().applyMutation(activeTrustedSigners).build());
        }

        /**
         * <p>
         * CloudFront automatically adds this field to the response if you’ve configured a cache behavior in this
         * distribution to serve private content using key groups. This field contains a list of key groups and the
         * public keys in each key group that CloudFront can use to verify the signatures of signed URLs or signed
         * cookies.
         * </p>
         * 
         * @param activeTrustedKeyGroups
         *        CloudFront automatically adds this field to the response if you’ve configured a cache behavior in this
         *        distribution to serve private content using key groups. This field contains a list of key groups and
         *        the public keys in each key group that CloudFront can use to verify the signatures of signed URLs or
         *        signed cookies.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder activeTrustedKeyGroups(ActiveTrustedKeyGroups activeTrustedKeyGroups);

        /**
         * <p>
         * CloudFront automatically adds this field to the response if you’ve configured a cache behavior in this
         * distribution to serve private content using key groups. This field contains a list of key groups and the
         * public keys in each key group that CloudFront can use to verify the signatures of signed URLs or signed
         * cookies.
         * </p>
         * This is a convenience that creates an instance of the {@link ActiveTrustedKeyGroups.Builder} avoiding the
         * need to create one manually via {@link ActiveTrustedKeyGroups#builder()}.
         *
         * When the {@link Consumer} completes, {@link ActiveTrustedKeyGroups.Builder#build()} is called immediately and
         * its result is passed to {@link #activeTrustedKeyGroups(ActiveTrustedKeyGroups)}.
         * 
         * @param activeTrustedKeyGroups
         *        a consumer that will call methods on {@link ActiveTrustedKeyGroups.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #activeTrustedKeyGroups(ActiveTrustedKeyGroups)
         */
        default Builder activeTrustedKeyGroups(Consumer<ActiveTrustedKeyGroups.Builder> activeTrustedKeyGroups) {
            return activeTrustedKeyGroups(ActiveTrustedKeyGroups.builder().applyMutation(activeTrustedKeyGroups).build());
        }

        /**
         * <p>
         * The current configuration information for the distribution. Send a <code>GET</code> request to the
         * <code>/<i>CloudFront API version</i>/distribution ID/config</code> resource.
         * </p>
         * 
         * @param distributionConfig
         *        The current configuration information for the distribution. Send a <code>GET</code> request to the
         *        <code>/<i>CloudFront API version</i>/distribution ID/config</code> resource.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder distributionConfig(DistributionConfig distributionConfig);

        /**
         * <p>
         * The current configuration information for the distribution. Send a <code>GET</code> request to the
         * <code>/<i>CloudFront API version</i>/distribution ID/config</code> resource.
         * </p>
         * This is a convenience that creates an instance of the {@link DistributionConfig.Builder} avoiding the need to
         * create one manually via {@link DistributionConfig#builder()}.
         *
         * When the {@link Consumer} completes, {@link DistributionConfig.Builder#build()} is called immediately and its
         * result is passed to {@link #distributionConfig(DistributionConfig)}.
         * 
         * @param distributionConfig
         *        a consumer that will call methods on {@link DistributionConfig.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #distributionConfig(DistributionConfig)
         */
        default Builder distributionConfig(Consumer<DistributionConfig.Builder> distributionConfig) {
            return distributionConfig(DistributionConfig.builder().applyMutation(distributionConfig).build());
        }

        /**
         * <p>
         * Amazon Web Services services in China customers must file for an Internet Content Provider (ICP) recordal if
         * they want to serve content publicly on an alternate domain name, also known as a CNAME, that they've added to
         * CloudFront. AliasICPRecordal provides the ICP recordal status for CNAMEs associated with distributions.
         * </p>
         * <p>
         * For more information about ICP recordals, see <a
         * href="https://docs.amazonaws.cn/en_us/aws/latest/userguide/accounts-and-credentials.html"> Signup, Accounts,
         * and Credentials</a> in <i>Getting Started with Amazon Web Services services in China</i>.
         * </p>
         * 
         * @param aliasICPRecordals
         *        Amazon Web Services services in China customers must file for an Internet Content Provider (ICP)
         *        recordal if they want to serve content publicly on an alternate domain name, also known as a CNAME,
         *        that they've added to CloudFront. AliasICPRecordal provides the ICP recordal status for CNAMEs
         *        associated with distributions.</p>
         *        <p>
         *        For more information about ICP recordals, see <a
         *        href="https://docs.amazonaws.cn/en_us/aws/latest/userguide/accounts-and-credentials.html"> Signup,
         *        Accounts, and Credentials</a> in <i>Getting Started with Amazon Web Services services in China</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder aliasICPRecordals(Collection<AliasICPRecordal> aliasICPRecordals);

        /**
         * <p>
         * Amazon Web Services services in China customers must file for an Internet Content Provider (ICP) recordal if
         * they want to serve content publicly on an alternate domain name, also known as a CNAME, that they've added to
         * CloudFront. AliasICPRecordal provides the ICP recordal status for CNAMEs associated with distributions.
         * </p>
         * <p>
         * For more information about ICP recordals, see <a
         * href="https://docs.amazonaws.cn/en_us/aws/latest/userguide/accounts-and-credentials.html"> Signup, Accounts,
         * and Credentials</a> in <i>Getting Started with Amazon Web Services services in China</i>.
         * </p>
         * 
         * @param aliasICPRecordals
         *        Amazon Web Services services in China customers must file for an Internet Content Provider (ICP)
         *        recordal if they want to serve content publicly on an alternate domain name, also known as a CNAME,
         *        that they've added to CloudFront. AliasICPRecordal provides the ICP recordal status for CNAMEs
         *        associated with distributions.</p>
         *        <p>
         *        For more information about ICP recordals, see <a
         *        href="https://docs.amazonaws.cn/en_us/aws/latest/userguide/accounts-and-credentials.html"> Signup,
         *        Accounts, and Credentials</a> in <i>Getting Started with Amazon Web Services services in China</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder aliasICPRecordals(AliasICPRecordal... aliasICPRecordals);

        /**
         * <p>
         * Amazon Web Services services in China customers must file for an Internet Content Provider (ICP) recordal if
         * they want to serve content publicly on an alternate domain name, also known as a CNAME, that they've added to
         * CloudFront. AliasICPRecordal provides the ICP recordal status for CNAMEs associated with distributions.
         * </p>
         * <p>
         * For more information about ICP recordals, see <a
         * href="https://docs.amazonaws.cn/en_us/aws/latest/userguide/accounts-and-credentials.html"> Signup, Accounts,
         * and Credentials</a> in <i>Getting Started with Amazon Web Services services in China</i>.
         * </p>
         * This is a convenience that creates an instance of the {@link List<AliasICPRecordal>.Builder} avoiding the
         * need to create one manually via {@link List<AliasICPRecordal>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<AliasICPRecordal>.Builder#build()} is called immediately and
         * its result is passed to {@link #aliasICPRecordals(List<AliasICPRecordal>)}.
         * 
         * @param aliasICPRecordals
         *        a consumer that will call methods on {@link List<AliasICPRecordal>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #aliasICPRecordals(List<AliasICPRecordal>)
         */
        Builder aliasICPRecordals(Consumer<AliasICPRecordal.Builder>... aliasICPRecordals);
    }

    static final class BuilderImpl implements Builder {
        private String id;

        private String arn;

        private String status;

        private Instant lastModifiedTime;

        private Integer inProgressInvalidationBatches;

        private String domainName;

        private ActiveTrustedSigners activeTrustedSigners;

        private ActiveTrustedKeyGroups activeTrustedKeyGroups;

        private DistributionConfig distributionConfig;

        private List<AliasICPRecordal> aliasICPRecordals = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(Distribution model) {
            id(model.id);
            arn(model.arn);
            status(model.status);
            lastModifiedTime(model.lastModifiedTime);
            inProgressInvalidationBatches(model.inProgressInvalidationBatches);
            domainName(model.domainName);
            activeTrustedSigners(model.activeTrustedSigners);
            activeTrustedKeyGroups(model.activeTrustedKeyGroups);
            distributionConfig(model.distributionConfig);
            aliasICPRecordals(model.aliasICPRecordals);
        }

        public final String getId() {
            return id;
        }

        public final void setId(String id) {
            this.id = id;
        }

        @Override
        @Transient
        public final Builder id(String id) {
            this.id = id;
            return this;
        }

        public final String getArn() {
            return arn;
        }

        public final void setArn(String arn) {
            this.arn = arn;
        }

        @Override
        @Transient
        public final Builder arn(String arn) {
            this.arn = arn;
            return this;
        }

        public final String getStatus() {
            return status;
        }

        public final void setStatus(String status) {
            this.status = status;
        }

        @Override
        @Transient
        public final Builder status(String status) {
            this.status = status;
            return this;
        }

        public final Instant getLastModifiedTime() {
            return lastModifiedTime;
        }

        public final void setLastModifiedTime(Instant lastModifiedTime) {
            this.lastModifiedTime = lastModifiedTime;
        }

        @Override
        @Transient
        public final Builder lastModifiedTime(Instant lastModifiedTime) {
            this.lastModifiedTime = lastModifiedTime;
            return this;
        }

        public final Integer getInProgressInvalidationBatches() {
            return inProgressInvalidationBatches;
        }

        public final void setInProgressInvalidationBatches(Integer inProgressInvalidationBatches) {
            this.inProgressInvalidationBatches = inProgressInvalidationBatches;
        }

        @Override
        @Transient
        public final Builder inProgressInvalidationBatches(Integer inProgressInvalidationBatches) {
            this.inProgressInvalidationBatches = inProgressInvalidationBatches;
            return this;
        }

        public final String getDomainName() {
            return domainName;
        }

        public final void setDomainName(String domainName) {
            this.domainName = domainName;
        }

        @Override
        @Transient
        public final Builder domainName(String domainName) {
            this.domainName = domainName;
            return this;
        }

        public final ActiveTrustedSigners.Builder getActiveTrustedSigners() {
            return activeTrustedSigners != null ? activeTrustedSigners.toBuilder() : null;
        }

        public final void setActiveTrustedSigners(ActiveTrustedSigners.BuilderImpl activeTrustedSigners) {
            this.activeTrustedSigners = activeTrustedSigners != null ? activeTrustedSigners.build() : null;
        }

        @Override
        @Transient
        public final Builder activeTrustedSigners(ActiveTrustedSigners activeTrustedSigners) {
            this.activeTrustedSigners = activeTrustedSigners;
            return this;
        }

        public final ActiveTrustedKeyGroups.Builder getActiveTrustedKeyGroups() {
            return activeTrustedKeyGroups != null ? activeTrustedKeyGroups.toBuilder() : null;
        }

        public final void setActiveTrustedKeyGroups(ActiveTrustedKeyGroups.BuilderImpl activeTrustedKeyGroups) {
            this.activeTrustedKeyGroups = activeTrustedKeyGroups != null ? activeTrustedKeyGroups.build() : null;
        }

        @Override
        @Transient
        public final Builder activeTrustedKeyGroups(ActiveTrustedKeyGroups activeTrustedKeyGroups) {
            this.activeTrustedKeyGroups = activeTrustedKeyGroups;
            return this;
        }

        public final DistributionConfig.Builder getDistributionConfig() {
            return distributionConfig != null ? distributionConfig.toBuilder() : null;
        }

        public final void setDistributionConfig(DistributionConfig.BuilderImpl distributionConfig) {
            this.distributionConfig = distributionConfig != null ? distributionConfig.build() : null;
        }

        @Override
        @Transient
        public final Builder distributionConfig(DistributionConfig distributionConfig) {
            this.distributionConfig = distributionConfig;
            return this;
        }

        public final List<AliasICPRecordal.Builder> getAliasICPRecordals() {
            List<AliasICPRecordal.Builder> result = AliasICPRecordalsCopier.copyToBuilder(this.aliasICPRecordals);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setAliasICPRecordals(Collection<AliasICPRecordal.BuilderImpl> aliasICPRecordals) {
            this.aliasICPRecordals = AliasICPRecordalsCopier.copyFromBuilder(aliasICPRecordals);
        }

        @Override
        @Transient
        public final Builder aliasICPRecordals(Collection<AliasICPRecordal> aliasICPRecordals) {
            this.aliasICPRecordals = AliasICPRecordalsCopier.copy(aliasICPRecordals);
            return this;
        }

        @Override
        @Transient
        @SafeVarargs
        public final Builder aliasICPRecordals(AliasICPRecordal... aliasICPRecordals) {
            aliasICPRecordals(Arrays.asList(aliasICPRecordals));
            return this;
        }

        @Override
        @Transient
        @SafeVarargs
        public final Builder aliasICPRecordals(Consumer<AliasICPRecordal.Builder>... aliasICPRecordals) {
            aliasICPRecordals(Stream.of(aliasICPRecordals).map(c -> AliasICPRecordal.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        @Override
        public Distribution build() {
            return new Distribution(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
