/*
 * Copyright 2012-2017 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.cloudfront.model;

import java.util.Arrays;
import java.util.Collection;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import javax.annotation.Generated;
import software.amazon.awssdk.core.runtime.TypeConverter;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * A complex type that controls whether CloudFront caches the response to requests using the specified HTTP methods.
 * There are two choices:
 * </p>
 * <ul>
 * <li>
 * <p>
 * CloudFront caches responses to <code>GET</code> and <code>HEAD</code> requests.
 * </p>
 * </li>
 * <li>
 * <p>
 * CloudFront caches responses to <code>GET</code>, <code>HEAD</code>, and <code>OPTIONS</code> requests.
 * </p>
 * </li>
 * </ul>
 * <p>
 * If you pick the second choice for your Amazon S3 Origin, you may need to forward Access-Control-Request-Method,
 * Access-Control-Request-Headers, and Origin headers for the responses to be cached correctly.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public class CachedMethods implements ToCopyableBuilder<CachedMethods.Builder, CachedMethods> {
    private final Integer quantity;

    private final List<String> items;

    private CachedMethods(BuilderImpl builder) {
        this.quantity = builder.quantity;
        this.items = builder.items;
    }

    /**
     * <p>
     * The number of HTTP methods for which you want CloudFront to cache responses. Valid values are <code>2</code> (for
     * caching responses to <code>GET</code> and <code>HEAD</code> requests) and <code>3</code> (for caching responses
     * to <code>GET</code>, <code>HEAD</code>, and <code>OPTIONS</code> requests).
     * </p>
     * 
     * @return The number of HTTP methods for which you want CloudFront to cache responses. Valid values are
     *         <code>2</code> (for caching responses to <code>GET</code> and <code>HEAD</code> requests) and
     *         <code>3</code> (for caching responses to <code>GET</code>, <code>HEAD</code>, and <code>OPTIONS</code>
     *         requests).
     */
    public Integer quantity() {
        return quantity;
    }

    /**
     * <p>
     * A complex type that contains the HTTP methods that you want CloudFront to cache responses to.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return A complex type that contains the HTTP methods that you want CloudFront to cache responses to.
     */
    public List<Method> items() {
        return TypeConverter.convert(items, Method::fromValue);
    }

    /**
     * <p>
     * A complex type that contains the HTTP methods that you want CloudFront to cache responses to.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return A complex type that contains the HTTP methods that you want CloudFront to cache responses to.
     */
    public List<String> itemsStrings() {
        return items;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(quantity());
        hashCode = 31 * hashCode + Objects.hashCode(itemsStrings());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof CachedMethods)) {
            return false;
        }
        CachedMethods other = (CachedMethods) obj;
        return Objects.equals(quantity(), other.quantity()) && Objects.equals(itemsStrings(), other.itemsStrings());
    }

    @Override
    public String toString() {
        StringBuilder sb = new StringBuilder("{");
        if (quantity() != null) {
            sb.append("Quantity: ").append(quantity()).append(",");
        }
        if (itemsStrings() != null) {
            sb.append("Items: ").append(itemsStrings()).append(",");
        }
        if (sb.length() > 1) {
            sb.setLength(sb.length() - 1);
        }
        sb.append("}");
        return sb.toString();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Quantity":
            return Optional.of(clazz.cast(quantity()));
        case "Items":
            return Optional.of(clazz.cast(itemsStrings()));
        default:
            return Optional.empty();
        }
    }

    public interface Builder extends CopyableBuilder<Builder, CachedMethods> {
        /**
         * <p>
         * The number of HTTP methods for which you want CloudFront to cache responses. Valid values are <code>2</code>
         * (for caching responses to <code>GET</code> and <code>HEAD</code> requests) and <code>3</code> (for caching
         * responses to <code>GET</code>, <code>HEAD</code>, and <code>OPTIONS</code> requests).
         * </p>
         * 
         * @param quantity
         *        The number of HTTP methods for which you want CloudFront to cache responses. Valid values are
         *        <code>2</code> (for caching responses to <code>GET</code> and <code>HEAD</code> requests) and
         *        <code>3</code> (for caching responses to <code>GET</code>, <code>HEAD</code>, and <code>OPTIONS</code>
         *        requests).
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder quantity(Integer quantity);

        /**
         * <p>
         * A complex type that contains the HTTP methods that you want CloudFront to cache responses to.
         * </p>
         * 
         * @param items
         *        A complex type that contains the HTTP methods that you want CloudFront to cache responses to.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder items(Collection<String> items);

        /**
         * <p>
         * A complex type that contains the HTTP methods that you want CloudFront to cache responses to.
         * </p>
         * 
         * @param items
         *        A complex type that contains the HTTP methods that you want CloudFront to cache responses to.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder items(String... items);
    }

    static final class BuilderImpl implements Builder {
        private Integer quantity;

        private List<String> items;

        private BuilderImpl() {
        }

        private BuilderImpl(CachedMethods model) {
            quantity(model.quantity);
            items(model.items);
        }

        public final Integer getQuantity() {
            return quantity;
        }

        @Override
        public final Builder quantity(Integer quantity) {
            this.quantity = quantity;
            return this;
        }

        public final void setQuantity(Integer quantity) {
            this.quantity = quantity;
        }

        public final Collection<String> getItems() {
            return items;
        }

        @Override
        public final Builder items(Collection<String> items) {
            this.items = MethodsListCopier.copy(items);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder items(String... items) {
            items(Arrays.asList(items));
            return this;
        }

        public final void setItems(Collection<String> items) {
            this.items = MethodsListCopier.copy(items);
        }

        @Override
        public CachedMethods build() {
            return new CachedMethods(this);
        }
    }
}
