/*
 * Copyright 2012-2017 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.cloudfront.model;

import java.util.Arrays;
import java.util.Collection;
import java.util.List;
import javax.annotation.Generated;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * A complex type that specifies the AWS accounts, if any, that you want to allow to create signed URLs for private
 * content.
 * </p>
 * <p>
 * If you want to require signed URLs in requests for objects in the target origin that match the
 * <code>PathPattern</code> for this cache behavior, specify <code>true</code> for <code>Enabled</code>, and specify the
 * applicable values for <code>Quantity</code> and <code>Items</code>. For more information, see <a
 * href="http://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/PrivateContent.html">Serving Private Content
 * through CloudFront</a> in the <i>Amazon Amazon CloudFront Developer Guide</i>.
 * </p>
 * <p>
 * If you don't want to require signed URLs in requests for objects that match <code>PathPattern</code>, specify
 * <code>false</code> for <code>Enabled</code> and <code>0</code> for <code>Quantity</code>. Omit <code>Items</code>.
 * </p>
 * <p>
 * To add, change, or remove one or more trusted signers, change <code>Enabled</code> to <code>true</code> (if it's
 * currently <code>false</code>), change <code>Quantity</code> as applicable, and specify all of the trusted signers
 * that you want to include in the updated distribution.
 * </p>
 * <p>
 * For more information about updating the distribution configuration, see <a>DistributionConfig</a> .
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public class TrustedSigners implements ToCopyableBuilder<TrustedSigners.Builder, TrustedSigners> {
    private final Boolean enabled;

    private final Integer quantity;

    private final List<String> items;

    private TrustedSigners(BuilderImpl builder) {
        this.enabled = builder.enabled;
        this.quantity = builder.quantity;
        this.items = builder.items;
    }

    /**
     * <p>
     * Specifies whether you want to require viewers to use signed URLs to access the files specified by
     * <code>PathPattern</code> and <code>TargetOriginId</code>.
     * </p>
     * 
     * @return Specifies whether you want to require viewers to use signed URLs to access the files specified by
     *         <code>PathPattern</code> and <code>TargetOriginId</code>.
     */
    public Boolean enabled() {
        return enabled;
    }

    /**
     * <p>
     * The number of trusted signers for this cache behavior.
     * </p>
     * 
     * @return The number of trusted signers for this cache behavior.
     */
    public Integer quantity() {
        return quantity;
    }

    /**
     * <p>
     * <b>Optional</b>: A complex type that contains trusted signers for this cache behavior. If <code>Quantity</code>
     * is <code>0</code>, you can omit <code>Items</code>.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return <b>Optional</b>: A complex type that contains trusted signers for this cache behavior. If
     *         <code>Quantity</code> is <code>0</code>, you can omit <code>Items</code>.
     */
    public List<String> items() {
        return items;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + ((enabled() == null) ? 0 : enabled().hashCode());
        hashCode = 31 * hashCode + ((quantity() == null) ? 0 : quantity().hashCode());
        hashCode = 31 * hashCode + ((items() == null) ? 0 : items().hashCode());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof TrustedSigners)) {
            return false;
        }
        TrustedSigners other = (TrustedSigners) obj;
        if (other.enabled() == null ^ this.enabled() == null) {
            return false;
        }
        if (other.enabled() != null && !other.enabled().equals(this.enabled())) {
            return false;
        }
        if (other.quantity() == null ^ this.quantity() == null) {
            return false;
        }
        if (other.quantity() != null && !other.quantity().equals(this.quantity())) {
            return false;
        }
        if (other.items() == null ^ this.items() == null) {
            return false;
        }
        if (other.items() != null && !other.items().equals(this.items())) {
            return false;
        }
        return true;
    }

    @Override
    public String toString() {
        StringBuilder sb = new StringBuilder();
        sb.append("{");
        if (enabled() != null) {
            sb.append("Enabled: ").append(enabled()).append(",");
        }
        if (quantity() != null) {
            sb.append("Quantity: ").append(quantity()).append(",");
        }
        if (items() != null) {
            sb.append("Items: ").append(items()).append(",");
        }
        sb.append("}");
        return sb.toString();
    }

    public interface Builder extends CopyableBuilder<Builder, TrustedSigners> {
        /**
         * <p>
         * Specifies whether you want to require viewers to use signed URLs to access the files specified by
         * <code>PathPattern</code> and <code>TargetOriginId</code>.
         * </p>
         * 
         * @param enabled
         *        Specifies whether you want to require viewers to use signed URLs to access the files specified by
         *        <code>PathPattern</code> and <code>TargetOriginId</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder enabled(Boolean enabled);

        /**
         * <p>
         * The number of trusted signers for this cache behavior.
         * </p>
         * 
         * @param quantity
         *        The number of trusted signers for this cache behavior.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder quantity(Integer quantity);

        /**
         * <p>
         * <b>Optional</b>: A complex type that contains trusted signers for this cache behavior. If
         * <code>Quantity</code> is <code>0</code>, you can omit <code>Items</code>.
         * </p>
         * 
         * @param items
         *        <b>Optional</b>: A complex type that contains trusted signers for this cache behavior. If
         *        <code>Quantity</code> is <code>0</code>, you can omit <code>Items</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder items(Collection<String> items);

        /**
         * <p>
         * <b>Optional</b>: A complex type that contains trusted signers for this cache behavior. If
         * <code>Quantity</code> is <code>0</code>, you can omit <code>Items</code>.
         * </p>
         * 
         * @param items
         *        <b>Optional</b>: A complex type that contains trusted signers for this cache behavior. If
         *        <code>Quantity</code> is <code>0</code>, you can omit <code>Items</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder items(String... items);
    }

    private static final class BuilderImpl implements Builder {
        private Boolean enabled;

        private Integer quantity;

        private List<String> items;

        private BuilderImpl() {
        }

        private BuilderImpl(TrustedSigners model) {
            setEnabled(model.enabled);
            setQuantity(model.quantity);
            setItems(model.items);
        }

        public final Boolean getEnabled() {
            return enabled;
        }

        @Override
        public final Builder enabled(Boolean enabled) {
            this.enabled = enabled;
            return this;
        }

        public final void setEnabled(Boolean enabled) {
            this.enabled = enabled;
        }

        public final Integer getQuantity() {
            return quantity;
        }

        @Override
        public final Builder quantity(Integer quantity) {
            this.quantity = quantity;
            return this;
        }

        public final void setQuantity(Integer quantity) {
            this.quantity = quantity;
        }

        public final Collection<String> getItems() {
            return items;
        }

        @Override
        public final Builder items(Collection<String> items) {
            this.items = AwsAccountNumberListCopier.copy(items);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder items(String... items) {
            items(Arrays.asList(items));
            return this;
        }

        public final void setItems(Collection<String> items) {
            this.items = AwsAccountNumberListCopier.copy(items);
        }

        @Override
        public TrustedSigners build() {
            return new TrustedSigners(this);
        }
    }
}
