/*
 * Copyright 2012-2017 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.cloudfront.model;

import java.util.Arrays;
import java.util.Collection;
import java.util.List;
import javax.annotation.Generated;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * A complex type that specifies the headers that you want CloudFront to forward to the origin for this cache behavior.
 * </p>
 * <p>
 * For the headers that you specify, CloudFront also caches separate versions of a specified object based on the header
 * values in viewer requests. For example, suppose viewer requests for <code>logo.jpg</code> contain a custom
 * <code>Product</code> header that has a value of either <code>Acme</code> or <code>Apex</code>, and you configure
 * CloudFront to cache your content based on values in the <code>Product</code> header. CloudFront forwards the
 * <code>Product</code> header to the origin and caches the response from the origin once for each header value. For
 * more information about caching based on header values, see <a
 * href="http://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/header-caching.html">How CloudFront Forwards
 * and Caches Headers</a> in the <i>Amazon CloudFront Developer Guide</i>.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public class Headers implements ToCopyableBuilder<Headers.Builder, Headers> {
    private final Integer quantity;

    private final List<String> items;

    private Headers(BuilderImpl builder) {
        this.quantity = builder.quantity;
        this.items = builder.items;
    }

    /**
     * <p>
     * The number of different headers that you want CloudFront to forward to the origin for this cache behavior. You
     * can configure each cache behavior in a web distribution to do one of the following:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <b>Forward all headers to your origin</b>: Specify <code>1</code> for <code>Quantity</code> and <code>*</code>
     * for <code>Name</code>.
     * </p>
     * <important>
     * <p>
     * If you configure CloudFront to forward all headers to your origin, CloudFront doesn't cache the objects
     * associated with this cache behavior. Instead, it sends every request to the origin.
     * </p>
     * </important></li>
     * <li>
     * <p>
     * <i>Forward a whitelist of headers you specify</i>: Specify the number of headers that you want to forward, and
     * specify the header names in <code>Name</code> elements. CloudFront caches your objects based on the values in all
     * of the specified headers. CloudFront also forwards the headers that it forwards by default, but it caches your
     * objects based only on the headers that you specify.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>Forward only the default headers</b>: Specify <code>0</code> for <code>Quantity</code> and omit
     * <code>Items</code>. In this configuration, CloudFront doesn't cache based on the values in the request headers.
     * </p>
     * </li>
     * </ul>
     * 
     * @return The number of different headers that you want CloudFront to forward to the origin for this cache
     *         behavior. You can configure each cache behavior in a web distribution to do one of the following:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <b>Forward all headers to your origin</b>: Specify <code>1</code> for <code>Quantity</code> and
     *         <code>*</code> for <code>Name</code>.
     *         </p>
     *         <important>
     *         <p>
     *         If you configure CloudFront to forward all headers to your origin, CloudFront doesn't cache the objects
     *         associated with this cache behavior. Instead, it sends every request to the origin.
     *         </p>
     *         </important></li>
     *         <li>
     *         <p>
     *         <i>Forward a whitelist of headers you specify</i>: Specify the number of headers that you want to
     *         forward, and specify the header names in <code>Name</code> elements. CloudFront caches your objects based
     *         on the values in all of the specified headers. CloudFront also forwards the headers that it forwards by
     *         default, but it caches your objects based only on the headers that you specify.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>Forward only the default headers</b>: Specify <code>0</code> for <code>Quantity</code> and omit
     *         <code>Items</code>. In this configuration, CloudFront doesn't cache based on the values in the request
     *         headers.
     *         </p>
     *         </li>
     */
    public Integer quantity() {
        return quantity;
    }

    /**
     * <p>
     * A complex type that contains one <code>Name</code> element for each header that you want CloudFront to forward to
     * the origin and to vary on for this cache behavior. If <code>Quantity</code> is <code>0</code>, omit
     * <code>Items</code>.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return A complex type that contains one <code>Name</code> element for each header that you want CloudFront to
     *         forward to the origin and to vary on for this cache behavior. If <code>Quantity</code> is <code>0</code>,
     *         omit <code>Items</code>.
     */
    public List<String> items() {
        return items;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + ((quantity() == null) ? 0 : quantity().hashCode());
        hashCode = 31 * hashCode + ((items() == null) ? 0 : items().hashCode());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof Headers)) {
            return false;
        }
        Headers other = (Headers) obj;
        if (other.quantity() == null ^ this.quantity() == null) {
            return false;
        }
        if (other.quantity() != null && !other.quantity().equals(this.quantity())) {
            return false;
        }
        if (other.items() == null ^ this.items() == null) {
            return false;
        }
        if (other.items() != null && !other.items().equals(this.items())) {
            return false;
        }
        return true;
    }

    @Override
    public String toString() {
        StringBuilder sb = new StringBuilder();
        sb.append("{");
        if (quantity() != null) {
            sb.append("Quantity: ").append(quantity()).append(",");
        }
        if (items() != null) {
            sb.append("Items: ").append(items()).append(",");
        }
        sb.append("}");
        return sb.toString();
    }

    public interface Builder extends CopyableBuilder<Builder, Headers> {
        /**
         * <p>
         * The number of different headers that you want CloudFront to forward to the origin for this cache behavior.
         * You can configure each cache behavior in a web distribution to do one of the following:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <b>Forward all headers to your origin</b>: Specify <code>1</code> for <code>Quantity</code> and
         * <code>*</code> for <code>Name</code>.
         * </p>
         * <important>
         * <p>
         * If you configure CloudFront to forward all headers to your origin, CloudFront doesn't cache the objects
         * associated with this cache behavior. Instead, it sends every request to the origin.
         * </p>
         * </important></li>
         * <li>
         * <p>
         * <i>Forward a whitelist of headers you specify</i>: Specify the number of headers that you want to forward,
         * and specify the header names in <code>Name</code> elements. CloudFront caches your objects based on the
         * values in all of the specified headers. CloudFront also forwards the headers that it forwards by default, but
         * it caches your objects based only on the headers that you specify.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b>Forward only the default headers</b>: Specify <code>0</code> for <code>Quantity</code> and omit
         * <code>Items</code>. In this configuration, CloudFront doesn't cache based on the values in the request
         * headers.
         * </p>
         * </li>
         * </ul>
         * 
         * @param quantity
         *        The number of different headers that you want CloudFront to forward to the origin for this cache
         *        behavior. You can configure each cache behavior in a web distribution to do one of the following:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <b>Forward all headers to your origin</b>: Specify <code>1</code> for <code>Quantity</code> and
         *        <code>*</code> for <code>Name</code>.
         *        </p>
         *        <important>
         *        <p>
         *        If you configure CloudFront to forward all headers to your origin, CloudFront doesn't cache the
         *        objects associated with this cache behavior. Instead, it sends every request to the origin.
         *        </p>
         *        </important></li>
         *        <li>
         *        <p>
         *        <i>Forward a whitelist of headers you specify</i>: Specify the number of headers that you want to
         *        forward, and specify the header names in <code>Name</code> elements. CloudFront caches your objects
         *        based on the values in all of the specified headers. CloudFront also forwards the headers that it
         *        forwards by default, but it caches your objects based only on the headers that you specify.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b>Forward only the default headers</b>: Specify <code>0</code> for <code>Quantity</code> and omit
         *        <code>Items</code>. In this configuration, CloudFront doesn't cache based on the values in the request
         *        headers.
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder quantity(Integer quantity);

        /**
         * <p>
         * A complex type that contains one <code>Name</code> element for each header that you want CloudFront to
         * forward to the origin and to vary on for this cache behavior. If <code>Quantity</code> is <code>0</code>,
         * omit <code>Items</code>.
         * </p>
         * 
         * @param items
         *        A complex type that contains one <code>Name</code> element for each header that you want CloudFront to
         *        forward to the origin and to vary on for this cache behavior. If <code>Quantity</code> is
         *        <code>0</code>, omit <code>Items</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder items(Collection<String> items);

        /**
         * <p>
         * A complex type that contains one <code>Name</code> element for each header that you want CloudFront to
         * forward to the origin and to vary on for this cache behavior. If <code>Quantity</code> is <code>0</code>,
         * omit <code>Items</code>.
         * </p>
         * 
         * @param items
         *        A complex type that contains one <code>Name</code> element for each header that you want CloudFront to
         *        forward to the origin and to vary on for this cache behavior. If <code>Quantity</code> is
         *        <code>0</code>, omit <code>Items</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder items(String... items);
    }

    private static final class BuilderImpl implements Builder {
        private Integer quantity;

        private List<String> items;

        private BuilderImpl() {
        }

        private BuilderImpl(Headers model) {
            setQuantity(model.quantity);
            setItems(model.items);
        }

        public final Integer getQuantity() {
            return quantity;
        }

        @Override
        public final Builder quantity(Integer quantity) {
            this.quantity = quantity;
            return this;
        }

        public final void setQuantity(Integer quantity) {
            this.quantity = quantity;
        }

        public final Collection<String> getItems() {
            return items;
        }

        @Override
        public final Builder items(Collection<String> items) {
            this.items = HeaderListCopier.copy(items);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder items(String... items) {
            items(Arrays.asList(items));
            return this;
        }

        public final void setItems(Collection<String> items) {
            this.items = HeaderListCopier.copy(items);
        }

        @SafeVarargs
        public final void setItems(String... items) {
            items(Arrays.asList(items));
        }

        @Override
        public Headers build() {
            return new Headers(this);
        }
    }
}
