/*
 * Copyright 2013-2018 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.cloudfront.model;

import java.util.Objects;
import java.util.Optional;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * A complex type that controls:
 * </p>
 * <ul>
 * <li>
 * <p>
 * Whether CloudFront replaces HTTP status codes in the 4xx and 5xx range with custom error messages before returning
 * the response to the viewer.
 * </p>
 * </li>
 * <li>
 * <p>
 * How long CloudFront caches HTTP status codes in the 4xx and 5xx range.
 * </p>
 * </li>
 * </ul>
 * <p>
 * For more information about custom error pages, see <a
 * href="http://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/custom-error-pages.html">Customizing Error
 * Responses</a> in the <i>Amazon CloudFront Developer Guide</i>.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class CustomErrorResponse implements ToCopyableBuilder<CustomErrorResponse.Builder, CustomErrorResponse> {
    private final Integer errorCode;

    private final String responsePagePath;

    private final String responseCode;

    private final Long errorCachingMinTTL;

    private CustomErrorResponse(BuilderImpl builder) {
        this.errorCode = builder.errorCode;
        this.responsePagePath = builder.responsePagePath;
        this.responseCode = builder.responseCode;
        this.errorCachingMinTTL = builder.errorCachingMinTTL;
    }

    /**
     * <p>
     * The HTTP status code for which you want to specify a custom error page and/or a caching duration.
     * </p>
     * 
     * @return The HTTP status code for which you want to specify a custom error page and/or a caching duration.
     */
    public Integer errorCode() {
        return errorCode;
    }

    /**
     * <p>
     * The path to the custom error page that you want CloudFront to return to a viewer when your origin returns the
     * HTTP status code specified by <code>ErrorCode</code>, for example, <code>/4xx-errors/403-forbidden.html</code>.
     * If you want to store your objects and your custom error pages in different locations, your distribution must
     * include a cache behavior for which the following is true:
     * </p>
     * <ul>
     * <li>
     * <p>
     * The value of <code>PathPattern</code> matches the path to your custom error messages. For example, suppose you
     * saved custom error pages for 4xx errors in an Amazon S3 bucket in a directory named <code>/4xx-errors</code>.
     * Your distribution must include a cache behavior for which the path pattern routes requests for your custom error
     * pages to that location, for example, <code>/4xx-errors/*</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * The value of <code>TargetOriginId</code> specifies the value of the <code>ID</code> element for the origin that
     * contains your custom error pages.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If you specify a value for <code>ResponsePagePath</code>, you must also specify a value for
     * <code>ResponseCode</code>. If you don't want to specify a value, include an empty element,
     * <code>&lt;ResponsePagePath&gt;</code>, in the XML document.
     * </p>
     * <p>
     * We recommend that you store custom error pages in an Amazon S3 bucket. If you store custom error pages on an HTTP
     * server and the server starts to return 5xx errors, CloudFront can't get the files that you want to return to
     * viewers because the origin server is unavailable.
     * </p>
     * 
     * @return The path to the custom error page that you want CloudFront to return to a viewer when your origin returns
     *         the HTTP status code specified by <code>ErrorCode</code>, for example,
     *         <code>/4xx-errors/403-forbidden.html</code>. If you want to store your objects and your custom error
     *         pages in different locations, your distribution must include a cache behavior for which the following is
     *         true:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         The value of <code>PathPattern</code> matches the path to your custom error messages. For example,
     *         suppose you saved custom error pages for 4xx errors in an Amazon S3 bucket in a directory named
     *         <code>/4xx-errors</code>. Your distribution must include a cache behavior for which the path pattern
     *         routes requests for your custom error pages to that location, for example, <code>/4xx-errors/*</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         The value of <code>TargetOriginId</code> specifies the value of the <code>ID</code> element for the
     *         origin that contains your custom error pages.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         If you specify a value for <code>ResponsePagePath</code>, you must also specify a value for
     *         <code>ResponseCode</code>. If you don't want to specify a value, include an empty element,
     *         <code>&lt;ResponsePagePath&gt;</code>, in the XML document.
     *         </p>
     *         <p>
     *         We recommend that you store custom error pages in an Amazon S3 bucket. If you store custom error pages on
     *         an HTTP server and the server starts to return 5xx errors, CloudFront can't get the files that you want
     *         to return to viewers because the origin server is unavailable.
     */
    public String responsePagePath() {
        return responsePagePath;
    }

    /**
     * <p>
     * The HTTP status code that you want CloudFront to return to the viewer along with the custom error page. There are
     * a variety of reasons that you might want CloudFront to return a status code different from the status code that
     * your origin returned to CloudFront, for example:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Some Internet devices (some firewalls and corporate proxies, for example) intercept HTTP 4xx and 5xx and prevent
     * the response from being returned to the viewer. If you substitute <code>200</code>, the response typically won't
     * be intercepted.
     * </p>
     * </li>
     * <li>
     * <p>
     * If you don't care about distinguishing among different client errors or server errors, you can specify
     * <code>400</code> or <code>500</code> as the <code>ResponseCode</code> for all 4xx or 5xx errors.
     * </p>
     * </li>
     * <li>
     * <p>
     * You might want to return a <code>200</code> status code (OK) and static website so your customers don't know that
     * your website is down.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If you specify a value for <code>ResponseCode</code>, you must also specify a value for
     * <code>ResponsePagePath</code>. If you don't want to specify a value, include an empty element,
     * <code>&lt;ResponseCode&gt;</code>, in the XML document.
     * </p>
     * 
     * @return The HTTP status code that you want CloudFront to return to the viewer along with the custom error page.
     *         There are a variety of reasons that you might want CloudFront to return a status code different from the
     *         status code that your origin returned to CloudFront, for example:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         Some Internet devices (some firewalls and corporate proxies, for example) intercept HTTP 4xx and 5xx and
     *         prevent the response from being returned to the viewer. If you substitute <code>200</code>, the response
     *         typically won't be intercepted.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         If you don't care about distinguishing among different client errors or server errors, you can specify
     *         <code>400</code> or <code>500</code> as the <code>ResponseCode</code> for all 4xx or 5xx errors.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         You might want to return a <code>200</code> status code (OK) and static website so your customers don't
     *         know that your website is down.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         If you specify a value for <code>ResponseCode</code>, you must also specify a value for
     *         <code>ResponsePagePath</code>. If you don't want to specify a value, include an empty element,
     *         <code>&lt;ResponseCode&gt;</code>, in the XML document.
     */
    public String responseCode() {
        return responseCode;
    }

    /**
     * <p>
     * The minimum amount of time, in seconds, that you want CloudFront to cache the HTTP status code specified in
     * <code>ErrorCode</code>. When this time period has elapsed, CloudFront queries your origin to see whether the
     * problem that caused the error has been resolved and the requested object is now available.
     * </p>
     * <p>
     * If you don't want to specify a value, include an empty element, <code>&lt;ErrorCachingMinTTL&gt;</code>, in the
     * XML document.
     * </p>
     * <p>
     * For more information, see <a
     * href="http://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/custom-error-pages.html">Customizing
     * Error Responses</a> in the <i>Amazon CloudFront Developer Guide</i>.
     * </p>
     * 
     * @return The minimum amount of time, in seconds, that you want CloudFront to cache the HTTP status code specified
     *         in <code>ErrorCode</code>. When this time period has elapsed, CloudFront queries your origin to see
     *         whether the problem that caused the error has been resolved and the requested object is now
     *         available.</p>
     *         <p>
     *         If you don't want to specify a value, include an empty element, <code>&lt;ErrorCachingMinTTL&gt;</code>,
     *         in the XML document.
     *         </p>
     *         <p>
     *         For more information, see <a
     *         href="http://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/custom-error-pages.html"
     *         >Customizing Error Responses</a> in the <i>Amazon CloudFront Developer Guide</i>.
     */
    public Long errorCachingMinTTL() {
        return errorCachingMinTTL;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(errorCode());
        hashCode = 31 * hashCode + Objects.hashCode(responsePagePath());
        hashCode = 31 * hashCode + Objects.hashCode(responseCode());
        hashCode = 31 * hashCode + Objects.hashCode(errorCachingMinTTL());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof CustomErrorResponse)) {
            return false;
        }
        CustomErrorResponse other = (CustomErrorResponse) obj;
        return Objects.equals(errorCode(), other.errorCode()) && Objects.equals(responsePagePath(), other.responsePagePath())
                && Objects.equals(responseCode(), other.responseCode())
                && Objects.equals(errorCachingMinTTL(), other.errorCachingMinTTL());
    }

    @Override
    public String toString() {
        return ToString.builder("CustomErrorResponse").add("ErrorCode", errorCode()).add("ResponsePagePath", responsePagePath())
                .add("ResponseCode", responseCode()).add("ErrorCachingMinTTL", errorCachingMinTTL()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "ErrorCode":
            return Optional.ofNullable(clazz.cast(errorCode()));
        case "ResponsePagePath":
            return Optional.ofNullable(clazz.cast(responsePagePath()));
        case "ResponseCode":
            return Optional.ofNullable(clazz.cast(responseCode()));
        case "ErrorCachingMinTTL":
            return Optional.ofNullable(clazz.cast(errorCachingMinTTL()));
        default:
            return Optional.empty();
        }
    }

    public interface Builder extends CopyableBuilder<Builder, CustomErrorResponse> {
        /**
         * <p>
         * The HTTP status code for which you want to specify a custom error page and/or a caching duration.
         * </p>
         * 
         * @param errorCode
         *        The HTTP status code for which you want to specify a custom error page and/or a caching duration.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder errorCode(Integer errorCode);

        /**
         * <p>
         * The path to the custom error page that you want CloudFront to return to a viewer when your origin returns the
         * HTTP status code specified by <code>ErrorCode</code>, for example,
         * <code>/4xx-errors/403-forbidden.html</code>. If you want to store your objects and your custom error pages in
         * different locations, your distribution must include a cache behavior for which the following is true:
         * </p>
         * <ul>
         * <li>
         * <p>
         * The value of <code>PathPattern</code> matches the path to your custom error messages. For example, suppose
         * you saved custom error pages for 4xx errors in an Amazon S3 bucket in a directory named
         * <code>/4xx-errors</code>. Your distribution must include a cache behavior for which the path pattern routes
         * requests for your custom error pages to that location, for example, <code>/4xx-errors/*</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * The value of <code>TargetOriginId</code> specifies the value of the <code>ID</code> element for the origin
         * that contains your custom error pages.
         * </p>
         * </li>
         * </ul>
         * <p>
         * If you specify a value for <code>ResponsePagePath</code>, you must also specify a value for
         * <code>ResponseCode</code>. If you don't want to specify a value, include an empty element,
         * <code>&lt;ResponsePagePath&gt;</code>, in the XML document.
         * </p>
         * <p>
         * We recommend that you store custom error pages in an Amazon S3 bucket. If you store custom error pages on an
         * HTTP server and the server starts to return 5xx errors, CloudFront can't get the files that you want to
         * return to viewers because the origin server is unavailable.
         * </p>
         * 
         * @param responsePagePath
         *        The path to the custom error page that you want CloudFront to return to a viewer when your origin
         *        returns the HTTP status code specified by <code>ErrorCode</code>, for example,
         *        <code>/4xx-errors/403-forbidden.html</code>. If you want to store your objects and your custom error
         *        pages in different locations, your distribution must include a cache behavior for which the following
         *        is true:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        The value of <code>PathPattern</code> matches the path to your custom error messages. For example,
         *        suppose you saved custom error pages for 4xx errors in an Amazon S3 bucket in a directory named
         *        <code>/4xx-errors</code>. Your distribution must include a cache behavior for which the path pattern
         *        routes requests for your custom error pages to that location, for example, <code>/4xx-errors/*</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        The value of <code>TargetOriginId</code> specifies the value of the <code>ID</code> element for the
         *        origin that contains your custom error pages.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        If you specify a value for <code>ResponsePagePath</code>, you must also specify a value for
         *        <code>ResponseCode</code>. If you don't want to specify a value, include an empty element,
         *        <code>&lt;ResponsePagePath&gt;</code>, in the XML document.
         *        </p>
         *        <p>
         *        We recommend that you store custom error pages in an Amazon S3 bucket. If you store custom error pages
         *        on an HTTP server and the server starts to return 5xx errors, CloudFront can't get the files that you
         *        want to return to viewers because the origin server is unavailable.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder responsePagePath(String responsePagePath);

        /**
         * <p>
         * The HTTP status code that you want CloudFront to return to the viewer along with the custom error page. There
         * are a variety of reasons that you might want CloudFront to return a status code different from the status
         * code that your origin returned to CloudFront, for example:
         * </p>
         * <ul>
         * <li>
         * <p>
         * Some Internet devices (some firewalls and corporate proxies, for example) intercept HTTP 4xx and 5xx and
         * prevent the response from being returned to the viewer. If you substitute <code>200</code>, the response
         * typically won't be intercepted.
         * </p>
         * </li>
         * <li>
         * <p>
         * If you don't care about distinguishing among different client errors or server errors, you can specify
         * <code>400</code> or <code>500</code> as the <code>ResponseCode</code> for all 4xx or 5xx errors.
         * </p>
         * </li>
         * <li>
         * <p>
         * You might want to return a <code>200</code> status code (OK) and static website so your customers don't know
         * that your website is down.
         * </p>
         * </li>
         * </ul>
         * <p>
         * If you specify a value for <code>ResponseCode</code>, you must also specify a value for
         * <code>ResponsePagePath</code>. If you don't want to specify a value, include an empty element,
         * <code>&lt;ResponseCode&gt;</code>, in the XML document.
         * </p>
         * 
         * @param responseCode
         *        The HTTP status code that you want CloudFront to return to the viewer along with the custom error
         *        page. There are a variety of reasons that you might want CloudFront to return a status code different
         *        from the status code that your origin returned to CloudFront, for example:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        Some Internet devices (some firewalls and corporate proxies, for example) intercept HTTP 4xx and 5xx
         *        and prevent the response from being returned to the viewer. If you substitute <code>200</code>, the
         *        response typically won't be intercepted.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        If you don't care about distinguishing among different client errors or server errors, you can specify
         *        <code>400</code> or <code>500</code> as the <code>ResponseCode</code> for all 4xx or 5xx errors.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        You might want to return a <code>200</code> status code (OK) and static website so your customers
         *        don't know that your website is down.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        If you specify a value for <code>ResponseCode</code>, you must also specify a value for
         *        <code>ResponsePagePath</code>. If you don't want to specify a value, include an empty element,
         *        <code>&lt;ResponseCode&gt;</code>, in the XML document.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder responseCode(String responseCode);

        /**
         * <p>
         * The minimum amount of time, in seconds, that you want CloudFront to cache the HTTP status code specified in
         * <code>ErrorCode</code>. When this time period has elapsed, CloudFront queries your origin to see whether the
         * problem that caused the error has been resolved and the requested object is now available.
         * </p>
         * <p>
         * If you don't want to specify a value, include an empty element, <code>&lt;ErrorCachingMinTTL&gt;</code>, in
         * the XML document.
         * </p>
         * <p>
         * For more information, see <a
         * href="http://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/custom-error-pages.html">Customizing
         * Error Responses</a> in the <i>Amazon CloudFront Developer Guide</i>.
         * </p>
         * 
         * @param errorCachingMinTTL
         *        The minimum amount of time, in seconds, that you want CloudFront to cache the HTTP status code
         *        specified in <code>ErrorCode</code>. When this time period has elapsed, CloudFront queries your origin
         *        to see whether the problem that caused the error has been resolved and the requested object is now
         *        available.</p>
         *        <p>
         *        If you don't want to specify a value, include an empty element,
         *        <code>&lt;ErrorCachingMinTTL&gt;</code>, in the XML document.
         *        </p>
         *        <p>
         *        For more information, see <a
         *        href="http://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/custom-error-pages.html"
         *        >Customizing Error Responses</a> in the <i>Amazon CloudFront Developer Guide</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder errorCachingMinTTL(Long errorCachingMinTTL);
    }

    static final class BuilderImpl implements Builder {
        private Integer errorCode;

        private String responsePagePath;

        private String responseCode;

        private Long errorCachingMinTTL;

        private BuilderImpl() {
        }

        private BuilderImpl(CustomErrorResponse model) {
            errorCode(model.errorCode);
            responsePagePath(model.responsePagePath);
            responseCode(model.responseCode);
            errorCachingMinTTL(model.errorCachingMinTTL);
        }

        public final Integer getErrorCode() {
            return errorCode;
        }

        @Override
        public final Builder errorCode(Integer errorCode) {
            this.errorCode = errorCode;
            return this;
        }

        public final void setErrorCode(Integer errorCode) {
            this.errorCode = errorCode;
        }

        public final String getResponsePagePath() {
            return responsePagePath;
        }

        @Override
        public final Builder responsePagePath(String responsePagePath) {
            this.responsePagePath = responsePagePath;
            return this;
        }

        public final void setResponsePagePath(String responsePagePath) {
            this.responsePagePath = responsePagePath;
        }

        public final String getResponseCode() {
            return responseCode;
        }

        @Override
        public final Builder responseCode(String responseCode) {
            this.responseCode = responseCode;
            return this;
        }

        public final void setResponseCode(String responseCode) {
            this.responseCode = responseCode;
        }

        public final Long getErrorCachingMinTTL() {
            return errorCachingMinTTL;
        }

        @Override
        public final Builder errorCachingMinTTL(Long errorCachingMinTTL) {
            this.errorCachingMinTTL = errorCachingMinTTL;
            return this;
        }

        public final void setErrorCachingMinTTL(Long errorCachingMinTTL) {
            this.errorCachingMinTTL = errorCachingMinTTL;
        }

        @Override
        public CustomErrorResponse build() {
            return new CustomErrorResponse(this);
        }
    }
}
