/*
 * Copyright 2013-2018 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.cloudfront.model;

import java.util.Objects;
import java.util.Optional;
import java.util.function.Consumer;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * This action deletes a web distribution. To delete a web distribution using the CloudFront API, perform the following
 * steps.
 * </p>
 * <p>
 * <b>To delete a web distribution using the CloudFront API:</b>
 * </p>
 * <ol>
 * <li>
 * <p>
 * Disable the web distribution
 * </p>
 * </li>
 * <li>
 * <p>
 * Submit a <code>GET Distribution Config</code> request to get the current configuration and the <code>Etag</code>
 * header for the distribution.
 * </p>
 * </li>
 * <li>
 * <p>
 * Update the XML document that was returned in the response to your <code>GET Distribution Config</code> request to
 * change the value of <code>Enabled</code> to <code>false</code>.
 * </p>
 * </li>
 * <li>
 * <p>
 * Submit a <code>PUT Distribution Config</code> request to update the configuration for your distribution. In the
 * request body, include the XML document that you updated in Step 3. Set the value of the HTTP <code>If-Match</code>
 * header to the value of the <code>ETag</code> header that CloudFront returned when you submitted the
 * <code>GET Distribution Config</code> request in Step 2.
 * </p>
 * </li>
 * <li>
 * <p>
 * Review the response to the <code>PUT Distribution Config</code> request to confirm that the distribution was
 * successfully disabled.
 * </p>
 * </li>
 * <li>
 * <p>
 * Submit a <code>GET Distribution</code> request to confirm that your changes have propagated. When propagation is
 * complete, the value of <code>Status</code> is <code>Deployed</code>.
 * </p>
 * </li>
 * <li>
 * <p>
 * Submit a <code>DELETE Distribution</code> request. Set the value of the HTTP <code>If-Match</code> header to the
 * value of the <code>ETag</code> header that CloudFront returned when you submitted the
 * <code>GET Distribution Config</code> request in Step 6.
 * </p>
 * </li>
 * <li>
 * <p>
 * Review the response to your <code>DELETE Distribution</code> request to confirm that the distribution was
 * successfully deleted.
 * </p>
 * </li>
 * </ol>
 * <p>
 * For information about deleting a distribution using the CloudFront console, see <a
 * href="http://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/HowToDeleteDistribution.html">Deleting a
 * Distribution</a> in the <i>Amazon CloudFront Developer Guide</i>.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class DeleteDistributionRequest extends CloudFrontRequest implements
        ToCopyableBuilder<DeleteDistributionRequest.Builder, DeleteDistributionRequest> {
    private final String id;

    private final String ifMatch;

    private DeleteDistributionRequest(BuilderImpl builder) {
        super(builder);
        this.id = builder.id;
        this.ifMatch = builder.ifMatch;
    }

    /**
     * <p>
     * The distribution ID.
     * </p>
     * 
     * @return The distribution ID.
     */
    public String id() {
        return id;
    }

    /**
     * <p>
     * The value of the <code>ETag</code> header that you received when you disabled the distribution. For example:
     * <code>E2QWRUHAPOMQZL</code>.
     * </p>
     * 
     * @return The value of the <code>ETag</code> header that you received when you disabled the distribution. For
     *         example: <code>E2QWRUHAPOMQZL</code>.
     */
    public String ifMatch() {
        return ifMatch;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(id());
        hashCode = 31 * hashCode + Objects.hashCode(ifMatch());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof DeleteDistributionRequest)) {
            return false;
        }
        DeleteDistributionRequest other = (DeleteDistributionRequest) obj;
        return Objects.equals(id(), other.id()) && Objects.equals(ifMatch(), other.ifMatch());
    }

    @Override
    public String toString() {
        return ToString.builder("DeleteDistributionRequest").add("Id", id()).add("IfMatch", ifMatch()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Id":
            return Optional.ofNullable(clazz.cast(id()));
        case "IfMatch":
            return Optional.ofNullable(clazz.cast(ifMatch()));
        default:
            return Optional.empty();
        }
    }

    public interface Builder extends CloudFrontRequest.Builder, CopyableBuilder<Builder, DeleteDistributionRequest> {
        /**
         * <p>
         * The distribution ID.
         * </p>
         * 
         * @param id
         *        The distribution ID.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder id(String id);

        /**
         * <p>
         * The value of the <code>ETag</code> header that you received when you disabled the distribution. For example:
         * <code>E2QWRUHAPOMQZL</code>.
         * </p>
         * 
         * @param ifMatch
         *        The value of the <code>ETag</code> header that you received when you disabled the distribution. For
         *        example: <code>E2QWRUHAPOMQZL</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder ifMatch(String ifMatch);

        @Override
        Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration);

        @Override
        Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer);
    }

    static final class BuilderImpl extends CloudFrontRequest.BuilderImpl implements Builder {
        private String id;

        private String ifMatch;

        private BuilderImpl() {
        }

        private BuilderImpl(DeleteDistributionRequest model) {
            super(model);
            id(model.id);
            ifMatch(model.ifMatch);
        }

        public final String getId() {
            return id;
        }

        @Override
        public final Builder id(String id) {
            this.id = id;
            return this;
        }

        public final void setId(String id) {
            this.id = id;
        }

        public final String getIfMatch() {
            return ifMatch;
        }

        @Override
        public final Builder ifMatch(String ifMatch) {
            this.ifMatch = ifMatch;
            return this;
        }

        public final void setIfMatch(String ifMatch) {
            this.ifMatch = ifMatch;
        }

        @Override
        public Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration) {
            super.overrideConfiguration(overrideConfiguration);
            return this;
        }

        @Override
        public Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer) {
            super.overrideConfiguration(builderConsumer);
            return this;
        }

        @Override
        public DeleteDistributionRequest build() {
            return new DeleteDistributionRequest(this);
        }
    }
}
