/*
 * Copyright 2013-2018 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.cloudfront.model;

import java.util.Arrays;
import java.util.Collection;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.Consumer;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.runtime.TypeConverter;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * A complex type that controls which HTTP methods CloudFront processes and forwards to your Amazon S3 bucket or your
 * custom origin. There are three choices:
 * </p>
 * <ul>
 * <li>
 * <p>
 * CloudFront forwards only <code>GET</code> and <code>HEAD</code> requests.
 * </p>
 * </li>
 * <li>
 * <p>
 * CloudFront forwards only <code>GET</code>, <code>HEAD</code>, and <code>OPTIONS</code> requests.
 * </p>
 * </li>
 * <li>
 * <p>
 * CloudFront forwards <code>GET, HEAD, OPTIONS, PUT, PATCH, POST</code>, and <code>DELETE</code> requests.
 * </p>
 * </li>
 * </ul>
 * <p>
 * If you pick the third choice, you may need to restrict access to your Amazon S3 bucket or to your custom origin so
 * users can't perform operations that you don't want them to. For example, you might not want users to have permissions
 * to delete objects from your origin.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class AllowedMethods implements ToCopyableBuilder<AllowedMethods.Builder, AllowedMethods> {
    private final Integer quantity;

    private final List<String> items;

    private final CachedMethods cachedMethods;

    private AllowedMethods(BuilderImpl builder) {
        this.quantity = builder.quantity;
        this.items = builder.items;
        this.cachedMethods = builder.cachedMethods;
    }

    /**
     * <p>
     * The number of HTTP methods that you want CloudFront to forward to your origin. Valid values are 2 (for
     * <code>GET</code> and <code>HEAD</code> requests), 3 (for <code>GET</code>, <code>HEAD</code>, and
     * <code>OPTIONS</code> requests) and 7 (for <code>GET, HEAD, OPTIONS, PUT, PATCH, POST</code>, and
     * <code>DELETE</code> requests).
     * </p>
     * 
     * @return The number of HTTP methods that you want CloudFront to forward to your origin. Valid values are 2 (for
     *         <code>GET</code> and <code>HEAD</code> requests), 3 (for <code>GET</code>, <code>HEAD</code>, and
     *         <code>OPTIONS</code> requests) and 7 (for <code>GET, HEAD, OPTIONS, PUT, PATCH, POST</code>, and
     *         <code>DELETE</code> requests).
     */
    public Integer quantity() {
        return quantity;
    }

    /**
     * <p>
     * A complex type that contains the HTTP methods that you want CloudFront to process and forward to your origin.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return A complex type that contains the HTTP methods that you want CloudFront to process and forward to your
     *         origin.
     */
    public List<Method> items() {
        return TypeConverter.convert(items, Method::fromValue);
    }

    /**
     * <p>
     * A complex type that contains the HTTP methods that you want CloudFront to process and forward to your origin.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return A complex type that contains the HTTP methods that you want CloudFront to process and forward to your
     *         origin.
     */
    public List<String> itemsAsStrings() {
        return items;
    }

    /**
     * Returns the value of the CachedMethods property for this object.
     * 
     * @return The value of the CachedMethods property for this object.
     */
    public CachedMethods cachedMethods() {
        return cachedMethods;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(quantity());
        hashCode = 31 * hashCode + Objects.hashCode(itemsAsStrings());
        hashCode = 31 * hashCode + Objects.hashCode(cachedMethods());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof AllowedMethods)) {
            return false;
        }
        AllowedMethods other = (AllowedMethods) obj;
        return Objects.equals(quantity(), other.quantity()) && Objects.equals(itemsAsStrings(), other.itemsAsStrings())
                && Objects.equals(cachedMethods(), other.cachedMethods());
    }

    @Override
    public String toString() {
        return ToString.builder("AllowedMethods").add("Quantity", quantity()).add("Items", itemsAsStrings())
                .add("CachedMethods", cachedMethods()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Quantity":
            return Optional.ofNullable(clazz.cast(quantity()));
        case "Items":
            return Optional.ofNullable(clazz.cast(itemsAsStrings()));
        case "CachedMethods":
            return Optional.ofNullable(clazz.cast(cachedMethods()));
        default:
            return Optional.empty();
        }
    }

    public interface Builder extends CopyableBuilder<Builder, AllowedMethods> {
        /**
         * <p>
         * The number of HTTP methods that you want CloudFront to forward to your origin. Valid values are 2 (for
         * <code>GET</code> and <code>HEAD</code> requests), 3 (for <code>GET</code>, <code>HEAD</code>, and
         * <code>OPTIONS</code> requests) and 7 (for <code>GET, HEAD, OPTIONS, PUT, PATCH, POST</code>, and
         * <code>DELETE</code> requests).
         * </p>
         * 
         * @param quantity
         *        The number of HTTP methods that you want CloudFront to forward to your origin. Valid values are 2 (for
         *        <code>GET</code> and <code>HEAD</code> requests), 3 (for <code>GET</code>, <code>HEAD</code>, and
         *        <code>OPTIONS</code> requests) and 7 (for <code>GET, HEAD, OPTIONS, PUT, PATCH, POST</code>, and
         *        <code>DELETE</code> requests).
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder quantity(Integer quantity);

        /**
         * <p>
         * A complex type that contains the HTTP methods that you want CloudFront to process and forward to your origin.
         * </p>
         * 
         * @param items
         *        A complex type that contains the HTTP methods that you want CloudFront to process and forward to your
         *        origin.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder itemsWithStrings(Collection<String> items);

        /**
         * <p>
         * A complex type that contains the HTTP methods that you want CloudFront to process and forward to your origin.
         * </p>
         * 
         * @param items
         *        A complex type that contains the HTTP methods that you want CloudFront to process and forward to your
         *        origin.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder itemsWithStrings(String... items);

        /**
         * <p>
         * A complex type that contains the HTTP methods that you want CloudFront to process and forward to your origin.
         * </p>
         * 
         * @param items
         *        A complex type that contains the HTTP methods that you want CloudFront to process and forward to your
         *        origin.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder items(Collection<Method> items);

        /**
         * <p>
         * A complex type that contains the HTTP methods that you want CloudFront to process and forward to your origin.
         * </p>
         * 
         * @param items
         *        A complex type that contains the HTTP methods that you want CloudFront to process and forward to your
         *        origin.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder items(Method... items);

        /**
         * Sets the value of the CachedMethods property for this object.
         *
         * @param cachedMethods
         *        The new value for the CachedMethods property for this object.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder cachedMethods(CachedMethods cachedMethods);

        /**
         * Sets the value of the CachedMethods property for this object.
         *
         * This is a convenience that creates an instance of the {@link CachedMethods.Builder} avoiding the need to
         * create one manually via {@link CachedMethods#builder()}.
         *
         * When the {@link Consumer} completes, {@link CachedMethods.Builder#build()} is called immediately and its
         * result is passed to {@link #cachedMethods(CachedMethods)}.
         * 
         * @param cachedMethods
         *        a consumer that will call methods on {@link CachedMethods.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #cachedMethods(CachedMethods)
         */
        default Builder cachedMethods(Consumer<CachedMethods.Builder> cachedMethods) {
            return cachedMethods(CachedMethods.builder().applyMutation(cachedMethods).build());
        }
    }

    static final class BuilderImpl implements Builder {
        private Integer quantity;

        private List<String> items = DefaultSdkAutoConstructList.getInstance();

        private CachedMethods cachedMethods;

        private BuilderImpl() {
        }

        private BuilderImpl(AllowedMethods model) {
            quantity(model.quantity);
            itemsWithStrings(model.items);
            cachedMethods(model.cachedMethods);
        }

        public final Integer getQuantity() {
            return quantity;
        }

        @Override
        public final Builder quantity(Integer quantity) {
            this.quantity = quantity;
            return this;
        }

        public final void setQuantity(Integer quantity) {
            this.quantity = quantity;
        }

        public final Collection<String> getItems() {
            return items;
        }

        @Override
        public final Builder itemsWithStrings(Collection<String> items) {
            this.items = MethodsListCopier.copy(items);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder itemsWithStrings(String... items) {
            itemsWithStrings(Arrays.asList(items));
            return this;
        }

        @Override
        public final Builder items(Collection<Method> items) {
            this.items = MethodsListCopier.copyEnumToString(items);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder items(Method... items) {
            items(Arrays.asList(items));
            return this;
        }

        public final void setItems(Collection<String> items) {
            this.items = MethodsListCopier.copy(items);
        }

        public final CachedMethods.Builder getCachedMethods() {
            return cachedMethods != null ? cachedMethods.toBuilder() : null;
        }

        @Override
        public final Builder cachedMethods(CachedMethods cachedMethods) {
            this.cachedMethods = cachedMethods;
            return this;
        }

        public final void setCachedMethods(CachedMethods.BuilderImpl cachedMethods) {
            this.cachedMethods = cachedMethods != null ? cachedMethods.build() : null;
        }

        @Override
        public AllowedMethods build() {
            return new AllowedMethods(this);
        }
    }
}
