/*
 * Copyright 2013-2018 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.cloudfront.paginators;

import java.util.Collections;
import java.util.Iterator;
import java.util.function.Function;
import javax.annotation.Generated;
import software.amazon.awssdk.core.pagination.PaginatedItemsIterable;
import software.amazon.awssdk.core.pagination.PaginatedResponsesIterator;
import software.amazon.awssdk.core.pagination.SdkIterable;
import software.amazon.awssdk.core.pagination.SyncPageFetcher;
import software.amazon.awssdk.services.cloudfront.CloudFrontClient;
import software.amazon.awssdk.services.cloudfront.model.CloudFrontOriginAccessIdentitySummary;
import software.amazon.awssdk.services.cloudfront.model.ListCloudFrontOriginAccessIdentitiesRequest;
import software.amazon.awssdk.services.cloudfront.model.ListCloudFrontOriginAccessIdentitiesResponse;

/**
 * <p>
 * Represents the output for the
 * {@link software.amazon.awssdk.services.cloudfront.CloudFrontClient#listCloudFrontOriginAccessIdentitiesPaginator(software.amazon.awssdk.services.cloudfront.model.ListCloudFrontOriginAccessIdentitiesRequest)}
 * operation which is a paginated operation. This class is an iterable of
 * {@link software.amazon.awssdk.services.cloudfront.model.ListCloudFrontOriginAccessIdentitiesResponse} that can be
 * used to iterate through all the response pages of the operation.
 * </p>
 * <p>
 * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet and
 * so there is no guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily
 * loading response pages by making service calls until there are no pages left or your iteration stops. If there are
 * errors in your request, you will see the failures only after you start iterating through the iterable.
 * </p>
 *
 * <p>
 * The following are few ways to iterate through the response pages:
 * </p>
 * 1) Using a Stream
 * 
 * <pre>
 * {@code
 * software.amazon.awssdk.services.cloudfront.paginators.ListCloudFrontOriginAccessIdentitiesIterable responses = client.listCloudFrontOriginAccessIdentitiesPaginator(request);
 * responses.stream().forEach(....);
 * }
 * </pre>
 *
 * 2) Using For loop
 * 
 * <pre>
 * {
 *     &#064;code
 *     software.amazon.awssdk.services.cloudfront.paginators.ListCloudFrontOriginAccessIdentitiesIterable responses = client
 *             .listCloudFrontOriginAccessIdentitiesPaginator(request);
 *     for (software.amazon.awssdk.services.cloudfront.model.ListCloudFrontOriginAccessIdentitiesResponse response : responses) {
 *         // do something;
 *     }
 * }
 * </pre>
 *
 * 3) Use iterator directly
 * 
 * <pre>
 * {@code
 * software.amazon.awssdk.services.cloudfront.paginators.ListCloudFrontOriginAccessIdentitiesIterable responses = client.listCloudFrontOriginAccessIdentitiesPaginator(request);
 * responses.iterator().forEachRemaining(....);
 * }
 * </pre>
 * <p>
 * <b>Note: If you prefer to have control on service calls, use the
 * {@link #listCloudFrontOriginAccessIdentities(software.amazon.awssdk.services.cloudfront.model.ListCloudFrontOriginAccessIdentitiesRequest)}
 * operation.</b>
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public class ListCloudFrontOriginAccessIdentitiesIterable implements SdkIterable<ListCloudFrontOriginAccessIdentitiesResponse> {
    private final CloudFrontClient client;

    private final ListCloudFrontOriginAccessIdentitiesRequest firstRequest;

    private final SyncPageFetcher nextPageFetcher;

    public ListCloudFrontOriginAccessIdentitiesIterable(final CloudFrontClient client,
            final ListCloudFrontOriginAccessIdentitiesRequest firstRequest) {
        this.client = client;
        this.firstRequest = firstRequest;
        this.nextPageFetcher = new ListCloudFrontOriginAccessIdentitiesResponseFetcher();
    }

    @Override
    public Iterator<ListCloudFrontOriginAccessIdentitiesResponse> iterator() {
        return new PaginatedResponsesIterator(nextPageFetcher);
    }

    /**
     * Returns an iterable to iterate through the paginated {@link ListCloudFrontOriginAccessIdentitiesResponse#items()}
     * member. The returned iterable is used to iterate through the results across all response pages and not a single
     * page.
     *
     * This method is useful if you are interested in iterating over the paginated member in the response pages instead
     * of the top level pages. Similar to iteration over pages, this method internally makes service calls to get the
     * next list of results until the iteration stops or there are no more results.
     */
    public final SdkIterable<CloudFrontOriginAccessIdentitySummary> items() {
        Function<ListCloudFrontOriginAccessIdentitiesResponse, Iterator<CloudFrontOriginAccessIdentitySummary>> getIterator = response -> {
            if (response != null && response.cloudFrontOriginAccessIdentityList() != null
                    && response.cloudFrontOriginAccessIdentityList().items() != null) {
                return response.cloudFrontOriginAccessIdentityList().items().iterator();
            }
            return Collections.emptyIterator();
        };
        return new PaginatedItemsIterable(this, getIterator);
    }

    /**
     * <p>
     * A helper method to resume the pages in case of unexpected failures. The method takes the last successful response
     * page as input and returns an instance of {@link ListCloudFrontOriginAccessIdentitiesIterable} that can be used to
     * retrieve the consecutive pages that follows the input page.
     * </p>
     */
    public final ListCloudFrontOriginAccessIdentitiesIterable resume(
            final ListCloudFrontOriginAccessIdentitiesResponse lastSuccessfulPage) {
        if (nextPageFetcher.hasNextPage(lastSuccessfulPage)) {
            return new ListCloudFrontOriginAccessIdentitiesIterable(client, firstRequest.toBuilder()
                    .marker(lastSuccessfulPage.cloudFrontOriginAccessIdentityList().nextMarker()).build());
        }
        return new ListCloudFrontOriginAccessIdentitiesIterable(client, firstRequest) {
            @Override
            public Iterator<ListCloudFrontOriginAccessIdentitiesResponse> iterator() {
                return Collections.emptyIterator();
            }
        };
    }

    private class ListCloudFrontOriginAccessIdentitiesResponseFetcher implements
            SyncPageFetcher<ListCloudFrontOriginAccessIdentitiesResponse> {
        @Override
        public boolean hasNextPage(ListCloudFrontOriginAccessIdentitiesResponse previousPage) {
            return previousPage.cloudFrontOriginAccessIdentityList().isTruncated().booleanValue();
        }

        @Override
        public ListCloudFrontOriginAccessIdentitiesResponse nextPage(ListCloudFrontOriginAccessIdentitiesResponse previousPage) {
            if (previousPage == null) {
                return client.listCloudFrontOriginAccessIdentities(firstRequest);
            }
            return client.listCloudFrontOriginAccessIdentities(firstRequest.toBuilder()
                    .marker(previousPage.cloudFrontOriginAccessIdentityList().nextMarker()).build());
        }
    }
}
