/*
 * Copyright 2013-2018 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.cloudfront.model;

import java.util.Arrays;
import java.util.Collection;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.Consumer;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import javax.annotation.Generated;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Lists the origin access identities for CloudFront.Send a <code>GET</code> request to the
 * <code>/<i>CloudFront API version</i>/origin-access-identity/cloudfront</code> resource. The response includes a
 * <code>CloudFrontOriginAccessIdentityList</code> element with zero or more
 * <code>CloudFrontOriginAccessIdentitySummary</code> child elements. By default, your entire list of origin access
 * identities is returned in one single page. If the list is long, you can paginate it using the <code>MaxItems</code>
 * and <code>Marker</code> parameters.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public class CloudFrontOriginAccessIdentityList implements
        ToCopyableBuilder<CloudFrontOriginAccessIdentityList.Builder, CloudFrontOriginAccessIdentityList> {
    private final String marker;

    private final String nextMarker;

    private final Integer maxItems;

    private final Boolean isTruncated;

    private final Integer quantity;

    private final List<CloudFrontOriginAccessIdentitySummary> items;

    private CloudFrontOriginAccessIdentityList(BuilderImpl builder) {
        this.marker = builder.marker;
        this.nextMarker = builder.nextMarker;
        this.maxItems = builder.maxItems;
        this.isTruncated = builder.isTruncated;
        this.quantity = builder.quantity;
        this.items = builder.items;
    }

    /**
     * <p>
     * Use this when paginating results to indicate where to begin in your list of origin access identities. The results
     * include identities in the list that occur after the marker. To get the next page of results, set the
     * <code>Marker</code> to the value of the <code>NextMarker</code> from the current page's response (which is also
     * the ID of the last identity on that page).
     * </p>
     * 
     * @return Use this when paginating results to indicate where to begin in your list of origin access identities. The
     *         results include identities in the list that occur after the marker. To get the next page of results, set
     *         the <code>Marker</code> to the value of the <code>NextMarker</code> from the current page's response
     *         (which is also the ID of the last identity on that page).
     */
    public String marker() {
        return marker;
    }

    /**
     * <p>
     * If <code>IsTruncated</code> is <code>true</code>, this element is present and contains the value you can use for
     * the <code>Marker</code> request parameter to continue listing your origin access identities where they left off.
     * </p>
     * 
     * @return If <code>IsTruncated</code> is <code>true</code>, this element is present and contains the value you can
     *         use for the <code>Marker</code> request parameter to continue listing your origin access identities where
     *         they left off.
     */
    public String nextMarker() {
        return nextMarker;
    }

    /**
     * <p>
     * The maximum number of origin access identities you want in the response body.
     * </p>
     * 
     * @return The maximum number of origin access identities you want in the response body.
     */
    public Integer maxItems() {
        return maxItems;
    }

    /**
     * <p>
     * A flag that indicates whether more origin access identities remain to be listed. If your results were truncated,
     * you can make a follow-up pagination request using the <code>Marker</code> request parameter to retrieve more
     * items in the list.
     * </p>
     * 
     * @return A flag that indicates whether more origin access identities remain to be listed. If your results were
     *         truncated, you can make a follow-up pagination request using the <code>Marker</code> request parameter to
     *         retrieve more items in the list.
     */
    public Boolean isTruncated() {
        return isTruncated;
    }

    /**
     * <p>
     * The number of CloudFront origin access identities that were created by the current AWS account.
     * </p>
     * 
     * @return The number of CloudFront origin access identities that were created by the current AWS account.
     */
    public Integer quantity() {
        return quantity;
    }

    /**
     * <p>
     * A complex type that contains one <code>CloudFrontOriginAccessIdentitySummary</code> element for each origin
     * access identity that was created by the current AWS account.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return A complex type that contains one <code>CloudFrontOriginAccessIdentitySummary</code> element for each
     *         origin access identity that was created by the current AWS account.
     */
    public List<CloudFrontOriginAccessIdentitySummary> items() {
        return items;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(marker());
        hashCode = 31 * hashCode + Objects.hashCode(nextMarker());
        hashCode = 31 * hashCode + Objects.hashCode(maxItems());
        hashCode = 31 * hashCode + Objects.hashCode(isTruncated());
        hashCode = 31 * hashCode + Objects.hashCode(quantity());
        hashCode = 31 * hashCode + Objects.hashCode(items());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof CloudFrontOriginAccessIdentityList)) {
            return false;
        }
        CloudFrontOriginAccessIdentityList other = (CloudFrontOriginAccessIdentityList) obj;
        return Objects.equals(marker(), other.marker()) && Objects.equals(nextMarker(), other.nextMarker())
                && Objects.equals(maxItems(), other.maxItems()) && Objects.equals(isTruncated(), other.isTruncated())
                && Objects.equals(quantity(), other.quantity()) && Objects.equals(items(), other.items());
    }

    @Override
    public String toString() {
        return ToString.builder("CloudFrontOriginAccessIdentityList").add("Marker", marker()).add("NextMarker", nextMarker())
                .add("MaxItems", maxItems()).add("IsTruncated", isTruncated()).add("Quantity", quantity()).add("Items", items())
                .build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Marker":
            return Optional.of(clazz.cast(marker()));
        case "NextMarker":
            return Optional.of(clazz.cast(nextMarker()));
        case "MaxItems":
            return Optional.of(clazz.cast(maxItems()));
        case "IsTruncated":
            return Optional.of(clazz.cast(isTruncated()));
        case "Quantity":
            return Optional.of(clazz.cast(quantity()));
        case "Items":
            return Optional.of(clazz.cast(items()));
        default:
            return Optional.empty();
        }
    }

    public interface Builder extends CopyableBuilder<Builder, CloudFrontOriginAccessIdentityList> {
        /**
         * <p>
         * Use this when paginating results to indicate where to begin in your list of origin access identities. The
         * results include identities in the list that occur after the marker. To get the next page of results, set the
         * <code>Marker</code> to the value of the <code>NextMarker</code> from the current page's response (which is
         * also the ID of the last identity on that page).
         * </p>
         * 
         * @param marker
         *        Use this when paginating results to indicate where to begin in your list of origin access identities.
         *        The results include identities in the list that occur after the marker. To get the next page of
         *        results, set the <code>Marker</code> to the value of the <code>NextMarker</code> from the current
         *        page's response (which is also the ID of the last identity on that page).
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder marker(String marker);

        /**
         * <p>
         * If <code>IsTruncated</code> is <code>true</code>, this element is present and contains the value you can use
         * for the <code>Marker</code> request parameter to continue listing your origin access identities where they
         * left off.
         * </p>
         * 
         * @param nextMarker
         *        If <code>IsTruncated</code> is <code>true</code>, this element is present and contains the value you
         *        can use for the <code>Marker</code> request parameter to continue listing your origin access
         *        identities where they left off.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder nextMarker(String nextMarker);

        /**
         * <p>
         * The maximum number of origin access identities you want in the response body.
         * </p>
         * 
         * @param maxItems
         *        The maximum number of origin access identities you want in the response body.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder maxItems(Integer maxItems);

        /**
         * <p>
         * A flag that indicates whether more origin access identities remain to be listed. If your results were
         * truncated, you can make a follow-up pagination request using the <code>Marker</code> request parameter to
         * retrieve more items in the list.
         * </p>
         * 
         * @param isTruncated
         *        A flag that indicates whether more origin access identities remain to be listed. If your results were
         *        truncated, you can make a follow-up pagination request using the <code>Marker</code> request parameter
         *        to retrieve more items in the list.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder isTruncated(Boolean isTruncated);

        /**
         * <p>
         * The number of CloudFront origin access identities that were created by the current AWS account.
         * </p>
         * 
         * @param quantity
         *        The number of CloudFront origin access identities that were created by the current AWS account.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder quantity(Integer quantity);

        /**
         * <p>
         * A complex type that contains one <code>CloudFrontOriginAccessIdentitySummary</code> element for each origin
         * access identity that was created by the current AWS account.
         * </p>
         * 
         * @param items
         *        A complex type that contains one <code>CloudFrontOriginAccessIdentitySummary</code> element for each
         *        origin access identity that was created by the current AWS account.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder items(Collection<CloudFrontOriginAccessIdentitySummary> items);

        /**
         * <p>
         * A complex type that contains one <code>CloudFrontOriginAccessIdentitySummary</code> element for each origin
         * access identity that was created by the current AWS account.
         * </p>
         * 
         * @param items
         *        A complex type that contains one <code>CloudFrontOriginAccessIdentitySummary</code> element for each
         *        origin access identity that was created by the current AWS account.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder items(CloudFrontOriginAccessIdentitySummary... items);

        /**
         * <p>
         * A complex type that contains one <code>CloudFrontOriginAccessIdentitySummary</code> element for each origin
         * access identity that was created by the current AWS account.
         * </p>
         * This is a convenience that creates an instance of the {@link List
         * <CloudFrontOriginAccessIdentitySummary>.Builder} avoiding the need to create one manually via {@link List
         * <CloudFrontOriginAccessIdentitySummary>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<CloudFrontOriginAccessIdentitySummary>.Builder#build()} is
         * called immediately and its result is passed to {@link #items(List<CloudFrontOriginAccessIdentitySummary>)}.
         * 
         * @param items
         *        a consumer that will call methods on {@link List<CloudFrontOriginAccessIdentitySummary>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #items(List<CloudFrontOriginAccessIdentitySummary>)
         */
        Builder items(Consumer<CloudFrontOriginAccessIdentitySummary.Builder>... items);
    }

    static final class BuilderImpl implements Builder {
        private String marker;

        private String nextMarker;

        private Integer maxItems;

        private Boolean isTruncated;

        private Integer quantity;

        private List<CloudFrontOriginAccessIdentitySummary> items;

        private BuilderImpl() {
        }

        private BuilderImpl(CloudFrontOriginAccessIdentityList model) {
            marker(model.marker);
            nextMarker(model.nextMarker);
            maxItems(model.maxItems);
            isTruncated(model.isTruncated);
            quantity(model.quantity);
            items(model.items);
        }

        public final String getMarker() {
            return marker;
        }

        @Override
        public final Builder marker(String marker) {
            this.marker = marker;
            return this;
        }

        public final void setMarker(String marker) {
            this.marker = marker;
        }

        public final String getNextMarker() {
            return nextMarker;
        }

        @Override
        public final Builder nextMarker(String nextMarker) {
            this.nextMarker = nextMarker;
            return this;
        }

        public final void setNextMarker(String nextMarker) {
            this.nextMarker = nextMarker;
        }

        public final Integer getMaxItems() {
            return maxItems;
        }

        @Override
        public final Builder maxItems(Integer maxItems) {
            this.maxItems = maxItems;
            return this;
        }

        public final void setMaxItems(Integer maxItems) {
            this.maxItems = maxItems;
        }

        public final Boolean getIsTruncated() {
            return isTruncated;
        }

        @Override
        public final Builder isTruncated(Boolean isTruncated) {
            this.isTruncated = isTruncated;
            return this;
        }

        public final void setIsTruncated(Boolean isTruncated) {
            this.isTruncated = isTruncated;
        }

        public final Integer getQuantity() {
            return quantity;
        }

        @Override
        public final Builder quantity(Integer quantity) {
            this.quantity = quantity;
            return this;
        }

        public final void setQuantity(Integer quantity) {
            this.quantity = quantity;
        }

        public final Collection<CloudFrontOriginAccessIdentitySummary.Builder> getItems() {
            return items != null ? items.stream().map(CloudFrontOriginAccessIdentitySummary::toBuilder)
                    .collect(Collectors.toList()) : null;
        }

        @Override
        public final Builder items(Collection<CloudFrontOriginAccessIdentitySummary> items) {
            this.items = CloudFrontOriginAccessIdentitySummaryListCopier.copy(items);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder items(CloudFrontOriginAccessIdentitySummary... items) {
            items(Arrays.asList(items));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder items(Consumer<CloudFrontOriginAccessIdentitySummary.Builder>... items) {
            items(Stream.of(items).map(c -> CloudFrontOriginAccessIdentitySummary.builder().apply(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final void setItems(Collection<CloudFrontOriginAccessIdentitySummary.BuilderImpl> items) {
            this.items = CloudFrontOriginAccessIdentitySummaryListCopier.copyFromBuilder(items);
        }

        @Override
        public CloudFrontOriginAccessIdentityList build() {
            return new CloudFrontOriginAccessIdentityList(this);
        }
    }
}
