/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.cloudformation.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Contains information about whether the resource's actual configuration differs, or has <i>drifted</i>, from its
 * expected configuration.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class StackResourceDriftInformation implements SdkPojo, Serializable,
        ToCopyableBuilder<StackResourceDriftInformation.Builder, StackResourceDriftInformation> {
    private static final SdkField<String> STACK_RESOURCE_DRIFT_STATUS_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("StackResourceDriftStatus")
            .getter(getter(StackResourceDriftInformation::stackResourceDriftStatusAsString))
            .setter(setter(Builder::stackResourceDriftStatus))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("StackResourceDriftStatus").build())
            .build();

    private static final SdkField<Instant> LAST_CHECK_TIMESTAMP_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("LastCheckTimestamp").getter(getter(StackResourceDriftInformation::lastCheckTimestamp))
            .setter(setter(Builder::lastCheckTimestamp))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("LastCheckTimestamp").build())
            .build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(
            STACK_RESOURCE_DRIFT_STATUS_FIELD, LAST_CHECK_TIMESTAMP_FIELD));

    private static final long serialVersionUID = 1L;

    private final String stackResourceDriftStatus;

    private final Instant lastCheckTimestamp;

    private StackResourceDriftInformation(BuilderImpl builder) {
        this.stackResourceDriftStatus = builder.stackResourceDriftStatus;
        this.lastCheckTimestamp = builder.lastCheckTimestamp;
    }

    /**
     * <p>
     * Status of the resource's actual configuration compared to its expected configuration
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>DELETED</code>: The resource differs from its expected configuration in that it has been deleted.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>MODIFIED</code>: The resource differs from its expected configuration.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>NOT_CHECKED</code>: AWS CloudFormation has not checked if the resource differs from its expected
     * configuration.
     * </p>
     * <p>
     * Any resources that do not currently support drift detection have a status of <code>NOT_CHECKED</code>. For more
     * information, see <a
     * href="https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/using-cfn-stack-drift-resource-list.html"
     * >Resources that Support Drift Detection</a>.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>IN_SYNC</code>: The resources's actual configuration matches its expected configuration.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #stackResourceDriftStatus} will return {@link StackResourceDriftStatus#UNKNOWN_TO_SDK_VERSION}. The raw
     * value returned by the service is available from {@link #stackResourceDriftStatusAsString}.
     * </p>
     * 
     * @return Status of the resource's actual configuration compared to its expected configuration</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>DELETED</code>: The resource differs from its expected configuration in that it has been deleted.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>MODIFIED</code>: The resource differs from its expected configuration.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>NOT_CHECKED</code>: AWS CloudFormation has not checked if the resource differs from its expected
     *         configuration.
     *         </p>
     *         <p>
     *         Any resources that do not currently support drift detection have a status of <code>NOT_CHECKED</code>.
     *         For more information, see <a href=
     *         "https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/using-cfn-stack-drift-resource-list.html"
     *         >Resources that Support Drift Detection</a>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>IN_SYNC</code>: The resources's actual configuration matches its expected configuration.
     *         </p>
     *         </li>
     * @see StackResourceDriftStatus
     */
    public final StackResourceDriftStatus stackResourceDriftStatus() {
        return StackResourceDriftStatus.fromValue(stackResourceDriftStatus);
    }

    /**
     * <p>
     * Status of the resource's actual configuration compared to its expected configuration
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>DELETED</code>: The resource differs from its expected configuration in that it has been deleted.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>MODIFIED</code>: The resource differs from its expected configuration.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>NOT_CHECKED</code>: AWS CloudFormation has not checked if the resource differs from its expected
     * configuration.
     * </p>
     * <p>
     * Any resources that do not currently support drift detection have a status of <code>NOT_CHECKED</code>. For more
     * information, see <a
     * href="https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/using-cfn-stack-drift-resource-list.html"
     * >Resources that Support Drift Detection</a>.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>IN_SYNC</code>: The resources's actual configuration matches its expected configuration.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #stackResourceDriftStatus} will return {@link StackResourceDriftStatus#UNKNOWN_TO_SDK_VERSION}. The raw
     * value returned by the service is available from {@link #stackResourceDriftStatusAsString}.
     * </p>
     * 
     * @return Status of the resource's actual configuration compared to its expected configuration</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>DELETED</code>: The resource differs from its expected configuration in that it has been deleted.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>MODIFIED</code>: The resource differs from its expected configuration.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>NOT_CHECKED</code>: AWS CloudFormation has not checked if the resource differs from its expected
     *         configuration.
     *         </p>
     *         <p>
     *         Any resources that do not currently support drift detection have a status of <code>NOT_CHECKED</code>.
     *         For more information, see <a href=
     *         "https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/using-cfn-stack-drift-resource-list.html"
     *         >Resources that Support Drift Detection</a>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>IN_SYNC</code>: The resources's actual configuration matches its expected configuration.
     *         </p>
     *         </li>
     * @see StackResourceDriftStatus
     */
    public final String stackResourceDriftStatusAsString() {
        return stackResourceDriftStatus;
    }

    /**
     * <p>
     * When AWS CloudFormation last checked if the resource had drifted from its expected configuration.
     * </p>
     * 
     * @return When AWS CloudFormation last checked if the resource had drifted from its expected configuration.
     */
    public final Instant lastCheckTimestamp() {
        return lastCheckTimestamp;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(stackResourceDriftStatusAsString());
        hashCode = 31 * hashCode + Objects.hashCode(lastCheckTimestamp());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof StackResourceDriftInformation)) {
            return false;
        }
        StackResourceDriftInformation other = (StackResourceDriftInformation) obj;
        return Objects.equals(stackResourceDriftStatusAsString(), other.stackResourceDriftStatusAsString())
                && Objects.equals(lastCheckTimestamp(), other.lastCheckTimestamp());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("StackResourceDriftInformation")
                .add("StackResourceDriftStatus", stackResourceDriftStatusAsString())
                .add("LastCheckTimestamp", lastCheckTimestamp()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "StackResourceDriftStatus":
            return Optional.ofNullable(clazz.cast(stackResourceDriftStatusAsString()));
        case "LastCheckTimestamp":
            return Optional.ofNullable(clazz.cast(lastCheckTimestamp()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<StackResourceDriftInformation, T> g) {
        return obj -> g.apply((StackResourceDriftInformation) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, StackResourceDriftInformation> {
        /**
         * <p>
         * Status of the resource's actual configuration compared to its expected configuration
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>DELETED</code>: The resource differs from its expected configuration in that it has been deleted.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>MODIFIED</code>: The resource differs from its expected configuration.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>NOT_CHECKED</code>: AWS CloudFormation has not checked if the resource differs from its expected
         * configuration.
         * </p>
         * <p>
         * Any resources that do not currently support drift detection have a status of <code>NOT_CHECKED</code>. For
         * more information, see <a href=
         * "https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/using-cfn-stack-drift-resource-list.html"
         * >Resources that Support Drift Detection</a>.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>IN_SYNC</code>: The resources's actual configuration matches its expected configuration.
         * </p>
         * </li>
         * </ul>
         * 
         * @param stackResourceDriftStatus
         *        Status of the resource's actual configuration compared to its expected configuration</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>DELETED</code>: The resource differs from its expected configuration in that it has been
         *        deleted.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>MODIFIED</code>: The resource differs from its expected configuration.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>NOT_CHECKED</code>: AWS CloudFormation has not checked if the resource differs from its expected
         *        configuration.
         *        </p>
         *        <p>
         *        Any resources that do not currently support drift detection have a status of <code>NOT_CHECKED</code>.
         *        For more information, see <a href=
         *        "https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/using-cfn-stack-drift-resource-list.html"
         *        >Resources that Support Drift Detection</a>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>IN_SYNC</code>: The resources's actual configuration matches its expected configuration.
         *        </p>
         *        </li>
         * @see StackResourceDriftStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see StackResourceDriftStatus
         */
        Builder stackResourceDriftStatus(String stackResourceDriftStatus);

        /**
         * <p>
         * Status of the resource's actual configuration compared to its expected configuration
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>DELETED</code>: The resource differs from its expected configuration in that it has been deleted.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>MODIFIED</code>: The resource differs from its expected configuration.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>NOT_CHECKED</code>: AWS CloudFormation has not checked if the resource differs from its expected
         * configuration.
         * </p>
         * <p>
         * Any resources that do not currently support drift detection have a status of <code>NOT_CHECKED</code>. For
         * more information, see <a href=
         * "https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/using-cfn-stack-drift-resource-list.html"
         * >Resources that Support Drift Detection</a>.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>IN_SYNC</code>: The resources's actual configuration matches its expected configuration.
         * </p>
         * </li>
         * </ul>
         * 
         * @param stackResourceDriftStatus
         *        Status of the resource's actual configuration compared to its expected configuration</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>DELETED</code>: The resource differs from its expected configuration in that it has been
         *        deleted.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>MODIFIED</code>: The resource differs from its expected configuration.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>NOT_CHECKED</code>: AWS CloudFormation has not checked if the resource differs from its expected
         *        configuration.
         *        </p>
         *        <p>
         *        Any resources that do not currently support drift detection have a status of <code>NOT_CHECKED</code>.
         *        For more information, see <a href=
         *        "https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/using-cfn-stack-drift-resource-list.html"
         *        >Resources that Support Drift Detection</a>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>IN_SYNC</code>: The resources's actual configuration matches its expected configuration.
         *        </p>
         *        </li>
         * @see StackResourceDriftStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see StackResourceDriftStatus
         */
        Builder stackResourceDriftStatus(StackResourceDriftStatus stackResourceDriftStatus);

        /**
         * <p>
         * When AWS CloudFormation last checked if the resource had drifted from its expected configuration.
         * </p>
         * 
         * @param lastCheckTimestamp
         *        When AWS CloudFormation last checked if the resource had drifted from its expected configuration.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder lastCheckTimestamp(Instant lastCheckTimestamp);
    }

    static final class BuilderImpl implements Builder {
        private String stackResourceDriftStatus;

        private Instant lastCheckTimestamp;

        private BuilderImpl() {
        }

        private BuilderImpl(StackResourceDriftInformation model) {
            stackResourceDriftStatus(model.stackResourceDriftStatus);
            lastCheckTimestamp(model.lastCheckTimestamp);
        }

        public final String getStackResourceDriftStatus() {
            return stackResourceDriftStatus;
        }

        @Override
        public final Builder stackResourceDriftStatus(String stackResourceDriftStatus) {
            this.stackResourceDriftStatus = stackResourceDriftStatus;
            return this;
        }

        @Override
        public final Builder stackResourceDriftStatus(StackResourceDriftStatus stackResourceDriftStatus) {
            this.stackResourceDriftStatus(stackResourceDriftStatus == null ? null : stackResourceDriftStatus.toString());
            return this;
        }

        public final void setStackResourceDriftStatus(String stackResourceDriftStatus) {
            this.stackResourceDriftStatus = stackResourceDriftStatus;
        }

        public final Instant getLastCheckTimestamp() {
            return lastCheckTimestamp;
        }

        @Override
        public final Builder lastCheckTimestamp(Instant lastCheckTimestamp) {
            this.lastCheckTimestamp = lastCheckTimestamp;
            return this;
        }

        public final void setLastCheckTimestamp(Instant lastCheckTimestamp) {
            this.lastCheckTimestamp = lastCheckTimestamp;
        }

        @Override
        public StackResourceDriftInformation build() {
            return new StackResourceDriftInformation(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
