/*
 * Copyright 2013-2018 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.cloudformation.paginators;

import java.util.Collections;
import java.util.Iterator;
import java.util.function.Function;
import javax.annotation.Generated;
import software.amazon.awssdk.core.pagination.PaginatedItemsIterable;
import software.amazon.awssdk.core.pagination.PaginatedResponsesIterator;
import software.amazon.awssdk.core.pagination.SdkIterable;
import software.amazon.awssdk.core.pagination.SyncPageFetcher;
import software.amazon.awssdk.services.cloudformation.CloudFormationClient;
import software.amazon.awssdk.services.cloudformation.model.ListStackResourcesRequest;
import software.amazon.awssdk.services.cloudformation.model.ListStackResourcesResponse;
import software.amazon.awssdk.services.cloudformation.model.StackResourceSummary;

/**
 * <p>
 * Represents the output for the
 * {@link software.amazon.awssdk.services.cloudformation.CloudFormationClient#listStackResourcesPaginator(software.amazon.awssdk.services.cloudformation.model.ListStackResourcesRequest)}
 * operation which is a paginated operation. This class is an iterable of
 * {@link software.amazon.awssdk.services.cloudformation.model.ListStackResourcesResponse} that can be used to iterate
 * through all the response pages of the operation.
 * </p>
 * <p>
 * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet and
 * so there is no guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily
 * loading response pages by making service calls until there are no pages left or your iteration stops. If there are
 * errors in your request, you will see the failures only after you start iterating through the iterable.
 * </p>
 *
 * <p>
 * The following are few ways to iterate through the response pages:
 * </p>
 * 1) Using a Stream
 * 
 * <pre>
 * {@code
 * software.amazon.awssdk.services.cloudformation.paginators.ListStackResourcesIterable responses = client.listStackResourcesPaginator(request);
 * responses.stream().forEach(....);
 * }
 * </pre>
 *
 * 2) Using For loop
 * 
 * <pre>
 * {
 *     &#064;code
 *     software.amazon.awssdk.services.cloudformation.paginators.ListStackResourcesIterable responses = client
 *             .listStackResourcesPaginator(request);
 *     for (software.amazon.awssdk.services.cloudformation.model.ListStackResourcesResponse response : responses) {
 *         // do something;
 *     }
 * }
 * </pre>
 *
 * 3) Use iterator directly
 * 
 * <pre>
 * {@code
 * software.amazon.awssdk.services.cloudformation.paginators.ListStackResourcesIterable responses = client.listStackResourcesPaginator(request);
 * responses.iterator().forEachRemaining(....);
 * }
 * </pre>
 * <p>
 * <b>Note: If you prefer to have control on service calls, use the
 * {@link #listStackResources(software.amazon.awssdk.services.cloudformation.model.ListStackResourcesRequest)}
 * operation.</b>
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public class ListStackResourcesIterable implements SdkIterable<ListStackResourcesResponse> {
    private final CloudFormationClient client;

    private final ListStackResourcesRequest firstRequest;

    private final SyncPageFetcher nextPageFetcher;

    public ListStackResourcesIterable(final CloudFormationClient client, final ListStackResourcesRequest firstRequest) {
        this.client = client;
        this.firstRequest = firstRequest;
        this.nextPageFetcher = new ListStackResourcesResponseFetcher();
    }

    @Override
    public Iterator<ListStackResourcesResponse> iterator() {
        return new PaginatedResponsesIterator(nextPageFetcher);
    }

    /**
     * Returns an iterable to iterate through the paginated {@link ListStackResourcesResponse#stackResourceSummaries()}
     * member. The returned iterable is used to iterate through the results across all response pages and not a single
     * page.
     *
     * This method is useful if you are interested in iterating over the paginated member in the response pages instead
     * of the top level pages. Similar to iteration over pages, this method internally makes service calls to get the
     * next list of results until the iteration stops or there are no more results.
     */
    public final SdkIterable<StackResourceSummary> stackResourceSummaries() {
        Function<ListStackResourcesResponse, Iterator<StackResourceSummary>> getIterator = response -> {
            if (response != null && response.stackResourceSummaries() != null) {
                return response.stackResourceSummaries().iterator();
            }
            return Collections.emptyIterator();
        };
        return new PaginatedItemsIterable(this, getIterator);
    }

    /**
     * <p>
     * A helper method to resume the pages in case of unexpected failures. The method takes the last successful response
     * page as input and returns an instance of {@link ListStackResourcesIterable} that can be used to retrieve the
     * consecutive pages that follows the input page.
     * </p>
     */
    public final ListStackResourcesIterable resume(final ListStackResourcesResponse lastSuccessfulPage) {
        if (nextPageFetcher.hasNextPage(lastSuccessfulPage)) {
            return new ListStackResourcesIterable(client, firstRequest.toBuilder().nextToken(lastSuccessfulPage.nextToken())
                    .build());
        }
        return new ListStackResourcesIterable(client, firstRequest) {
            @Override
            public Iterator<ListStackResourcesResponse> iterator() {
                return Collections.emptyIterator();
            }
        };
    }

    private class ListStackResourcesResponseFetcher implements SyncPageFetcher<ListStackResourcesResponse> {
        @Override
        public boolean hasNextPage(ListStackResourcesResponse previousPage) {
            return previousPage.nextToken() != null;
        }

        @Override
        public ListStackResourcesResponse nextPage(ListStackResourcesResponse previousPage) {
            if (previousPage == null) {
                return client.listStackResources(firstRequest);
            }
            return client.listStackResources(firstRequest.toBuilder().nextToken(previousPage.nextToken()).build());
        }
    }
}
