/*
 * Copyright 2013-2018 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.cloudformation.paginators;

import java.util.Collections;
import java.util.Iterator;
import java.util.concurrent.CompletableFuture;
import java.util.function.Function;
import javax.annotation.Generated;
import org.reactivestreams.Subscriber;
import software.amazon.awssdk.core.pagination.async.AsyncPageFetcher;
import software.amazon.awssdk.core.pagination.async.EmptySubscription;
import software.amazon.awssdk.core.pagination.async.PaginatedItemsPublisher;
import software.amazon.awssdk.core.pagination.async.ResponsesSubscription;
import software.amazon.awssdk.core.pagination.async.SdkPublisher;
import software.amazon.awssdk.services.cloudformation.CloudFormationAsyncClient;
import software.amazon.awssdk.services.cloudformation.model.DescribeStackEventsRequest;
import software.amazon.awssdk.services.cloudformation.model.DescribeStackEventsResponse;
import software.amazon.awssdk.services.cloudformation.model.StackEvent;

/**
 * <p>
 * Represents the output for the
 * {@link software.amazon.awssdk.services.cloudformation.CloudFormationAsyncClient#describeStackEventsPaginator(software.amazon.awssdk.services.cloudformation.model.DescribeStackEventsRequest)}
 * operation which is a paginated operation. This class is a type of {@link org.reactivestreams.Publisher} which can be
 * used to provide a sequence of
 * {@link software.amazon.awssdk.services.cloudformation.model.DescribeStackEventsResponse} response pages as per demand
 * from the subscriber.
 * </p>
 * <p>
 * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet and
 * so there is no guarantee that the request is valid. If there are errors in your request, you will see the failures
 * only after you start streaming the data. The subscribe method should be called as a request to start streaming data.
 * For more info, see {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the
 * subscribe method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data
 * from the starting request.
 * </p>
 *
 * <p>
 * The following are few ways to use the response class:
 * </p>
 * 1) Using the forEach helper method
 * 
 * <pre>
 * {@code
 * software.amazon.awssdk.services.cloudformation.paginators.DescribeStackEventsPublisher publisher = client.describeStackEventsPaginator(request);
 * CompletableFuture<Void> future = publisher.forEach(res -> { // Do something with the response });
 * future.get();
 * }
 * </pre>
 *
 * 2) Using a custom subscriber
 * 
 * <pre>
 * {@code
 * software.amazon.awssdk.services.cloudformation.paginators.DescribeStackEventsPublisher publisher = client.describeStackEventsPaginator(request);
 * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.cloudformation.model.DescribeStackEventsResponse>() {
 * 
 * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
 * 
 * 
 * public void onNext(software.amazon.awssdk.services.cloudformation.model.DescribeStackEventsResponse response) { //... };
 * });}
 * </pre>
 * 
 * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
 * <p>
 * <b>Note: If you prefer to have control on service calls, use the
 * {@link #describeStackEvents(software.amazon.awssdk.services.cloudformation.model.DescribeStackEventsRequest)}
 * operation.</b>
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public class DescribeStackEventsPublisher implements SdkPublisher<DescribeStackEventsResponse> {
    private final CloudFormationAsyncClient client;

    private final DescribeStackEventsRequest firstRequest;

    private final AsyncPageFetcher nextPageFetcher;

    private boolean isLastPage;

    public DescribeStackEventsPublisher(final CloudFormationAsyncClient client, final DescribeStackEventsRequest firstRequest) {
        this(client, firstRequest, false);
    }

    private DescribeStackEventsPublisher(final CloudFormationAsyncClient client, final DescribeStackEventsRequest firstRequest,
            final boolean isLastPage) {
        this.client = client;
        this.firstRequest = firstRequest;
        this.isLastPage = isLastPage;
        this.nextPageFetcher = new DescribeStackEventsResponseFetcher();
    }

    @Override
    public void subscribe(Subscriber<? super DescribeStackEventsResponse> subscriber) {
        subscriber.onSubscribe(new ResponsesSubscription(subscriber, nextPageFetcher));
    }

    /**
     * Returns a publisher that can be used to get a stream of data. You need to subscribe to the publisher to request
     * the stream of data. The publisher has a helper forEach method that takes in a {@link java.util.function.Consumer}
     * and then applies that consumer to each response returned by the service.
     */
    public final SdkPublisher<StackEvent> stackEvents() {
        Function<DescribeStackEventsResponse, Iterator<StackEvent>> getIterator = response -> {
            if (response != null && response.stackEvents() != null) {
                return response.stackEvents().iterator();
            }
            return Collections.emptyIterator();
        };
        return new PaginatedItemsPublisher(new DescribeStackEventsResponseFetcher(), getIterator, isLastPage);
    }

    /**
     * <p>
     * A helper method to resume the pages in case of unexpected failures. The method takes the last successful response
     * page as input and returns an instance of {@link DescribeStackEventsPublisher} that can be used to retrieve the
     * consecutive pages that follows the input page.
     * </p>
     */
    public final DescribeStackEventsPublisher resume(final DescribeStackEventsResponse lastSuccessfulPage) {
        if (nextPageFetcher.hasNextPage(lastSuccessfulPage)) {
            return new DescribeStackEventsPublisher(client, firstRequest.toBuilder().nextToken(lastSuccessfulPage.nextToken())
                    .build());
        }
        return new DescribeStackEventsPublisher(client, firstRequest, true) {
            @Override
            public void subscribe(Subscriber<? super DescribeStackEventsResponse> subscriber) {
                subscriber.onSubscribe(new EmptySubscription(subscriber));
            }
        };
    }

    private class DescribeStackEventsResponseFetcher implements AsyncPageFetcher<DescribeStackEventsResponse> {
        @Override
        public boolean hasNextPage(final DescribeStackEventsResponse previousPage) {
            return previousPage.nextToken() != null;
        }

        @Override
        public CompletableFuture<DescribeStackEventsResponse> nextPage(final DescribeStackEventsResponse previousPage) {
            if (previousPage == null) {
                return client.describeStackEvents(firstRequest);
            }
            return client.describeStackEvents(firstRequest.toBuilder().nextToken(previousPage.nextToken()).build());
        }
    }
}
