/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.chimesdkidentity;

import java.util.Collections;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsAsyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.awscore.internal.AwsProtocolMetadata;
import software.amazon.awssdk.awscore.internal.AwsServiceProtocol;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkPlugin;
import software.amazon.awssdk.core.SdkRequest;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.AsyncClientHandler;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.chimesdkidentity.internal.ChimeSdkIdentityServiceClientConfigurationBuilder;
import software.amazon.awssdk.services.chimesdkidentity.model.BadRequestException;
import software.amazon.awssdk.services.chimesdkidentity.model.ChimeSdkIdentityException;
import software.amazon.awssdk.services.chimesdkidentity.model.ConflictException;
import software.amazon.awssdk.services.chimesdkidentity.model.CreateAppInstanceAdminRequest;
import software.amazon.awssdk.services.chimesdkidentity.model.CreateAppInstanceAdminResponse;
import software.amazon.awssdk.services.chimesdkidentity.model.CreateAppInstanceBotRequest;
import software.amazon.awssdk.services.chimesdkidentity.model.CreateAppInstanceBotResponse;
import software.amazon.awssdk.services.chimesdkidentity.model.CreateAppInstanceRequest;
import software.amazon.awssdk.services.chimesdkidentity.model.CreateAppInstanceResponse;
import software.amazon.awssdk.services.chimesdkidentity.model.CreateAppInstanceUserRequest;
import software.amazon.awssdk.services.chimesdkidentity.model.CreateAppInstanceUserResponse;
import software.amazon.awssdk.services.chimesdkidentity.model.DeleteAppInstanceAdminRequest;
import software.amazon.awssdk.services.chimesdkidentity.model.DeleteAppInstanceAdminResponse;
import software.amazon.awssdk.services.chimesdkidentity.model.DeleteAppInstanceBotRequest;
import software.amazon.awssdk.services.chimesdkidentity.model.DeleteAppInstanceBotResponse;
import software.amazon.awssdk.services.chimesdkidentity.model.DeleteAppInstanceRequest;
import software.amazon.awssdk.services.chimesdkidentity.model.DeleteAppInstanceResponse;
import software.amazon.awssdk.services.chimesdkidentity.model.DeleteAppInstanceUserRequest;
import software.amazon.awssdk.services.chimesdkidentity.model.DeleteAppInstanceUserResponse;
import software.amazon.awssdk.services.chimesdkidentity.model.DeregisterAppInstanceUserEndpointRequest;
import software.amazon.awssdk.services.chimesdkidentity.model.DeregisterAppInstanceUserEndpointResponse;
import software.amazon.awssdk.services.chimesdkidentity.model.DescribeAppInstanceAdminRequest;
import software.amazon.awssdk.services.chimesdkidentity.model.DescribeAppInstanceAdminResponse;
import software.amazon.awssdk.services.chimesdkidentity.model.DescribeAppInstanceBotRequest;
import software.amazon.awssdk.services.chimesdkidentity.model.DescribeAppInstanceBotResponse;
import software.amazon.awssdk.services.chimesdkidentity.model.DescribeAppInstanceRequest;
import software.amazon.awssdk.services.chimesdkidentity.model.DescribeAppInstanceResponse;
import software.amazon.awssdk.services.chimesdkidentity.model.DescribeAppInstanceUserEndpointRequest;
import software.amazon.awssdk.services.chimesdkidentity.model.DescribeAppInstanceUserEndpointResponse;
import software.amazon.awssdk.services.chimesdkidentity.model.DescribeAppInstanceUserRequest;
import software.amazon.awssdk.services.chimesdkidentity.model.DescribeAppInstanceUserResponse;
import software.amazon.awssdk.services.chimesdkidentity.model.ForbiddenException;
import software.amazon.awssdk.services.chimesdkidentity.model.GetAppInstanceRetentionSettingsRequest;
import software.amazon.awssdk.services.chimesdkidentity.model.GetAppInstanceRetentionSettingsResponse;
import software.amazon.awssdk.services.chimesdkidentity.model.ListAppInstanceAdminsRequest;
import software.amazon.awssdk.services.chimesdkidentity.model.ListAppInstanceAdminsResponse;
import software.amazon.awssdk.services.chimesdkidentity.model.ListAppInstanceBotsRequest;
import software.amazon.awssdk.services.chimesdkidentity.model.ListAppInstanceBotsResponse;
import software.amazon.awssdk.services.chimesdkidentity.model.ListAppInstanceUserEndpointsRequest;
import software.amazon.awssdk.services.chimesdkidentity.model.ListAppInstanceUserEndpointsResponse;
import software.amazon.awssdk.services.chimesdkidentity.model.ListAppInstanceUsersRequest;
import software.amazon.awssdk.services.chimesdkidentity.model.ListAppInstanceUsersResponse;
import software.amazon.awssdk.services.chimesdkidentity.model.ListAppInstancesRequest;
import software.amazon.awssdk.services.chimesdkidentity.model.ListAppInstancesResponse;
import software.amazon.awssdk.services.chimesdkidentity.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.chimesdkidentity.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.chimesdkidentity.model.NotFoundException;
import software.amazon.awssdk.services.chimesdkidentity.model.PutAppInstanceRetentionSettingsRequest;
import software.amazon.awssdk.services.chimesdkidentity.model.PutAppInstanceRetentionSettingsResponse;
import software.amazon.awssdk.services.chimesdkidentity.model.PutAppInstanceUserExpirationSettingsRequest;
import software.amazon.awssdk.services.chimesdkidentity.model.PutAppInstanceUserExpirationSettingsResponse;
import software.amazon.awssdk.services.chimesdkidentity.model.RegisterAppInstanceUserEndpointRequest;
import software.amazon.awssdk.services.chimesdkidentity.model.RegisterAppInstanceUserEndpointResponse;
import software.amazon.awssdk.services.chimesdkidentity.model.ResourceLimitExceededException;
import software.amazon.awssdk.services.chimesdkidentity.model.ServiceFailureException;
import software.amazon.awssdk.services.chimesdkidentity.model.ServiceUnavailableException;
import software.amazon.awssdk.services.chimesdkidentity.model.TagResourceRequest;
import software.amazon.awssdk.services.chimesdkidentity.model.TagResourceResponse;
import software.amazon.awssdk.services.chimesdkidentity.model.ThrottledClientException;
import software.amazon.awssdk.services.chimesdkidentity.model.UnauthorizedClientException;
import software.amazon.awssdk.services.chimesdkidentity.model.UntagResourceRequest;
import software.amazon.awssdk.services.chimesdkidentity.model.UntagResourceResponse;
import software.amazon.awssdk.services.chimesdkidentity.model.UpdateAppInstanceBotRequest;
import software.amazon.awssdk.services.chimesdkidentity.model.UpdateAppInstanceBotResponse;
import software.amazon.awssdk.services.chimesdkidentity.model.UpdateAppInstanceRequest;
import software.amazon.awssdk.services.chimesdkidentity.model.UpdateAppInstanceResponse;
import software.amazon.awssdk.services.chimesdkidentity.model.UpdateAppInstanceUserEndpointRequest;
import software.amazon.awssdk.services.chimesdkidentity.model.UpdateAppInstanceUserEndpointResponse;
import software.amazon.awssdk.services.chimesdkidentity.model.UpdateAppInstanceUserRequest;
import software.amazon.awssdk.services.chimesdkidentity.model.UpdateAppInstanceUserResponse;
import software.amazon.awssdk.services.chimesdkidentity.transform.CreateAppInstanceAdminRequestMarshaller;
import software.amazon.awssdk.services.chimesdkidentity.transform.CreateAppInstanceBotRequestMarshaller;
import software.amazon.awssdk.services.chimesdkidentity.transform.CreateAppInstanceRequestMarshaller;
import software.amazon.awssdk.services.chimesdkidentity.transform.CreateAppInstanceUserRequestMarshaller;
import software.amazon.awssdk.services.chimesdkidentity.transform.DeleteAppInstanceAdminRequestMarshaller;
import software.amazon.awssdk.services.chimesdkidentity.transform.DeleteAppInstanceBotRequestMarshaller;
import software.amazon.awssdk.services.chimesdkidentity.transform.DeleteAppInstanceRequestMarshaller;
import software.amazon.awssdk.services.chimesdkidentity.transform.DeleteAppInstanceUserRequestMarshaller;
import software.amazon.awssdk.services.chimesdkidentity.transform.DeregisterAppInstanceUserEndpointRequestMarshaller;
import software.amazon.awssdk.services.chimesdkidentity.transform.DescribeAppInstanceAdminRequestMarshaller;
import software.amazon.awssdk.services.chimesdkidentity.transform.DescribeAppInstanceBotRequestMarshaller;
import software.amazon.awssdk.services.chimesdkidentity.transform.DescribeAppInstanceRequestMarshaller;
import software.amazon.awssdk.services.chimesdkidentity.transform.DescribeAppInstanceUserEndpointRequestMarshaller;
import software.amazon.awssdk.services.chimesdkidentity.transform.DescribeAppInstanceUserRequestMarshaller;
import software.amazon.awssdk.services.chimesdkidentity.transform.GetAppInstanceRetentionSettingsRequestMarshaller;
import software.amazon.awssdk.services.chimesdkidentity.transform.ListAppInstanceAdminsRequestMarshaller;
import software.amazon.awssdk.services.chimesdkidentity.transform.ListAppInstanceBotsRequestMarshaller;
import software.amazon.awssdk.services.chimesdkidentity.transform.ListAppInstanceUserEndpointsRequestMarshaller;
import software.amazon.awssdk.services.chimesdkidentity.transform.ListAppInstanceUsersRequestMarshaller;
import software.amazon.awssdk.services.chimesdkidentity.transform.ListAppInstancesRequestMarshaller;
import software.amazon.awssdk.services.chimesdkidentity.transform.ListTagsForResourceRequestMarshaller;
import software.amazon.awssdk.services.chimesdkidentity.transform.PutAppInstanceRetentionSettingsRequestMarshaller;
import software.amazon.awssdk.services.chimesdkidentity.transform.PutAppInstanceUserExpirationSettingsRequestMarshaller;
import software.amazon.awssdk.services.chimesdkidentity.transform.RegisterAppInstanceUserEndpointRequestMarshaller;
import software.amazon.awssdk.services.chimesdkidentity.transform.TagResourceRequestMarshaller;
import software.amazon.awssdk.services.chimesdkidentity.transform.UntagResourceRequestMarshaller;
import software.amazon.awssdk.services.chimesdkidentity.transform.UpdateAppInstanceBotRequestMarshaller;
import software.amazon.awssdk.services.chimesdkidentity.transform.UpdateAppInstanceRequestMarshaller;
import software.amazon.awssdk.services.chimesdkidentity.transform.UpdateAppInstanceUserEndpointRequestMarshaller;
import software.amazon.awssdk.services.chimesdkidentity.transform.UpdateAppInstanceUserRequestMarshaller;
import software.amazon.awssdk.utils.CompletableFutureUtils;

/**
 * Internal implementation of {@link ChimeSdkIdentityAsyncClient}.
 *
 * @see ChimeSdkIdentityAsyncClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultChimeSdkIdentityAsyncClient implements ChimeSdkIdentityAsyncClient {
    private static final Logger log = LoggerFactory.getLogger(DefaultChimeSdkIdentityAsyncClient.class);

    private static final AwsProtocolMetadata protocolMetadata = AwsProtocolMetadata.builder()
            .serviceProtocol(AwsServiceProtocol.REST_JSON).build();

    private final AsyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultChimeSdkIdentityAsyncClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsAsyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    /**
     * <p>
     * Creates an Amazon Chime SDK messaging <code>AppInstance</code> under an AWS account. Only SDK messaging customers
     * use this API. <code>CreateAppInstance</code> supports idempotency behavior as described in the AWS API Standard.
     * </p>
     * <p>
     * identity
     * </p>
     *
     * @param createAppInstanceRequest
     * @return A Java Future containing the result of the CreateAppInstance operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException The input parameters don't match the service's restrictions.</li>
     *         <li>ConflictException The request could not be processed because of conflict in the current state of the
     *         resource.</li>
     *         <li>ForbiddenException The client is permanently forbidden from making the request.</li>
     *         <li>ResourceLimitExceededException The request exceeds the resource limit.</li>
     *         <li>ThrottledClientException The client exceeded its request rate limit.</li>
     *         <li>UnauthorizedClientException The client is not currently authorized to make the request.</li>
     *         <li>ServiceUnavailableException The service is currently unavailable.</li>
     *         <li>ServiceFailureException The service encountered an unexpected error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ChimeSdkIdentityException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample ChimeSdkIdentityAsyncClient.CreateAppInstance
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/chime-sdk-identity-2021-04-20/CreateAppInstance"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateAppInstanceResponse> createAppInstance(CreateAppInstanceRequest createAppInstanceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createAppInstanceRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createAppInstanceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Chime SDK Identity");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateAppInstance");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateAppInstanceResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateAppInstanceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateAppInstanceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateAppInstanceRequest, CreateAppInstanceResponse>()
                            .withOperationName("CreateAppInstance").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new CreateAppInstanceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createAppInstanceRequest));
            CompletableFuture<CreateAppInstanceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Promotes an <code>AppInstanceUser</code> or <code>AppInstanceBot</code> to an <code>AppInstanceAdmin</code>. The
     * promoted entity can perform the following actions.
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>ChannelModerator</code> actions across all channels in the <code>AppInstance</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>DeleteChannelMessage</code> actions.
     * </p>
     * </li>
     * </ul>
     * <p>
     * Only an <code>AppInstanceUser</code> and <code>AppInstanceBot</code> can be promoted to an
     * <code>AppInstanceAdmin</code> role.
     * </p>
     *
     * @param createAppInstanceAdminRequest
     * @return A Java Future containing the result of the CreateAppInstanceAdmin operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException The input parameters don't match the service's restrictions.</li>
     *         <li>ConflictException The request could not be processed because of conflict in the current state of the
     *         resource.</li>
     *         <li>ForbiddenException The client is permanently forbidden from making the request.</li>
     *         <li>ResourceLimitExceededException The request exceeds the resource limit.</li>
     *         <li>ThrottledClientException The client exceeded its request rate limit.</li>
     *         <li>UnauthorizedClientException The client is not currently authorized to make the request.</li>
     *         <li>ServiceUnavailableException The service is currently unavailable.</li>
     *         <li>ServiceFailureException The service encountered an unexpected error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ChimeSdkIdentityException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample ChimeSdkIdentityAsyncClient.CreateAppInstanceAdmin
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/chime-sdk-identity-2021-04-20/CreateAppInstanceAdmin"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateAppInstanceAdminResponse> createAppInstanceAdmin(
            CreateAppInstanceAdminRequest createAppInstanceAdminRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createAppInstanceAdminRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createAppInstanceAdminRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Chime SDK Identity");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateAppInstanceAdmin");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateAppInstanceAdminResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateAppInstanceAdminResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateAppInstanceAdminResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateAppInstanceAdminRequest, CreateAppInstanceAdminResponse>()
                            .withOperationName("CreateAppInstanceAdmin").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new CreateAppInstanceAdminRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createAppInstanceAdminRequest));
            CompletableFuture<CreateAppInstanceAdminResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a bot under an Amazon Chime <code>AppInstance</code>. The request consists of a unique
     * <code>Configuration</code> and <code>Name</code> for that bot.
     * </p>
     *
     * @param createAppInstanceBotRequest
     * @return A Java Future containing the result of the CreateAppInstanceBot operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException The input parameters don't match the service's restrictions.</li>
     *         <li>ConflictException The request could not be processed because of conflict in the current state of the
     *         resource.</li>
     *         <li>ForbiddenException The client is permanently forbidden from making the request.</li>
     *         <li>ResourceLimitExceededException The request exceeds the resource limit.</li>
     *         <li>ThrottledClientException The client exceeded its request rate limit.</li>
     *         <li>UnauthorizedClientException The client is not currently authorized to make the request.</li>
     *         <li>ServiceUnavailableException The service is currently unavailable.</li>
     *         <li>ServiceFailureException The service encountered an unexpected error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ChimeSdkIdentityException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample ChimeSdkIdentityAsyncClient.CreateAppInstanceBot
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/chime-sdk-identity-2021-04-20/CreateAppInstanceBot"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateAppInstanceBotResponse> createAppInstanceBot(
            CreateAppInstanceBotRequest createAppInstanceBotRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createAppInstanceBotRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createAppInstanceBotRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Chime SDK Identity");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateAppInstanceBot");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateAppInstanceBotResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateAppInstanceBotResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateAppInstanceBotResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateAppInstanceBotRequest, CreateAppInstanceBotResponse>()
                            .withOperationName("CreateAppInstanceBot").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new CreateAppInstanceBotRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createAppInstanceBotRequest));
            CompletableFuture<CreateAppInstanceBotResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a user under an Amazon Chime <code>AppInstance</code>. The request consists of a unique
     * <code>appInstanceUserId</code> and <code>Name</code> for that user.
     * </p>
     *
     * @param createAppInstanceUserRequest
     * @return A Java Future containing the result of the CreateAppInstanceUser operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException The input parameters don't match the service's restrictions.</li>
     *         <li>ConflictException The request could not be processed because of conflict in the current state of the
     *         resource.</li>
     *         <li>ForbiddenException The client is permanently forbidden from making the request.</li>
     *         <li>ResourceLimitExceededException The request exceeds the resource limit.</li>
     *         <li>ThrottledClientException The client exceeded its request rate limit.</li>
     *         <li>UnauthorizedClientException The client is not currently authorized to make the request.</li>
     *         <li>ServiceUnavailableException The service is currently unavailable.</li>
     *         <li>ServiceFailureException The service encountered an unexpected error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ChimeSdkIdentityException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample ChimeSdkIdentityAsyncClient.CreateAppInstanceUser
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/chime-sdk-identity-2021-04-20/CreateAppInstanceUser"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateAppInstanceUserResponse> createAppInstanceUser(
            CreateAppInstanceUserRequest createAppInstanceUserRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createAppInstanceUserRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createAppInstanceUserRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Chime SDK Identity");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateAppInstanceUser");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateAppInstanceUserResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateAppInstanceUserResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateAppInstanceUserResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateAppInstanceUserRequest, CreateAppInstanceUserResponse>()
                            .withOperationName("CreateAppInstanceUser").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new CreateAppInstanceUserRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createAppInstanceUserRequest));
            CompletableFuture<CreateAppInstanceUserResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes an <code>AppInstance</code> and all associated data asynchronously.
     * </p>
     *
     * @param deleteAppInstanceRequest
     * @return A Java Future containing the result of the DeleteAppInstance operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException The input parameters don't match the service's restrictions.</li>
     *         <li>ForbiddenException The client is permanently forbidden from making the request.</li>
     *         <li>ResourceLimitExceededException The request exceeds the resource limit.</li>
     *         <li>ThrottledClientException The client exceeded its request rate limit.</li>
     *         <li>UnauthorizedClientException The client is not currently authorized to make the request.</li>
     *         <li>ServiceUnavailableException The service is currently unavailable.</li>
     *         <li>ServiceFailureException The service encountered an unexpected error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ChimeSdkIdentityException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample ChimeSdkIdentityAsyncClient.DeleteAppInstance
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/chime-sdk-identity-2021-04-20/DeleteAppInstance"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteAppInstanceResponse> deleteAppInstance(DeleteAppInstanceRequest deleteAppInstanceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteAppInstanceRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteAppInstanceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Chime SDK Identity");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteAppInstance");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteAppInstanceResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteAppInstanceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteAppInstanceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteAppInstanceRequest, DeleteAppInstanceResponse>()
                            .withOperationName("DeleteAppInstance").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DeleteAppInstanceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteAppInstanceRequest));
            CompletableFuture<DeleteAppInstanceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Demotes an <code>AppInstanceAdmin</code> to an <code>AppInstanceUser</code> or <code>AppInstanceBot</code>. This
     * action does not delete the user.
     * </p>
     *
     * @param deleteAppInstanceAdminRequest
     * @return A Java Future containing the result of the DeleteAppInstanceAdmin operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException The input parameters don't match the service's restrictions.</li>
     *         <li>ConflictException The request could not be processed because of conflict in the current state of the
     *         resource.</li>
     *         <li>ForbiddenException The client is permanently forbidden from making the request.</li>
     *         <li>ResourceLimitExceededException The request exceeds the resource limit.</li>
     *         <li>ThrottledClientException The client exceeded its request rate limit.</li>
     *         <li>UnauthorizedClientException The client is not currently authorized to make the request.</li>
     *         <li>ServiceUnavailableException The service is currently unavailable.</li>
     *         <li>ServiceFailureException The service encountered an unexpected error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ChimeSdkIdentityException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample ChimeSdkIdentityAsyncClient.DeleteAppInstanceAdmin
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/chime-sdk-identity-2021-04-20/DeleteAppInstanceAdmin"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteAppInstanceAdminResponse> deleteAppInstanceAdmin(
            DeleteAppInstanceAdminRequest deleteAppInstanceAdminRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteAppInstanceAdminRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteAppInstanceAdminRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Chime SDK Identity");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteAppInstanceAdmin");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteAppInstanceAdminResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteAppInstanceAdminResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteAppInstanceAdminResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteAppInstanceAdminRequest, DeleteAppInstanceAdminResponse>()
                            .withOperationName("DeleteAppInstanceAdmin").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DeleteAppInstanceAdminRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteAppInstanceAdminRequest));
            CompletableFuture<DeleteAppInstanceAdminResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes an <code>AppInstanceBot</code>.
     * </p>
     *
     * @param deleteAppInstanceBotRequest
     * @return A Java Future containing the result of the DeleteAppInstanceBot operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException The input parameters don't match the service's restrictions.</li>
     *         <li>ConflictException The request could not be processed because of conflict in the current state of the
     *         resource.</li>
     *         <li>ForbiddenException The client is permanently forbidden from making the request.</li>
     *         <li>ResourceLimitExceededException The request exceeds the resource limit.</li>
     *         <li>ThrottledClientException The client exceeded its request rate limit.</li>
     *         <li>UnauthorizedClientException The client is not currently authorized to make the request.</li>
     *         <li>ServiceUnavailableException The service is currently unavailable.</li>
     *         <li>ServiceFailureException The service encountered an unexpected error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ChimeSdkIdentityException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample ChimeSdkIdentityAsyncClient.DeleteAppInstanceBot
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/chime-sdk-identity-2021-04-20/DeleteAppInstanceBot"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteAppInstanceBotResponse> deleteAppInstanceBot(
            DeleteAppInstanceBotRequest deleteAppInstanceBotRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteAppInstanceBotRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteAppInstanceBotRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Chime SDK Identity");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteAppInstanceBot");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteAppInstanceBotResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteAppInstanceBotResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteAppInstanceBotResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteAppInstanceBotRequest, DeleteAppInstanceBotResponse>()
                            .withOperationName("DeleteAppInstanceBot").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DeleteAppInstanceBotRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteAppInstanceBotRequest));
            CompletableFuture<DeleteAppInstanceBotResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes an <code>AppInstanceUser</code>.
     * </p>
     *
     * @param deleteAppInstanceUserRequest
     * @return A Java Future containing the result of the DeleteAppInstanceUser operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException The input parameters don't match the service's restrictions.</li>
     *         <li>ConflictException The request could not be processed because of conflict in the current state of the
     *         resource.</li>
     *         <li>ForbiddenException The client is permanently forbidden from making the request.</li>
     *         <li>ResourceLimitExceededException The request exceeds the resource limit.</li>
     *         <li>ThrottledClientException The client exceeded its request rate limit.</li>
     *         <li>UnauthorizedClientException The client is not currently authorized to make the request.</li>
     *         <li>ServiceUnavailableException The service is currently unavailable.</li>
     *         <li>ServiceFailureException The service encountered an unexpected error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ChimeSdkIdentityException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample ChimeSdkIdentityAsyncClient.DeleteAppInstanceUser
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/chime-sdk-identity-2021-04-20/DeleteAppInstanceUser"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteAppInstanceUserResponse> deleteAppInstanceUser(
            DeleteAppInstanceUserRequest deleteAppInstanceUserRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteAppInstanceUserRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteAppInstanceUserRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Chime SDK Identity");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteAppInstanceUser");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteAppInstanceUserResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteAppInstanceUserResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteAppInstanceUserResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteAppInstanceUserRequest, DeleteAppInstanceUserResponse>()
                            .withOperationName("DeleteAppInstanceUser").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DeleteAppInstanceUserRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteAppInstanceUserRequest));
            CompletableFuture<DeleteAppInstanceUserResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deregisters an <code>AppInstanceUserEndpoint</code>.
     * </p>
     *
     * @param deregisterAppInstanceUserEndpointRequest
     * @return A Java Future containing the result of the DeregisterAppInstanceUserEndpoint operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException The input parameters don't match the service's restrictions.</li>
     *         <li>ForbiddenException The client is permanently forbidden from making the request.</li>
     *         <li>ThrottledClientException The client exceeded its request rate limit.</li>
     *         <li>UnauthorizedClientException The client is not currently authorized to make the request.</li>
     *         <li>ServiceUnavailableException The service is currently unavailable.</li>
     *         <li>ServiceFailureException The service encountered an unexpected error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ChimeSdkIdentityException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample ChimeSdkIdentityAsyncClient.DeregisterAppInstanceUserEndpoint
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/chime-sdk-identity-2021-04-20/DeregisterAppInstanceUserEndpoint"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeregisterAppInstanceUserEndpointResponse> deregisterAppInstanceUserEndpoint(
            DeregisterAppInstanceUserEndpointRequest deregisterAppInstanceUserEndpointRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deregisterAppInstanceUserEndpointRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                deregisterAppInstanceUserEndpointRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Chime SDK Identity");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeregisterAppInstanceUserEndpoint");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeregisterAppInstanceUserEndpointResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, DeregisterAppInstanceUserEndpointResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeregisterAppInstanceUserEndpointResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeregisterAppInstanceUserEndpointRequest, DeregisterAppInstanceUserEndpointResponse>()
                            .withOperationName("DeregisterAppInstanceUserEndpoint").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DeregisterAppInstanceUserEndpointRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deregisterAppInstanceUserEndpointRequest));
            CompletableFuture<DeregisterAppInstanceUserEndpointResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns the full details of an <code>AppInstance</code>.
     * </p>
     *
     * @param describeAppInstanceRequest
     * @return A Java Future containing the result of the DescribeAppInstance operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException The input parameters don't match the service's restrictions.</li>
     *         <li>ForbiddenException The client is permanently forbidden from making the request.</li>
     *         <li>ThrottledClientException The client exceeded its request rate limit.</li>
     *         <li>UnauthorizedClientException The client is not currently authorized to make the request.</li>
     *         <li>ServiceUnavailableException The service is currently unavailable.</li>
     *         <li>ServiceFailureException The service encountered an unexpected error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ChimeSdkIdentityException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample ChimeSdkIdentityAsyncClient.DescribeAppInstance
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/chime-sdk-identity-2021-04-20/DescribeAppInstance"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeAppInstanceResponse> describeAppInstance(
            DescribeAppInstanceRequest describeAppInstanceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeAppInstanceRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeAppInstanceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Chime SDK Identity");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeAppInstance");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeAppInstanceResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeAppInstanceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeAppInstanceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeAppInstanceRequest, DescribeAppInstanceResponse>()
                            .withOperationName("DescribeAppInstance").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DescribeAppInstanceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(describeAppInstanceRequest));
            CompletableFuture<DescribeAppInstanceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns the full details of an <code>AppInstanceAdmin</code>.
     * </p>
     *
     * @param describeAppInstanceAdminRequest
     * @return A Java Future containing the result of the DescribeAppInstanceAdmin operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException The input parameters don't match the service's restrictions.</li>
     *         <li>ForbiddenException The client is permanently forbidden from making the request.</li>
     *         <li>ThrottledClientException The client exceeded its request rate limit.</li>
     *         <li>UnauthorizedClientException The client is not currently authorized to make the request.</li>
     *         <li>ServiceUnavailableException The service is currently unavailable.</li>
     *         <li>ServiceFailureException The service encountered an unexpected error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ChimeSdkIdentityException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample ChimeSdkIdentityAsyncClient.DescribeAppInstanceAdmin
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/chime-sdk-identity-2021-04-20/DescribeAppInstanceAdmin"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeAppInstanceAdminResponse> describeAppInstanceAdmin(
            DescribeAppInstanceAdminRequest describeAppInstanceAdminRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeAppInstanceAdminRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeAppInstanceAdminRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Chime SDK Identity");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeAppInstanceAdmin");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeAppInstanceAdminResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeAppInstanceAdminResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeAppInstanceAdminResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeAppInstanceAdminRequest, DescribeAppInstanceAdminResponse>()
                            .withOperationName("DescribeAppInstanceAdmin").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DescribeAppInstanceAdminRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(describeAppInstanceAdminRequest));
            CompletableFuture<DescribeAppInstanceAdminResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * The <code>AppInstanceBot's</code> information.
     * </p>
     *
     * @param describeAppInstanceBotRequest
     * @return A Java Future containing the result of the DescribeAppInstanceBot operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException The input parameters don't match the service's restrictions.</li>
     *         <li>ForbiddenException The client is permanently forbidden from making the request.</li>
     *         <li>ThrottledClientException The client exceeded its request rate limit.</li>
     *         <li>UnauthorizedClientException The client is not currently authorized to make the request.</li>
     *         <li>NotFoundException One or more of the resources in the request does not exist in the system.</li>
     *         <li>ServiceUnavailableException The service is currently unavailable.</li>
     *         <li>ServiceFailureException The service encountered an unexpected error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ChimeSdkIdentityException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample ChimeSdkIdentityAsyncClient.DescribeAppInstanceBot
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/chime-sdk-identity-2021-04-20/DescribeAppInstanceBot"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeAppInstanceBotResponse> describeAppInstanceBot(
            DescribeAppInstanceBotRequest describeAppInstanceBotRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeAppInstanceBotRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeAppInstanceBotRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Chime SDK Identity");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeAppInstanceBot");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeAppInstanceBotResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeAppInstanceBotResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeAppInstanceBotResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeAppInstanceBotRequest, DescribeAppInstanceBotResponse>()
                            .withOperationName("DescribeAppInstanceBot").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DescribeAppInstanceBotRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(describeAppInstanceBotRequest));
            CompletableFuture<DescribeAppInstanceBotResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns the full details of an <code>AppInstanceUser</code>.
     * </p>
     *
     * @param describeAppInstanceUserRequest
     * @return A Java Future containing the result of the DescribeAppInstanceUser operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException The input parameters don't match the service's restrictions.</li>
     *         <li>ForbiddenException The client is permanently forbidden from making the request.</li>
     *         <li>ThrottledClientException The client exceeded its request rate limit.</li>
     *         <li>UnauthorizedClientException The client is not currently authorized to make the request.</li>
     *         <li>ServiceUnavailableException The service is currently unavailable.</li>
     *         <li>ServiceFailureException The service encountered an unexpected error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ChimeSdkIdentityException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample ChimeSdkIdentityAsyncClient.DescribeAppInstanceUser
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/chime-sdk-identity-2021-04-20/DescribeAppInstanceUser"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeAppInstanceUserResponse> describeAppInstanceUser(
            DescribeAppInstanceUserRequest describeAppInstanceUserRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeAppInstanceUserRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeAppInstanceUserRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Chime SDK Identity");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeAppInstanceUser");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeAppInstanceUserResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeAppInstanceUserResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeAppInstanceUserResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeAppInstanceUserRequest, DescribeAppInstanceUserResponse>()
                            .withOperationName("DescribeAppInstanceUser").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DescribeAppInstanceUserRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(describeAppInstanceUserRequest));
            CompletableFuture<DescribeAppInstanceUserResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns the full details of an <code>AppInstanceUserEndpoint</code>.
     * </p>
     *
     * @param describeAppInstanceUserEndpointRequest
     * @return A Java Future containing the result of the DescribeAppInstanceUserEndpoint operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException The input parameters don't match the service's restrictions.</li>
     *         <li>ForbiddenException The client is permanently forbidden from making the request.</li>
     *         <li>ThrottledClientException The client exceeded its request rate limit.</li>
     *         <li>UnauthorizedClientException The client is not currently authorized to make the request.</li>
     *         <li>ServiceUnavailableException The service is currently unavailable.</li>
     *         <li>ServiceFailureException The service encountered an unexpected error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ChimeSdkIdentityException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample ChimeSdkIdentityAsyncClient.DescribeAppInstanceUserEndpoint
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/chime-sdk-identity-2021-04-20/DescribeAppInstanceUserEndpoint"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeAppInstanceUserEndpointResponse> describeAppInstanceUserEndpoint(
            DescribeAppInstanceUserEndpointRequest describeAppInstanceUserEndpointRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeAppInstanceUserEndpointRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                describeAppInstanceUserEndpointRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Chime SDK Identity");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeAppInstanceUserEndpoint");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeAppInstanceUserEndpointResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeAppInstanceUserEndpointResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeAppInstanceUserEndpointResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeAppInstanceUserEndpointRequest, DescribeAppInstanceUserEndpointResponse>()
                            .withOperationName("DescribeAppInstanceUserEndpoint").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DescribeAppInstanceUserEndpointRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(describeAppInstanceUserEndpointRequest));
            CompletableFuture<DescribeAppInstanceUserEndpointResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Gets the retention settings for an <code>AppInstance</code>.
     * </p>
     *
     * @param getAppInstanceRetentionSettingsRequest
     * @return A Java Future containing the result of the GetAppInstanceRetentionSettings operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException The input parameters don't match the service's restrictions.</li>
     *         <li>ForbiddenException The client is permanently forbidden from making the request.</li>
     *         <li>ThrottledClientException The client exceeded its request rate limit.</li>
     *         <li>UnauthorizedClientException The client is not currently authorized to make the request.</li>
     *         <li>ServiceUnavailableException The service is currently unavailable.</li>
     *         <li>ServiceFailureException The service encountered an unexpected error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ChimeSdkIdentityException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample ChimeSdkIdentityAsyncClient.GetAppInstanceRetentionSettings
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/chime-sdk-identity-2021-04-20/GetAppInstanceRetentionSettings"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetAppInstanceRetentionSettingsResponse> getAppInstanceRetentionSettings(
            GetAppInstanceRetentionSettingsRequest getAppInstanceRetentionSettingsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getAppInstanceRetentionSettingsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                getAppInstanceRetentionSettingsRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Chime SDK Identity");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetAppInstanceRetentionSettings");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetAppInstanceRetentionSettingsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetAppInstanceRetentionSettingsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetAppInstanceRetentionSettingsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetAppInstanceRetentionSettingsRequest, GetAppInstanceRetentionSettingsResponse>()
                            .withOperationName("GetAppInstanceRetentionSettings").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetAppInstanceRetentionSettingsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getAppInstanceRetentionSettingsRequest));
            CompletableFuture<GetAppInstanceRetentionSettingsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns a list of the administrators in the <code>AppInstance</code>.
     * </p>
     *
     * @param listAppInstanceAdminsRequest
     * @return A Java Future containing the result of the ListAppInstanceAdmins operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException The input parameters don't match the service's restrictions.</li>
     *         <li>ForbiddenException The client is permanently forbidden from making the request.</li>
     *         <li>ResourceLimitExceededException The request exceeds the resource limit.</li>
     *         <li>ThrottledClientException The client exceeded its request rate limit.</li>
     *         <li>UnauthorizedClientException The client is not currently authorized to make the request.</li>
     *         <li>ServiceUnavailableException The service is currently unavailable.</li>
     *         <li>ServiceFailureException The service encountered an unexpected error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ChimeSdkIdentityException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample ChimeSdkIdentityAsyncClient.ListAppInstanceAdmins
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/chime-sdk-identity-2021-04-20/ListAppInstanceAdmins"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListAppInstanceAdminsResponse> listAppInstanceAdmins(
            ListAppInstanceAdminsRequest listAppInstanceAdminsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listAppInstanceAdminsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listAppInstanceAdminsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Chime SDK Identity");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListAppInstanceAdmins");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListAppInstanceAdminsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListAppInstanceAdminsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListAppInstanceAdminsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListAppInstanceAdminsRequest, ListAppInstanceAdminsResponse>()
                            .withOperationName("ListAppInstanceAdmins").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListAppInstanceAdminsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listAppInstanceAdminsRequest));
            CompletableFuture<ListAppInstanceAdminsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists all <code>AppInstanceBots</code> created under a single <code>AppInstance</code>.
     * </p>
     *
     * @param listAppInstanceBotsRequest
     * @return A Java Future containing the result of the ListAppInstanceBots operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException The input parameters don't match the service's restrictions.</li>
     *         <li>ForbiddenException The client is permanently forbidden from making the request.</li>
     *         <li>ResourceLimitExceededException The request exceeds the resource limit.</li>
     *         <li>ThrottledClientException The client exceeded its request rate limit.</li>
     *         <li>UnauthorizedClientException The client is not currently authorized to make the request.</li>
     *         <li>ServiceUnavailableException The service is currently unavailable.</li>
     *         <li>ServiceFailureException The service encountered an unexpected error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ChimeSdkIdentityException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample ChimeSdkIdentityAsyncClient.ListAppInstanceBots
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/chime-sdk-identity-2021-04-20/ListAppInstanceBots"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListAppInstanceBotsResponse> listAppInstanceBots(
            ListAppInstanceBotsRequest listAppInstanceBotsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listAppInstanceBotsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listAppInstanceBotsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Chime SDK Identity");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListAppInstanceBots");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListAppInstanceBotsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListAppInstanceBotsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListAppInstanceBotsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListAppInstanceBotsRequest, ListAppInstanceBotsResponse>()
                            .withOperationName("ListAppInstanceBots").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListAppInstanceBotsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listAppInstanceBotsRequest));
            CompletableFuture<ListAppInstanceBotsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists all the <code>AppInstanceUserEndpoints</code> created under a single <code>AppInstanceUser</code>.
     * </p>
     *
     * @param listAppInstanceUserEndpointsRequest
     * @return A Java Future containing the result of the ListAppInstanceUserEndpoints operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException The input parameters don't match the service's restrictions.</li>
     *         <li>ForbiddenException The client is permanently forbidden from making the request.</li>
     *         <li>ThrottledClientException The client exceeded its request rate limit.</li>
     *         <li>UnauthorizedClientException The client is not currently authorized to make the request.</li>
     *         <li>ServiceUnavailableException The service is currently unavailable.</li>
     *         <li>ServiceFailureException The service encountered an unexpected error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ChimeSdkIdentityException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample ChimeSdkIdentityAsyncClient.ListAppInstanceUserEndpoints
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/chime-sdk-identity-2021-04-20/ListAppInstanceUserEndpoints"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListAppInstanceUserEndpointsResponse> listAppInstanceUserEndpoints(
            ListAppInstanceUserEndpointsRequest listAppInstanceUserEndpointsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listAppInstanceUserEndpointsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listAppInstanceUserEndpointsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Chime SDK Identity");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListAppInstanceUserEndpoints");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListAppInstanceUserEndpointsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListAppInstanceUserEndpointsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListAppInstanceUserEndpointsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListAppInstanceUserEndpointsRequest, ListAppInstanceUserEndpointsResponse>()
                            .withOperationName("ListAppInstanceUserEndpoints").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListAppInstanceUserEndpointsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listAppInstanceUserEndpointsRequest));
            CompletableFuture<ListAppInstanceUserEndpointsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * List all <code>AppInstanceUsers</code> created under a single <code>AppInstance</code>.
     * </p>
     *
     * @param listAppInstanceUsersRequest
     * @return A Java Future containing the result of the ListAppInstanceUsers operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException The input parameters don't match the service's restrictions.</li>
     *         <li>ForbiddenException The client is permanently forbidden from making the request.</li>
     *         <li>ThrottledClientException The client exceeded its request rate limit.</li>
     *         <li>UnauthorizedClientException The client is not currently authorized to make the request.</li>
     *         <li>ServiceUnavailableException The service is currently unavailable.</li>
     *         <li>ServiceFailureException The service encountered an unexpected error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ChimeSdkIdentityException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample ChimeSdkIdentityAsyncClient.ListAppInstanceUsers
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/chime-sdk-identity-2021-04-20/ListAppInstanceUsers"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListAppInstanceUsersResponse> listAppInstanceUsers(
            ListAppInstanceUsersRequest listAppInstanceUsersRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listAppInstanceUsersRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listAppInstanceUsersRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Chime SDK Identity");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListAppInstanceUsers");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListAppInstanceUsersResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListAppInstanceUsersResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListAppInstanceUsersResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListAppInstanceUsersRequest, ListAppInstanceUsersResponse>()
                            .withOperationName("ListAppInstanceUsers").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListAppInstanceUsersRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listAppInstanceUsersRequest));
            CompletableFuture<ListAppInstanceUsersResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists all Amazon Chime <code>AppInstance</code>s created under a single AWS account.
     * </p>
     *
     * @param listAppInstancesRequest
     * @return A Java Future containing the result of the ListAppInstances operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException The input parameters don't match the service's restrictions.</li>
     *         <li>ForbiddenException The client is permanently forbidden from making the request.</li>
     *         <li>ThrottledClientException The client exceeded its request rate limit.</li>
     *         <li>UnauthorizedClientException The client is not currently authorized to make the request.</li>
     *         <li>ServiceUnavailableException The service is currently unavailable.</li>
     *         <li>ServiceFailureException The service encountered an unexpected error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ChimeSdkIdentityException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample ChimeSdkIdentityAsyncClient.ListAppInstances
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/chime-sdk-identity-2021-04-20/ListAppInstances"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListAppInstancesResponse> listAppInstances(ListAppInstancesRequest listAppInstancesRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listAppInstancesRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listAppInstancesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Chime SDK Identity");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListAppInstances");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListAppInstancesResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListAppInstancesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListAppInstancesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListAppInstancesRequest, ListAppInstancesResponse>()
                            .withOperationName("ListAppInstances").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListAppInstancesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listAppInstancesRequest));
            CompletableFuture<ListAppInstancesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists the tags applied to an Amazon Chime SDK identity resource.
     * </p>
     *
     * @param listTagsForResourceRequest
     * @return A Java Future containing the result of the ListTagsForResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException The input parameters don't match the service's restrictions.</li>
     *         <li>ForbiddenException The client is permanently forbidden from making the request.</li>
     *         <li>UnauthorizedClientException The client is not currently authorized to make the request.</li>
     *         <li>ThrottledClientException The client exceeded its request rate limit.</li>
     *         <li>ServiceUnavailableException The service is currently unavailable.</li>
     *         <li>ServiceFailureException The service encountered an unexpected error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ChimeSdkIdentityException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample ChimeSdkIdentityAsyncClient.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/chime-sdk-identity-2021-04-20/ListTagsForResource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListTagsForResourceResponse> listTagsForResource(
            ListTagsForResourceRequest listTagsForResourceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listTagsForResourceRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTagsForResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Chime SDK Identity");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTagsForResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListTagsForResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListTagsForResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListTagsForResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListTagsForResourceRequest, ListTagsForResourceResponse>()
                            .withOperationName("ListTagsForResource").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListTagsForResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listTagsForResourceRequest));
            CompletableFuture<ListTagsForResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Sets the amount of time in days that a given <code>AppInstance</code> retains data.
     * </p>
     *
     * @param putAppInstanceRetentionSettingsRequest
     * @return A Java Future containing the result of the PutAppInstanceRetentionSettings operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException The input parameters don't match the service's restrictions.</li>
     *         <li>ForbiddenException The client is permanently forbidden from making the request.</li>
     *         <li>ThrottledClientException The client exceeded its request rate limit.</li>
     *         <li>UnauthorizedClientException The client is not currently authorized to make the request.</li>
     *         <li>ServiceUnavailableException The service is currently unavailable.</li>
     *         <li>ServiceFailureException The service encountered an unexpected error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ChimeSdkIdentityException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample ChimeSdkIdentityAsyncClient.PutAppInstanceRetentionSettings
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/chime-sdk-identity-2021-04-20/PutAppInstanceRetentionSettings"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<PutAppInstanceRetentionSettingsResponse> putAppInstanceRetentionSettings(
            PutAppInstanceRetentionSettingsRequest putAppInstanceRetentionSettingsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(putAppInstanceRetentionSettingsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                putAppInstanceRetentionSettingsRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Chime SDK Identity");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "PutAppInstanceRetentionSettings");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<PutAppInstanceRetentionSettingsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, PutAppInstanceRetentionSettingsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<PutAppInstanceRetentionSettingsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<PutAppInstanceRetentionSettingsRequest, PutAppInstanceRetentionSettingsResponse>()
                            .withOperationName("PutAppInstanceRetentionSettings").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new PutAppInstanceRetentionSettingsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(putAppInstanceRetentionSettingsRequest));
            CompletableFuture<PutAppInstanceRetentionSettingsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Sets the number of days before the <code>AppInstanceUser</code> is automatically deleted.
     * </p>
     * <note>
     * <p>
     * A background process deletes expired <code>AppInstanceUsers</code> within 6 hours of expiration. Actual deletion
     * times may vary.
     * </p>
     * <p>
     * Expired <code>AppInstanceUsers</code> that have not yet been deleted appear as active, and you can update their
     * expiration settings. The system honors the new settings.
     * </p>
     * </note>
     *
     * @param putAppInstanceUserExpirationSettingsRequest
     * @return A Java Future containing the result of the PutAppInstanceUserExpirationSettings operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException The input parameters don't match the service's restrictions.</li>
     *         <li>ConflictException The request could not be processed because of conflict in the current state of the
     *         resource.</li>
     *         <li>ForbiddenException The client is permanently forbidden from making the request.</li>
     *         <li>ThrottledClientException The client exceeded its request rate limit.</li>
     *         <li>UnauthorizedClientException The client is not currently authorized to make the request.</li>
     *         <li>ServiceUnavailableException The service is currently unavailable.</li>
     *         <li>ServiceFailureException The service encountered an unexpected error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ChimeSdkIdentityException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample ChimeSdkIdentityAsyncClient.PutAppInstanceUserExpirationSettings
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/chime-sdk-identity-2021-04-20/PutAppInstanceUserExpirationSettings"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<PutAppInstanceUserExpirationSettingsResponse> putAppInstanceUserExpirationSettings(
            PutAppInstanceUserExpirationSettingsRequest putAppInstanceUserExpirationSettingsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(putAppInstanceUserExpirationSettingsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                putAppInstanceUserExpirationSettingsRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Chime SDK Identity");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "PutAppInstanceUserExpirationSettings");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<PutAppInstanceUserExpirationSettingsResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, PutAppInstanceUserExpirationSettingsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<PutAppInstanceUserExpirationSettingsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<PutAppInstanceUserExpirationSettingsRequest, PutAppInstanceUserExpirationSettingsResponse>()
                            .withOperationName("PutAppInstanceUserExpirationSettings").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new PutAppInstanceUserExpirationSettingsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(putAppInstanceUserExpirationSettingsRequest));
            CompletableFuture<PutAppInstanceUserExpirationSettingsResponse> whenCompleted = executeFuture
                    .whenComplete((r, e) -> {
                        metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
                    });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Registers an endpoint under an Amazon Chime <code>AppInstanceUser</code>. The endpoint receives messages for a
     * user. For push notifications, the endpoint is a mobile device used to receive mobile push notifications for a
     * user.
     * </p>
     *
     * @param registerAppInstanceUserEndpointRequest
     * @return A Java Future containing the result of the RegisterAppInstanceUserEndpoint operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException The input parameters don't match the service's restrictions.</li>
     *         <li>ConflictException The request could not be processed because of conflict in the current state of the
     *         resource.</li>
     *         <li>ForbiddenException The client is permanently forbidden from making the request.</li>
     *         <li>ResourceLimitExceededException The request exceeds the resource limit.</li>
     *         <li>ThrottledClientException The client exceeded its request rate limit.</li>
     *         <li>UnauthorizedClientException The client is not currently authorized to make the request.</li>
     *         <li>ServiceUnavailableException The service is currently unavailable.</li>
     *         <li>ServiceFailureException The service encountered an unexpected error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ChimeSdkIdentityException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample ChimeSdkIdentityAsyncClient.RegisterAppInstanceUserEndpoint
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/chime-sdk-identity-2021-04-20/RegisterAppInstanceUserEndpoint"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<RegisterAppInstanceUserEndpointResponse> registerAppInstanceUserEndpoint(
            RegisterAppInstanceUserEndpointRequest registerAppInstanceUserEndpointRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(registerAppInstanceUserEndpointRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                registerAppInstanceUserEndpointRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Chime SDK Identity");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "RegisterAppInstanceUserEndpoint");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<RegisterAppInstanceUserEndpointResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, RegisterAppInstanceUserEndpointResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<RegisterAppInstanceUserEndpointResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<RegisterAppInstanceUserEndpointRequest, RegisterAppInstanceUserEndpointResponse>()
                            .withOperationName("RegisterAppInstanceUserEndpoint").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new RegisterAppInstanceUserEndpointRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(registerAppInstanceUserEndpointRequest));
            CompletableFuture<RegisterAppInstanceUserEndpointResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Applies the specified tags to the specified Amazon Chime SDK identity resource.
     * </p>
     *
     * @param tagResourceRequest
     * @return A Java Future containing the result of the TagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException The input parameters don't match the service's restrictions.</li>
     *         <li>ForbiddenException The client is permanently forbidden from making the request.</li>
     *         <li>UnauthorizedClientException The client is not currently authorized to make the request.</li>
     *         <li>ResourceLimitExceededException The request exceeds the resource limit.</li>
     *         <li>ThrottledClientException The client exceeded its request rate limit.</li>
     *         <li>ServiceUnavailableException The service is currently unavailable.</li>
     *         <li>ServiceFailureException The service encountered an unexpected error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ChimeSdkIdentityException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample ChimeSdkIdentityAsyncClient.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/chime-sdk-identity-2021-04-20/TagResource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<TagResourceResponse> tagResource(TagResourceRequest tagResourceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(tagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, tagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Chime SDK Identity");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TagResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<TagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    TagResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<TagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<TagResourceRequest, TagResourceResponse>()
                            .withOperationName("TagResource").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new TagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(tagResourceRequest));
            CompletableFuture<TagResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Removes the specified tags from the specified Amazon Chime SDK identity resource.
     * </p>
     *
     * @param untagResourceRequest
     * @return A Java Future containing the result of the UntagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException The input parameters don't match the service's restrictions.</li>
     *         <li>ForbiddenException The client is permanently forbidden from making the request.</li>
     *         <li>UnauthorizedClientException The client is not currently authorized to make the request.</li>
     *         <li>ThrottledClientException The client exceeded its request rate limit.</li>
     *         <li>ServiceUnavailableException The service is currently unavailable.</li>
     *         <li>ServiceFailureException The service encountered an unexpected error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ChimeSdkIdentityException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample ChimeSdkIdentityAsyncClient.UntagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/chime-sdk-identity-2021-04-20/UntagResource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UntagResourceResponse> untagResource(UntagResourceRequest untagResourceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(untagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, untagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Chime SDK Identity");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UntagResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UntagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    UntagResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UntagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UntagResourceRequest, UntagResourceResponse>()
                            .withOperationName("UntagResource").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new UntagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(untagResourceRequest));
            CompletableFuture<UntagResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates <code>AppInstance</code> metadata.
     * </p>
     *
     * @param updateAppInstanceRequest
     * @return A Java Future containing the result of the UpdateAppInstance operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException The input parameters don't match the service's restrictions.</li>
     *         <li>ConflictException The request could not be processed because of conflict in the current state of the
     *         resource.</li>
     *         <li>ForbiddenException The client is permanently forbidden from making the request.</li>
     *         <li>ThrottledClientException The client exceeded its request rate limit.</li>
     *         <li>UnauthorizedClientException The client is not currently authorized to make the request.</li>
     *         <li>ServiceUnavailableException The service is currently unavailable.</li>
     *         <li>ServiceFailureException The service encountered an unexpected error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ChimeSdkIdentityException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample ChimeSdkIdentityAsyncClient.UpdateAppInstance
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/chime-sdk-identity-2021-04-20/UpdateAppInstance"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateAppInstanceResponse> updateAppInstance(UpdateAppInstanceRequest updateAppInstanceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateAppInstanceRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateAppInstanceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Chime SDK Identity");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateAppInstance");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateAppInstanceResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateAppInstanceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateAppInstanceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateAppInstanceRequest, UpdateAppInstanceResponse>()
                            .withOperationName("UpdateAppInstance").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new UpdateAppInstanceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(updateAppInstanceRequest));
            CompletableFuture<UpdateAppInstanceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates the name and metadata of an <code>AppInstanceBot</code>.
     * </p>
     *
     * @param updateAppInstanceBotRequest
     * @return A Java Future containing the result of the UpdateAppInstanceBot operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException The input parameters don't match the service's restrictions.</li>
     *         <li>ConflictException The request could not be processed because of conflict in the current state of the
     *         resource.</li>
     *         <li>ForbiddenException The client is permanently forbidden from making the request.</li>
     *         <li>ResourceLimitExceededException The request exceeds the resource limit.</li>
     *         <li>ThrottledClientException The client exceeded its request rate limit.</li>
     *         <li>UnauthorizedClientException The client is not currently authorized to make the request.</li>
     *         <li>ServiceUnavailableException The service is currently unavailable.</li>
     *         <li>ServiceFailureException The service encountered an unexpected error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ChimeSdkIdentityException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample ChimeSdkIdentityAsyncClient.UpdateAppInstanceBot
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/chime-sdk-identity-2021-04-20/UpdateAppInstanceBot"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateAppInstanceBotResponse> updateAppInstanceBot(
            UpdateAppInstanceBotRequest updateAppInstanceBotRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateAppInstanceBotRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateAppInstanceBotRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Chime SDK Identity");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateAppInstanceBot");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateAppInstanceBotResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateAppInstanceBotResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateAppInstanceBotResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateAppInstanceBotRequest, UpdateAppInstanceBotResponse>()
                            .withOperationName("UpdateAppInstanceBot").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new UpdateAppInstanceBotRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(updateAppInstanceBotRequest));
            CompletableFuture<UpdateAppInstanceBotResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates the details of an <code>AppInstanceUser</code>. You can update names and metadata.
     * </p>
     *
     * @param updateAppInstanceUserRequest
     * @return A Java Future containing the result of the UpdateAppInstanceUser operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException The input parameters don't match the service's restrictions.</li>
     *         <li>ConflictException The request could not be processed because of conflict in the current state of the
     *         resource.</li>
     *         <li>ForbiddenException The client is permanently forbidden from making the request.</li>
     *         <li>ResourceLimitExceededException The request exceeds the resource limit.</li>
     *         <li>ThrottledClientException The client exceeded its request rate limit.</li>
     *         <li>UnauthorizedClientException The client is not currently authorized to make the request.</li>
     *         <li>ServiceUnavailableException The service is currently unavailable.</li>
     *         <li>ServiceFailureException The service encountered an unexpected error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ChimeSdkIdentityException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample ChimeSdkIdentityAsyncClient.UpdateAppInstanceUser
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/chime-sdk-identity-2021-04-20/UpdateAppInstanceUser"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateAppInstanceUserResponse> updateAppInstanceUser(
            UpdateAppInstanceUserRequest updateAppInstanceUserRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateAppInstanceUserRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateAppInstanceUserRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Chime SDK Identity");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateAppInstanceUser");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateAppInstanceUserResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateAppInstanceUserResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateAppInstanceUserResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateAppInstanceUserRequest, UpdateAppInstanceUserResponse>()
                            .withOperationName("UpdateAppInstanceUser").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new UpdateAppInstanceUserRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(updateAppInstanceUserRequest));
            CompletableFuture<UpdateAppInstanceUserResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates the details of an <code>AppInstanceUserEndpoint</code>. You can update the name and
     * <code>AllowMessage</code> values.
     * </p>
     *
     * @param updateAppInstanceUserEndpointRequest
     * @return A Java Future containing the result of the UpdateAppInstanceUserEndpoint operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException The input parameters don't match the service's restrictions.</li>
     *         <li>ConflictException The request could not be processed because of conflict in the current state of the
     *         resource.</li>
     *         <li>ForbiddenException The client is permanently forbidden from making the request.</li>
     *         <li>ThrottledClientException The client exceeded its request rate limit.</li>
     *         <li>UnauthorizedClientException The client is not currently authorized to make the request.</li>
     *         <li>ServiceUnavailableException The service is currently unavailable.</li>
     *         <li>ServiceFailureException The service encountered an unexpected error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ChimeSdkIdentityException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample ChimeSdkIdentityAsyncClient.UpdateAppInstanceUserEndpoint
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/chime-sdk-identity-2021-04-20/UpdateAppInstanceUserEndpoint"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateAppInstanceUserEndpointResponse> updateAppInstanceUserEndpoint(
            UpdateAppInstanceUserEndpointRequest updateAppInstanceUserEndpointRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateAppInstanceUserEndpointRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                updateAppInstanceUserEndpointRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Chime SDK Identity");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateAppInstanceUserEndpoint");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateAppInstanceUserEndpointResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateAppInstanceUserEndpointResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateAppInstanceUserEndpointResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateAppInstanceUserEndpointRequest, UpdateAppInstanceUserEndpointResponse>()
                            .withOperationName("UpdateAppInstanceUserEndpoint").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new UpdateAppInstanceUserEndpointRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(updateAppInstanceUserEndpointRequest));
            CompletableFuture<UpdateAppInstanceUserEndpointResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    @Override
    public final ChimeSdkIdentityServiceClientConfiguration serviceClientConfiguration() {
        return new ChimeSdkIdentityServiceClientConfigurationBuilder(this.clientConfiguration.toBuilder()).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(ChimeSdkIdentityException::builder)
                .protocol(AwsJsonProtocol.REST_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("UnauthorizedClientException")
                                .exceptionBuilderSupplier(UnauthorizedClientException::builder).httpStatusCode(401).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("NotFoundException")
                                .exceptionBuilderSupplier(NotFoundException::builder).httpStatusCode(404).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ServiceFailureException")
                                .exceptionBuilderSupplier(ServiceFailureException::builder).httpStatusCode(500).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ConflictException")
                                .exceptionBuilderSupplier(ConflictException::builder).httpStatusCode(409).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ThrottledClientException")
                                .exceptionBuilderSupplier(ThrottledClientException::builder).httpStatusCode(429).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ServiceUnavailableException")
                                .exceptionBuilderSupplier(ServiceUnavailableException::builder).httpStatusCode(503).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ForbiddenException")
                                .exceptionBuilderSupplier(ForbiddenException::builder).httpStatusCode(403).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceLimitExceededException")
                                .exceptionBuilderSupplier(ResourceLimitExceededException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("BadRequestException")
                                .exceptionBuilderSupplier(BadRequestException::builder).httpStatusCode(400).build());
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private SdkClientConfiguration updateSdkClientConfiguration(SdkRequest request, SdkClientConfiguration clientConfiguration) {
        List<SdkPlugin> plugins = request.overrideConfiguration().map(c -> c.plugins()).orElse(Collections.emptyList());
        SdkClientConfiguration.Builder configuration = clientConfiguration.toBuilder();
        if (plugins.isEmpty()) {
            return configuration.build();
        }
        ChimeSdkIdentityServiceClientConfigurationBuilder serviceConfigBuilder = new ChimeSdkIdentityServiceClientConfigurationBuilder(
                configuration);
        for (SdkPlugin plugin : plugins) {
            plugin.configureClient(serviceConfigBuilder);
        }
        return configuration.build();
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
