/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.budgets.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * A notification that's associated with a budget. A budget can have up to ten notifications.
 * </p>
 * <p>
 * Each notification must have at least one subscriber. A notification can have one SNS subscriber and up to 10 email
 * subscribers, for a total of 11 subscribers.
 * </p>
 * <p>
 * For example, if you have a budget for 200 dollars and you want to be notified when you go over 160 dollars, create a
 * notification with the following parameters:
 * </p>
 * <ul>
 * <li>
 * <p>
 * A notificationType of <code>ACTUAL</code>
 * </p>
 * </li>
 * <li>
 * <p>
 * A <code>thresholdType</code> of <code>PERCENTAGE</code>
 * </p>
 * </li>
 * <li>
 * <p>
 * A <code>comparisonOperator</code> of <code>GREATER_THAN</code>
 * </p>
 * </li>
 * <li>
 * <p>
 * A notification <code>threshold</code> of <code>80</code>
 * </p>
 * </li>
 * </ul>
 */
@Generated("software.amazon.awssdk:codegen")
public final class Notification implements SdkPojo, Serializable, ToCopyableBuilder<Notification.Builder, Notification> {
    private static final SdkField<String> NOTIFICATION_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("NotificationType").getter(getter(Notification::notificationTypeAsString))
            .setter(setter(Builder::notificationType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("NotificationType").build()).build();

    private static final SdkField<String> COMPARISON_OPERATOR_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ComparisonOperator").getter(getter(Notification::comparisonOperatorAsString))
            .setter(setter(Builder::comparisonOperator))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ComparisonOperator").build())
            .build();

    private static final SdkField<Double> THRESHOLD_FIELD = SdkField.<Double> builder(MarshallingType.DOUBLE)
            .memberName("Threshold").getter(getter(Notification::threshold)).setter(setter(Builder::threshold))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Threshold").build()).build();

    private static final SdkField<String> THRESHOLD_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ThresholdType").getter(getter(Notification::thresholdTypeAsString))
            .setter(setter(Builder::thresholdType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ThresholdType").build()).build();

    private static final SdkField<String> NOTIFICATION_STATE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("NotificationState").getter(getter(Notification::notificationStateAsString))
            .setter(setter(Builder::notificationState))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("NotificationState").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(NOTIFICATION_TYPE_FIELD,
            COMPARISON_OPERATOR_FIELD, THRESHOLD_FIELD, THRESHOLD_TYPE_FIELD, NOTIFICATION_STATE_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String notificationType;

    private final String comparisonOperator;

    private final Double threshold;

    private final String thresholdType;

    private final String notificationState;

    private Notification(BuilderImpl builder) {
        this.notificationType = builder.notificationType;
        this.comparisonOperator = builder.comparisonOperator;
        this.threshold = builder.threshold;
        this.thresholdType = builder.thresholdType;
        this.notificationState = builder.notificationState;
    }

    /**
     * <p>
     * Specifies whether the notification is for how much you have spent (<code>ACTUAL</code>) or for how much that
     * you're forecasted to spend (<code>FORECASTED</code>).
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #notificationType}
     * will return {@link NotificationType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #notificationTypeAsString}.
     * </p>
     * 
     * @return Specifies whether the notification is for how much you have spent (<code>ACTUAL</code>) or for how much
     *         that you're forecasted to spend (<code>FORECASTED</code>).
     * @see NotificationType
     */
    public final NotificationType notificationType() {
        return NotificationType.fromValue(notificationType);
    }

    /**
     * <p>
     * Specifies whether the notification is for how much you have spent (<code>ACTUAL</code>) or for how much that
     * you're forecasted to spend (<code>FORECASTED</code>).
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #notificationType}
     * will return {@link NotificationType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #notificationTypeAsString}.
     * </p>
     * 
     * @return Specifies whether the notification is for how much you have spent (<code>ACTUAL</code>) or for how much
     *         that you're forecasted to spend (<code>FORECASTED</code>).
     * @see NotificationType
     */
    public final String notificationTypeAsString() {
        return notificationType;
    }

    /**
     * <p>
     * The comparison that's used for this notification.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #comparisonOperator} will return {@link ComparisonOperator#UNKNOWN_TO_SDK_VERSION}. The raw value returned
     * by the service is available from {@link #comparisonOperatorAsString}.
     * </p>
     * 
     * @return The comparison that's used for this notification.
     * @see ComparisonOperator
     */
    public final ComparisonOperator comparisonOperator() {
        return ComparisonOperator.fromValue(comparisonOperator);
    }

    /**
     * <p>
     * The comparison that's used for this notification.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #comparisonOperator} will return {@link ComparisonOperator#UNKNOWN_TO_SDK_VERSION}. The raw value returned
     * by the service is available from {@link #comparisonOperatorAsString}.
     * </p>
     * 
     * @return The comparison that's used for this notification.
     * @see ComparisonOperator
     */
    public final String comparisonOperatorAsString() {
        return comparisonOperator;
    }

    /**
     * <p>
     * The threshold that's associated with a notification. Thresholds are always a percentage, and many customers find
     * value being alerted between 50% - 200% of the budgeted amount. The maximum limit for your threshold is 1,000,000%
     * above the budgeted amount.
     * </p>
     * 
     * @return The threshold that's associated with a notification. Thresholds are always a percentage, and many
     *         customers find value being alerted between 50% - 200% of the budgeted amount. The maximum limit for your
     *         threshold is 1,000,000% above the budgeted amount.
     */
    public final Double threshold() {
        return threshold;
    }

    /**
     * <p>
     * The type of threshold for a notification. For <code>ABSOLUTE_VALUE</code> thresholds, Amazon Web Services
     * notifies you when you go over or are forecasted to go over your total cost threshold. For <code>PERCENTAGE</code>
     * thresholds, Amazon Web Services notifies you when you go over or are forecasted to go over a certain percentage
     * of your forecasted spend. For example, if you have a budget for 200 dollars and you have a
     * <code>PERCENTAGE</code> threshold of 80%, Amazon Web Services notifies you when you go over 160 dollars.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #thresholdType}
     * will return {@link ThresholdType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #thresholdTypeAsString}.
     * </p>
     * 
     * @return The type of threshold for a notification. For <code>ABSOLUTE_VALUE</code> thresholds, Amazon Web Services
     *         notifies you when you go over or are forecasted to go over your total cost threshold. For
     *         <code>PERCENTAGE</code> thresholds, Amazon Web Services notifies you when you go over or are forecasted
     *         to go over a certain percentage of your forecasted spend. For example, if you have a budget for 200
     *         dollars and you have a <code>PERCENTAGE</code> threshold of 80%, Amazon Web Services notifies you when
     *         you go over 160 dollars.
     * @see ThresholdType
     */
    public final ThresholdType thresholdType() {
        return ThresholdType.fromValue(thresholdType);
    }

    /**
     * <p>
     * The type of threshold for a notification. For <code>ABSOLUTE_VALUE</code> thresholds, Amazon Web Services
     * notifies you when you go over or are forecasted to go over your total cost threshold. For <code>PERCENTAGE</code>
     * thresholds, Amazon Web Services notifies you when you go over or are forecasted to go over a certain percentage
     * of your forecasted spend. For example, if you have a budget for 200 dollars and you have a
     * <code>PERCENTAGE</code> threshold of 80%, Amazon Web Services notifies you when you go over 160 dollars.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #thresholdType}
     * will return {@link ThresholdType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #thresholdTypeAsString}.
     * </p>
     * 
     * @return The type of threshold for a notification. For <code>ABSOLUTE_VALUE</code> thresholds, Amazon Web Services
     *         notifies you when you go over or are forecasted to go over your total cost threshold. For
     *         <code>PERCENTAGE</code> thresholds, Amazon Web Services notifies you when you go over or are forecasted
     *         to go over a certain percentage of your forecasted spend. For example, if you have a budget for 200
     *         dollars and you have a <code>PERCENTAGE</code> threshold of 80%, Amazon Web Services notifies you when
     *         you go over 160 dollars.
     * @see ThresholdType
     */
    public final String thresholdTypeAsString() {
        return thresholdType;
    }

    /**
     * <p>
     * Specifies whether this notification is in alarm. If a budget notification is in the <code>ALARM</code> state, you
     * passed the set threshold for the budget.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #notificationState}
     * will return {@link NotificationState#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #notificationStateAsString}.
     * </p>
     * 
     * @return Specifies whether this notification is in alarm. If a budget notification is in the <code>ALARM</code>
     *         state, you passed the set threshold for the budget.
     * @see NotificationState
     */
    public final NotificationState notificationState() {
        return NotificationState.fromValue(notificationState);
    }

    /**
     * <p>
     * Specifies whether this notification is in alarm. If a budget notification is in the <code>ALARM</code> state, you
     * passed the set threshold for the budget.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #notificationState}
     * will return {@link NotificationState#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #notificationStateAsString}.
     * </p>
     * 
     * @return Specifies whether this notification is in alarm. If a budget notification is in the <code>ALARM</code>
     *         state, you passed the set threshold for the budget.
     * @see NotificationState
     */
    public final String notificationStateAsString() {
        return notificationState;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(notificationTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(comparisonOperatorAsString());
        hashCode = 31 * hashCode + Objects.hashCode(threshold());
        hashCode = 31 * hashCode + Objects.hashCode(thresholdTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(notificationStateAsString());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof Notification)) {
            return false;
        }
        Notification other = (Notification) obj;
        return Objects.equals(notificationTypeAsString(), other.notificationTypeAsString())
                && Objects.equals(comparisonOperatorAsString(), other.comparisonOperatorAsString())
                && Objects.equals(threshold(), other.threshold())
                && Objects.equals(thresholdTypeAsString(), other.thresholdTypeAsString())
                && Objects.equals(notificationStateAsString(), other.notificationStateAsString());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("Notification").add("NotificationType", notificationTypeAsString())
                .add("ComparisonOperator", comparisonOperatorAsString()).add("Threshold", threshold())
                .add("ThresholdType", thresholdTypeAsString()).add("NotificationState", notificationStateAsString()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "NotificationType":
            return Optional.ofNullable(clazz.cast(notificationTypeAsString()));
        case "ComparisonOperator":
            return Optional.ofNullable(clazz.cast(comparisonOperatorAsString()));
        case "Threshold":
            return Optional.ofNullable(clazz.cast(threshold()));
        case "ThresholdType":
            return Optional.ofNullable(clazz.cast(thresholdTypeAsString()));
        case "NotificationState":
            return Optional.ofNullable(clazz.cast(notificationStateAsString()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("NotificationType", NOTIFICATION_TYPE_FIELD);
        map.put("ComparisonOperator", COMPARISON_OPERATOR_FIELD);
        map.put("Threshold", THRESHOLD_FIELD);
        map.put("ThresholdType", THRESHOLD_TYPE_FIELD);
        map.put("NotificationState", NOTIFICATION_STATE_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<Notification, T> g) {
        return obj -> g.apply((Notification) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, Notification> {
        /**
         * <p>
         * Specifies whether the notification is for how much you have spent (<code>ACTUAL</code>) or for how much that
         * you're forecasted to spend (<code>FORECASTED</code>).
         * </p>
         * 
         * @param notificationType
         *        Specifies whether the notification is for how much you have spent (<code>ACTUAL</code>) or for how
         *        much that you're forecasted to spend (<code>FORECASTED</code>).
         * @see NotificationType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see NotificationType
         */
        Builder notificationType(String notificationType);

        /**
         * <p>
         * Specifies whether the notification is for how much you have spent (<code>ACTUAL</code>) or for how much that
         * you're forecasted to spend (<code>FORECASTED</code>).
         * </p>
         * 
         * @param notificationType
         *        Specifies whether the notification is for how much you have spent (<code>ACTUAL</code>) or for how
         *        much that you're forecasted to spend (<code>FORECASTED</code>).
         * @see NotificationType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see NotificationType
         */
        Builder notificationType(NotificationType notificationType);

        /**
         * <p>
         * The comparison that's used for this notification.
         * </p>
         * 
         * @param comparisonOperator
         *        The comparison that's used for this notification.
         * @see ComparisonOperator
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ComparisonOperator
         */
        Builder comparisonOperator(String comparisonOperator);

        /**
         * <p>
         * The comparison that's used for this notification.
         * </p>
         * 
         * @param comparisonOperator
         *        The comparison that's used for this notification.
         * @see ComparisonOperator
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ComparisonOperator
         */
        Builder comparisonOperator(ComparisonOperator comparisonOperator);

        /**
         * <p>
         * The threshold that's associated with a notification. Thresholds are always a percentage, and many customers
         * find value being alerted between 50% - 200% of the budgeted amount. The maximum limit for your threshold is
         * 1,000,000% above the budgeted amount.
         * </p>
         * 
         * @param threshold
         *        The threshold that's associated with a notification. Thresholds are always a percentage, and many
         *        customers find value being alerted between 50% - 200% of the budgeted amount. The maximum limit for
         *        your threshold is 1,000,000% above the budgeted amount.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder threshold(Double threshold);

        /**
         * <p>
         * The type of threshold for a notification. For <code>ABSOLUTE_VALUE</code> thresholds, Amazon Web Services
         * notifies you when you go over or are forecasted to go over your total cost threshold. For
         * <code>PERCENTAGE</code> thresholds, Amazon Web Services notifies you when you go over or are forecasted to go
         * over a certain percentage of your forecasted spend. For example, if you have a budget for 200 dollars and you
         * have a <code>PERCENTAGE</code> threshold of 80%, Amazon Web Services notifies you when you go over 160
         * dollars.
         * </p>
         * 
         * @param thresholdType
         *        The type of threshold for a notification. For <code>ABSOLUTE_VALUE</code> thresholds, Amazon Web
         *        Services notifies you when you go over or are forecasted to go over your total cost threshold. For
         *        <code>PERCENTAGE</code> thresholds, Amazon Web Services notifies you when you go over or are
         *        forecasted to go over a certain percentage of your forecasted spend. For example, if you have a budget
         *        for 200 dollars and you have a <code>PERCENTAGE</code> threshold of 80%, Amazon Web Services notifies
         *        you when you go over 160 dollars.
         * @see ThresholdType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ThresholdType
         */
        Builder thresholdType(String thresholdType);

        /**
         * <p>
         * The type of threshold for a notification. For <code>ABSOLUTE_VALUE</code> thresholds, Amazon Web Services
         * notifies you when you go over or are forecasted to go over your total cost threshold. For
         * <code>PERCENTAGE</code> thresholds, Amazon Web Services notifies you when you go over or are forecasted to go
         * over a certain percentage of your forecasted spend. For example, if you have a budget for 200 dollars and you
         * have a <code>PERCENTAGE</code> threshold of 80%, Amazon Web Services notifies you when you go over 160
         * dollars.
         * </p>
         * 
         * @param thresholdType
         *        The type of threshold for a notification. For <code>ABSOLUTE_VALUE</code> thresholds, Amazon Web
         *        Services notifies you when you go over or are forecasted to go over your total cost threshold. For
         *        <code>PERCENTAGE</code> thresholds, Amazon Web Services notifies you when you go over or are
         *        forecasted to go over a certain percentage of your forecasted spend. For example, if you have a budget
         *        for 200 dollars and you have a <code>PERCENTAGE</code> threshold of 80%, Amazon Web Services notifies
         *        you when you go over 160 dollars.
         * @see ThresholdType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ThresholdType
         */
        Builder thresholdType(ThresholdType thresholdType);

        /**
         * <p>
         * Specifies whether this notification is in alarm. If a budget notification is in the <code>ALARM</code> state,
         * you passed the set threshold for the budget.
         * </p>
         * 
         * @param notificationState
         *        Specifies whether this notification is in alarm. If a budget notification is in the <code>ALARM</code>
         *        state, you passed the set threshold for the budget.
         * @see NotificationState
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see NotificationState
         */
        Builder notificationState(String notificationState);

        /**
         * <p>
         * Specifies whether this notification is in alarm. If a budget notification is in the <code>ALARM</code> state,
         * you passed the set threshold for the budget.
         * </p>
         * 
         * @param notificationState
         *        Specifies whether this notification is in alarm. If a budget notification is in the <code>ALARM</code>
         *        state, you passed the set threshold for the budget.
         * @see NotificationState
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see NotificationState
         */
        Builder notificationState(NotificationState notificationState);
    }

    static final class BuilderImpl implements Builder {
        private String notificationType;

        private String comparisonOperator;

        private Double threshold;

        private String thresholdType;

        private String notificationState;

        private BuilderImpl() {
        }

        private BuilderImpl(Notification model) {
            notificationType(model.notificationType);
            comparisonOperator(model.comparisonOperator);
            threshold(model.threshold);
            thresholdType(model.thresholdType);
            notificationState(model.notificationState);
        }

        public final String getNotificationType() {
            return notificationType;
        }

        public final void setNotificationType(String notificationType) {
            this.notificationType = notificationType;
        }

        @Override
        public final Builder notificationType(String notificationType) {
            this.notificationType = notificationType;
            return this;
        }

        @Override
        public final Builder notificationType(NotificationType notificationType) {
            this.notificationType(notificationType == null ? null : notificationType.toString());
            return this;
        }

        public final String getComparisonOperator() {
            return comparisonOperator;
        }

        public final void setComparisonOperator(String comparisonOperator) {
            this.comparisonOperator = comparisonOperator;
        }

        @Override
        public final Builder comparisonOperator(String comparisonOperator) {
            this.comparisonOperator = comparisonOperator;
            return this;
        }

        @Override
        public final Builder comparisonOperator(ComparisonOperator comparisonOperator) {
            this.comparisonOperator(comparisonOperator == null ? null : comparisonOperator.toString());
            return this;
        }

        public final Double getThreshold() {
            return threshold;
        }

        public final void setThreshold(Double threshold) {
            this.threshold = threshold;
        }

        @Override
        public final Builder threshold(Double threshold) {
            this.threshold = threshold;
            return this;
        }

        public final String getThresholdType() {
            return thresholdType;
        }

        public final void setThresholdType(String thresholdType) {
            this.thresholdType = thresholdType;
        }

        @Override
        public final Builder thresholdType(String thresholdType) {
            this.thresholdType = thresholdType;
            return this;
        }

        @Override
        public final Builder thresholdType(ThresholdType thresholdType) {
            this.thresholdType(thresholdType == null ? null : thresholdType.toString());
            return this;
        }

        public final String getNotificationState() {
            return notificationState;
        }

        public final void setNotificationState(String notificationState) {
            this.notificationState = notificationState;
        }

        @Override
        public final Builder notificationState(String notificationState) {
            this.notificationState = notificationState;
            return this;
        }

        @Override
        public final Builder notificationState(NotificationState notificationState) {
            this.notificationState(notificationState == null ? null : notificationState.toString());
            return this;
        }

        @Override
        public Notification build() {
            return new Notification(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
