/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.braket;

import java.util.Collections;
import java.util.List;
import java.util.function.Consumer;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.awscore.client.handler.AwsSyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.ApiName;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.client.handler.SyncClientHandler;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.core.util.VersionInfo;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.braket.model.AccessDeniedException;
import software.amazon.awssdk.services.braket.model.BraketException;
import software.amazon.awssdk.services.braket.model.BraketRequest;
import software.amazon.awssdk.services.braket.model.CancelJobRequest;
import software.amazon.awssdk.services.braket.model.CancelJobResponse;
import software.amazon.awssdk.services.braket.model.CancelQuantumTaskRequest;
import software.amazon.awssdk.services.braket.model.CancelQuantumTaskResponse;
import software.amazon.awssdk.services.braket.model.ConflictException;
import software.amazon.awssdk.services.braket.model.CreateJobRequest;
import software.amazon.awssdk.services.braket.model.CreateJobResponse;
import software.amazon.awssdk.services.braket.model.CreateQuantumTaskRequest;
import software.amazon.awssdk.services.braket.model.CreateQuantumTaskResponse;
import software.amazon.awssdk.services.braket.model.DeviceOfflineException;
import software.amazon.awssdk.services.braket.model.DeviceRetiredException;
import software.amazon.awssdk.services.braket.model.GetDeviceRequest;
import software.amazon.awssdk.services.braket.model.GetDeviceResponse;
import software.amazon.awssdk.services.braket.model.GetJobRequest;
import software.amazon.awssdk.services.braket.model.GetJobResponse;
import software.amazon.awssdk.services.braket.model.GetQuantumTaskRequest;
import software.amazon.awssdk.services.braket.model.GetQuantumTaskResponse;
import software.amazon.awssdk.services.braket.model.InternalServiceException;
import software.amazon.awssdk.services.braket.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.braket.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.braket.model.ResourceNotFoundException;
import software.amazon.awssdk.services.braket.model.SearchDevicesRequest;
import software.amazon.awssdk.services.braket.model.SearchDevicesResponse;
import software.amazon.awssdk.services.braket.model.SearchJobsRequest;
import software.amazon.awssdk.services.braket.model.SearchJobsResponse;
import software.amazon.awssdk.services.braket.model.SearchQuantumTasksRequest;
import software.amazon.awssdk.services.braket.model.SearchQuantumTasksResponse;
import software.amazon.awssdk.services.braket.model.ServiceQuotaExceededException;
import software.amazon.awssdk.services.braket.model.TagResourceRequest;
import software.amazon.awssdk.services.braket.model.TagResourceResponse;
import software.amazon.awssdk.services.braket.model.ThrottlingException;
import software.amazon.awssdk.services.braket.model.UntagResourceRequest;
import software.amazon.awssdk.services.braket.model.UntagResourceResponse;
import software.amazon.awssdk.services.braket.model.ValidationException;
import software.amazon.awssdk.services.braket.paginators.SearchDevicesIterable;
import software.amazon.awssdk.services.braket.paginators.SearchJobsIterable;
import software.amazon.awssdk.services.braket.paginators.SearchQuantumTasksIterable;
import software.amazon.awssdk.services.braket.transform.CancelJobRequestMarshaller;
import software.amazon.awssdk.services.braket.transform.CancelQuantumTaskRequestMarshaller;
import software.amazon.awssdk.services.braket.transform.CreateJobRequestMarshaller;
import software.amazon.awssdk.services.braket.transform.CreateQuantumTaskRequestMarshaller;
import software.amazon.awssdk.services.braket.transform.GetDeviceRequestMarshaller;
import software.amazon.awssdk.services.braket.transform.GetJobRequestMarshaller;
import software.amazon.awssdk.services.braket.transform.GetQuantumTaskRequestMarshaller;
import software.amazon.awssdk.services.braket.transform.ListTagsForResourceRequestMarshaller;
import software.amazon.awssdk.services.braket.transform.SearchDevicesRequestMarshaller;
import software.amazon.awssdk.services.braket.transform.SearchJobsRequestMarshaller;
import software.amazon.awssdk.services.braket.transform.SearchQuantumTasksRequestMarshaller;
import software.amazon.awssdk.services.braket.transform.TagResourceRequestMarshaller;
import software.amazon.awssdk.services.braket.transform.UntagResourceRequestMarshaller;
import software.amazon.awssdk.utils.Logger;

/**
 * Internal implementation of {@link BraketClient}.
 *
 * @see BraketClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultBraketClient implements BraketClient {
    private static final Logger log = Logger.loggerFor(DefaultBraketClient.class);

    private final SyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultBraketClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsSyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    /**
     * <p>
     * Cancels an Amazon Braket job.
     * </p>
     *
     * @param cancelJobRequest
     * @return Result of the CancelJob operation returned by the service.
     * @throws ResourceNotFoundException
     *         The specified resource was not found.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ConflictException
     *         An error occurred due to a conflict.
     * @throws ThrottlingException
     *         The throttling rate limit is met.
     * @throws InternalServiceException
     *         The request processing has failed because of an unknown error, exception, or failure.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an AWS service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BraketException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BraketClient.CancelJob
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/braket-2019-09-01/CancelJob" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CancelJobResponse cancelJob(CancelJobRequest cancelJobRequest) throws ResourceNotFoundException,
            AccessDeniedException, ConflictException, ThrottlingException, InternalServiceException, ValidationException,
            AwsServiceException, SdkClientException, BraketException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CancelJobResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                CancelJobResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, cancelJobRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Braket");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CancelJob");

            return clientHandler.execute(new ClientExecutionParams<CancelJobRequest, CancelJobResponse>()
                    .withOperationName("CancelJob").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(cancelJobRequest)
                    .withMetricCollector(apiCallMetricCollector).withMarshaller(new CancelJobRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Cancels the specified task.
     * </p>
     *
     * @param cancelQuantumTaskRequest
     * @return Result of the CancelQuantumTask operation returned by the service.
     * @throws ResourceNotFoundException
     *         The specified resource was not found.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ConflictException
     *         An error occurred due to a conflict.
     * @throws ThrottlingException
     *         The throttling rate limit is met.
     * @throws InternalServiceException
     *         The request processing has failed because of an unknown error, exception, or failure.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an AWS service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BraketException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BraketClient.CancelQuantumTask
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/braket-2019-09-01/CancelQuantumTask" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CancelQuantumTaskResponse cancelQuantumTask(CancelQuantumTaskRequest cancelQuantumTaskRequest)
            throws ResourceNotFoundException, AccessDeniedException, ConflictException, ThrottlingException,
            InternalServiceException, ValidationException, AwsServiceException, SdkClientException, BraketException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CancelQuantumTaskResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                CancelQuantumTaskResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, cancelQuantumTaskRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Braket");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CancelQuantumTask");

            return clientHandler.execute(new ClientExecutionParams<CancelQuantumTaskRequest, CancelQuantumTaskResponse>()
                    .withOperationName("CancelQuantumTask").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(cancelQuantumTaskRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CancelQuantumTaskRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Creates an Amazon Braket job.
     * </p>
     *
     * @param createJobRequest
     * @return Result of the CreateJob operation returned by the service.
     * @throws ConflictException
     *         An error occurred due to a conflict.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ThrottlingException
     *         The throttling rate limit is met.
     * @throws DeviceRetiredException
     *         The specified device has been retired.
     * @throws InternalServiceException
     *         The request processing has failed because of an unknown error, exception, or failure.
     * @throws ServiceQuotaExceededException
     *         The request failed because a service quota is exceeded.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an AWS service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BraketException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BraketClient.CreateJob
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/braket-2019-09-01/CreateJob" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CreateJobResponse createJob(CreateJobRequest createJobRequest) throws ConflictException, AccessDeniedException,
            ThrottlingException, DeviceRetiredException, InternalServiceException, ServiceQuotaExceededException,
            ValidationException, AwsServiceException, SdkClientException, BraketException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateJobResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                CreateJobResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createJobRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Braket");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateJob");

            return clientHandler.execute(new ClientExecutionParams<CreateJobRequest, CreateJobResponse>()
                    .withOperationName("CreateJob").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(createJobRequest)
                    .withMetricCollector(apiCallMetricCollector).withMarshaller(new CreateJobRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Creates a quantum task.
     * </p>
     *
     * @param createQuantumTaskRequest
     * @return Result of the CreateQuantumTask operation returned by the service.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ThrottlingException
     *         The throttling rate limit is met.
     * @throws DeviceOfflineException
     *         The specified device is currently offline.
     * @throws DeviceRetiredException
     *         The specified device has been retired.
     * @throws InternalServiceException
     *         The request processing has failed because of an unknown error, exception, or failure.
     * @throws ServiceQuotaExceededException
     *         The request failed because a service quota is exceeded.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an AWS service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BraketException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BraketClient.CreateQuantumTask
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/braket-2019-09-01/CreateQuantumTask" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CreateQuantumTaskResponse createQuantumTask(CreateQuantumTaskRequest createQuantumTaskRequest)
            throws AccessDeniedException, ThrottlingException, DeviceOfflineException, DeviceRetiredException,
            InternalServiceException, ServiceQuotaExceededException, ValidationException, AwsServiceException,
            SdkClientException, BraketException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateQuantumTaskResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                CreateQuantumTaskResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createQuantumTaskRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Braket");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateQuantumTask");

            return clientHandler.execute(new ClientExecutionParams<CreateQuantumTaskRequest, CreateQuantumTaskResponse>()
                    .withOperationName("CreateQuantumTask").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(createQuantumTaskRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CreateQuantumTaskRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Retrieves the devices available in Amazon Braket.
     * </p>
     * <note>
     * <p>
     * For backwards compatibility with older versions of BraketSchemas, OpenQASM information is omitted from GetDevice
     * API calls. To get this information the user-agent needs to present a recent version of the BraketSchemas (1.8.0
     * or later). The Braket SDK automatically reports this for you. If you do not see OpenQASM results in the GetDevice
     * response when using a Braket SDK, you may need to set AWS_EXECUTION_ENV environment variable to configure
     * user-agent. See the code examples provided below for how to do this for the AWS CLI, Boto3, and the Go, Java, and
     * JavaScript/TypeScript SDKs.
     * </p>
     * </note>
     *
     * @param getDeviceRequest
     * @return Result of the GetDevice operation returned by the service.
     * @throws ResourceNotFoundException
     *         The specified resource was not found.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ThrottlingException
     *         The throttling rate limit is met.
     * @throws InternalServiceException
     *         The request processing has failed because of an unknown error, exception, or failure.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an AWS service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BraketException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BraketClient.GetDevice
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/braket-2019-09-01/GetDevice" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public GetDeviceResponse getDevice(GetDeviceRequest getDeviceRequest) throws ResourceNotFoundException,
            AccessDeniedException, ThrottlingException, InternalServiceException, ValidationException, AwsServiceException,
            SdkClientException, BraketException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetDeviceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetDeviceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getDeviceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Braket");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetDevice");

            return clientHandler.execute(new ClientExecutionParams<GetDeviceRequest, GetDeviceResponse>()
                    .withOperationName("GetDevice").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(getDeviceRequest)
                    .withMetricCollector(apiCallMetricCollector).withMarshaller(new GetDeviceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Retrieves the specified Amazon Braket job.
     * </p>
     *
     * @param getJobRequest
     * @return Result of the GetJob operation returned by the service.
     * @throws ResourceNotFoundException
     *         The specified resource was not found.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ThrottlingException
     *         The throttling rate limit is met.
     * @throws InternalServiceException
     *         The request processing has failed because of an unknown error, exception, or failure.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an AWS service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BraketException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BraketClient.GetJob
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/braket-2019-09-01/GetJob" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public GetJobResponse getJob(GetJobRequest getJobRequest) throws ResourceNotFoundException, AccessDeniedException,
            ThrottlingException, InternalServiceException, ValidationException, AwsServiceException, SdkClientException,
            BraketException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetJobResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetJobResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getJobRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Braket");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetJob");

            return clientHandler.execute(new ClientExecutionParams<GetJobRequest, GetJobResponse>().withOperationName("GetJob")
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler).withInput(getJobRequest)
                    .withMetricCollector(apiCallMetricCollector).withMarshaller(new GetJobRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Retrieves the specified quantum task.
     * </p>
     *
     * @param getQuantumTaskRequest
     * @return Result of the GetQuantumTask operation returned by the service.
     * @throws ResourceNotFoundException
     *         The specified resource was not found.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ThrottlingException
     *         The throttling rate limit is met.
     * @throws InternalServiceException
     *         The request processing has failed because of an unknown error, exception, or failure.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an AWS service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BraketException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BraketClient.GetQuantumTask
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/braket-2019-09-01/GetQuantumTask" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public GetQuantumTaskResponse getQuantumTask(GetQuantumTaskRequest getQuantumTaskRequest) throws ResourceNotFoundException,
            AccessDeniedException, ThrottlingException, InternalServiceException, ValidationException, AwsServiceException,
            SdkClientException, BraketException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetQuantumTaskResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetQuantumTaskResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getQuantumTaskRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Braket");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetQuantumTask");

            return clientHandler.execute(new ClientExecutionParams<GetQuantumTaskRequest, GetQuantumTaskResponse>()
                    .withOperationName("GetQuantumTask").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(getQuantumTaskRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetQuantumTaskRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Shows the tags associated with this resource.
     * </p>
     *
     * @param listTagsForResourceRequest
     * @return Result of the ListTagsForResource operation returned by the service.
     * @throws ResourceNotFoundException
     *         The specified resource was not found.
     * @throws InternalServiceException
     *         The request processing has failed because of an unknown error, exception, or failure.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an AWS service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BraketException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BraketClient.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/braket-2019-09-01/ListTagsForResource" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public ListTagsForResourceResponse listTagsForResource(ListTagsForResourceRequest listTagsForResourceRequest)
            throws ResourceNotFoundException, InternalServiceException, ValidationException, AwsServiceException,
            SdkClientException, BraketException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListTagsForResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListTagsForResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTagsForResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Braket");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTagsForResource");

            return clientHandler.execute(new ClientExecutionParams<ListTagsForResourceRequest, ListTagsForResourceResponse>()
                    .withOperationName("ListTagsForResource").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(listTagsForResourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListTagsForResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Searches for devices using the specified filters.
     * </p>
     *
     * @param searchDevicesRequest
     * @return Result of the SearchDevices operation returned by the service.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ThrottlingException
     *         The throttling rate limit is met.
     * @throws InternalServiceException
     *         The request processing has failed because of an unknown error, exception, or failure.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an AWS service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BraketException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BraketClient.SearchDevices
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/braket-2019-09-01/SearchDevices" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public SearchDevicesResponse searchDevices(SearchDevicesRequest searchDevicesRequest) throws AccessDeniedException,
            ThrottlingException, InternalServiceException, ValidationException, AwsServiceException, SdkClientException,
            BraketException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<SearchDevicesResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                SearchDevicesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, searchDevicesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Braket");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "SearchDevices");

            return clientHandler.execute(new ClientExecutionParams<SearchDevicesRequest, SearchDevicesResponse>()
                    .withOperationName("SearchDevices").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(searchDevicesRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new SearchDevicesRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Searches for devices using the specified filters.
     * </p>
     * <br/>
     * <p>
     * This is a variant of {@link #searchDevices(software.amazon.awssdk.services.braket.model.SearchDevicesRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.braket.paginators.SearchDevicesIterable responses = client.searchDevicesPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.braket.paginators.SearchDevicesIterable responses = client.searchDevicesPaginator(request);
     *     for (software.amazon.awssdk.services.braket.model.SearchDevicesResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.braket.paginators.SearchDevicesIterable responses = client.searchDevicesPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #searchDevices(software.amazon.awssdk.services.braket.model.SearchDevicesRequest)} operation.</b>
     * </p>
     *
     * @param searchDevicesRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ThrottlingException
     *         The throttling rate limit is met.
     * @throws InternalServiceException
     *         The request processing has failed because of an unknown error, exception, or failure.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an AWS service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BraketException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BraketClient.SearchDevices
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/braket-2019-09-01/SearchDevices" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public SearchDevicesIterable searchDevicesPaginator(SearchDevicesRequest searchDevicesRequest) throws AccessDeniedException,
            ThrottlingException, InternalServiceException, ValidationException, AwsServiceException, SdkClientException,
            BraketException {
        return new SearchDevicesIterable(this, applyPaginatorUserAgent(searchDevicesRequest));
    }

    /**
     * <p>
     * Searches for Amazon Braket jobs that match the specified filter values.
     * </p>
     *
     * @param searchJobsRequest
     * @return Result of the SearchJobs operation returned by the service.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ThrottlingException
     *         The throttling rate limit is met.
     * @throws InternalServiceException
     *         The request processing has failed because of an unknown error, exception, or failure.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an AWS service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BraketException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BraketClient.SearchJobs
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/braket-2019-09-01/SearchJobs" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public SearchJobsResponse searchJobs(SearchJobsRequest searchJobsRequest) throws AccessDeniedException, ThrottlingException,
            InternalServiceException, ValidationException, AwsServiceException, SdkClientException, BraketException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<SearchJobsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                SearchJobsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, searchJobsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Braket");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "SearchJobs");

            return clientHandler
                    .execute(new ClientExecutionParams<SearchJobsRequest, SearchJobsResponse>().withOperationName("SearchJobs")
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(searchJobsRequest).withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new SearchJobsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Searches for Amazon Braket jobs that match the specified filter values.
     * </p>
     * <br/>
     * <p>
     * This is a variant of {@link #searchJobs(software.amazon.awssdk.services.braket.model.SearchJobsRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.braket.paginators.SearchJobsIterable responses = client.searchJobsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.braket.paginators.SearchJobsIterable responses = client.searchJobsPaginator(request);
     *     for (software.amazon.awssdk.services.braket.model.SearchJobsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.braket.paginators.SearchJobsIterable responses = client.searchJobsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #searchJobs(software.amazon.awssdk.services.braket.model.SearchJobsRequest)} operation.</b>
     * </p>
     *
     * @param searchJobsRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ThrottlingException
     *         The throttling rate limit is met.
     * @throws InternalServiceException
     *         The request processing has failed because of an unknown error, exception, or failure.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an AWS service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BraketException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BraketClient.SearchJobs
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/braket-2019-09-01/SearchJobs" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public SearchJobsIterable searchJobsPaginator(SearchJobsRequest searchJobsRequest) throws AccessDeniedException,
            ThrottlingException, InternalServiceException, ValidationException, AwsServiceException, SdkClientException,
            BraketException {
        return new SearchJobsIterable(this, applyPaginatorUserAgent(searchJobsRequest));
    }

    /**
     * <p>
     * Searches for tasks that match the specified filter values.
     * </p>
     *
     * @param searchQuantumTasksRequest
     * @return Result of the SearchQuantumTasks operation returned by the service.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ThrottlingException
     *         The throttling rate limit is met.
     * @throws InternalServiceException
     *         The request processing has failed because of an unknown error, exception, or failure.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an AWS service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BraketException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BraketClient.SearchQuantumTasks
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/braket-2019-09-01/SearchQuantumTasks" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public SearchQuantumTasksResponse searchQuantumTasks(SearchQuantumTasksRequest searchQuantumTasksRequest)
            throws AccessDeniedException, ThrottlingException, InternalServiceException, ValidationException,
            AwsServiceException, SdkClientException, BraketException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<SearchQuantumTasksResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, SearchQuantumTasksResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, searchQuantumTasksRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Braket");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "SearchQuantumTasks");

            return clientHandler.execute(new ClientExecutionParams<SearchQuantumTasksRequest, SearchQuantumTasksResponse>()
                    .withOperationName("SearchQuantumTasks").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(searchQuantumTasksRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new SearchQuantumTasksRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Searches for tasks that match the specified filter values.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #searchQuantumTasks(software.amazon.awssdk.services.braket.model.SearchQuantumTasksRequest)} operation.
     * The return type is a custom iterable that can be used to iterate through all the pages. SDK will internally
     * handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.braket.paginators.SearchQuantumTasksIterable responses = client.searchQuantumTasksPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.braket.paginators.SearchQuantumTasksIterable responses = client
     *             .searchQuantumTasksPaginator(request);
     *     for (software.amazon.awssdk.services.braket.model.SearchQuantumTasksResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.braket.paginators.SearchQuantumTasksIterable responses = client.searchQuantumTasksPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #searchQuantumTasks(software.amazon.awssdk.services.braket.model.SearchQuantumTasksRequest)}
     * operation.</b>
     * </p>
     *
     * @param searchQuantumTasksRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ThrottlingException
     *         The throttling rate limit is met.
     * @throws InternalServiceException
     *         The request processing has failed because of an unknown error, exception, or failure.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an AWS service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BraketException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BraketClient.SearchQuantumTasks
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/braket-2019-09-01/SearchQuantumTasks" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public SearchQuantumTasksIterable searchQuantumTasksPaginator(SearchQuantumTasksRequest searchQuantumTasksRequest)
            throws AccessDeniedException, ThrottlingException, InternalServiceException, ValidationException,
            AwsServiceException, SdkClientException, BraketException {
        return new SearchQuantumTasksIterable(this, applyPaginatorUserAgent(searchQuantumTasksRequest));
    }

    /**
     * <p>
     * Add a tag to the specified resource.
     * </p>
     *
     * @param tagResourceRequest
     * @return Result of the TagResource operation returned by the service.
     * @throws ResourceNotFoundException
     *         The specified resource was not found.
     * @throws InternalServiceException
     *         The request processing has failed because of an unknown error, exception, or failure.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an AWS service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BraketException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BraketClient.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/braket-2019-09-01/TagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public TagResourceResponse tagResource(TagResourceRequest tagResourceRequest) throws ResourceNotFoundException,
            InternalServiceException, ValidationException, AwsServiceException, SdkClientException, BraketException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<TagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                TagResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, tagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Braket");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TagResource");

            return clientHandler.execute(new ClientExecutionParams<TagResourceRequest, TagResourceResponse>()
                    .withOperationName("TagResource").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(tagResourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new TagResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Remove tags from a resource.
     * </p>
     *
     * @param untagResourceRequest
     * @return Result of the UntagResource operation returned by the service.
     * @throws ResourceNotFoundException
     *         The specified resource was not found.
     * @throws InternalServiceException
     *         The request processing has failed because of an unknown error, exception, or failure.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an AWS service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BraketException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BraketClient.UntagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/braket-2019-09-01/UntagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public UntagResourceResponse untagResource(UntagResourceRequest untagResourceRequest) throws ResourceNotFoundException,
            InternalServiceException, ValidationException, AwsServiceException, SdkClientException, BraketException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UntagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UntagResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, untagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Braket");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UntagResource");

            return clientHandler.execute(new ClientExecutionParams<UntagResourceRequest, UntagResourceResponse>()
                    .withOperationName("UntagResource").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(untagResourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UntagResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    private <T extends BraketRequest> T applyPaginatorUserAgent(T request) {
        Consumer<AwsRequestOverrideConfiguration.Builder> userAgentApplier = b -> b.addApiName(ApiName.builder()
                .version(VersionInfo.SDK_VERSION).name("PAGINATED").build());
        AwsRequestOverrideConfiguration overrideConfiguration = request.overrideConfiguration()
                .map(c -> c.toBuilder().applyMutation(userAgentApplier).build())
                .orElse((AwsRequestOverrideConfiguration.builder().applyMutation(userAgentApplier).build()));
        return (T) request.toBuilder().overrideConfiguration(overrideConfiguration).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(BraketException::builder)
                .protocol(AwsJsonProtocol.REST_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("AccessDeniedException")
                                .exceptionBuilderSupplier(AccessDeniedException::builder).httpStatusCode(403).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ConflictException")
                                .exceptionBuilderSupplier(ConflictException::builder).httpStatusCode(409).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFoundException")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).httpStatusCode(404).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("DeviceRetiredException")
                                .exceptionBuilderSupplier(DeviceRetiredException::builder).httpStatusCode(410).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ThrottlingException")
                                .exceptionBuilderSupplier(ThrottlingException::builder).httpStatusCode(429).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ValidationException")
                                .exceptionBuilderSupplier(ValidationException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException")
                                .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).httpStatusCode(402).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalServiceException")
                                .exceptionBuilderSupplier(InternalServiceException::builder).httpStatusCode(500).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("DeviceOfflineException")
                                .exceptionBuilderSupplier(DeviceOfflineException::builder).httpStatusCode(424).build());
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
