/*
 * Copyright 2013-2018 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.appstream.model;

import java.util.Arrays;
import java.util.Collection;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.core.protocol.ProtocolMarshaller;
import software.amazon.awssdk.core.protocol.StructuredPojo;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.services.appstream.transform.StorageConnectorMarshaller;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Describes a connector to enable persistent storage for users.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class StorageConnector implements StructuredPojo, ToCopyableBuilder<StorageConnector.Builder, StorageConnector> {
    private final String connectorType;

    private final String resourceIdentifier;

    private final List<String> domains;

    private StorageConnector(BuilderImpl builder) {
        this.connectorType = builder.connectorType;
        this.resourceIdentifier = builder.resourceIdentifier;
        this.domains = builder.domains;
    }

    /**
     * <p>
     * The type of storage connector.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #connectorType}
     * will return {@link StorageConnectorType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #connectorTypeAsString}.
     * </p>
     * 
     * @return The type of storage connector.
     * @see StorageConnectorType
     */
    public StorageConnectorType connectorType() {
        return StorageConnectorType.fromValue(connectorType);
    }

    /**
     * <p>
     * The type of storage connector.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #connectorType}
     * will return {@link StorageConnectorType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #connectorTypeAsString}.
     * </p>
     * 
     * @return The type of storage connector.
     * @see StorageConnectorType
     */
    public String connectorTypeAsString() {
        return connectorType;
    }

    /**
     * <p>
     * The ARN of the storage connector.
     * </p>
     * 
     * @return The ARN of the storage connector.
     */
    public String resourceIdentifier() {
        return resourceIdentifier;
    }

    /**
     * <p>
     * The names of the domains for the G Suite account.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return The names of the domains for the G Suite account.
     */
    public List<String> domains() {
        return domains;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(connectorTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(resourceIdentifier());
        hashCode = 31 * hashCode + Objects.hashCode(domains());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof StorageConnector)) {
            return false;
        }
        StorageConnector other = (StorageConnector) obj;
        return Objects.equals(connectorTypeAsString(), other.connectorTypeAsString())
                && Objects.equals(resourceIdentifier(), other.resourceIdentifier()) && Objects.equals(domains(), other.domains());
    }

    @Override
    public String toString() {
        return ToString.builder("StorageConnector").add("ConnectorType", connectorTypeAsString())
                .add("ResourceIdentifier", resourceIdentifier()).add("Domains", domains()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "ConnectorType":
            return Optional.ofNullable(clazz.cast(connectorTypeAsString()));
        case "ResourceIdentifier":
            return Optional.ofNullable(clazz.cast(resourceIdentifier()));
        case "Domains":
            return Optional.ofNullable(clazz.cast(domains()));
        default:
            return Optional.empty();
        }
    }

    @SdkInternalApi
    @Override
    public void marshall(ProtocolMarshaller protocolMarshaller) {
        StorageConnectorMarshaller.getInstance().marshall(this, protocolMarshaller);
    }

    public interface Builder extends CopyableBuilder<Builder, StorageConnector> {
        /**
         * <p>
         * The type of storage connector.
         * </p>
         * 
         * @param connectorType
         *        The type of storage connector.
         * @see StorageConnectorType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see StorageConnectorType
         */
        Builder connectorType(String connectorType);

        /**
         * <p>
         * The type of storage connector.
         * </p>
         * 
         * @param connectorType
         *        The type of storage connector.
         * @see StorageConnectorType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see StorageConnectorType
         */
        Builder connectorType(StorageConnectorType connectorType);

        /**
         * <p>
         * The ARN of the storage connector.
         * </p>
         * 
         * @param resourceIdentifier
         *        The ARN of the storage connector.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder resourceIdentifier(String resourceIdentifier);

        /**
         * <p>
         * The names of the domains for the G Suite account.
         * </p>
         * 
         * @param domains
         *        The names of the domains for the G Suite account.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder domains(Collection<String> domains);

        /**
         * <p>
         * The names of the domains for the G Suite account.
         * </p>
         * 
         * @param domains
         *        The names of the domains for the G Suite account.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder domains(String... domains);
    }

    static final class BuilderImpl implements Builder {
        private String connectorType;

        private String resourceIdentifier;

        private List<String> domains = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(StorageConnector model) {
            connectorType(model.connectorType);
            resourceIdentifier(model.resourceIdentifier);
            domains(model.domains);
        }

        public final String getConnectorType() {
            return connectorType;
        }

        @Override
        public final Builder connectorType(String connectorType) {
            this.connectorType = connectorType;
            return this;
        }

        @Override
        public final Builder connectorType(StorageConnectorType connectorType) {
            this.connectorType(connectorType.toString());
            return this;
        }

        public final void setConnectorType(String connectorType) {
            this.connectorType = connectorType;
        }

        public final String getResourceIdentifier() {
            return resourceIdentifier;
        }

        @Override
        public final Builder resourceIdentifier(String resourceIdentifier) {
            this.resourceIdentifier = resourceIdentifier;
            return this;
        }

        public final void setResourceIdentifier(String resourceIdentifier) {
            this.resourceIdentifier = resourceIdentifier;
        }

        public final Collection<String> getDomains() {
            return domains;
        }

        @Override
        public final Builder domains(Collection<String> domains) {
            this.domains = DomainListCopier.copy(domains);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder domains(String... domains) {
            domains(Arrays.asList(domains));
            return this;
        }

        public final void setDomains(Collection<String> domains) {
            this.domains = DomainListCopier.copy(domains);
        }

        @Override
        public StorageConnector build() {
            return new StorageConnector(this);
        }
    }
}
