/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.apprunner.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.MapTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructMap;
import software.amazon.awssdk.core.util.SdkAutoConstructMap;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Describes the basic configuration needed for building and running an App Runner service. This type doesn't support
 * the full set of possible configuration options. Fur full configuration capabilities, use a
 * <code>apprunner.yaml</code> file in the source code repository.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class CodeConfigurationValues implements SdkPojo, Serializable,
        ToCopyableBuilder<CodeConfigurationValues.Builder, CodeConfigurationValues> {
    private static final SdkField<String> RUNTIME_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Runtime")
            .getter(getter(CodeConfigurationValues::runtimeAsString)).setter(setter(Builder::runtime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Runtime").build()).build();

    private static final SdkField<String> BUILD_COMMAND_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("BuildCommand").getter(getter(CodeConfigurationValues::buildCommand))
            .setter(setter(Builder::buildCommand))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("BuildCommand").build()).build();

    private static final SdkField<String> START_COMMAND_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("StartCommand").getter(getter(CodeConfigurationValues::startCommand))
            .setter(setter(Builder::startCommand))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("StartCommand").build()).build();

    private static final SdkField<String> PORT_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Port")
            .getter(getter(CodeConfigurationValues::port)).setter(setter(Builder::port))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Port").build()).build();

    private static final SdkField<Map<String, String>> RUNTIME_ENVIRONMENT_VARIABLES_FIELD = SdkField
            .<Map<String, String>> builder(MarshallingType.MAP)
            .memberName("RuntimeEnvironmentVariables")
            .getter(getter(CodeConfigurationValues::runtimeEnvironmentVariables))
            .setter(setter(Builder::runtimeEnvironmentVariables))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("RuntimeEnvironmentVariables")
                    .build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final SdkField<Map<String, String>> RUNTIME_ENVIRONMENT_SECRETS_FIELD = SdkField
            .<Map<String, String>> builder(MarshallingType.MAP)
            .memberName("RuntimeEnvironmentSecrets")
            .getter(getter(CodeConfigurationValues::runtimeEnvironmentSecrets))
            .setter(setter(Builder::runtimeEnvironmentSecrets))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("RuntimeEnvironmentSecrets").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(RUNTIME_FIELD,
            BUILD_COMMAND_FIELD, START_COMMAND_FIELD, PORT_FIELD, RUNTIME_ENVIRONMENT_VARIABLES_FIELD,
            RUNTIME_ENVIRONMENT_SECRETS_FIELD));

    private static final long serialVersionUID = 1L;

    private final String runtime;

    private final String buildCommand;

    private final String startCommand;

    private final String port;

    private final Map<String, String> runtimeEnvironmentVariables;

    private final Map<String, String> runtimeEnvironmentSecrets;

    private CodeConfigurationValues(BuilderImpl builder) {
        this.runtime = builder.runtime;
        this.buildCommand = builder.buildCommand;
        this.startCommand = builder.startCommand;
        this.port = builder.port;
        this.runtimeEnvironmentVariables = builder.runtimeEnvironmentVariables;
        this.runtimeEnvironmentSecrets = builder.runtimeEnvironmentSecrets;
    }

    /**
     * <p>
     * A runtime environment type for building and running an App Runner service. It represents a programming language
     * runtime.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #runtime} will
     * return {@link Runtime#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #runtimeAsString}.
     * </p>
     * 
     * @return A runtime environment type for building and running an App Runner service. It represents a programming
     *         language runtime.
     * @see Runtime
     */
    public final Runtime runtime() {
        return Runtime.fromValue(runtime);
    }

    /**
     * <p>
     * A runtime environment type for building and running an App Runner service. It represents a programming language
     * runtime.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #runtime} will
     * return {@link Runtime#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #runtimeAsString}.
     * </p>
     * 
     * @return A runtime environment type for building and running an App Runner service. It represents a programming
     *         language runtime.
     * @see Runtime
     */
    public final String runtimeAsString() {
        return runtime;
    }

    /**
     * <p>
     * The command App Runner runs to build your application.
     * </p>
     * 
     * @return The command App Runner runs to build your application.
     */
    public final String buildCommand() {
        return buildCommand;
    }

    /**
     * <p>
     * The command App Runner runs to start your application.
     * </p>
     * 
     * @return The command App Runner runs to start your application.
     */
    public final String startCommand() {
        return startCommand;
    }

    /**
     * <p>
     * The port that your application listens to in the container.
     * </p>
     * <p>
     * Default: <code>8080</code>
     * </p>
     * 
     * @return The port that your application listens to in the container.</p>
     *         <p>
     *         Default: <code>8080</code>
     */
    public final String port() {
        return port;
    }

    /**
     * For responses, this returns true if the service returned a value for the RuntimeEnvironmentVariables property.
     * This DOES NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the
     * property). This is useful because the SDK will never return a null collection or map, but you may need to
     * differentiate between the service returning nothing (or null) and the service returning an empty collection or
     * map. For requests, this returns true if a value for the property was specified in the request builder, and false
     * if a value was not specified.
     */
    public final boolean hasRuntimeEnvironmentVariables() {
        return runtimeEnvironmentVariables != null && !(runtimeEnvironmentVariables instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * The environment variables that are available to your running App Runner service. An array of key-value pairs.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasRuntimeEnvironmentVariables} method.
     * </p>
     * 
     * @return The environment variables that are available to your running App Runner service. An array of key-value
     *         pairs.
     */
    public final Map<String, String> runtimeEnvironmentVariables() {
        return runtimeEnvironmentVariables;
    }

    /**
     * For responses, this returns true if the service returned a value for the RuntimeEnvironmentSecrets property. This
     * DOES NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the
     * property). This is useful because the SDK will never return a null collection or map, but you may need to
     * differentiate between the service returning nothing (or null) and the service returning an empty collection or
     * map. For requests, this returns true if a value for the property was specified in the request builder, and false
     * if a value was not specified.
     */
    public final boolean hasRuntimeEnvironmentSecrets() {
        return runtimeEnvironmentSecrets != null && !(runtimeEnvironmentSecrets instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * An array of key-value pairs representing the secrets and parameters that get referenced to your service as an
     * environment variable. The supported values are either the full Amazon Resource Name (ARN) of the Secrets Manager
     * secret or the full ARN of the parameter in the Amazon Web Services Systems Manager Parameter Store.
     * </p>
     * <note>
     * <ul>
     * <li>
     * <p>
     * If the Amazon Web Services Systems Manager Parameter Store parameter exists in the same Amazon Web Services
     * Region as the service that you're launching, you can use either the full ARN or name of the secret. If the
     * parameter exists in a different Region, then the full ARN must be specified.
     * </p>
     * </li>
     * <li>
     * <p>
     * Currently, cross account referencing of Amazon Web Services Systems Manager Parameter Store parameter is not
     * supported.
     * </p>
     * </li>
     * </ul>
     * </note>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasRuntimeEnvironmentSecrets} method.
     * </p>
     * 
     * @return An array of key-value pairs representing the secrets and parameters that get referenced to your service
     *         as an environment variable. The supported values are either the full Amazon Resource Name (ARN) of the
     *         Secrets Manager secret or the full ARN of the parameter in the Amazon Web Services Systems Manager
     *         Parameter Store.</p> <note>
     *         <ul>
     *         <li>
     *         <p>
     *         If the Amazon Web Services Systems Manager Parameter Store parameter exists in the same Amazon Web
     *         Services Region as the service that you're launching, you can use either the full ARN or name of the
     *         secret. If the parameter exists in a different Region, then the full ARN must be specified.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Currently, cross account referencing of Amazon Web Services Systems Manager Parameter Store parameter is
     *         not supported.
     *         </p>
     *         </li>
     *         </ul>
     */
    public final Map<String, String> runtimeEnvironmentSecrets() {
        return runtimeEnvironmentSecrets;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(runtimeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(buildCommand());
        hashCode = 31 * hashCode + Objects.hashCode(startCommand());
        hashCode = 31 * hashCode + Objects.hashCode(port());
        hashCode = 31 * hashCode + Objects.hashCode(hasRuntimeEnvironmentVariables() ? runtimeEnvironmentVariables() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasRuntimeEnvironmentSecrets() ? runtimeEnvironmentSecrets() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof CodeConfigurationValues)) {
            return false;
        }
        CodeConfigurationValues other = (CodeConfigurationValues) obj;
        return Objects.equals(runtimeAsString(), other.runtimeAsString()) && Objects.equals(buildCommand(), other.buildCommand())
                && Objects.equals(startCommand(), other.startCommand()) && Objects.equals(port(), other.port())
                && hasRuntimeEnvironmentVariables() == other.hasRuntimeEnvironmentVariables()
                && Objects.equals(runtimeEnvironmentVariables(), other.runtimeEnvironmentVariables())
                && hasRuntimeEnvironmentSecrets() == other.hasRuntimeEnvironmentSecrets()
                && Objects.equals(runtimeEnvironmentSecrets(), other.runtimeEnvironmentSecrets());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString
                .builder("CodeConfigurationValues")
                .add("Runtime", runtimeAsString())
                .add("BuildCommand", buildCommand() == null ? null : "*** Sensitive Data Redacted ***")
                .add("StartCommand", startCommand() == null ? null : "*** Sensitive Data Redacted ***")
                .add("Port", port())
                .add("RuntimeEnvironmentVariables",
                        runtimeEnvironmentVariables() == null ? null : "*** Sensitive Data Redacted ***")
                .add("RuntimeEnvironmentSecrets", runtimeEnvironmentSecrets() == null ? null : "*** Sensitive Data Redacted ***")
                .build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Runtime":
            return Optional.ofNullable(clazz.cast(runtimeAsString()));
        case "BuildCommand":
            return Optional.ofNullable(clazz.cast(buildCommand()));
        case "StartCommand":
            return Optional.ofNullable(clazz.cast(startCommand()));
        case "Port":
            return Optional.ofNullable(clazz.cast(port()));
        case "RuntimeEnvironmentVariables":
            return Optional.ofNullable(clazz.cast(runtimeEnvironmentVariables()));
        case "RuntimeEnvironmentSecrets":
            return Optional.ofNullable(clazz.cast(runtimeEnvironmentSecrets()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<CodeConfigurationValues, T> g) {
        return obj -> g.apply((CodeConfigurationValues) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, CodeConfigurationValues> {
        /**
         * <p>
         * A runtime environment type for building and running an App Runner service. It represents a programming
         * language runtime.
         * </p>
         * 
         * @param runtime
         *        A runtime environment type for building and running an App Runner service. It represents a programming
         *        language runtime.
         * @see Runtime
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see Runtime
         */
        Builder runtime(String runtime);

        /**
         * <p>
         * A runtime environment type for building and running an App Runner service. It represents a programming
         * language runtime.
         * </p>
         * 
         * @param runtime
         *        A runtime environment type for building and running an App Runner service. It represents a programming
         *        language runtime.
         * @see Runtime
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see Runtime
         */
        Builder runtime(Runtime runtime);

        /**
         * <p>
         * The command App Runner runs to build your application.
         * </p>
         * 
         * @param buildCommand
         *        The command App Runner runs to build your application.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder buildCommand(String buildCommand);

        /**
         * <p>
         * The command App Runner runs to start your application.
         * </p>
         * 
         * @param startCommand
         *        The command App Runner runs to start your application.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder startCommand(String startCommand);

        /**
         * <p>
         * The port that your application listens to in the container.
         * </p>
         * <p>
         * Default: <code>8080</code>
         * </p>
         * 
         * @param port
         *        The port that your application listens to in the container.</p>
         *        <p>
         *        Default: <code>8080</code>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder port(String port);

        /**
         * <p>
         * The environment variables that are available to your running App Runner service. An array of key-value pairs.
         * </p>
         * 
         * @param runtimeEnvironmentVariables
         *        The environment variables that are available to your running App Runner service. An array of key-value
         *        pairs.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder runtimeEnvironmentVariables(Map<String, String> runtimeEnvironmentVariables);

        /**
         * <p>
         * An array of key-value pairs representing the secrets and parameters that get referenced to your service as an
         * environment variable. The supported values are either the full Amazon Resource Name (ARN) of the Secrets
         * Manager secret or the full ARN of the parameter in the Amazon Web Services Systems Manager Parameter Store.
         * </p>
         * <note>
         * <ul>
         * <li>
         * <p>
         * If the Amazon Web Services Systems Manager Parameter Store parameter exists in the same Amazon Web Services
         * Region as the service that you're launching, you can use either the full ARN or name of the secret. If the
         * parameter exists in a different Region, then the full ARN must be specified.
         * </p>
         * </li>
         * <li>
         * <p>
         * Currently, cross account referencing of Amazon Web Services Systems Manager Parameter Store parameter is not
         * supported.
         * </p>
         * </li>
         * </ul>
         * </note>
         * 
         * @param runtimeEnvironmentSecrets
         *        An array of key-value pairs representing the secrets and parameters that get referenced to your
         *        service as an environment variable. The supported values are either the full Amazon Resource Name
         *        (ARN) of the Secrets Manager secret or the full ARN of the parameter in the Amazon Web Services
         *        Systems Manager Parameter Store.</p> <note>
         *        <ul>
         *        <li>
         *        <p>
         *        If the Amazon Web Services Systems Manager Parameter Store parameter exists in the same Amazon Web
         *        Services Region as the service that you're launching, you can use either the full ARN or name of the
         *        secret. If the parameter exists in a different Region, then the full ARN must be specified.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Currently, cross account referencing of Amazon Web Services Systems Manager Parameter Store parameter
         *        is not supported.
         *        </p>
         *        </li>
         *        </ul>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder runtimeEnvironmentSecrets(Map<String, String> runtimeEnvironmentSecrets);
    }

    static final class BuilderImpl implements Builder {
        private String runtime;

        private String buildCommand;

        private String startCommand;

        private String port;

        private Map<String, String> runtimeEnvironmentVariables = DefaultSdkAutoConstructMap.getInstance();

        private Map<String, String> runtimeEnvironmentSecrets = DefaultSdkAutoConstructMap.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(CodeConfigurationValues model) {
            runtime(model.runtime);
            buildCommand(model.buildCommand);
            startCommand(model.startCommand);
            port(model.port);
            runtimeEnvironmentVariables(model.runtimeEnvironmentVariables);
            runtimeEnvironmentSecrets(model.runtimeEnvironmentSecrets);
        }

        public final String getRuntime() {
            return runtime;
        }

        public final void setRuntime(String runtime) {
            this.runtime = runtime;
        }

        @Override
        public final Builder runtime(String runtime) {
            this.runtime = runtime;
            return this;
        }

        @Override
        public final Builder runtime(Runtime runtime) {
            this.runtime(runtime == null ? null : runtime.toString());
            return this;
        }

        public final String getBuildCommand() {
            return buildCommand;
        }

        public final void setBuildCommand(String buildCommand) {
            this.buildCommand = buildCommand;
        }

        @Override
        public final Builder buildCommand(String buildCommand) {
            this.buildCommand = buildCommand;
            return this;
        }

        public final String getStartCommand() {
            return startCommand;
        }

        public final void setStartCommand(String startCommand) {
            this.startCommand = startCommand;
        }

        @Override
        public final Builder startCommand(String startCommand) {
            this.startCommand = startCommand;
            return this;
        }

        public final String getPort() {
            return port;
        }

        public final void setPort(String port) {
            this.port = port;
        }

        @Override
        public final Builder port(String port) {
            this.port = port;
            return this;
        }

        public final Map<String, String> getRuntimeEnvironmentVariables() {
            if (runtimeEnvironmentVariables instanceof SdkAutoConstructMap) {
                return null;
            }
            return runtimeEnvironmentVariables;
        }

        public final void setRuntimeEnvironmentVariables(Map<String, String> runtimeEnvironmentVariables) {
            this.runtimeEnvironmentVariables = RuntimeEnvironmentVariablesCopier.copy(runtimeEnvironmentVariables);
        }

        @Override
        public final Builder runtimeEnvironmentVariables(Map<String, String> runtimeEnvironmentVariables) {
            this.runtimeEnvironmentVariables = RuntimeEnvironmentVariablesCopier.copy(runtimeEnvironmentVariables);
            return this;
        }

        public final Map<String, String> getRuntimeEnvironmentSecrets() {
            if (runtimeEnvironmentSecrets instanceof SdkAutoConstructMap) {
                return null;
            }
            return runtimeEnvironmentSecrets;
        }

        public final void setRuntimeEnvironmentSecrets(Map<String, String> runtimeEnvironmentSecrets) {
            this.runtimeEnvironmentSecrets = RuntimeEnvironmentSecretsCopier.copy(runtimeEnvironmentSecrets);
        }

        @Override
        public final Builder runtimeEnvironmentSecrets(Map<String, String> runtimeEnvironmentSecrets) {
            this.runtimeEnvironmentSecrets = RuntimeEnvironmentSecretsCopier.copy(runtimeEnvironmentSecrets);
            return this;
        }

        @Override
        public CodeConfigurationValues build() {
            return new CodeConfigurationValues(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
