/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.apigateway.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.MapTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructMap;
import software.amazon.awssdk.core.util.SdkAutoConstructMap;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Represents a method response of a given HTTP status code returned to the client. The method response is passed from
 * the back end through the associated integration response that can be transformed using a mapping template.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class MethodResponse implements SdkPojo, Serializable, ToCopyableBuilder<MethodResponse.Builder, MethodResponse> {
    private static final SdkField<String> STATUS_CODE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("statusCode").getter(getter(MethodResponse::statusCode)).setter(setter(Builder::statusCode))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("statusCode").build()).build();

    private static final SdkField<Map<String, Boolean>> RESPONSE_PARAMETERS_FIELD = SdkField
            .<Map<String, Boolean>> builder(MarshallingType.MAP)
            .memberName("responseParameters")
            .getter(getter(MethodResponse::responseParameters))
            .setter(setter(Builder::responseParameters))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("responseParameters").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final SdkField<Map<String, String>> RESPONSE_MODELS_FIELD = SdkField
            .<Map<String, String>> builder(MarshallingType.MAP)
            .memberName("responseModels")
            .getter(getter(MethodResponse::responseModels))
            .setter(setter(Builder::responseModels))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("responseModels").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(STATUS_CODE_FIELD,
            RESPONSE_PARAMETERS_FIELD, RESPONSE_MODELS_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = Collections
            .unmodifiableMap(new HashMap<String, SdkField<?>>() {
                {
                    put("statusCode", STATUS_CODE_FIELD);
                    put("responseParameters", RESPONSE_PARAMETERS_FIELD);
                    put("responseModels", RESPONSE_MODELS_FIELD);
                }
            });

    private static final long serialVersionUID = 1L;

    private final String statusCodeValue;

    private final Map<String, Boolean> responseParameters;

    private final Map<String, String> responseModels;

    private MethodResponse(BuilderImpl builder) {
        this.statusCodeValue = builder.statusCodeValue;
        this.responseParameters = builder.responseParameters;
        this.responseModels = builder.responseModels;
    }

    /**
     * <p>
     * The method response's status code.
     * </p>
     * 
     * @return The method response's status code.
     */
    public final String statusCode() {
        return statusCodeValue;
    }

    /**
     * For responses, this returns true if the service returned a value for the ResponseParameters property. This DOES
     * NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasResponseParameters() {
        return responseParameters != null && !(responseParameters instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * A key-value map specifying required or optional response parameters that API Gateway can send back to the caller.
     * A key defines a method response header and the value specifies whether the associated method response header is
     * required or not. The expression of the key must match the pattern <code>method.response.header.{name}</code>,
     * where <code>name</code> is a valid and unique header name. API Gateway passes certain integration response data
     * to the method response headers specified here according to the mapping you prescribe in the API's
     * IntegrationResponse. The integration response data that can be mapped include an integration response header
     * expressed in <code>integration.response.header.{name}</code>, a static value enclosed within a pair of single
     * quotes (e.g., <code>'application/json'</code>), or a JSON expression from the back-end response payload in the
     * form of <code>integration.response.body.{JSON-expression}</code>, where <code>JSON-expression</code> is a valid
     * JSON expression without the <code>$</code> prefix.)
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasResponseParameters} method.
     * </p>
     * 
     * @return A key-value map specifying required or optional response parameters that API Gateway can send back to the
     *         caller. A key defines a method response header and the value specifies whether the associated method
     *         response header is required or not. The expression of the key must match the pattern
     *         <code>method.response.header.{name}</code>, where <code>name</code> is a valid and unique header name.
     *         API Gateway passes certain integration response data to the method response headers specified here
     *         according to the mapping you prescribe in the API's IntegrationResponse. The integration response data
     *         that can be mapped include an integration response header expressed in
     *         <code>integration.response.header.{name}</code>, a static value enclosed within a pair of single quotes
     *         (e.g., <code>'application/json'</code>), or a JSON expression from the back-end response payload in the
     *         form of <code>integration.response.body.{JSON-expression}</code>, where <code>JSON-expression</code> is a
     *         valid JSON expression without the <code>$</code> prefix.)
     */
    public final Map<String, Boolean> responseParameters() {
        return responseParameters;
    }

    /**
     * For responses, this returns true if the service returned a value for the ResponseModels property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasResponseModels() {
        return responseModels != null && !(responseModels instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * Specifies the Model resources used for the response's content-type. Response models are represented as a
     * key/value map, with a content-type as the key and a Model name as the value.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasResponseModels} method.
     * </p>
     * 
     * @return Specifies the Model resources used for the response's content-type. Response models are represented as a
     *         key/value map, with a content-type as the key and a Model name as the value.
     */
    public final Map<String, String> responseModels() {
        return responseModels;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(statusCode());
        hashCode = 31 * hashCode + Objects.hashCode(hasResponseParameters() ? responseParameters() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasResponseModels() ? responseModels() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof MethodResponse)) {
            return false;
        }
        MethodResponse other = (MethodResponse) obj;
        return Objects.equals(statusCode(), other.statusCode()) && hasResponseParameters() == other.hasResponseParameters()
                && Objects.equals(responseParameters(), other.responseParameters())
                && hasResponseModels() == other.hasResponseModels() && Objects.equals(responseModels(), other.responseModels());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("MethodResponse").add("StatusCode", statusCode())
                .add("ResponseParameters", hasResponseParameters() ? responseParameters() : null)
                .add("ResponseModels", hasResponseModels() ? responseModels() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "statusCode":
            return Optional.ofNullable(clazz.cast(statusCode()));
        case "responseParameters":
            return Optional.ofNullable(clazz.cast(responseParameters()));
        case "responseModels":
            return Optional.ofNullable(clazz.cast(responseModels()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static <T> Function<Object, T> getter(Function<MethodResponse, T> g) {
        return obj -> g.apply((MethodResponse) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, MethodResponse> {
        /**
         * <p>
         * The method response's status code.
         * </p>
         * 
         * @param statusCodeValue
         *        The method response's status code.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder statusCode(String statusCodeValue);

        /**
         * <p>
         * A key-value map specifying required or optional response parameters that API Gateway can send back to the
         * caller. A key defines a method response header and the value specifies whether the associated method response
         * header is required or not. The expression of the key must match the pattern
         * <code>method.response.header.{name}</code>, where <code>name</code> is a valid and unique header name. API
         * Gateway passes certain integration response data to the method response headers specified here according to
         * the mapping you prescribe in the API's IntegrationResponse. The integration response data that can be mapped
         * include an integration response header expressed in <code>integration.response.header.{name}</code>, a static
         * value enclosed within a pair of single quotes (e.g., <code>'application/json'</code>), or a JSON expression
         * from the back-end response payload in the form of <code>integration.response.body.{JSON-expression}</code>,
         * where <code>JSON-expression</code> is a valid JSON expression without the <code>$</code> prefix.)
         * </p>
         * 
         * @param responseParameters
         *        A key-value map specifying required or optional response parameters that API Gateway can send back to
         *        the caller. A key defines a method response header and the value specifies whether the associated
         *        method response header is required or not. The expression of the key must match the pattern
         *        <code>method.response.header.{name}</code>, where <code>name</code> is a valid and unique header name.
         *        API Gateway passes certain integration response data to the method response headers specified here
         *        according to the mapping you prescribe in the API's IntegrationResponse. The integration response data
         *        that can be mapped include an integration response header expressed in
         *        <code>integration.response.header.{name}</code>, a static value enclosed within a pair of single
         *        quotes (e.g., <code>'application/json'</code>), or a JSON expression from the back-end response
         *        payload in the form of <code>integration.response.body.{JSON-expression}</code>, where
         *        <code>JSON-expression</code> is a valid JSON expression without the <code>$</code> prefix.)
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder responseParameters(Map<String, Boolean> responseParameters);

        /**
         * <p>
         * Specifies the Model resources used for the response's content-type. Response models are represented as a
         * key/value map, with a content-type as the key and a Model name as the value.
         * </p>
         * 
         * @param responseModels
         *        Specifies the Model resources used for the response's content-type. Response models are represented as
         *        a key/value map, with a content-type as the key and a Model name as the value.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder responseModels(Map<String, String> responseModels);
    }

    static final class BuilderImpl implements Builder {
        private String statusCodeValue;

        private Map<String, Boolean> responseParameters = DefaultSdkAutoConstructMap.getInstance();

        private Map<String, String> responseModels = DefaultSdkAutoConstructMap.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(MethodResponse model) {
            statusCode(model.statusCodeValue);
            responseParameters(model.responseParameters);
            responseModels(model.responseModels);
        }

        public final String getStatusCode() {
            return statusCodeValue;
        }

        public final void setStatusCode(String statusCodeValue) {
            this.statusCodeValue = statusCodeValue;
        }

        @Override
        public final Builder statusCode(String statusCodeValue) {
            this.statusCodeValue = statusCodeValue;
            return this;
        }

        public final Map<String, Boolean> getResponseParameters() {
            if (responseParameters instanceof SdkAutoConstructMap) {
                return null;
            }
            return responseParameters;
        }

        public final void setResponseParameters(Map<String, Boolean> responseParameters) {
            this.responseParameters = MapOfStringToBooleanCopier.copy(responseParameters);
        }

        @Override
        public final Builder responseParameters(Map<String, Boolean> responseParameters) {
            this.responseParameters = MapOfStringToBooleanCopier.copy(responseParameters);
            return this;
        }

        public final Map<String, String> getResponseModels() {
            if (responseModels instanceof SdkAutoConstructMap) {
                return null;
            }
            return responseModels;
        }

        public final void setResponseModels(Map<String, String> responseModels) {
            this.responseModels = MapOfStringToStringCopier.copy(responseModels);
        }

        @Override
        public final Builder responseModels(Map<String, String> responseModels) {
            this.responseModels = MapOfStringToStringCopier.copy(responseModels);
            return this;
        }

        @Override
        public MethodResponse build() {
            return new MethodResponse(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
