/*
 * Copyright 2013-2018 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.amplify;

import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsSyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.client.handler.SyncClientHandler;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.amplify.model.AmplifyException;
import software.amazon.awssdk.services.amplify.model.BadRequestException;
import software.amazon.awssdk.services.amplify.model.CreateAppRequest;
import software.amazon.awssdk.services.amplify.model.CreateAppResponse;
import software.amazon.awssdk.services.amplify.model.CreateBranchRequest;
import software.amazon.awssdk.services.amplify.model.CreateBranchResponse;
import software.amazon.awssdk.services.amplify.model.CreateDomainAssociationRequest;
import software.amazon.awssdk.services.amplify.model.CreateDomainAssociationResponse;
import software.amazon.awssdk.services.amplify.model.DeleteAppRequest;
import software.amazon.awssdk.services.amplify.model.DeleteAppResponse;
import software.amazon.awssdk.services.amplify.model.DeleteBranchRequest;
import software.amazon.awssdk.services.amplify.model.DeleteBranchResponse;
import software.amazon.awssdk.services.amplify.model.DeleteDomainAssociationRequest;
import software.amazon.awssdk.services.amplify.model.DeleteDomainAssociationResponse;
import software.amazon.awssdk.services.amplify.model.DeleteJobRequest;
import software.amazon.awssdk.services.amplify.model.DeleteJobResponse;
import software.amazon.awssdk.services.amplify.model.DependentServiceFailureException;
import software.amazon.awssdk.services.amplify.model.GetAppRequest;
import software.amazon.awssdk.services.amplify.model.GetAppResponse;
import software.amazon.awssdk.services.amplify.model.GetBranchRequest;
import software.amazon.awssdk.services.amplify.model.GetBranchResponse;
import software.amazon.awssdk.services.amplify.model.GetDomainAssociationRequest;
import software.amazon.awssdk.services.amplify.model.GetDomainAssociationResponse;
import software.amazon.awssdk.services.amplify.model.GetJobRequest;
import software.amazon.awssdk.services.amplify.model.GetJobResponse;
import software.amazon.awssdk.services.amplify.model.InternalFailureException;
import software.amazon.awssdk.services.amplify.model.LimitExceededException;
import software.amazon.awssdk.services.amplify.model.ListAppsRequest;
import software.amazon.awssdk.services.amplify.model.ListAppsResponse;
import software.amazon.awssdk.services.amplify.model.ListBranchesRequest;
import software.amazon.awssdk.services.amplify.model.ListBranchesResponse;
import software.amazon.awssdk.services.amplify.model.ListDomainAssociationsRequest;
import software.amazon.awssdk.services.amplify.model.ListDomainAssociationsResponse;
import software.amazon.awssdk.services.amplify.model.ListJobsRequest;
import software.amazon.awssdk.services.amplify.model.ListJobsResponse;
import software.amazon.awssdk.services.amplify.model.NotFoundException;
import software.amazon.awssdk.services.amplify.model.StartJobRequest;
import software.amazon.awssdk.services.amplify.model.StartJobResponse;
import software.amazon.awssdk.services.amplify.model.StopJobRequest;
import software.amazon.awssdk.services.amplify.model.StopJobResponse;
import software.amazon.awssdk.services.amplify.model.UnauthorizedException;
import software.amazon.awssdk.services.amplify.model.UpdateAppRequest;
import software.amazon.awssdk.services.amplify.model.UpdateAppResponse;
import software.amazon.awssdk.services.amplify.model.UpdateBranchRequest;
import software.amazon.awssdk.services.amplify.model.UpdateBranchResponse;
import software.amazon.awssdk.services.amplify.model.UpdateDomainAssociationRequest;
import software.amazon.awssdk.services.amplify.model.UpdateDomainAssociationResponse;
import software.amazon.awssdk.services.amplify.transform.CreateAppRequestMarshaller;
import software.amazon.awssdk.services.amplify.transform.CreateBranchRequestMarshaller;
import software.amazon.awssdk.services.amplify.transform.CreateDomainAssociationRequestMarshaller;
import software.amazon.awssdk.services.amplify.transform.DeleteAppRequestMarshaller;
import software.amazon.awssdk.services.amplify.transform.DeleteBranchRequestMarshaller;
import software.amazon.awssdk.services.amplify.transform.DeleteDomainAssociationRequestMarshaller;
import software.amazon.awssdk.services.amplify.transform.DeleteJobRequestMarshaller;
import software.amazon.awssdk.services.amplify.transform.GetAppRequestMarshaller;
import software.amazon.awssdk.services.amplify.transform.GetBranchRequestMarshaller;
import software.amazon.awssdk.services.amplify.transform.GetDomainAssociationRequestMarshaller;
import software.amazon.awssdk.services.amplify.transform.GetJobRequestMarshaller;
import software.amazon.awssdk.services.amplify.transform.ListAppsRequestMarshaller;
import software.amazon.awssdk.services.amplify.transform.ListBranchesRequestMarshaller;
import software.amazon.awssdk.services.amplify.transform.ListDomainAssociationsRequestMarshaller;
import software.amazon.awssdk.services.amplify.transform.ListJobsRequestMarshaller;
import software.amazon.awssdk.services.amplify.transform.StartJobRequestMarshaller;
import software.amazon.awssdk.services.amplify.transform.StopJobRequestMarshaller;
import software.amazon.awssdk.services.amplify.transform.UpdateAppRequestMarshaller;
import software.amazon.awssdk.services.amplify.transform.UpdateBranchRequestMarshaller;
import software.amazon.awssdk.services.amplify.transform.UpdateDomainAssociationRequestMarshaller;

/**
 * Internal implementation of {@link AmplifyClient}.
 *
 * @see AmplifyClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultAmplifyClient implements AmplifyClient {
    private final SyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultAmplifyClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsSyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    /**
     * <p>
     * Creates a new Amplify App.
     * </p>
     *
     * @param createAppRequest
     *        Request structure used to create Apps in Amplify.
     * @return Result of the CreateApp operation returned by the service.
     * @throws BadRequestException
     *         Exception thrown when a request contains unexpected data.
     * @throws UnauthorizedException
     *         Exception thrown when an operation fails due to a lack of access.
     * @throws InternalFailureException
     *         Exception thrown when the service fails to perform an operation due to an internal issue.
     * @throws LimitExceededException
     *         Exception thrown when a resource could not be created because of service limits.
     * @throws DependentServiceFailureException
     *         Exception thrown when an operation fails due to a dependent service throwing an exception.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AmplifyException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AmplifyClient.CreateApp
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/amplify-2017-07-25/CreateApp" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CreateAppResponse createApp(CreateAppRequest createAppRequest) throws BadRequestException, UnauthorizedException,
            InternalFailureException, LimitExceededException, DependentServiceFailureException, AwsServiceException,
            SdkClientException, AmplifyException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateAppResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                CreateAppResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<CreateAppRequest, CreateAppResponse>()
                .withOperationName("CreateApp").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(createAppRequest)
                .withMarshaller(new CreateAppRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Creates a new Branch for an Amplify App.
     * </p>
     *
     * @param createBranchRequest
     *        Request structure for a branch create request.
     * @return Result of the CreateBranch operation returned by the service.
     * @throws BadRequestException
     *         Exception thrown when a request contains unexpected data.
     * @throws UnauthorizedException
     *         Exception thrown when an operation fails due to a lack of access.
     * @throws NotFoundException
     *         Exception thrown when an entity has not been found during an operation.
     * @throws InternalFailureException
     *         Exception thrown when the service fails to perform an operation due to an internal issue.
     * @throws LimitExceededException
     *         Exception thrown when a resource could not be created because of service limits.
     * @throws DependentServiceFailureException
     *         Exception thrown when an operation fails due to a dependent service throwing an exception.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AmplifyException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AmplifyClient.CreateBranch
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/amplify-2017-07-25/CreateBranch" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CreateBranchResponse createBranch(CreateBranchRequest createBranchRequest) throws BadRequestException,
            UnauthorizedException, NotFoundException, InternalFailureException, LimitExceededException,
            DependentServiceFailureException, AwsServiceException, SdkClientException, AmplifyException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateBranchResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                CreateBranchResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<CreateBranchRequest, CreateBranchResponse>()
                .withOperationName("CreateBranch").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(createBranchRequest)
                .withMarshaller(new CreateBranchRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Create a new DomainAssociation on an App
     * </p>
     *
     * @param createDomainAssociationRequest
     *        Request structure for create Domain Association request.
     * @return Result of the CreateDomainAssociation operation returned by the service.
     * @throws BadRequestException
     *         Exception thrown when a request contains unexpected data.
     * @throws UnauthorizedException
     *         Exception thrown when an operation fails due to a lack of access.
     * @throws NotFoundException
     *         Exception thrown when an entity has not been found during an operation.
     * @throws InternalFailureException
     *         Exception thrown when the service fails to perform an operation due to an internal issue.
     * @throws LimitExceededException
     *         Exception thrown when a resource could not be created because of service limits.
     * @throws DependentServiceFailureException
     *         Exception thrown when an operation fails due to a dependent service throwing an exception.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AmplifyException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AmplifyClient.CreateDomainAssociation
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/amplify-2017-07-25/CreateDomainAssociation"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CreateDomainAssociationResponse createDomainAssociation(CreateDomainAssociationRequest createDomainAssociationRequest)
            throws BadRequestException, UnauthorizedException, NotFoundException, InternalFailureException,
            LimitExceededException, DependentServiceFailureException, AwsServiceException, SdkClientException, AmplifyException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateDomainAssociationResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, CreateDomainAssociationResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<CreateDomainAssociationRequest, CreateDomainAssociationResponse>()
                .withOperationName("CreateDomainAssociation").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(createDomainAssociationRequest)
                .withMarshaller(new CreateDomainAssociationRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Delete an existing Amplify App by appId.
     * </p>
     *
     * @param deleteAppRequest
     *        Request structure for an Amplify App delete request.
     * @return Result of the DeleteApp operation returned by the service.
     * @throws BadRequestException
     *         Exception thrown when a request contains unexpected data.
     * @throws NotFoundException
     *         Exception thrown when an entity has not been found during an operation.
     * @throws UnauthorizedException
     *         Exception thrown when an operation fails due to a lack of access.
     * @throws InternalFailureException
     *         Exception thrown when the service fails to perform an operation due to an internal issue.
     * @throws DependentServiceFailureException
     *         Exception thrown when an operation fails due to a dependent service throwing an exception.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AmplifyException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AmplifyClient.DeleteApp
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/amplify-2017-07-25/DeleteApp" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DeleteAppResponse deleteApp(DeleteAppRequest deleteAppRequest) throws BadRequestException, NotFoundException,
            UnauthorizedException, InternalFailureException, DependentServiceFailureException, AwsServiceException,
            SdkClientException, AmplifyException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteAppResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DeleteAppResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<DeleteAppRequest, DeleteAppResponse>()
                .withOperationName("DeleteApp").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(deleteAppRequest)
                .withMarshaller(new DeleteAppRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Deletes a branch for an Amplify App.
     * </p>
     *
     * @param deleteBranchRequest
     *        Request structure for delete branch request.
     * @return Result of the DeleteBranch operation returned by the service.
     * @throws BadRequestException
     *         Exception thrown when a request contains unexpected data.
     * @throws UnauthorizedException
     *         Exception thrown when an operation fails due to a lack of access.
     * @throws NotFoundException
     *         Exception thrown when an entity has not been found during an operation.
     * @throws InternalFailureException
     *         Exception thrown when the service fails to perform an operation due to an internal issue.
     * @throws DependentServiceFailureException
     *         Exception thrown when an operation fails due to a dependent service throwing an exception.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AmplifyException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AmplifyClient.DeleteBranch
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/amplify-2017-07-25/DeleteBranch" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DeleteBranchResponse deleteBranch(DeleteBranchRequest deleteBranchRequest) throws BadRequestException,
            UnauthorizedException, NotFoundException, InternalFailureException, DependentServiceFailureException,
            AwsServiceException, SdkClientException, AmplifyException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteBranchResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DeleteBranchResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<DeleteBranchRequest, DeleteBranchResponse>()
                .withOperationName("DeleteBranch").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(deleteBranchRequest)
                .withMarshaller(new DeleteBranchRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Deletes a DomainAssociation.
     * </p>
     *
     * @param deleteDomainAssociationRequest
     *        Request structure for the delete Domain Association request.
     * @return Result of the DeleteDomainAssociation operation returned by the service.
     * @throws BadRequestException
     *         Exception thrown when a request contains unexpected data.
     * @throws UnauthorizedException
     *         Exception thrown when an operation fails due to a lack of access.
     * @throws NotFoundException
     *         Exception thrown when an entity has not been found during an operation.
     * @throws InternalFailureException
     *         Exception thrown when the service fails to perform an operation due to an internal issue.
     * @throws DependentServiceFailureException
     *         Exception thrown when an operation fails due to a dependent service throwing an exception.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AmplifyException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AmplifyClient.DeleteDomainAssociation
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/amplify-2017-07-25/DeleteDomainAssociation"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DeleteDomainAssociationResponse deleteDomainAssociation(DeleteDomainAssociationRequest deleteDomainAssociationRequest)
            throws BadRequestException, UnauthorizedException, NotFoundException, InternalFailureException,
            DependentServiceFailureException, AwsServiceException, SdkClientException, AmplifyException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteDomainAssociationResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DeleteDomainAssociationResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<DeleteDomainAssociationRequest, DeleteDomainAssociationResponse>()
                .withOperationName("DeleteDomainAssociation").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(deleteDomainAssociationRequest)
                .withMarshaller(new DeleteDomainAssociationRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Delete a job, for an Amplify branch, part of Amplify App.
     * </p>
     *
     * @param deleteJobRequest
     *        Request structure for delete job request.
     * @return Result of the DeleteJob operation returned by the service.
     * @throws BadRequestException
     *         Exception thrown when a request contains unexpected data.
     * @throws UnauthorizedException
     *         Exception thrown when an operation fails due to a lack of access.
     * @throws InternalFailureException
     *         Exception thrown when the service fails to perform an operation due to an internal issue.
     * @throws NotFoundException
     *         Exception thrown when an entity has not been found during an operation.
     * @throws LimitExceededException
     *         Exception thrown when a resource could not be created because of service limits.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AmplifyException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AmplifyClient.DeleteJob
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/amplify-2017-07-25/DeleteJob" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DeleteJobResponse deleteJob(DeleteJobRequest deleteJobRequest) throws BadRequestException, UnauthorizedException,
            InternalFailureException, NotFoundException, LimitExceededException, AwsServiceException, SdkClientException,
            AmplifyException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteJobResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DeleteJobResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<DeleteJobRequest, DeleteJobResponse>()
                .withOperationName("DeleteJob").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(deleteJobRequest)
                .withMarshaller(new DeleteJobRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Retrieves an existing Amplify App by appId.
     * </p>
     *
     * @param getAppRequest
     *        Request structure for get App request.
     * @return Result of the GetApp operation returned by the service.
     * @throws BadRequestException
     *         Exception thrown when a request contains unexpected data.
     * @throws NotFoundException
     *         Exception thrown when an entity has not been found during an operation.
     * @throws UnauthorizedException
     *         Exception thrown when an operation fails due to a lack of access.
     * @throws InternalFailureException
     *         Exception thrown when the service fails to perform an operation due to an internal issue.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AmplifyException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AmplifyClient.GetApp
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/amplify-2017-07-25/GetApp" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public GetAppResponse getApp(GetAppRequest getAppRequest) throws BadRequestException, NotFoundException,
            UnauthorizedException, InternalFailureException, AwsServiceException, SdkClientException, AmplifyException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetAppResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetAppResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<GetAppRequest, GetAppResponse>().withOperationName("GetApp")
                .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler).withInput(getAppRequest)
                .withMarshaller(new GetAppRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Retrieves a branch for an Amplify App.
     * </p>
     *
     * @param getBranchRequest
     *        Result structure for get branch request.
     * @return Result of the GetBranch operation returned by the service.
     * @throws BadRequestException
     *         Exception thrown when a request contains unexpected data.
     * @throws UnauthorizedException
     *         Exception thrown when an operation fails due to a lack of access.
     * @throws NotFoundException
     *         Exception thrown when an entity has not been found during an operation.
     * @throws InternalFailureException
     *         Exception thrown when the service fails to perform an operation due to an internal issue.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AmplifyException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AmplifyClient.GetBranch
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/amplify-2017-07-25/GetBranch" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public GetBranchResponse getBranch(GetBranchRequest getBranchRequest) throws BadRequestException, UnauthorizedException,
            NotFoundException, InternalFailureException, AwsServiceException, SdkClientException, AmplifyException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetBranchResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetBranchResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<GetBranchRequest, GetBranchResponse>()
                .withOperationName("GetBranch").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(getBranchRequest)
                .withMarshaller(new GetBranchRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Retrieves domain info that corresponds to an appId and domainName.
     * </p>
     *
     * @param getDomainAssociationRequest
     *        Request structure for the get Domain Association request.
     * @return Result of the GetDomainAssociation operation returned by the service.
     * @throws BadRequestException
     *         Exception thrown when a request contains unexpected data.
     * @throws UnauthorizedException
     *         Exception thrown when an operation fails due to a lack of access.
     * @throws NotFoundException
     *         Exception thrown when an entity has not been found during an operation.
     * @throws InternalFailureException
     *         Exception thrown when the service fails to perform an operation due to an internal issue.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AmplifyException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AmplifyClient.GetDomainAssociation
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/amplify-2017-07-25/GetDomainAssociation" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public GetDomainAssociationResponse getDomainAssociation(GetDomainAssociationRequest getDomainAssociationRequest)
            throws BadRequestException, UnauthorizedException, NotFoundException, InternalFailureException, AwsServiceException,
            SdkClientException, AmplifyException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetDomainAssociationResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, GetDomainAssociationResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<GetDomainAssociationRequest, GetDomainAssociationResponse>()
                .withOperationName("GetDomainAssociation").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(getDomainAssociationRequest)
                .withMarshaller(new GetDomainAssociationRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Get a job for a branch, part of an Amplify App.
     * </p>
     *
     * @param getJobRequest
     *        Request structure for get job request.
     * @return Result of the GetJob operation returned by the service.
     * @throws BadRequestException
     *         Exception thrown when a request contains unexpected data.
     * @throws UnauthorizedException
     *         Exception thrown when an operation fails due to a lack of access.
     * @throws InternalFailureException
     *         Exception thrown when the service fails to perform an operation due to an internal issue.
     * @throws NotFoundException
     *         Exception thrown when an entity has not been found during an operation.
     * @throws LimitExceededException
     *         Exception thrown when a resource could not be created because of service limits.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AmplifyException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AmplifyClient.GetJob
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/amplify-2017-07-25/GetJob" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public GetJobResponse getJob(GetJobRequest getJobRequest) throws BadRequestException, UnauthorizedException,
            InternalFailureException, NotFoundException, LimitExceededException, AwsServiceException, SdkClientException,
            AmplifyException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetJobResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetJobResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<GetJobRequest, GetJobResponse>().withOperationName("GetJob")
                .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler).withInput(getJobRequest)
                .withMarshaller(new GetJobRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Lists existing Amplify Apps.
     * </p>
     *
     * @param listAppsRequest
     *        Request structure for an Amplify App list request.
     * @return Result of the ListApps operation returned by the service.
     * @throws BadRequestException
     *         Exception thrown when a request contains unexpected data.
     * @throws UnauthorizedException
     *         Exception thrown when an operation fails due to a lack of access.
     * @throws InternalFailureException
     *         Exception thrown when the service fails to perform an operation due to an internal issue.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AmplifyException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AmplifyClient.ListApps
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/amplify-2017-07-25/ListApps" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListAppsResponse listApps(ListAppsRequest listAppsRequest) throws BadRequestException, UnauthorizedException,
            InternalFailureException, AwsServiceException, SdkClientException, AmplifyException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListAppsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListAppsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<ListAppsRequest, ListAppsResponse>().withOperationName("ListApps")
                .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler).withInput(listAppsRequest)
                .withMarshaller(new ListAppsRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Lists branches for an Amplify App.
     * </p>
     *
     * @param listBranchesRequest
     *        Request structure for list branches request.
     * @return Result of the ListBranches operation returned by the service.
     * @throws BadRequestException
     *         Exception thrown when a request contains unexpected data.
     * @throws UnauthorizedException
     *         Exception thrown when an operation fails due to a lack of access.
     * @throws InternalFailureException
     *         Exception thrown when the service fails to perform an operation due to an internal issue.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AmplifyException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AmplifyClient.ListBranches
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/amplify-2017-07-25/ListBranches" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListBranchesResponse listBranches(ListBranchesRequest listBranchesRequest) throws BadRequestException,
            UnauthorizedException, InternalFailureException, AwsServiceException, SdkClientException, AmplifyException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListBranchesResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListBranchesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<ListBranchesRequest, ListBranchesResponse>()
                .withOperationName("ListBranches").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(listBranchesRequest)
                .withMarshaller(new ListBranchesRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * List domains with an app
     * </p>
     *
     * @param listDomainAssociationsRequest
     *        Request structure for the list Domain Associations request.
     * @return Result of the ListDomainAssociations operation returned by the service.
     * @throws BadRequestException
     *         Exception thrown when a request contains unexpected data.
     * @throws UnauthorizedException
     *         Exception thrown when an operation fails due to a lack of access.
     * @throws InternalFailureException
     *         Exception thrown when the service fails to perform an operation due to an internal issue.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AmplifyException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AmplifyClient.ListDomainAssociations
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/amplify-2017-07-25/ListDomainAssociations" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public ListDomainAssociationsResponse listDomainAssociations(ListDomainAssociationsRequest listDomainAssociationsRequest)
            throws BadRequestException, UnauthorizedException, InternalFailureException, AwsServiceException, SdkClientException,
            AmplifyException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListDomainAssociationsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListDomainAssociationsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<ListDomainAssociationsRequest, ListDomainAssociationsResponse>()
                .withOperationName("ListDomainAssociations").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(listDomainAssociationsRequest)
                .withMarshaller(new ListDomainAssociationsRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * List Jobs for a branch, part of an Amplify App.
     * </p>
     *
     * @param listJobsRequest
     *        Request structure for list job request.
     * @return Result of the ListJobs operation returned by the service.
     * @throws BadRequestException
     *         Exception thrown when a request contains unexpected data.
     * @throws UnauthorizedException
     *         Exception thrown when an operation fails due to a lack of access.
     * @throws InternalFailureException
     *         Exception thrown when the service fails to perform an operation due to an internal issue.
     * @throws LimitExceededException
     *         Exception thrown when a resource could not be created because of service limits.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AmplifyException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AmplifyClient.ListJobs
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/amplify-2017-07-25/ListJobs" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListJobsResponse listJobs(ListJobsRequest listJobsRequest) throws BadRequestException, UnauthorizedException,
            InternalFailureException, LimitExceededException, AwsServiceException, SdkClientException, AmplifyException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListJobsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListJobsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<ListJobsRequest, ListJobsResponse>().withOperationName("ListJobs")
                .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler).withInput(listJobsRequest)
                .withMarshaller(new ListJobsRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Starts a new job for a branch, part of an Amplify App.
     * </p>
     *
     * @param startJobRequest
     *        Request structure for Start job request.
     * @return Result of the StartJob operation returned by the service.
     * @throws BadRequestException
     *         Exception thrown when a request contains unexpected data.
     * @throws UnauthorizedException
     *         Exception thrown when an operation fails due to a lack of access.
     * @throws InternalFailureException
     *         Exception thrown when the service fails to perform an operation due to an internal issue.
     * @throws NotFoundException
     *         Exception thrown when an entity has not been found during an operation.
     * @throws LimitExceededException
     *         Exception thrown when a resource could not be created because of service limits.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AmplifyException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AmplifyClient.StartJob
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/amplify-2017-07-25/StartJob" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public StartJobResponse startJob(StartJobRequest startJobRequest) throws BadRequestException, UnauthorizedException,
            InternalFailureException, NotFoundException, LimitExceededException, AwsServiceException, SdkClientException,
            AmplifyException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<StartJobResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                StartJobResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<StartJobRequest, StartJobResponse>().withOperationName("StartJob")
                .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler).withInput(startJobRequest)
                .withMarshaller(new StartJobRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Stop a job that is in progress, for an Amplify branch, part of Amplify App.
     * </p>
     *
     * @param stopJobRequest
     *        Request structure for stop job request.
     * @return Result of the StopJob operation returned by the service.
     * @throws BadRequestException
     *         Exception thrown when a request contains unexpected data.
     * @throws UnauthorizedException
     *         Exception thrown when an operation fails due to a lack of access.
     * @throws InternalFailureException
     *         Exception thrown when the service fails to perform an operation due to an internal issue.
     * @throws NotFoundException
     *         Exception thrown when an entity has not been found during an operation.
     * @throws LimitExceededException
     *         Exception thrown when a resource could not be created because of service limits.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AmplifyException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AmplifyClient.StopJob
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/amplify-2017-07-25/StopJob" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public StopJobResponse stopJob(StopJobRequest stopJobRequest) throws BadRequestException, UnauthorizedException,
            InternalFailureException, NotFoundException, LimitExceededException, AwsServiceException, SdkClientException,
            AmplifyException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<StopJobResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                StopJobResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<StopJobRequest, StopJobResponse>().withOperationName("StopJob")
                .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler).withInput(stopJobRequest)
                .withMarshaller(new StopJobRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Updates an existing Amplify App.
     * </p>
     *
     * @param updateAppRequest
     *        Request structure for update App request.
     * @return Result of the UpdateApp operation returned by the service.
     * @throws BadRequestException
     *         Exception thrown when a request contains unexpected data.
     * @throws NotFoundException
     *         Exception thrown when an entity has not been found during an operation.
     * @throws UnauthorizedException
     *         Exception thrown when an operation fails due to a lack of access.
     * @throws InternalFailureException
     *         Exception thrown when the service fails to perform an operation due to an internal issue.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AmplifyException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AmplifyClient.UpdateApp
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/amplify-2017-07-25/UpdateApp" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public UpdateAppResponse updateApp(UpdateAppRequest updateAppRequest) throws BadRequestException, NotFoundException,
            UnauthorizedException, InternalFailureException, AwsServiceException, SdkClientException, AmplifyException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateAppResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UpdateAppResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<UpdateAppRequest, UpdateAppResponse>()
                .withOperationName("UpdateApp").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(updateAppRequest)
                .withMarshaller(new UpdateAppRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Updates a branch for an Amplify App.
     * </p>
     *
     * @param updateBranchRequest
     *        Request structure for update branch request.
     * @return Result of the UpdateBranch operation returned by the service.
     * @throws BadRequestException
     *         Exception thrown when a request contains unexpected data.
     * @throws UnauthorizedException
     *         Exception thrown when an operation fails due to a lack of access.
     * @throws NotFoundException
     *         Exception thrown when an entity has not been found during an operation.
     * @throws InternalFailureException
     *         Exception thrown when the service fails to perform an operation due to an internal issue.
     * @throws DependentServiceFailureException
     *         Exception thrown when an operation fails due to a dependent service throwing an exception.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AmplifyException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AmplifyClient.UpdateBranch
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/amplify-2017-07-25/UpdateBranch" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public UpdateBranchResponse updateBranch(UpdateBranchRequest updateBranchRequest) throws BadRequestException,
            UnauthorizedException, NotFoundException, InternalFailureException, DependentServiceFailureException,
            AwsServiceException, SdkClientException, AmplifyException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateBranchResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UpdateBranchResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<UpdateBranchRequest, UpdateBranchResponse>()
                .withOperationName("UpdateBranch").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(updateBranchRequest)
                .withMarshaller(new UpdateBranchRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Create a new DomainAssociation on an App
     * </p>
     *
     * @param updateDomainAssociationRequest
     *        Request structure for update Domain Association request.
     * @return Result of the UpdateDomainAssociation operation returned by the service.
     * @throws BadRequestException
     *         Exception thrown when a request contains unexpected data.
     * @throws UnauthorizedException
     *         Exception thrown when an operation fails due to a lack of access.
     * @throws NotFoundException
     *         Exception thrown when an entity has not been found during an operation.
     * @throws InternalFailureException
     *         Exception thrown when the service fails to perform an operation due to an internal issue.
     * @throws DependentServiceFailureException
     *         Exception thrown when an operation fails due to a dependent service throwing an exception.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AmplifyException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AmplifyClient.UpdateDomainAssociation
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/amplify-2017-07-25/UpdateDomainAssociation"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public UpdateDomainAssociationResponse updateDomainAssociation(UpdateDomainAssociationRequest updateDomainAssociationRequest)
            throws BadRequestException, UnauthorizedException, NotFoundException, InternalFailureException,
            DependentServiceFailureException, AwsServiceException, SdkClientException, AmplifyException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateDomainAssociationResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, UpdateDomainAssociationResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<UpdateDomainAssociationRequest, UpdateDomainAssociationResponse>()
                .withOperationName("UpdateDomainAssociation").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(updateDomainAssociationRequest)
                .withMarshaller(new UpdateDomainAssociationRequestMarshaller(protocolFactory)));
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(AmplifyException::builder)
                .protocol(AwsJsonProtocol.REST_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalFailureException")
                                .exceptionBuilderSupplier(InternalFailureException::builder).httpStatusCode(500).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("DependentServiceFailureException")
                                .exceptionBuilderSupplier(DependentServiceFailureException::builder).httpStatusCode(503).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("NotFoundException")
                                .exceptionBuilderSupplier(NotFoundException::builder).httpStatusCode(404).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("UnauthorizedException")
                                .exceptionBuilderSupplier(UnauthorizedException::builder).httpStatusCode(401).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("BadRequestException")
                                .exceptionBuilderSupplier(BadRequestException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("LimitExceededException")
                                .exceptionBuilderSupplier(LimitExceededException::builder).httpStatusCode(429).build());
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
