/**
 * <h1>Kinesis Analytics Flink</h1>
 * <p>
 * This package provides constructs for creating Kinesis Analytics Flink
 * applications. To learn more about using using managed Flink applications, see
 * the <a href="https://docs.aws.amazon.com/kinesisanalytics/latest/java/">AWS developer
 * guide</a>.
 * <p>
 * <h2>Creating Flink Applications</h2>
 * <p>
 * To create a new Flink application, use the <code>Application</code> construct:
 * <p>
 * <blockquote><pre>
 * import path.*;
 * import software.amazon.awscdk.core.*;
 * import software.amazon.awscdk.core.*;
 * import software.amazon.awscdk.services.cloudwatch.*;
 * 
 * App app = new App();
 * Stack stack = new Stack(app, "FlinkAppTest");
 * 
 * Application flinkApp = Application.Builder.create(stack, "App")
 *         .code(ApplicationCode.fromAsset(join(__dirname, "code-asset")))
 *         .runtime(Runtime.FLINK_1_11)
 *         .build();
 * 
 * Alarm.Builder.create(stack, "Alarm")
 *         .metric(flinkApp.metricFullRestarts())
 *         .evaluationPeriods(1)
 *         .threshold(3)
 *         .build();
 * 
 * app.synth();
 * </pre></blockquote>
 * <p>
 * The <code>code</code> property can use <code>fromAsset</code> as shown above to reference a local jar
 * file in s3 or <code>fromBucket</code> to reference a file in s3.
 * <p>
 * <blockquote><pre>
 * import path.*;
 * import software.amazon.awscdk.services.s3.assets.*;
 * import software.amazon.awscdk.core.*;
 * import software.amazon.awscdk.core.*;
 * 
 * App app = new App();
 * Stack stack = new Stack(app, "FlinkAppCodeFromBucketTest");
 * 
 * Asset asset = Asset.Builder.create(stack, "CodeAsset")
 *         .path(join(__dirname, "code-asset"))
 *         .build();
 * IBucket bucket = asset.getBucket();
 * String fileKey = asset.getS3ObjectKey();
 * 
 * Application.Builder.create(stack, "App")
 *         .code(ApplicationCode.fromBucket(bucket, fileKey))
 *         .runtime(Runtime.FLINK_1_11)
 *         .build();
 * 
 * app.synth();
 * </pre></blockquote>
 * <p>
 * The <code>propertyGroups</code> property provides a way of passing arbitrary runtime
 * properties to your Flink application. You can use the
 * aws-kinesisanalytics-runtime library to <a href="https://docs.aws.amazon.com/kinesisanalytics/latest/java/how-properties.html#how-properties-access">retrieve these
 * properties</a>.
 * <p>
 * <blockquote><pre>
 * Bucket bucket;
 * 
 * Application flinkApp = Application.Builder.create(this, "Application")
 *         .propertyGroups(PropertyGroups.builder()
 *                 .FlinkApplicationProperties(Map.of(
 *                         "inputStreamName", "my-input-kinesis-stream",
 *                         "outputStreamName", "my-output-kinesis-stream"))
 *                 .build())
 *         // ...
 *         .runtime(Runtime.FLINK_1_13)
 *         .code(ApplicationCode.fromBucket(bucket, "my-app.jar"))
 *         .build();
 * </pre></blockquote>
 * <p>
 * Flink applications also have specific configuration for passing parameters
 * when the Flink job starts. These include parameters for checkpointing,
 * snapshotting, monitoring, and parallelism.
 * <p>
 * <blockquote><pre>
 * Bucket bucket;
 * 
 * Application flinkApp = Application.Builder.create(this, "Application")
 *         .code(ApplicationCode.fromBucket(bucket, "my-app.jar"))
 *         .runtime(Runtime.FLINK_1_13)
 *         .checkpointingEnabled(true) // default is true
 *         .checkpointInterval(Duration.seconds(30)) // default is 1 minute
 *         .minPauseBetweenCheckpoints(Duration.seconds(10)) // default is 5 seconds
 *         .logLevel(LogLevel.ERROR) // default is INFO
 *         .metricsLevel(MetricsLevel.PARALLELISM) // default is APPLICATION
 *         .autoScalingEnabled(false) // default is true
 *         .parallelism(32) // default is 1
 *         .parallelismPerKpu(2) // default is 1
 *         .snapshotsEnabled(false) // default is true
 *         .logGroup(new LogGroup(this, "LogGroup"))
 *         .build();
 * </pre></blockquote>
 */
package software.amazon.awscdk.services.kinesis.analytics.flink;
