/**
 * <h1>AWS::IoTEvents Construct Library</h1>
 * <p>
 * AWS IoT Events enables you to monitor your equipment or device fleets for
 * failures or changes in operation, and to trigger actions when such events
 * occur.
 * <p>
 * <h2>Installation</h2>
 * <p>
 * Install the module:
 * <p>
 * <blockquote><pre>
 * $ npm i &#64;aws-cdk/aws-iotevents
 * </pre></blockquote>
 * <p>
 * Import it into your code:
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.core.*;
 * </pre></blockquote>
 * <p>
 * <h2><code>DetectorModel</code></h2>
 * <p>
 * The following example creates an AWS IoT Events detector model to your stack.
 * The detector model need a reference to at least one AWS IoT Events input.
 * AWS IoT Events inputs enable the detector to get MQTT payload values from IoT Core rules.
 * <p>
 * You can define built-in actions to use a timer or set a variable, or send data to other AWS resources.
 * See also <a href="https://docs.aws.amazon.com/cdk/api/v1/docs/aws-iotevents-actions-readme.html">&#64;aws-cdk/aws-iotevents-actions</a> for other actions.
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.core.*;
 * import software.amazon.awscdk.core.*;
 * import software.amazon.awscdk.core.*;
 * 
 * IFunction func;
 * 
 * 
 * Input input = Input.Builder.create(this, "MyInput")
 *         .inputName("my_input") // optional
 *         .attributeJsonPaths(List.of("payload.deviceId", "payload.temperature"))
 *         .build();
 * 
 * State warmState = State.Builder.create()
 *         .stateName("warm")
 *         .onEnter(List.of(Event.builder()
 *                 .eventName("test-enter-event")
 *                 .condition(Expression.currentInput(input))
 *                 .actions(List.of(new LambdaInvokeAction(func)))
 *                 .build()))
 *         .onInput(List.of(Event.builder() // optional
 *                 .eventName("test-input-event")
 *                 .actions(List.of(new LambdaInvokeAction(func))).build()))
 *         .onExit(List.of(Event.builder() // optional
 *                 .eventName("test-exit-event")
 *                 .actions(List.of(new LambdaInvokeAction(func))).build()))
 *         .build();
 * State coldState = State.Builder.create()
 *         .stateName("cold")
 *         .build();
 * 
 * // transit to coldState when temperature is less than 15
 * warmState.transitionTo(coldState, TransitionOptions.builder()
 *         .eventName("to_coldState") // optional property, default by combining the names of the States
 *         .when(Expression.lt(Expression.inputAttribute(input, "payload.temperature"), Expression.fromString("15")))
 *         .executing(List.of(new LambdaInvokeAction(func)))
 *         .build());
 * // transit to warmState when temperature is greater than or equal to 15
 * coldState.transitionTo(warmState, TransitionOptions.builder()
 *         .when(Expression.gte(Expression.inputAttribute(input, "payload.temperature"), Expression.fromString("15")))
 *         .build());
 * 
 * DetectorModel.Builder.create(this, "MyDetectorModel")
 *         .detectorModelName("test-detector-model") // optional
 *         .description("test-detector-model-description") // optional property, default is none
 *         .evaluationMethod(EventEvaluation.SERIAL) // optional property, default is iotevents.EventEvaluation.BATCH
 *         .detectorKey("payload.deviceId") // optional property, default is none and single detector instance will be created and all inputs will be routed to it
 *         .initialState(warmState)
 *         .build();
 * </pre></blockquote>
 * <p>
 * To grant permissions to put messages in the input,
 * you can use the <code>grantWrite()</code> method:
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.core.*;
 * import software.amazon.awscdk.core.*;
 * 
 * IGrantable grantable;
 * 
 * IInput input = Input.fromInputName(this, "MyInput", "my_input");
 * 
 * input.grantWrite(grantable);
 * </pre></blockquote>
 */
package software.amazon.awscdk.services.iotevents;
