/**
 * <h1>CDK Construct library for higher-level ECS Constructs</h1>
 * <p>
 * This library provides higher-level Amazon ECS constructs which follow common architectural patterns. It contains:
 * <p>
 * <ul>
 * <li>Application Load Balanced Services</li>
 * <li>Network Load Balanced Services</li>
 * <li>Queue Processing Services</li>
 * <li>Scheduled Tasks (cron jobs)</li>
 * <li>Additional Examples</li>
 * </ul>
 * <p>
 * <h2>Application Load Balanced Services</h2>
 * <p>
 * To define an Amazon ECS service that is behind an application load balancer, instantiate one of the following:
 * <p>
 * <ul>
 * <li><code>ApplicationLoadBalancedEc2Service</code></li>
 * </ul>
 * <p>
 * <blockquote><pre>
 * Cluster cluster;
 * 
 * ApplicationLoadBalancedEc2Service loadBalancedEcsService = ApplicationLoadBalancedEc2Service.Builder.create(this, "Service")
 *         .cluster(cluster)
 *         .memoryLimitMiB(1024)
 *         .taskImageOptions(ApplicationLoadBalancedTaskImageOptions.builder()
 *                 .image(ContainerImage.fromRegistry("test"))
 *                 .environment(Map.of(
 *                         "TEST_ENVIRONMENT_VARIABLE1", "test environment variable 1 value",
 *                         "TEST_ENVIRONMENT_VARIABLE2", "test environment variable 2 value"))
 *                 .build())
 *         .desiredCount(2)
 *         .build();
 * </pre></blockquote>
 * <p>
 * <ul>
 * <li><code>ApplicationLoadBalancedFargateService</code></li>
 * </ul>
 * <p>
 * <blockquote><pre>
 * Cluster cluster;
 * 
 * ApplicationLoadBalancedFargateService loadBalancedFargateService = ApplicationLoadBalancedFargateService.Builder.create(this, "Service")
 *         .cluster(cluster)
 *         .memoryLimitMiB(1024)
 *         .cpu(512)
 *         .taskImageOptions(ApplicationLoadBalancedTaskImageOptions.builder()
 *                 .image(ContainerImage.fromRegistry("amazon/amazon-ecs-sample"))
 *                 .build())
 *         .build();
 * 
 * loadBalancedFargateService.targetGroup.configureHealthCheck(HealthCheck.builder()
 *         .path("/custom-health-path")
 *         .build());
 * </pre></blockquote>
 * <p>
 * Instead of providing a cluster you can specify a VPC and CDK will create a new ECS cluster.
 * If you deploy multiple services CDK will only create one cluster per VPC.
 * <p>
 * You can omit <code>cluster</code> and <code>vpc</code> to let CDK create a new VPC with two AZs and create a cluster inside this VPC.
 * <p>
 * You can customize the health check for your target group; otherwise it defaults to <code>HTTP</code> over port <code>80</code> hitting path <code>/</code>.
 * <p>
 * Fargate services will use the <code>LATEST</code> platform version by default, but you can override by providing a value for the <code>platformVersion</code> property in the constructor.
 * <p>
 * Fargate services use the default VPC Security Group unless one or more are provided using the <code>securityGroups</code> property in the constructor.
 * <p>
 * By setting <code>redirectHTTP</code> to true, CDK will automatically create a listener on port 80 that redirects HTTP traffic to the HTTPS port.
 * <p>
 * If you specify the option <code>recordType</code> you can decide if you want the construct to use CNAME or Route53-Aliases as record sets.
 * <p>
 * If you need to encrypt the traffic between the load balancer and the ECS tasks, you can set the <code>targetProtocol</code> to <code>HTTPS</code>.
 * <p>
 * Additionally, if more than one application target group are needed, instantiate one of the following:
 * <p>
 * <ul>
 * <li><code>ApplicationMultipleTargetGroupsEc2Service</code></li>
 * </ul>
 * <p>
 * <blockquote><pre>
 * // One application load balancer with one listener and two target groups.
 * Cluster cluster;
 * 
 * ApplicationMultipleTargetGroupsEc2Service loadBalancedEc2Service = ApplicationMultipleTargetGroupsEc2Service.Builder.create(this, "Service")
 *         .cluster(cluster)
 *         .memoryLimitMiB(256)
 *         .taskImageOptions(ApplicationLoadBalancedTaskImageProps.builder()
 *                 .image(ContainerImage.fromRegistry("amazon/amazon-ecs-sample"))
 *                 .build())
 *         .targetGroups(List.of(ApplicationTargetProps.builder()
 *                 .containerPort(80)
 *                 .build(), ApplicationTargetProps.builder()
 *                 .containerPort(90)
 *                 .pathPattern("a/b/c")
 *                 .priority(10)
 *                 .build()))
 *         .build();
 * </pre></blockquote>
 * <p>
 * <ul>
 * <li><code>ApplicationMultipleTargetGroupsFargateService</code></li>
 * </ul>
 * <p>
 * <blockquote><pre>
 * // One application load balancer with one listener and two target groups.
 * Cluster cluster;
 * 
 * ApplicationMultipleTargetGroupsFargateService loadBalancedFargateService = ApplicationMultipleTargetGroupsFargateService.Builder.create(this, "Service")
 *         .cluster(cluster)
 *         .memoryLimitMiB(1024)
 *         .cpu(512)
 *         .taskImageOptions(ApplicationLoadBalancedTaskImageProps.builder()
 *                 .image(ContainerImage.fromRegistry("amazon/amazon-ecs-sample"))
 *                 .build())
 *         .targetGroups(List.of(ApplicationTargetProps.builder()
 *                 .containerPort(80)
 *                 .build(), ApplicationTargetProps.builder()
 *                 .containerPort(90)
 *                 .pathPattern("a/b/c")
 *                 .priority(10)
 *                 .build()))
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h2>Network Load Balanced Services</h2>
 * <p>
 * To define an Amazon ECS service that is behind a network load balancer, instantiate one of the following:
 * <p>
 * <ul>
 * <li><code>NetworkLoadBalancedEc2Service</code></li>
 * </ul>
 * <p>
 * <blockquote><pre>
 * Cluster cluster;
 * 
 * NetworkLoadBalancedEc2Service loadBalancedEcsService = NetworkLoadBalancedEc2Service.Builder.create(this, "Service")
 *         .cluster(cluster)
 *         .memoryLimitMiB(1024)
 *         .taskImageOptions(NetworkLoadBalancedTaskImageOptions.builder()
 *                 .image(ContainerImage.fromRegistry("test"))
 *                 .environment(Map.of(
 *                         "TEST_ENVIRONMENT_VARIABLE1", "test environment variable 1 value",
 *                         "TEST_ENVIRONMENT_VARIABLE2", "test environment variable 2 value"))
 *                 .build())
 *         .desiredCount(2)
 *         .build();
 * </pre></blockquote>
 * <p>
 * <ul>
 * <li><code>NetworkLoadBalancedFargateService</code></li>
 * </ul>
 * <p>
 * <blockquote><pre>
 * Cluster cluster;
 * 
 * NetworkLoadBalancedFargateService loadBalancedFargateService = NetworkLoadBalancedFargateService.Builder.create(this, "Service")
 *         .cluster(cluster)
 *         .memoryLimitMiB(1024)
 *         .cpu(512)
 *         .taskImageOptions(NetworkLoadBalancedTaskImageOptions.builder()
 *                 .image(ContainerImage.fromRegistry("amazon/amazon-ecs-sample"))
 *                 .build())
 *         .build();
 * </pre></blockquote>
 * <p>
 * The CDK will create a new Amazon ECS cluster if you specify a VPC and omit <code>cluster</code>. If you deploy multiple services the CDK will only create one cluster per VPC.
 * <p>
 * If <code>cluster</code> and <code>vpc</code> are omitted, the CDK creates a new VPC with subnets in two Availability Zones and a cluster within this VPC.
 * <p>
 * If you specify the option <code>recordType</code> you can decide if you want the construct to use CNAME or Route53-Aliases as record sets.
 * <p>
 * Additionally, if more than one network target group is needed, instantiate one of the following:
 * <p>
 * <ul>
 * <li>NetworkMultipleTargetGroupsEc2Service</li>
 * </ul>
 * <p>
 * <blockquote><pre>
 * // Two network load balancers, each with their own listener and target group.
 * Cluster cluster;
 * 
 * NetworkMultipleTargetGroupsEc2Service loadBalancedEc2Service = NetworkMultipleTargetGroupsEc2Service.Builder.create(this, "Service")
 *         .cluster(cluster)
 *         .memoryLimitMiB(256)
 *         .taskImageOptions(NetworkLoadBalancedTaskImageProps.builder()
 *                 .image(ContainerImage.fromRegistry("amazon/amazon-ecs-sample"))
 *                 .build())
 *         .loadBalancers(List.of(NetworkLoadBalancerProps.builder()
 *                 .name("lb1")
 *                 .listeners(List.of(NetworkListenerProps.builder()
 *                         .name("listener1")
 *                         .build()))
 *                 .build(), NetworkLoadBalancerProps.builder()
 *                 .name("lb2")
 *                 .listeners(List.of(NetworkListenerProps.builder()
 *                         .name("listener2")
 *                         .build()))
 *                 .build()))
 *         .targetGroups(List.of(NetworkTargetProps.builder()
 *                 .containerPort(80)
 *                 .listener("listener1")
 *                 .build(), NetworkTargetProps.builder()
 *                 .containerPort(90)
 *                 .listener("listener2")
 *                 .build()))
 *         .build();
 * </pre></blockquote>
 * <p>
 * <ul>
 * <li>NetworkMultipleTargetGroupsFargateService</li>
 * </ul>
 * <p>
 * <blockquote><pre>
 * // Two network load balancers, each with their own listener and target group.
 * Cluster cluster;
 * 
 * NetworkMultipleTargetGroupsFargateService loadBalancedFargateService = NetworkMultipleTargetGroupsFargateService.Builder.create(this, "Service")
 *         .cluster(cluster)
 *         .memoryLimitMiB(512)
 *         .taskImageOptions(NetworkLoadBalancedTaskImageProps.builder()
 *                 .image(ContainerImage.fromRegistry("amazon/amazon-ecs-sample"))
 *                 .build())
 *         .loadBalancers(List.of(NetworkLoadBalancerProps.builder()
 *                 .name("lb1")
 *                 .listeners(List.of(NetworkListenerProps.builder()
 *                         .name("listener1")
 *                         .build()))
 *                 .build(), NetworkLoadBalancerProps.builder()
 *                 .name("lb2")
 *                 .listeners(List.of(NetworkListenerProps.builder()
 *                         .name("listener2")
 *                         .build()))
 *                 .build()))
 *         .targetGroups(List.of(NetworkTargetProps.builder()
 *                 .containerPort(80)
 *                 .listener("listener1")
 *                 .build(), NetworkTargetProps.builder()
 *                 .containerPort(90)
 *                 .listener("listener2")
 *                 .build()))
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h2>Queue Processing Services</h2>
 * <p>
 * To define a service that creates a queue and reads from that queue, instantiate one of the following:
 * <p>
 * <ul>
 * <li><code>QueueProcessingEc2Service</code></li>
 * </ul>
 * <p>
 * <blockquote><pre>
 * Cluster cluster;
 * 
 * QueueProcessingEc2Service queueProcessingEc2Service = QueueProcessingEc2Service.Builder.create(this, "Service")
 *         .cluster(cluster)
 *         .memoryLimitMiB(1024)
 *         .image(ContainerImage.fromRegistry("test"))
 *         .command(List.of("-c", "4", "amazon.com"))
 *         .enableLogging(false)
 *         .desiredTaskCount(2)
 *         .environment(Map.of(
 *                 "TEST_ENVIRONMENT_VARIABLE1", "test environment variable 1 value",
 *                 "TEST_ENVIRONMENT_VARIABLE2", "test environment variable 2 value"))
 *         .maxScalingCapacity(5)
 *         .containerName("test")
 *         .build();
 * </pre></blockquote>
 * <p>
 * <ul>
 * <li><code>QueueProcessingFargateService</code></li>
 * </ul>
 * <p>
 * <blockquote><pre>
 * Cluster cluster;
 * 
 * QueueProcessingFargateService queueProcessingFargateService = QueueProcessingFargateService.Builder.create(this, "Service")
 *         .cluster(cluster)
 *         .memoryLimitMiB(512)
 *         .image(ContainerImage.fromRegistry("test"))
 *         .command(List.of("-c", "4", "amazon.com"))
 *         .enableLogging(false)
 *         .desiredTaskCount(2)
 *         .environment(Map.of(
 *                 "TEST_ENVIRONMENT_VARIABLE1", "test environment variable 1 value",
 *                 "TEST_ENVIRONMENT_VARIABLE2", "test environment variable 2 value"))
 *         .maxScalingCapacity(5)
 *         .containerName("test")
 *         .build();
 * </pre></blockquote>
 * <p>
 * when queue not provided by user, CDK will create a primary queue and a dead letter queue with default redrive policy and attach permission to the task to be able to access the primary queue.
 * <p>
 * <h2>Scheduled Tasks</h2>
 * <p>
 * To define a task that runs periodically, there are 2 options:
 * <p>
 * <ul>
 * <li><code>ScheduledEc2Task</code></li>
 * </ul>
 * <p>
 * <blockquote><pre>
 * // Instantiate an Amazon EC2 Task to run at a scheduled interval
 * Cluster cluster;
 * 
 * ScheduledEc2Task ecsScheduledTask = ScheduledEc2Task.Builder.create(this, "ScheduledTask")
 *         .cluster(cluster)
 *         .scheduledEc2TaskImageOptions(ScheduledEc2TaskImageOptions.builder()
 *                 .image(ContainerImage.fromRegistry("amazon/amazon-ecs-sample"))
 *                 .memoryLimitMiB(256)
 *                 .environment(Map.of("name", "TRIGGER", "value", "CloudWatch Events"))
 *                 .build())
 *         .schedule(Schedule.expression("rate(1 minute)"))
 *         .enabled(true)
 *         .ruleName("sample-scheduled-task-rule")
 *         .build();
 * </pre></blockquote>
 * <p>
 * <ul>
 * <li><code>ScheduledFargateTask</code></li>
 * </ul>
 * <p>
 * <blockquote><pre>
 * Cluster cluster;
 * 
 * ScheduledFargateTask scheduledFargateTask = ScheduledFargateTask.Builder.create(this, "ScheduledFargateTask")
 *         .cluster(cluster)
 *         .scheduledFargateTaskImageOptions(ScheduledFargateTaskImageOptions.builder()
 *                 .image(ContainerImage.fromRegistry("amazon/amazon-ecs-sample"))
 *                 .memoryLimitMiB(512)
 *                 .build())
 *         .schedule(Schedule.expression("rate(1 minute)"))
 *         .platformVersion(FargatePlatformVersion.LATEST)
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h2>Additional Examples</h2>
 * <p>
 * In addition to using the constructs, users can also add logic to customize these constructs:
 * <p>
 * <h3>Configure HTTPS on an ApplicationLoadBalancedFargateService</h3>
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.services.route53.HostedZone;
 * import software.amazon.awscdk.services.certificatemanager.Certificate;
 * import software.amazon.awscdk.services.elasticloadbalancingv2.SslPolicy;
 * 
 * Vpc vpc;
 * Cluster cluster;
 * 
 * 
 * IHostedZone domainZone = HostedZone.fromLookup(this, "Zone", HostedZoneProviderProps.builder().domainName("example.com").build());
 * ICertificate certificate = Certificate.fromCertificateArn(this, "Cert", "arn:aws:acm:us-east-1:123456:certificate/abcdefg");
 * ApplicationLoadBalancedFargateService loadBalancedFargateService = ApplicationLoadBalancedFargateService.Builder.create(this, "Service")
 *         .vpc(vpc)
 *         .cluster(cluster)
 *         .certificate(certificate)
 *         .sslPolicy(SslPolicy.RECOMMENDED)
 *         .domainName("api.example.com")
 *         .domainZone(domainZone)
 *         .redirectHTTP(true)
 *         .taskImageOptions(ApplicationLoadBalancedTaskImageOptions.builder()
 *                 .image(ContainerImage.fromRegistry("amazon/amazon-ecs-sample"))
 *                 .build())
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h3>Add Schedule-Based Auto-Scaling to an ApplicationLoadBalancedFargateService</h3>
 * <p>
 * <blockquote><pre>
 * Cluster cluster;
 * 
 * ApplicationLoadBalancedFargateService loadBalancedFargateService = ApplicationLoadBalancedFargateService.Builder.create(this, "Service")
 *         .cluster(cluster)
 *         .memoryLimitMiB(1024)
 *         .desiredCount(1)
 *         .cpu(512)
 *         .taskImageOptions(ApplicationLoadBalancedTaskImageOptions.builder()
 *                 .image(ContainerImage.fromRegistry("amazon/amazon-ecs-sample"))
 *                 .build())
 *         .build();
 * 
 * ScalableTaskCount scalableTarget = loadBalancedFargateService.service.autoScaleTaskCount(EnableScalingProps.builder()
 *         .minCapacity(5)
 *         .maxCapacity(20)
 *         .build());
 * 
 * scalableTarget.scaleOnSchedule("DaytimeScaleDown", ScalingSchedule.builder()
 *         .schedule(Schedule.cron(CronOptions.builder().hour("8").minute("0").build()))
 *         .minCapacity(1)
 *         .build());
 * 
 * scalableTarget.scaleOnSchedule("EveningRushScaleUp", ScalingSchedule.builder()
 *         .schedule(Schedule.cron(CronOptions.builder().hour("20").minute("0").build()))
 *         .minCapacity(10)
 *         .build());
 * </pre></blockquote>
 * <p>
 * <h3>Add Metric-Based Auto-Scaling to an ApplicationLoadBalancedFargateService</h3>
 * <p>
 * <blockquote><pre>
 * Cluster cluster;
 * 
 * ApplicationLoadBalancedFargateService loadBalancedFargateService = ApplicationLoadBalancedFargateService.Builder.create(this, "Service")
 *         .cluster(cluster)
 *         .memoryLimitMiB(1024)
 *         .desiredCount(1)
 *         .cpu(512)
 *         .taskImageOptions(ApplicationLoadBalancedTaskImageOptions.builder()
 *                 .image(ContainerImage.fromRegistry("amazon/amazon-ecs-sample"))
 *                 .build())
 *         .build();
 * 
 * ScalableTaskCount scalableTarget = loadBalancedFargateService.service.autoScaleTaskCount(EnableScalingProps.builder()
 *         .minCapacity(1)
 *         .maxCapacity(20)
 *         .build());
 * 
 * scalableTarget.scaleOnCpuUtilization("CpuScaling", CpuUtilizationScalingProps.builder()
 *         .targetUtilizationPercent(50)
 *         .build());
 * 
 * scalableTarget.scaleOnMemoryUtilization("MemoryScaling", MemoryUtilizationScalingProps.builder()
 *         .targetUtilizationPercent(50)
 *         .build());
 * </pre></blockquote>
 * <p>
 * <h3>Change the default Deployment Controller</h3>
 * <p>
 * <blockquote><pre>
 * Cluster cluster;
 * 
 * ApplicationLoadBalancedFargateService loadBalancedFargateService = ApplicationLoadBalancedFargateService.Builder.create(this, "Service")
 *         .cluster(cluster)
 *         .memoryLimitMiB(1024)
 *         .desiredCount(1)
 *         .cpu(512)
 *         .taskImageOptions(ApplicationLoadBalancedTaskImageOptions.builder()
 *                 .image(ContainerImage.fromRegistry("amazon/amazon-ecs-sample"))
 *                 .build())
 *         .deploymentController(DeploymentController.builder()
 *                 .type(DeploymentControllerType.CODE_DEPLOY)
 *                 .build())
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h3>Deployment circuit breaker and rollback</h3>
 * <p>
 * Amazon ECS <a href="https://aws.amazon.com/tw/blogs/containers/announcing-amazon-ecs-deployment-circuit-breaker/">deployment circuit breaker</a>
 * automatically rolls back unhealthy service deployments without the need for manual intervention. Use <code>circuitBreaker</code> to enable
 * deployment circuit breaker and optionally enable <code>rollback</code> for automatic rollback. See <a href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/deployment-type-ecs.html">Using the deployment circuit breaker</a>
 * for more details.
 * <p>
 * <blockquote><pre>
 * Cluster cluster;
 * 
 * ApplicationLoadBalancedFargateService service = ApplicationLoadBalancedFargateService.Builder.create(this, "Service")
 *         .cluster(cluster)
 *         .memoryLimitMiB(1024)
 *         .desiredCount(1)
 *         .cpu(512)
 *         .taskImageOptions(ApplicationLoadBalancedTaskImageOptions.builder()
 *                 .image(ContainerImage.fromRegistry("amazon/amazon-ecs-sample"))
 *                 .build())
 *         .circuitBreaker(DeploymentCircuitBreaker.builder().rollback(true).build())
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h3>Set deployment configuration on QueueProcessingService</h3>
 * <p>
 * <blockquote><pre>
 * Cluster cluster;
 * 
 * QueueProcessingFargateService queueProcessingFargateService = QueueProcessingFargateService.Builder.create(this, "Service")
 *         .cluster(cluster)
 *         .memoryLimitMiB(512)
 *         .image(ContainerImage.fromRegistry("test"))
 *         .command(List.of("-c", "4", "amazon.com"))
 *         .enableLogging(false)
 *         .desiredTaskCount(2)
 *         .environment(Map.of())
 *         .maxScalingCapacity(5)
 *         .maxHealthyPercent(200)
 *         .minHealthyPercent(66)
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h3>Set taskSubnets and securityGroups for QueueProcessingFargateService</h3>
 * <p>
 * <blockquote><pre>
 * Vpc vpc;
 * SecurityGroup securityGroup;
 * 
 * QueueProcessingFargateService queueProcessingFargateService = QueueProcessingFargateService.Builder.create(this, "Service")
 *         .vpc(vpc)
 *         .memoryLimitMiB(512)
 *         .image(ContainerImage.fromRegistry("test"))
 *         .securityGroups(List.of(securityGroup))
 *         .taskSubnets(SubnetSelection.builder().subnetType(SubnetType.PRIVATE_ISOLATED).build())
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h3>Define tasks with public IPs for QueueProcessingFargateService</h3>
 * <p>
 * <blockquote><pre>
 * Vpc vpc;
 * 
 * QueueProcessingFargateService queueProcessingFargateService = QueueProcessingFargateService.Builder.create(this, "Service")
 *         .vpc(vpc)
 *         .memoryLimitMiB(512)
 *         .image(ContainerImage.fromRegistry("test"))
 *         .assignPublicIp(true)
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h3>Define tasks with custom queue parameters for QueueProcessingFargateService</h3>
 * <p>
 * <blockquote><pre>
 * Vpc vpc;
 * 
 * QueueProcessingFargateService queueProcessingFargateService = QueueProcessingFargateService.Builder.create(this, "Service")
 *         .vpc(vpc)
 *         .memoryLimitMiB(512)
 *         .image(ContainerImage.fromRegistry("test"))
 *         .maxReceiveCount(42)
 *         .retentionPeriod(Duration.days(7))
 *         .visibilityTimeout(Duration.minutes(5))
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h3>Set capacityProviderStrategies for QueueProcessingFargateService</h3>
 * <p>
 * <blockquote><pre>
 * Cluster cluster;
 * 
 * cluster.enableFargateCapacityProviders();
 * 
 * QueueProcessingFargateService queueProcessingFargateService = QueueProcessingFargateService.Builder.create(this, "Service")
 *         .cluster(cluster)
 *         .memoryLimitMiB(512)
 *         .image(ContainerImage.fromRegistry("test"))
 *         .capacityProviderStrategies(List.of(CapacityProviderStrategy.builder()
 *                 .capacityProvider("FARGATE_SPOT")
 *                 .weight(2)
 *                 .build(), CapacityProviderStrategy.builder()
 *                 .capacityProvider("FARGATE")
 *                 .weight(1)
 *                 .build()))
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h3>Set a custom container-level Healthcheck for QueueProcessingFargateService</h3>
 * <p>
 * <blockquote><pre>
 * Vpc vpc;
 * SecurityGroup securityGroup;
 * 
 * QueueProcessingFargateService queueProcessingFargateService = QueueProcessingFargateService.Builder.create(this, "Service")
 *         .vpc(vpc)
 *         .memoryLimitMiB(512)
 *         .image(ContainerImage.fromRegistry("test"))
 *         .healthCheck(HealthCheck.builder()
 *                 .command(List.of("CMD-SHELL", "curl -f http://localhost/ || exit 1"))
 *                 // the properties below are optional
 *                 .interval(Duration.minutes(30))
 *                 .retries(123)
 *                 .startPeriod(Duration.minutes(30))
 *                 .timeout(Duration.minutes(30))
 *                 .build())
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h3>Set capacityProviderStrategies for QueueProcessingEc2Service</h3>
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.core.*;
 * 
 * 
 * Vpc vpc = Vpc.Builder.create(this, "Vpc").maxAzs(1).build();
 * Cluster cluster = Cluster.Builder.create(this, "EcsCluster").vpc(vpc).build();
 * AutoScalingGroup autoScalingGroup = AutoScalingGroup.Builder.create(this, "asg")
 *         .vpc(vpc)
 *         .instanceType(InstanceType.of(InstanceClass.BURSTABLE2, InstanceSize.MICRO))
 *         .machineImage(EcsOptimizedImage.amazonLinux2())
 *         .build();
 * AsgCapacityProvider capacityProvider = AsgCapacityProvider.Builder.create(this, "provider")
 *         .autoScalingGroup(autoScalingGroup)
 *         .build();
 * cluster.addAsgCapacityProvider(capacityProvider);
 * 
 * QueueProcessingFargateService queueProcessingFargateService = QueueProcessingFargateService.Builder.create(this, "Service")
 *         .cluster(cluster)
 *         .memoryLimitMiB(512)
 *         .image(ContainerImage.fromRegistry("test"))
 *         .capacityProviderStrategies(List.of(CapacityProviderStrategy.builder()
 *                 .capacityProvider(capacityProvider.getCapacityProviderName())
 *                 .build()))
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h3>Select specific vpc subnets for ApplicationLoadBalancedFargateService</h3>
 * <p>
 * <blockquote><pre>
 * Cluster cluster;
 * 
 * ApplicationLoadBalancedFargateService loadBalancedFargateService = ApplicationLoadBalancedFargateService.Builder.create(this, "Service")
 *         .cluster(cluster)
 *         .memoryLimitMiB(1024)
 *         .desiredCount(1)
 *         .cpu(512)
 *         .taskImageOptions(ApplicationLoadBalancedTaskImageOptions.builder()
 *                 .image(ContainerImage.fromRegistry("amazon/amazon-ecs-sample"))
 *                 .build())
 *         .taskSubnets(SubnetSelection.builder()
 *                 .subnets(List.of(Subnet.fromSubnetId(this, "subnet", "VpcISOLATEDSubnet1Subnet80F07FA0")))
 *                 .build())
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h3>Set PlatformVersion for ScheduledFargateTask</h3>
 * <p>
 * <blockquote><pre>
 * Cluster cluster;
 * 
 * ScheduledFargateTask scheduledFargateTask = ScheduledFargateTask.Builder.create(this, "ScheduledFargateTask")
 *         .cluster(cluster)
 *         .scheduledFargateTaskImageOptions(ScheduledFargateTaskImageOptions.builder()
 *                 .image(ContainerImage.fromRegistry("amazon/amazon-ecs-sample"))
 *                 .memoryLimitMiB(512)
 *                 .build())
 *         .schedule(Schedule.expression("rate(1 minute)"))
 *         .platformVersion(FargatePlatformVersion.VERSION1_4)
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h3>Set SecurityGroups for ScheduledFargateTask</h3>
 * <p>
 * <blockquote><pre>
 * Vpc vpc = Vpc.Builder.create(this, "Vpc").maxAzs(1).build();
 * Cluster cluster = Cluster.Builder.create(this, "EcsCluster").vpc(vpc).build();
 * SecurityGroup securityGroup = SecurityGroup.Builder.create(this, "SG").vpc(vpc).build();
 * 
 * ScheduledFargateTask scheduledFargateTask = ScheduledFargateTask.Builder.create(this, "ScheduledFargateTask")
 *         .cluster(cluster)
 *         .scheduledFargateTaskImageOptions(ScheduledFargateTaskImageOptions.builder()
 *                 .image(ContainerImage.fromRegistry("amazon/amazon-ecs-sample"))
 *                 .memoryLimitMiB(512)
 *                 .build())
 *         .schedule(Schedule.expression("rate(1 minute)"))
 *         .securityGroups(List.of(securityGroup))
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h3>Use the REMOVE_DEFAULT_DESIRED_COUNT feature flag</h3>
 * <p>
 * The REMOVE_DEFAULT_DESIRED_COUNT feature flag is used to override the default desiredCount that is autogenerated by the CDK. This will set the desiredCount of any service created by any of the following constructs to be undefined.
 * <p>
 * <ul>
 * <li>ApplicationLoadBalancedEc2Service</li>
 * <li>ApplicationLoadBalancedFargateService</li>
 * <li>NetworkLoadBalancedEc2Service</li>
 * <li>NetworkLoadBalancedFargateService</li>
 * <li>QueueProcessingEc2Service</li>
 * <li>QueueProcessingFargateService</li>
 * </ul>
 * <p>
 * If a desiredCount is not passed in as input to the above constructs, CloudFormation will either create a new service to start up with a desiredCount of 1, or update an existing service to start up with the same desiredCount as prior to the update.
 * <p>
 * To enable the feature flag, ensure that the REMOVE_DEFAULT_DESIRED_COUNT flag within an application stack context is set to true, like so:
 * <p>
 * <blockquote><pre>
 * Stack stack;
 * 
 * stack.node.setContext(ECS_REMOVE_DEFAULT_DESIRED_COUNT, true);
 * </pre></blockquote>
 * <p>
 * The following is an example of an application with the REMOVE_DEFAULT_DESIRED_COUNT feature flag enabled:
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.core.App;
 * import software.amazon.awscdk.core.Stack;
 * import software.amazon.awscdk.core.*;
 * import software.amazon.awscdk.core.*;
 * import software.amazon.awscdk.core.*;
 * import software.amazon.awscdk.core.*;
 * import path.*;
 * 
 * App app = new App();
 * 
 * Stack stack = new Stack(app, "aws-ecs-patterns-queue");
 * stack.node.setContext(ECS_REMOVE_DEFAULT_DESIRED_COUNT, true);
 * 
 * Vpc vpc = Vpc.Builder.create(stack, "VPC")
 *         .maxAzs(2)
 *         .build();
 * 
 * QueueProcessingFargateService.Builder.create(stack, "QueueProcessingService")
 *         .vpc(vpc)
 *         .memoryLimitMiB(512)
 *         .image(new AssetImage(join(__dirname, "..", "sqs-reader")))
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h3>Deploy application and metrics sidecar</h3>
 * <p>
 * The following is an example of deploying an application along with a metrics sidecar container that utilizes <code>dockerLabels</code> for discovery:
 * <p>
 * <blockquote><pre>
 * Cluster cluster;
 * Vpc vpc;
 * 
 * ApplicationLoadBalancedFargateService service = ApplicationLoadBalancedFargateService.Builder.create(this, "Service")
 *         .cluster(cluster)
 *         .vpc(vpc)
 *         .desiredCount(1)
 *         .taskImageOptions(ApplicationLoadBalancedTaskImageOptions.builder()
 *                 .image(ContainerImage.fromRegistry("amazon/amazon-ecs-sample"))
 *                 .dockerLabels(Map.of(
 *                         "application.label.one", "first_label",
 *                         "application.label.two", "second_label"))
 *                 .build())
 *         .build();
 * 
 * service.taskDefinition.addContainer("Sidecar", ContainerDefinitionOptions.builder()
 *         .image(ContainerImage.fromRegistry("example/metrics-sidecar"))
 *         .build());
 * </pre></blockquote>
 * <p>
 * <h3>Select specific load balancer name ApplicationLoadBalancedFargateService</h3>
 * <p>
 * <blockquote><pre>
 * Cluster cluster;
 * 
 * ApplicationLoadBalancedFargateService loadBalancedFargateService = ApplicationLoadBalancedFargateService.Builder.create(this, "Service")
 *         .cluster(cluster)
 *         .memoryLimitMiB(1024)
 *         .desiredCount(1)
 *         .cpu(512)
 *         .taskImageOptions(ApplicationLoadBalancedTaskImageOptions.builder()
 *                 .image(ContainerImage.fromRegistry("amazon/amazon-ecs-sample"))
 *                 .build())
 *         .taskSubnets(SubnetSelection.builder()
 *                 .subnets(List.of(Subnet.fromSubnetId(this, "subnet", "VpcISOLATEDSubnet1Subnet80F07FA0")))
 *                 .build())
 *         .loadBalancerName("application-lb-name")
 *         .build();
 * </pre></blockquote>
 */
package software.amazon.awscdk.services.ecs.patterns;
