/**
 * <h1>Amazon Cognito Identity Pool Construct Library</h1>
 * <p>
 * <blockquote>
 * <p>
 * <strong>Identity Pools are in a separate module while the API is being stabilized. Once we stabilize the module, they will</strong>
 * <strong>be included into the stable <a href="../aws-cognito">aws-cognito</a> library. Please provide feedback on this experience by</strong>
 * <strong>creating an <a href="https://github.com/aws/aws-cdk/issues/new/choose">issue here</a></strong>
 * <p>
 * </blockquote>
 * <p>
 * <a href="https://docs.aws.amazon.com/cognito/latest/developerguide/cognito-identity.html">Amazon Cognito Identity Pools</a> enable you to grant your users access to other AWS services.
 * <p>
 * Identity Pools are one of the two main components of <a href="https://docs.aws.amazon.com/cognito/latest/developerguide/what-is-amazon-cognito.html">Amazon Cognito</a>, which provides authentication, authorization, and
 * user management for your web and mobile apps. Your users can sign in directly with a user name and password, or through
 * a third party such as Facebook, Amazon, Google or Apple.
 * <p>
 * The other main component in Amazon Cognito is <a href="https://docs.aws.amazon.com/cognito/latest/developerguide/cognito-user-identity-pools.html">user pools</a>. User Pools are user directories that provide sign-up and
 * sign-in options for your app users.
 * <p>
 * This module is part of the <a href="https://github.com/aws/aws-cdk">AWS Cloud Development Kit</a> project.
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.services.cognito.identitypool.IdentityPool;
 * import software.amazon.awscdk.services.cognito.identitypool.UserPoolAuthenticationProvider;
 * </pre></blockquote>
 * <p>
 * <h2>Table of Contents</h2>
 * <p>
 * <ul>
 * <li><a href="#identity-pools">Identity Pools</a>
 * <p>
 * <ul>
 * <li><a href="#authenticated-and-unauthenticated-identities">Authenticated and Unauthenticated Identities</a></li>
 * <li><a href="#authentication-providers">Authentication Providers</a>
 * <p>
 * <ul>
 * <li><a href="#user-pool-authentication-provider">User Pool Authentication Provider</a></li>
 * <li><a href="#server-side-token-check">Server Side Token Check</a></li>
 * <li><a href="#associating-an-external-provider-directly">Associating an External Provider Directly</a></li>
 * <li><a href="#openid-connect-and-saml">OpenIdConnect and Saml</a></li>
 * <li><a href="#custom-providers">Custom Providers</a></li>
 * </ul></li>
 * <li><a href="#role-mapping">Role Mapping</a>
 * <p>
 * <ul>
 * <li><a href="#provider-urls">Provider Urls</a></li>
 * </ul></li>
 * <li><a href="#authentication-flow">Authentication Flow</a></li>
 * <li><a href="#cognito-sync">Cognito Sync</a></li>
 * <li><a href="#importing-identity-pools">Importing Identity Pools</a></li>
 * </ul></li>
 * </ul>
 * <p>
 * <h2>Identity Pools</h2>
 * <p>
 * Identity pools provide temporary AWS credentials for users who are guests (unauthenticated) and for users who have been
 * authenticated and received a token. An identity pool is a store of user identity data specific to an account.
 * <p>
 * Identity pools can be used in conjunction with Cognito User Pools or by accessing external federated identity providers
 * directly. Learn more at <a href="https://docs.aws.amazon.com/cognito/latest/developerguide/cognito-identity.html">Amazon Cognito Identity Pools</a>.
 * <p>
 * <h3>Authenticated and Unauthenticated Identities</h3>
 * <p>
 * Identity pools define two types of identities: authenticated(<code>user</code>) and unauthenticated (<code>guest</code>). Every identity in
 * an identity pool is either authenticated or unauthenticated. Each identity pool has a default role for authenticated
 * identities, and a default role for unauthenticated identities. Absent other overriding rules (see below), these are the
 * roles that will be assumed by the corresponding users in the authentication process.
 * <p>
 * A basic Identity Pool with minimal configuration has no required props, with default authenticated (user) and
 * unauthenticated (guest) roles applied to the identity pool:
 * <p>
 * <blockquote><pre>
 * new IdentityPool(this, "myIdentityPool");
 * </pre></blockquote>
 * <p>
 * By default, both the authenticated and unauthenticated roles will have no permissions attached. Grant permissions
 * to roles using the public <code>authenticatedRole</code> and <code>unauthenticatedRole</code> properties:
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.core.*;
 * Table table;
 * 
 * 
 * IdentityPool identityPool = new IdentityPool(this, "myIdentityPool");
 * 
 * // Grant permissions to authenticated users
 * table.grantReadWriteData(identityPool.getAuthenticatedRole());
 * // Grant permissions to unauthenticated guest users
 * table.grantReadData(identityPool.getUnauthenticatedRole());
 * 
 * //Or add policy statements straight to the role
 * identityPool.authenticatedRole.addToPrincipalPolicy(PolicyStatement.Builder.create()
 *         .effect(Effect.ALLOW)
 *         .actions(List.of("dynamodb:*"))
 *         .resources(List.of("*"))
 *         .build());
 * </pre></blockquote>
 * <p>
 * The default roles can also be supplied in <code>IdentityPoolProps</code>:
 * <p>
 * <blockquote><pre>
 * Stack stack = new Stack();
 * Role authenticatedRole = Role.Builder.create(this, "authRole")
 *         .assumedBy(new ServicePrincipal("service.amazonaws.com"))
 *         .build();
 * Role unauthenticatedRole = Role.Builder.create(this, "unauthRole")
 *         .assumedBy(new ServicePrincipal("service.amazonaws.com"))
 *         .build();
 * IdentityPool identityPool = IdentityPool.Builder.create(this, "TestIdentityPoolActions")
 *         .authenticatedRole(authenticatedRole)
 *         .unauthenticatedRole(unauthenticatedRole)
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h3>Authentication Providers</h3>
 * <p>
 * Authenticated identities belong to users who are authenticated by a public login provider (Amazon Cognito user pools,
 * Login with Amazon, Sign in with Apple, Facebook, Google, SAML, or any OpenID Connect Providers) or a developer provider
 * (your own backend authentication process).
 * <p>
 * <a href="https://docs.aws.amazon.com/cognito/latest/developerguide/external-identity-providers.html">Authentication providers</a> can be associated with an Identity Pool by first associating them with a Cognito User Pool or by
 * associating the provider directly with the identity pool.
 * <p>
 * <h4>User Pool Authentication Provider</h4>
 * <p>
 * In order to attach a user pool to an identity pool as an authentication provider, the identity pool needs properties
 * from both the user pool and the user pool client. For this reason identity pools use a <code>UserPoolAuthenticationProvider</code>
 * to gather the necessary properties from the user pool constructs.
 * <p>
 * <blockquote><pre>
 * UserPool userPool = new UserPool(this, "Pool");
 * 
 * IdentityPool.Builder.create(this, "myidentitypool")
 *         .identityPoolName("myidentitypool")
 *         .authenticationProviders(IdentityPoolAuthenticationProviders.builder()
 *                 .userPools(List.of(UserPoolAuthenticationProvider.Builder.create().userPool(userPool).build()))
 *                 .build())
 *         .build();
 * </pre></blockquote>
 * <p>
 * User pools can also be associated with an identity pool after instantiation. The Identity Pool's <code>addUserPoolAuthentication</code> method
 * returns the User Pool Client that has been created:
 * <p>
 * <blockquote><pre>
 * IdentityPool identityPool;
 * 
 * UserPool userPool = new UserPool(this, "Pool");
 * void userPoolClient = identityPool.addUserPoolAuthentication(UserPoolAuthenticationProvider.Builder.create()
 *         .userPool(userPool)
 *         .build());
 * </pre></blockquote>
 * <p>
 * <h4>Server Side Token Check</h4>
 * <p>
 * With the <code>IdentityPool</code> CDK Construct, by default the pool is configured to check with the integrated user pools to
 * make sure that the user has not been globally signed out or deleted before the identity pool provides an OIDC token or
 * AWS credentials for the user.
 * <p>
 * If the user is signed out or deleted, the identity pool will return a 400 Not Authorized error. This setting can be
 * disabled, however, in several ways.
 * <p>
 * Setting <code>disableServerSideTokenCheck</code> to true will change the default behavior to no server side token check. Learn
 * more <a href="https://docs.aws.amazon.com/cognitoidentity/latest/APIReference/API_CognitoIdentityProvider.html#CognitoIdentity-Type-CognitoIdentityProvider-ServerSideTokenCheck">here</a>:
 * <p>
 * <blockquote><pre>
 * IdentityPool identityPool;
 * 
 * UserPool userPool = new UserPool(this, "Pool");
 * identityPool.addUserPoolAuthentication(UserPoolAuthenticationProvider.Builder.create()
 *         .userPool(userPool)
 *         .disableServerSideTokenCheck(true)
 *         .build());
 * </pre></blockquote>
 * <p>
 * <h4>Associating an External Provider Directly</h4>
 * <p>
 * One or more <a href="https://docs.aws.amazon.com/cognito/latest/developerguide/external-identity-providers.html">external identity providers</a> can be associated with an identity pool directly using
 * <code>authenticationProviders</code>:
 * <p>
 * <blockquote><pre>
 * IdentityPool.Builder.create(this, "myidentitypool")
 *         .identityPoolName("myidentitypool")
 *         .authenticationProviders(IdentityPoolAuthenticationProviders.builder()
 *                 .amazon(IdentityPoolAmazonLoginProvider.builder()
 *                         .appId("amzn1.application.12312k3j234j13rjiwuenf")
 *                         .build())
 *                 .facebook(IdentityPoolFacebookLoginProvider.builder()
 *                         .appId("1234567890123")
 *                         .build())
 *                 .google(IdentityPoolGoogleLoginProvider.builder()
 *                         .clientId("12345678012.apps.googleusercontent.com")
 *                         .build())
 *                 .apple(IdentityPoolAppleLoginProvider.builder()
 *                         .servicesId("com.myappleapp.auth")
 *                         .build())
 *                 .twitter(IdentityPoolTwitterLoginProvider.builder()
 *                         .consumerKey("my-twitter-id")
 *                         .consumerSecret("my-twitter-secret")
 *                         .build())
 *                 .build())
 *         .build();
 * </pre></blockquote>
 * <p>
 * To associate more than one provider of the same type with the identity pool, use User
 * Pools, OpenIdConnect, or SAML. Only one provider per external service can be attached directly to the identity pool.
 * <p>
 * <h4>OpenId Connect and Saml</h4>
 * <p>
 * <a href="https://docs.aws.amazon.com/cognito/latest/developerguide/open-id.html">OpenID Connect</a> is an open standard for
 * authentication that is supported by a number of login providers. Amazon Cognito supports linking of identities with
 * OpenID Connect providers that are configured through <a href="http://aws.amazon.com/iam/">AWS Identity and Access Management</a>.
 * <p>
 * An identity provider that supports <a href="https://docs.aws.amazon.com/cognito/latest/developerguide/saml-identity-provider.html">Security Assertion Markup Language 2.0 (SAML 2.0)</a> can be used to provide a simple
 * onboarding flow for users. The SAML-supporting identity provider specifies the IAM roles that can be assumed by users
 * so that different users can be granted different sets of permissions. Associating an OpenId Connect or Saml provider
 * with an identity pool:
 * <p>
 * <blockquote><pre>
 * OpenIdConnectProvider openIdConnectProvider;
 * SamlProvider samlProvider;
 * 
 * 
 * IdentityPool.Builder.create(this, "myidentitypool")
 *         .identityPoolName("myidentitypool")
 *         .authenticationProviders(IdentityPoolAuthenticationProviders.builder()
 *                 .openIdConnectProviders(List.of(openIdConnectProvider))
 *                 .samlProviders(List.of(samlProvider))
 *                 .build())
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h4>Custom Providers</h4>
 * <p>
 * The identity pool's behavior can be customized further using custom <a href="https://docs.aws.amazon.com/cognito/latest/developerguide/developer-authenticated-identities.html">developer authenticated identities</a>.
 * With developer authenticated identities, users can be registered and authenticated via an existing authentication
 * process while still using Amazon Cognito to synchronize user data and access AWS resources.
 * <p>
 * Like the supported external providers, though, only one custom provider can be directly associated with the identity
 * pool.
 * <p>
 * <blockquote><pre>
 * OpenIdConnectProvider openIdConnectProvider;
 * 
 * IdentityPool.Builder.create(this, "myidentitypool")
 *         .identityPoolName("myidentitypool")
 *         .authenticationProviders(IdentityPoolAuthenticationProviders.builder()
 *                 .google(IdentityPoolGoogleLoginProvider.builder()
 *                         .clientId("12345678012.apps.googleusercontent.com")
 *                         .build())
 *                 .openIdConnectProviders(List.of(openIdConnectProvider))
 *                 .customProvider("my-custom-provider.example.com")
 *                 .build())
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h3>Role Mapping</h3>
 * <p>
 * In addition to setting default roles for authenticated and unauthenticated users, identity pools can also be used to
 * define rules to choose the role for each user based on claims in the user's ID token by using Role Mapping. When using
 * role mapping, it's important to be aware of some of the permissions the role will need. An in depth
 * review of roles and role mapping can be found <a href="https://docs.aws.amazon.com/cognito/latest/developerguide/role-based-access-control.html">here</a>.
 * <p>
 * Using a <a href="https://docs.aws.amazon.com/cognito/latest/developerguide/role-based-access-control.html#using-tokens-to-assign-roles-to-users">token-based approach</a> to role mapping will allow mapped roles to be passed through the <code>cognito:roles</code> or
 * <code>cognito:preferred_role</code> claims from the identity provider:
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.services.cognito.identitypool.IdentityPoolProviderUrl;
 * 
 * 
 * IdentityPool.Builder.create(this, "myidentitypool")
 *         .identityPoolName("myidentitypool")
 *         .roleMappings(List.of(IdentityPoolRoleMapping.builder()
 *                 .providerUrl(IdentityPoolProviderUrl.AMAZON)
 *                 .useToken(true)
 *                 .build()))
 *         .build();
 * </pre></blockquote>
 * <p>
 * Using a rule-based approach to role mapping allows roles to be assigned based on custom claims passed from the identity  provider:
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.services.cognito.identitypool.IdentityPoolProviderUrl;
 * import software.amazon.awscdk.services.cognito.identitypool.RoleMappingMatchType;
 * 
 * Role adminRole;
 * Role nonAdminRole;
 * 
 * IdentityPool.Builder.create(this, "myidentitypool")
 *         .identityPoolName("myidentitypool")
 *         // Assign specific roles to users based on whether or not the custom admin claim is passed from the identity provider
 *         .roleMappings(List.of(IdentityPoolRoleMapping.builder()
 *                 .providerUrl(IdentityPoolProviderUrl.AMAZON)
 *                 .rules(List.of(RoleMappingRule.builder()
 *                         .claim("custom:admin")
 *                         .claimValue("admin")
 *                         .mappedRole(adminRole)
 *                         .build(), RoleMappingRule.builder()
 *                         .claim("custom:admin")
 *                         .claimValue("admin")
 *                         .matchType(RoleMappingMatchType.NOTEQUAL)
 *                         .mappedRole(nonAdminRole)
 *                         .build()))
 *                 .build()))
 *         .build();
 * </pre></blockquote>
 * <p>
 * Role mappings can also be added after instantiation with the Identity Pool's <code>addRoleMappings</code> method:
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.services.cognito.identitypool.IdentityPoolRoleMapping;
 * 
 * IdentityPool identityPool;
 * IdentityPoolRoleMapping myAddedRoleMapping1;
 * IdentityPoolRoleMapping myAddedRoleMapping2;
 * IdentityPoolRoleMapping myAddedRoleMapping3;
 * 
 * 
 * identityPool.addRoleMappings(myAddedRoleMapping1, myAddedRoleMapping2, myAddedRoleMapping3);
 * </pre></blockquote>
 * <p>
 * <h4>Provider Urls</h4>
 * <p>
 * Role mappings must be associated with the url of an Identity Provider which can be supplied
 * <code>IdentityPoolProviderUrl</code>. Supported Providers have static Urls that can be used:
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.services.cognito.identitypool.IdentityPoolProviderUrl;
 * 
 * 
 * IdentityPool.Builder.create(this, "myidentitypool")
 *         .identityPoolName("myidentitypool")
 *         .roleMappings(List.of(IdentityPoolRoleMapping.builder()
 *                 .providerUrl(IdentityPoolProviderUrl.FACEBOOK)
 *                 .useToken(true)
 *                 .build()))
 *         .build();
 * </pre></blockquote>
 * <p>
 * For identity providers that don't have static Urls, a custom Url or User Pool Client Url can be supplied:
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.services.cognito.identitypool.IdentityPoolProviderUrl;
 * 
 * 
 * IdentityPool.Builder.create(this, "myidentitypool")
 *         .identityPoolName("myidentitypool")
 *         .roleMappings(List.of(IdentityPoolRoleMapping.builder()
 *                 .providerUrl(IdentityPoolProviderUrl.userPool("cognito-idp.my-idp-region.amazonaws.com/my-idp-region_abcdefghi:app_client_id"))
 *                 .useToken(true)
 *                 .build(), IdentityPoolRoleMapping.builder()
 *                 .providerUrl(IdentityPoolProviderUrl.custom("my-custom-provider.com"))
 *                 .useToken(true)
 *                 .build()))
 *         .build();
 * </pre></blockquote>
 * <p>
 * See <a href="https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cognito-identitypoolroleattachment-rolemapping.html#cfn-cognito-identitypoolroleattachment-rolemapping-identityprovider">here</a> for more information.
 * <p>
 * <h3>Authentication Flow</h3>
 * <p>
 * Identity Pool <a href="https://docs.aws.amazon.com/cognito/latest/developerguide/authentication-flow.html">Authentication Flow</a> defaults to the enhanced, simplified flow. The Classic (basic) Authentication Flow
 * can also be implemented using <code>allowClassicFlow</code>:
 * <p>
 * <blockquote><pre>
 * IdentityPool.Builder.create(this, "myidentitypool")
 *         .identityPoolName("myidentitypool")
 *         .allowClassicFlow(true)
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h3>Cognito Sync</h3>
 * <p>
 * It's now recommended to integrate <a href="https://aws.amazon.com/appsync/">AWS AppSync</a> for synchronizing app data across devices, so
 * Cognito Sync features like <code>PushSync</code>, <code>CognitoEvents</code>, and <code>CognitoStreams</code> are not a part of <code>IdentityPool</code>. More
 * information can be found <a href="https://docs.aws.amazon.com/cognito/latest/developerguide/cognito-sync.html">here</a>.
 * <p>
 * <h3>Importing Identity Pools</h3>
 * <p>
 * You can import existing identity pools into your stack using Identity Pool static methods with the Identity Pool Id or
 * Arn:
 * <p>
 * <blockquote><pre>
 * IdentityPool.fromIdentityPoolId(this, "my-imported-identity-pool", "us-east-1:dj2823ryiwuhef937");
 * IdentityPool.fromIdentityPoolArn(this, "my-imported-identity-pool", "arn:aws:cognito-identity:us-east-1:123456789012:identitypool/us-east-1:dj2823ryiwuhef937");
 * </pre></blockquote>
 */
package software.amazon.awscdk.services.cognito.identitypool;
