/**
 * <h1>Amazon EC2 Auto Scaling Construct Library</h1>
 * <p>
 * This module is part of the <a href="https://github.com/aws/aws-cdk">AWS Cloud Development Kit</a> project.
 * <p>
 * <h2>Auto Scaling Group</h2>
 * <p>
 * An <code>AutoScalingGroup</code> represents a number of instances on which you run your code. You
 * pick the size of the fleet, the instance type and the OS image:
 * <p>
 * <blockquote><pre>
 * Vpc vpc;
 * 
 * 
 * AutoScalingGroup.Builder.create(this, "ASG")
 *         .vpc(vpc)
 *         .instanceType(InstanceType.of(InstanceClass.BURSTABLE2, InstanceSize.MICRO))
 *         .machineImage(new AmazonLinuxImage())
 *         .build();
 * </pre></blockquote>
 * <p>
 * NOTE: AutoScalingGroup has an property called <code>allowAllOutbound</code> (allowing the instances to contact the
 * internet) which is set to <code>true</code> by default. Be sure to set this to <code>false</code>  if you don't want
 * your instances to be able to start arbitrary connections. Alternatively, you can specify an existing security
 * group to attach to the instances that are launched, rather than have the group create a new one.
 * <p>
 * <blockquote><pre>
 * Vpc vpc;
 * 
 * 
 * SecurityGroup mySecurityGroup = SecurityGroup.Builder.create(this, "SecurityGroup").vpc(vpc).build();
 * AutoScalingGroup.Builder.create(this, "ASG")
 *         .vpc(vpc)
 *         .instanceType(InstanceType.of(InstanceClass.BURSTABLE2, InstanceSize.MICRO))
 *         .machineImage(new AmazonLinuxImage())
 *         .securityGroup(mySecurityGroup)
 *         .build();
 * </pre></blockquote>
 * <p>
 * Alternatively you can create an <code>AutoScalingGroup</code> from a <code>LaunchTemplate</code>:
 * <p>
 * <blockquote><pre>
 * Vpc vpc;
 * LaunchTemplate launchTemplate;
 * 
 * 
 * AutoScalingGroup.Builder.create(this, "ASG")
 *         .vpc(vpc)
 *         .launchTemplate(launchTemplate)
 *         .build();
 * </pre></blockquote>
 * <p>
 * To launch a mixture of Spot and on-demand instances, and/or with multiple instance types, you can create an <code>AutoScalingGroup</code> from a MixedInstancesPolicy:
 * <p>
 * <blockquote><pre>
 * Vpc vpc;
 * LaunchTemplate launchTemplate1;
 * LaunchTemplate launchTemplate2;
 * 
 * 
 * AutoScalingGroup.Builder.create(this, "ASG")
 *         .vpc(vpc)
 *         .mixedInstancesPolicy(MixedInstancesPolicy.builder()
 *                 .instancesDistribution(InstancesDistribution.builder()
 *                         .onDemandPercentageAboveBaseCapacity(50)
 *                         .build())
 *                 .launchTemplate(launchTemplate1)
 *                 .launchTemplateOverrides(List.of(LaunchTemplateOverrides.builder().instanceType(new InstanceType("t3.micro")).build(), LaunchTemplateOverrides.builder().instanceType(new InstanceType("t3a.micro")).build(), LaunchTemplateOverrides.builder().instanceType(new InstanceType("t4g.micro")).launchTemplate(launchTemplate2).build()))
 *                 .build())
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h2>Machine Images (AMIs)</h2>
 * <p>
 * AMIs control the OS that gets launched when you start your EC2 instance. The EC2
 * library contains constructs to select the AMI you want to use.
 * <p>
 * Depending on the type of AMI, you select it a different way.
 * <p>
 * The latest version of Amazon Linux and Microsoft Windows images are
 * selectable by instantiating one of these classes:
 * <p>
 * <blockquote><pre>
 * // Pick a Windows edition to use
 * WindowsImage windows = new WindowsImage(WindowsVersion.WINDOWS_SERVER_2019_ENGLISH_FULL_BASE);
 * 
 * // Pick the right Amazon Linux edition. All arguments shown are optional
 * // and will default to these values when omitted.
 * AmazonLinuxImage amznLinux = AmazonLinuxImage.Builder.create()
 *         .generation(AmazonLinuxGeneration.AMAZON_LINUX)
 *         .edition(AmazonLinuxEdition.STANDARD)
 *         .virtualization(AmazonLinuxVirt.HVM)
 *         .storage(AmazonLinuxStorage.GENERAL_PURPOSE)
 *         .build();
 * 
 * // For other custom (Linux) images, instantiate a `GenericLinuxImage` with
 * // a map giving the AMI to in for each region:
 * 
 * GenericLinuxImage linux = new GenericLinuxImage(Map.of(
 *         "us-east-1", "ami-97785bed",
 *         "eu-west-1", "ami-12345678"));
 * </pre></blockquote>
 * <p>
 * <blockquote>
 * <p>
 * NOTE: The Amazon Linux images selected will be cached in your <code>cdk.json</code>, so that your
 * AutoScalingGroups don't automatically change out from under you when you're making unrelated
 * changes. To update to the latest version of Amazon Linux, remove the cache entry from the <code>context</code>
 * section of your <code>cdk.json</code>.
 * <p>
 * We will add command-line options to make this step easier in the future.
 * <p>
 * </blockquote>
 * <p>
 * <h2>AutoScaling Instance Counts</h2>
 * <p>
 * AutoScalingGroups make it possible to raise and lower the number of instances in the group,
 * in response to (or in advance of) changes in workload.
 * <p>
 * When you create your AutoScalingGroup, you specify a <code>minCapacity</code> and a
 * <code>maxCapacity</code>. AutoScaling policies that respond to metrics will never go higher
 * or lower than the indicated capacity (but scheduled scaling actions might, see
 * below).
 * <p>
 * There are three ways to scale your capacity:
 * <p>
 * <ul>
 * <li><strong>In response to a metric</strong> (also known as step scaling); for example, you
 * might want to scale out if the CPU usage across your cluster starts to rise,
 * and scale in when it drops again.</li>
 * <li><strong>By trying to keep a certain metric around a given value</strong> (also known as
 * target tracking scaling); you might want to automatically scale out and in to
 * keep your CPU usage around 50%.</li>
 * <li><strong>On a schedule</strong>; you might want to organize your scaling around traffic
 * flows you expect, by scaling out in the morning and scaling in in the
 * evening.</li>
 * </ul>
 * <p>
 * The general pattern of autoscaling will look like this:
 * <p>
 * <blockquote><pre>
 * Vpc vpc;
 * InstanceType instanceType;
 * IMachineImage machineImage;
 * 
 * 
 * AutoScalingGroup autoScalingGroup = AutoScalingGroup.Builder.create(this, "ASG")
 *         .vpc(vpc)
 *         .instanceType(instanceType)
 *         .machineImage(machineImage)
 * 
 *         .minCapacity(5)
 *         .maxCapacity(100)
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h3>Step Scaling</h3>
 * <p>
 * This type of scaling scales in and out in deterministics steps that you
 * configure, in response to metric values. For example, your scaling strategy to
 * scale in response to a metric that represents your average worker pool usage
 * might look like this:
 * <p>
 * <blockquote><pre>
 *  Scaling        -1          (no change)          +1       +3
 *             │        │                       │        │        │
 *             ├────────┼───────────────────────┼────────┼────────┤
 *             │        │                       │        │        │
 * Worker use  0%      10%                     50%       70%     100%
 * </pre></blockquote>
 * <p>
 * (Note that this is not necessarily a recommended scaling strategy, but it's
 * a possible one. You will have to determine what thresholds are right for you).
 * <p>
 * Note that in order to set up this scaling strategy, you will have to emit a
 * metric representing your worker utilization from your instances. After that,
 * you would configure the scaling something like this:
 * <p>
 * <blockquote><pre>
 * AutoScalingGroup autoScalingGroup;
 * 
 * 
 * Metric workerUtilizationMetric = Metric.Builder.create()
 *         .namespace("MyService")
 *         .metricName("WorkerUtilization")
 *         .build();
 * 
 * autoScalingGroup.scaleOnMetric("ScaleToCPU", BasicStepScalingPolicyProps.builder()
 *         .metric(workerUtilizationMetric)
 *         .scalingSteps(List.of(ScalingInterval.builder().upper(10).change(-1).build(), ScalingInterval.builder().lower(50).change(+1).build(), ScalingInterval.builder().lower(70).change(+3).build()))
 * 
 *         // Change this to AdjustmentType.PERCENT_CHANGE_IN_CAPACITY to interpret the
 *         // 'change' numbers before as percentages instead of capacity counts.
 *         .adjustmentType(AdjustmentType.CHANGE_IN_CAPACITY)
 *         .build());
 * </pre></blockquote>
 * <p>
 * The AutoScaling construct library will create the required CloudWatch alarms and
 * AutoScaling policies for you.
 * <p>
 * <h3>Target Tracking Scaling</h3>
 * <p>
 * This type of scaling scales in and out in order to keep a metric around a value
 * you prefer. There are four types of predefined metrics you can track, or you can
 * choose to track a custom metric. If you do choose to track a custom metric,
 * be aware that the metric has to represent instance utilization in some way
 * (AutoScaling will scale out if the metric is higher than the target, and scale
 * in if the metric is lower than the target).
 * <p>
 * If you configure multiple target tracking policies, AutoScaling will use the
 * one that yields the highest capacity.
 * <p>
 * The following example scales to keep the CPU usage of your instances around
 * 50% utilization:
 * <p>
 * <blockquote><pre>
 * AutoScalingGroup autoScalingGroup;
 * 
 * 
 * autoScalingGroup.scaleOnCpuUtilization("KeepSpareCPU", CpuUtilizationScalingProps.builder()
 *         .targetUtilizationPercent(50)
 *         .build());
 * </pre></blockquote>
 * <p>
 * To scale on average network traffic in and out of your instances:
 * <p>
 * <blockquote><pre>
 * AutoScalingGroup autoScalingGroup;
 * 
 * 
 * autoScalingGroup.scaleOnIncomingBytes("LimitIngressPerInstance", NetworkUtilizationScalingProps.builder()
 *         .targetBytesPerSecond(10 * 1024 * 1024)
 *         .build());
 * autoScalingGroup.scaleOnOutgoingBytes("LimitEgressPerInstance", NetworkUtilizationScalingProps.builder()
 *         .targetBytesPerSecond(10 * 1024 * 1024)
 *         .build());
 * </pre></blockquote>
 * <p>
 * To scale on the average request count per instance (only works for
 * AutoScalingGroups that have been attached to Application Load
 * Balancers):
 * <p>
 * <blockquote><pre>
 * AutoScalingGroup autoScalingGroup;
 * 
 * 
 * autoScalingGroup.scaleOnRequestCount("LimitRPS", RequestCountScalingProps.builder()
 *         .targetRequestsPerSecond(1000)
 *         .build());
 * </pre></blockquote>
 * <p>
 * <h3>Scheduled Scaling</h3>
 * <p>
 * This type of scaling is used to change capacities based on time. It works by
 * changing <code>minCapacity</code>, <code>maxCapacity</code> and <code>desiredCapacity</code> of the
 * AutoScalingGroup, and so can be used for two purposes:
 * <p>
 * <ul>
 * <li>Scale in and out on a schedule by setting the <code>minCapacity</code> high or
 * the <code>maxCapacity</code> low.</li>
 * <li>Still allow the regular scaling actions to do their job, but restrict
 * the range they can scale over (by setting both <code>minCapacity</code> and
 * <code>maxCapacity</code> but changing their range over time).</li>
 * </ul>
 * <p>
 * A schedule is expressed as a cron expression. The <code>Schedule</code> class has a <code>cron</code> method to help build cron expressions.
 * <p>
 * The following example scales the fleet out in the morning, going back to natural
 * scaling (all the way down to 1 instance if necessary) at night:
 * <p>
 * <blockquote><pre>
 * AutoScalingGroup autoScalingGroup;
 * 
 * 
 * autoScalingGroup.scaleOnSchedule("PrescaleInTheMorning", BasicScheduledActionProps.builder()
 *         .schedule(Schedule.cron(CronOptions.builder().hour("8").minute("0").build()))
 *         .minCapacity(20)
 *         .build());
 * 
 * autoScalingGroup.scaleOnSchedule("AllowDownscalingAtNight", BasicScheduledActionProps.builder()
 *         .schedule(Schedule.cron(CronOptions.builder().hour("20").minute("0").build()))
 *         .minCapacity(1)
 *         .build());
 * </pre></blockquote>
 * <p>
 * <h2>Configuring Instances using CloudFormation Init</h2>
 * <p>
 * It is possible to use the CloudFormation Init mechanism to configure the
 * instances in the AutoScalingGroup. You can write files to it, run commands,
 * start services, etc. See the documentation of
 * <a href="https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-init.html">AWS::CloudFormation::Init</a>
 * and the documentation of CDK's <code>aws-ec2</code> library for more information.
 * <p>
 * When you specify a CloudFormation Init configuration for an AutoScalingGroup:
 * <p>
 * <ul>
 * <li>you <em>must</em> also specify <code>signals</code> to configure how long CloudFormation
 * should wait for the instances to successfully configure themselves.</li>
 * <li>you <em>should</em> also specify an <code>updatePolicy</code> to configure how instances
 * should be updated when the AutoScalingGroup is updated (for example,
 * when the AMI is updated). If you don't specify an update policy, a <em>rolling
 * update</em> is chosen by default.</li>
 * </ul>
 * <p>
 * Here's an example of using CloudFormation Init to write a file to the
 * instance hosts on startup:
 * <p>
 * <blockquote><pre>
 * Vpc vpc;
 * InstanceType instanceType;
 * IMachineImage machineImage;
 * 
 * 
 * AutoScalingGroup.Builder.create(this, "ASG")
 *         .vpc(vpc)
 *         .instanceType(instanceType)
 *         .machineImage(machineImage)
 * 
 *         // ...
 * 
 *         .init(CloudFormationInit.fromElements(InitFile.fromString("/etc/my_instance", "This got written during instance startup")))
 *         .signals(Signals.waitForAll(SignalsOptions.builder()
 *                 .timeout(Duration.minutes(10))
 *                 .build()))
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h2>Signals</h2>
 * <p>
 * In normal operation, CloudFormation will send a Create or Update command to
 * an AutoScalingGroup and proceed with the rest of the deployment without waiting
 * for the <em>instances in the AutoScalingGroup</em>.
 * <p>
 * Configure <code>signals</code> to tell CloudFormation to wait for a specific number of
 * instances in the AutoScalingGroup to have been started (or failed to start)
 * before moving on. An instance is supposed to execute the
 * <a href="https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/cfn-signal.html"><code>cfn-signal</code></a>
 * program as part of its startup to indicate whether it was started
 * successfully or not.
 * <p>
 * If you use CloudFormation Init support (described in the previous section),
 * the appropriate call to <code>cfn-signal</code> is automatically added to the
 * AutoScalingGroup's UserData. If you don't use the <code>signals</code> directly, you are
 * responsible for adding such a call yourself.
 * <p>
 * The following type of <code>Signals</code> are available:
 * <p>
 * <ul>
 * <li><code>Signals.waitForAll([options])</code>: wait for all of <code>desiredCapacity</code> amount of instances
 * to have started (recommended).</li>
 * <li><code>Signals.waitForMinCapacity([options])</code>: wait for a <code>minCapacity</code> amount of instances
 * to have started (use this if waiting for all instances takes too long and you are happy
 * with a minimum count of healthy hosts).</li>
 * <li><code>Signals.waitForCount(count, [options])</code>: wait for a specific amount of instances to have
 * started.</li>
 * </ul>
 * <p>
 * There are two <code>options</code> you can configure:
 * <p>
 * <ul>
 * <li><code>timeout</code>: maximum time a host startup is allowed to take. If a host does not report
 * success within this time, it is considered a failure. Default is 5 minutes.</li>
 * <li><code>minSuccessPercentage</code>: percentage of hosts that needs to be healthy in order for the
 * update to succeed. If you set this value lower than 100, some percentage of hosts may
 * report failure, while still considering the deployment a success. Default is 100%.</li>
 * </ul>
 * <p>
 * <h2>Update Policy</h2>
 * <p>
 * The <em>update policy</em> describes what should happen to running instances when the definition
 * of the AutoScalingGroup is changed. For example, if you add a command to the UserData
 * of an AutoScalingGroup, do the existing instances get replaced with new instances that
 * have executed the new UserData? Or do the "old" instances just keep on running?
 * <p>
 * It is recommended to always use an update policy, otherwise the current state of your
 * instances also depends the previous state of your instances, rather than just on your
 * source code. This degrades the reproducibility of your deployments.
 * <p>
 * The following update policies are available:
 * <p>
 * <ul>
 * <li><code>UpdatePolicy.none()</code>: leave existing instances alone (not recommended).</li>
 * <li><code>UpdatePolicy.rollingUpdate([options])</code>: progressively replace the existing
 * instances with new instances, in small batches. At any point in time,
 * roughly the same amount of total instances will be running. If the deployment
 * needs to be rolled back, the fresh instances will be replaced with the "old"
 * configuration again.</li>
 * <li><code>UpdatePolicy.replacingUpdate([options])</code>: build a completely fresh copy
 * of the new AutoScalingGroup next to the old one. Once the AutoScalingGroup
 * has been successfully created (and the instances started, if <code>signals</code> is
 * configured on the AutoScalingGroup), the old AutoScalingGroup is deleted.
 * If the deployment needs to be rolled back, the new AutoScalingGroup is
 * deleted and the old one is left unchanged.</li>
 * </ul>
 * <p>
 * <h2>Allowing Connections</h2>
 * <p>
 * See the documentation of the <code>&#64;aws-cdk/aws-ec2</code> package for more information
 * about allowing connections between resources backed by instances.
 * <p>
 * <h2>Max Instance Lifetime</h2>
 * <p>
 * To enable the max instance lifetime support, specify <code>maxInstanceLifetime</code> property
 * for the <code>AutoscalingGroup</code> resource. The value must be between 7 and 365 days(inclusive).
 * To clear a previously set value, leave this property undefined.
 * <p>
 * <h2>Instance Monitoring</h2>
 * <p>
 * To disable detailed instance monitoring, specify <code>instanceMonitoring</code> property
 * for the <code>AutoscalingGroup</code> resource as <code>Monitoring.BASIC</code>. Otherwise detailed monitoring
 * will be enabled.
 * <p>
 * <h2>Monitoring Group Metrics</h2>
 * <p>
 * Group metrics are used to monitor group level properties; they describe the group rather than any of its instances (e.g GroupMaxSize, the group maximum size). To enable group metrics monitoring, use the <code>groupMetrics</code> property.
 * All group metrics are reported in a granularity of 1 minute at no additional charge.
 * <p>
 * See <a href="https://docs.aws.amazon.com/autoscaling/ec2/userguide/as-instance-monitoring.html#as-group-metrics">EC2 docs</a> for a list of all available group metrics.
 * <p>
 * To enable group metrics monitoring using the <code>groupMetrics</code> property:
 * <p>
 * <blockquote><pre>
 * Vpc vpc;
 * InstanceType instanceType;
 * IMachineImage machineImage;
 * 
 * 
 * // Enable monitoring of all group metrics
 * // Enable monitoring of all group metrics
 * AutoScalingGroup.Builder.create(this, "ASG")
 *         .vpc(vpc)
 *         .instanceType(instanceType)
 *         .machineImage(machineImage)
 * 
 *         // ...
 * 
 *         .groupMetrics(List.of(GroupMetrics.all()))
 *         .build();
 * 
 * // Enable monitoring for a subset of group metrics
 * // Enable monitoring for a subset of group metrics
 * AutoScalingGroup.Builder.create(this, "ASG")
 *         .vpc(vpc)
 *         .instanceType(instanceType)
 *         .machineImage(machineImage)
 * 
 *         // ...
 * 
 *         .groupMetrics(List.of(new GroupMetrics(GroupMetric.MIN_SIZE, GroupMetric.MAX_SIZE)))
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h2>Termination policies</h2>
 * <p>
 * Auto Scaling uses <a href="https://docs.aws.amazon.com/autoscaling/ec2/userguide/as-instance-termination.html">termination policies</a>
 * to determine which instances it terminates first during scale-in events. You
 * can specify one or more termination policies with the <code>terminationPolicies</code>
 * property:
 * <p>
 * <blockquote><pre>
 * Vpc vpc;
 * InstanceType instanceType;
 * IMachineImage machineImage;
 * 
 * 
 * AutoScalingGroup.Builder.create(this, "ASG")
 *         .vpc(vpc)
 *         .instanceType(instanceType)
 *         .machineImage(machineImage)
 * 
 *         // ...
 * 
 *         .terminationPolicies(List.of(TerminationPolicy.OLDEST_INSTANCE, TerminationPolicy.DEFAULT))
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h2>Protecting new instances from being terminated on scale-in</h2>
 * <p>
 * By default, Auto Scaling can terminate an instance at any time after launch when
 * scaling in an Auto Scaling Group, subject to the group's <a href="https://docs.aws.amazon.com/autoscaling/ec2/userguide/as-instance-termination.html">termination
 * policy</a>.
 * <p>
 * However, you may wish to protect newly-launched instances from being scaled in
 * if they are going to run critical applications that should not be prematurely
 * terminated. EC2 Capacity Providers for Amazon ECS requires this attribute be
 * set to <code>true</code>.
 * <p>
 * <blockquote><pre>
 * Vpc vpc;
 * InstanceType instanceType;
 * IMachineImage machineImage;
 * 
 * 
 * AutoScalingGroup.Builder.create(this, "ASG")
 *         .vpc(vpc)
 *         .instanceType(instanceType)
 *         .machineImage(machineImage)
 * 
 *         // ...
 * 
 *         .newInstancesProtectedFromScaleIn(true)
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h2>Configuring Instance Metadata Service (IMDS)</h2>
 * <p>
 * <h3>Toggling IMDSv1</h3>
 * <p>
 * You can configure <a href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/ec2-instance-metadata.html">EC2 Instance Metadata Service</a> options to either
 * allow both IMDSv1 and IMDSv2 or enforce IMDSv2 when interacting with the IMDS.
 * <p>
 * To do this for a single <code>AutoScalingGroup</code>, you can use set the <code>requireImdsv2</code> property.
 * The example below demonstrates IMDSv2 being required on a single <code>AutoScalingGroup</code>:
 * <p>
 * <blockquote><pre>
 * Vpc vpc;
 * InstanceType instanceType;
 * IMachineImage machineImage;
 * 
 * 
 * AutoScalingGroup.Builder.create(this, "ASG")
 *         .vpc(vpc)
 *         .instanceType(instanceType)
 *         .machineImage(machineImage)
 * 
 *         // ...
 * 
 *         .requireImdsv2(true)
 *         .build();
 * </pre></blockquote>
 * <p>
 * You can also use <code>AutoScalingGroupRequireImdsv2Aspect</code> to apply the operation to multiple AutoScalingGroups.
 * The example below demonstrates the <code>AutoScalingGroupRequireImdsv2Aspect</code> being used to require IMDSv2 for all AutoScalingGroups in a stack:
 * <p>
 * <blockquote><pre>
 * AutoScalingGroupRequireImdsv2Aspect aspect = new AutoScalingGroupRequireImdsv2Aspect();
 * 
 * Aspects.of(this).add(aspect);
 * </pre></blockquote>
 * <p>
 * <h2>Warm Pool</h2>
 * <p>
 * Auto Scaling offers <a href="https://docs.aws.amazon.com/autoscaling/ec2/userguide/ec2-auto-scaling-warm-pools.html">a warm pool</a> which gives an ability to decrease latency for applications that have exceptionally long boot times. You can create a warm pool with default parameters as below:
 * <p>
 * <blockquote><pre>
 * AutoScalingGroup autoScalingGroup;
 * 
 * 
 * autoScalingGroup.addWarmPool();
 * </pre></blockquote>
 * <p>
 * You can also customize a warm pool by configuring parameters:
 * <p>
 * <blockquote><pre>
 * AutoScalingGroup autoScalingGroup;
 * 
 * 
 * autoScalingGroup.addWarmPool(WarmPoolOptions.builder()
 *         .minSize(1)
 *         .reuseOnScaleIn(true)
 *         .build());
 * </pre></blockquote>
 * <p>
 * <h2>Future work</h2>
 * <p>
 * <ul>
 * <li>[ ] CloudWatch Events (impossible to add currently as the AutoScalingGroup ARN is
 * necessary to make this rule and this cannot be accessed from CloudFormation).</li>
 * </ul>
 */
package software.amazon.awscdk.services.autoscaling;
