/**
 * <h1>AWS::AppRunner Construct Library</h1>
 * <p>
 * This module is part of the <a href="https://github.com/aws/aws-cdk">AWS Cloud Development Kit</a> project.
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.core.*;
 * </pre></blockquote>
 * <p>
 * <h2>Introduction</h2>
 * <p>
 * AWS App Runner is a fully managed service that makes it easy for developers to quickly deploy containerized web applications and APIs, at scale and with no prior infrastructure experience required. Start with your source code or a container image. App Runner automatically builds and deploys the web application and load balances traffic with encryption. App Runner also scales up or down automatically to meet your traffic needs. With App Runner, rather than thinking about servers or scaling, you have more time to focus on your applications.
 * <p>
 * <h2>Service</h2>
 * <p>
 * The <code>Service</code> construct allows you to create AWS App Runner services with <code>ECR Public</code>, <code>ECR</code> or <code>Github</code> with the <code>source</code> property in the following scenarios:
 * <p>
 * <ul>
 * <li><code>Source.fromEcr()</code> - To define the source repository from <code>ECR</code>.</li>
 * <li><code>Source.fromEcrPublic()</code> - To define the source repository from <code>ECR Public</code>.</li>
 * <li><code>Source.fromGitHub()</code> - To define the source repository from the <code>Github repository</code>.</li>
 * <li><code>Source.fromAsset()</code> - To define the source from local asset directory.</li>
 * </ul>
 * <p>
 * <h2>ECR Public</h2>
 * <p>
 * To create a <code>Service</code> with ECR Public:
 * <p>
 * <blockquote><pre>
 * Service.Builder.create(this, "Service")
 *         .source(Source.fromEcrPublic(EcrPublicProps.builder()
 *                 .imageConfiguration(ImageConfiguration.builder().port(8000).build())
 *                 .imageIdentifier("public.ecr.aws/aws-containers/hello-app-runner:latest")
 *                 .build()))
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h2>ECR</h2>
 * <p>
 * To create a <code>Service</code> from an existing ECR repository:
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.core.*;
 * 
 * 
 * Service.Builder.create(this, "Service")
 *         .source(Source.fromEcr(EcrProps.builder()
 *                 .imageConfiguration(ImageConfiguration.builder().port(80).build())
 *                 .repository(Repository.fromRepositoryName(this, "NginxRepository", "nginx"))
 *                 .tagOrDigest("latest")
 *                 .build()))
 *         .build();
 * </pre></blockquote>
 * <p>
 * To create a <code>Service</code> from local docker image asset directory  built and pushed to Amazon ECR:
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.core.*;
 * 
 * 
 * DockerImageAsset imageAsset = DockerImageAsset.Builder.create(this, "ImageAssets")
 *         .directory(join(__dirname, "./docker.assets"))
 *         .build();
 * Service.Builder.create(this, "Service")
 *         .source(Source.fromAsset(AssetProps.builder()
 *                 .imageConfiguration(ImageConfiguration.builder().port(8000).build())
 *                 .asset(imageAsset)
 *                 .build()))
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h2>GitHub</h2>
 * <p>
 * To create a <code>Service</code> from the GitHub repository, you need to specify an existing App Runner <code>Connection</code>.
 * <p>
 * See <a href="https://docs.aws.amazon.com/apprunner/latest/dg/manage-connections.html">Managing App Runner connections</a> for more details.
 * <p>
 * <blockquote><pre>
 * Service.Builder.create(this, "Service")
 *         .source(Source.fromGitHub(GithubRepositoryProps.builder()
 *                 .repositoryUrl("https://github.com/aws-containers/hello-app-runner")
 *                 .branch("main")
 *                 .configurationSource(ConfigurationSourceType.REPOSITORY)
 *                 .connection(GitHubConnection.fromConnectionArn("CONNECTION_ARN"))
 *                 .build()))
 *         .build();
 * </pre></blockquote>
 * <p>
 * Use <code>codeConfigurationValues</code> to override configuration values with the <code>API</code> configuration source type.
 * <p>
 * <blockquote><pre>
 * Service.Builder.create(this, "Service")
 *         .source(Source.fromGitHub(GithubRepositoryProps.builder()
 *                 .repositoryUrl("https://github.com/aws-containers/hello-app-runner")
 *                 .branch("main")
 *                 .configurationSource(ConfigurationSourceType.API)
 *                 .codeConfigurationValues(CodeConfigurationValues.builder()
 *                         .runtime(Runtime.PYTHON_3)
 *                         .port("8000")
 *                         .startCommand("python app.py")
 *                         .buildCommand("yum install -y pycairo &amp;&amp; pip install -r requirements.txt")
 *                         .build())
 *                 .connection(GitHubConnection.fromConnectionArn("CONNECTION_ARN"))
 *                 .build()))
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h2>IAM Roles</h2>
 * <p>
 * You are allowed to define <code>instanceRole</code> and <code>accessRole</code> for the <code>Service</code>.
 * <p>
 * <code>instanceRole</code> - The IAM role that provides permissions to your App Runner service. These are permissions that
 * your code needs when it calls any AWS APIs.
 * <p>
 * <code>accessRole</code> - The IAM role that grants the App Runner service access to a source repository. It's required for
 * ECR image repositories (but not for ECR Public repositories). If not defined, a new access role will be generated
 * when required.
 * <p>
 * See <a href="https://docs.aws.amazon.com/apprunner/latest/dg/security_iam_service-with-iam.html#security_iam_service-with-iam-roles">App Runner IAM Roles</a> for more details.
 * <p>
 * <h2>VPC Connector</h2>
 * <p>
 * To associate an App Runner service with a custom VPC, define <code>vpcConnector</code> for the service.
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.core.*;
 * 
 * 
 * Vpc vpc = Vpc.Builder.create(this, "Vpc")
 *         .cidr("10.0.0.0/16")
 *         .build();
 * 
 * VpcConnector vpcConnector = VpcConnector.Builder.create(this, "VpcConnector")
 *         .vpc(vpc)
 *         .vpcSubnets(vpc.selectSubnets(SubnetSelection.builder().subnetType(SubnetType.PUBLIC).build()))
 *         .vpcConnectorName("MyVpcConnector")
 *         .build();
 * 
 * Service.Builder.create(this, "Service")
 *         .source(Source.fromEcrPublic(EcrPublicProps.builder()
 *                 .imageConfiguration(ImageConfiguration.builder().port(8000).build())
 *                 .imageIdentifier("public.ecr.aws/aws-containers/hello-app-runner:latest")
 *                 .build()))
 *         .vpcConnector(vpcConnector)
 *         .build();
 * </pre></blockquote>
 */
package software.amazon.awscdk.services.apprunner;
