/**
 * <h1>AWS Amplify Construct Library</h1>
 * <p>
 * The AWS Amplify Console provides a Git-based workflow for deploying and hosting fullstack serverless web applications. A fullstack serverless app consists of a backend built with cloud resources such as GraphQL or REST APIs, file and data storage, and a frontend built with single page application frameworks such as React, Angular, Vue, or Gatsby.
 * <p>
 * <h2>Setting up an app with branches, custom rules and a domain</h2>
 * <p>
 * To set up an Amplify Console app, define an <code>App</code>:
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.core.*;
 * 
 * 
 * App amplifyApp = App.Builder.create(this, "MyApp")
 *         .sourceCodeProvider(GitHubSourceCodeProvider.Builder.create()
 *                 .owner("&lt;user&gt;")
 *                 .repository("&lt;repo&gt;")
 *                 .oauthToken(SecretValue.secretsManager("my-github-token"))
 *                 .build())
 *         .buildSpec(BuildSpec.fromObjectToYaml(Map.of(
 *                 // Alternatively add a `amplify.yml` to the repo
 *                 "version", "1.0",
 *                 "frontend", Map.of(
 *                         "phases", Map.of(
 *                                 "preBuild", Map.of(
 *                                         "commands", List.of("yarn")),
 *                                 "build", Map.of(
 *                                         "commands", List.of("yarn build"))),
 *                         "artifacts", Map.of(
 *                                 "baseDirectory", "public",
 *                                 "files", -"**&#47;*")))))
 *         .build();
 * </pre></blockquote>
 * <p>
 * To connect your <code>App</code> to GitLab, use the <code>GitLabSourceCodeProvider</code>:
 * <p>
 * <blockquote><pre>
 * App amplifyApp = App.Builder.create(this, "MyApp")
 *         .sourceCodeProvider(GitLabSourceCodeProvider.Builder.create()
 *                 .owner("&lt;user&gt;")
 *                 .repository("&lt;repo&gt;")
 *                 .oauthToken(SecretValue.secretsManager("my-gitlab-token"))
 *                 .build())
 *         .build();
 * </pre></blockquote>
 * <p>
 * To connect your <code>App</code> to CodeCommit, use the <code>CodeCommitSourceCodeProvider</code>:
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.core.*;
 * 
 * 
 * Repository repository = Repository.Builder.create(this, "Repo")
 *         .repositoryName("my-repo")
 *         .build();
 * 
 * App amplifyApp = App.Builder.create(this, "App")
 *         .sourceCodeProvider(CodeCommitSourceCodeProvider.Builder.create().repository(repository).build())
 *         .build();
 * </pre></blockquote>
 * <p>
 * The IAM role associated with the <code>App</code> will automatically be granted the permission
 * to pull the CodeCommit repository.
 * <p>
 * Add branches:
 * <p>
 * <blockquote><pre>
 * App amplifyApp;
 * 
 * 
 * Branch master = amplifyApp.addBranch("master"); // `id` will be used as repo branch name
 * Branch dev = amplifyApp.addBranch("dev", BranchOptions.builder()
 *         .performanceMode(true)
 *         .build());
 * dev.addEnvironment("STAGE", "dev");
 * </pre></blockquote>
 * <p>
 * Auto build and pull request preview are enabled by default.
 * <p>
 * Add custom rules for redirection:
 * <p>
 * <blockquote><pre>
 * App amplifyApp;
 * 
 * amplifyApp.addCustomRule(Map.of(
 *         "source", "/docs/specific-filename.html",
 *         "target", "/documents/different-filename.html",
 *         "status", RedirectStatus.TEMPORARY_REDIRECT));
 * </pre></blockquote>
 * <p>
 * When working with a single page application (SPA), use the
 * <code>CustomRule.SINGLE_PAGE_APPLICATION_REDIRECT</code> to set up a 200
 * rewrite for all files to <code>index.html</code> except for the following
 * file extensions: css, gif, ico, jpg, js, png, txt, svg, woff,
 * ttf, map, json, webmanifest.
 * <p>
 * <blockquote><pre>
 * App mySinglePageApp;
 * 
 * 
 * mySinglePageApp.addCustomRule(CustomRule.SINGLE_PAGE_APPLICATION_REDIRECT);
 * </pre></blockquote>
 * <p>
 * Add a domain and map sub domains to branches:
 * <p>
 * <blockquote><pre>
 * App amplifyApp;
 * Branch master;
 * Branch dev;
 * 
 * 
 * Domain domain = amplifyApp.addDomain("example.com", DomainOptions.builder()
 *         .enableAutoSubdomain(true) // in case subdomains should be auto registered for branches
 *         .autoSubdomainCreationPatterns(List.of("*", "pr*"))
 *         .build());
 * domain.mapRoot(master); // map master branch to domain root
 * domain.mapSubDomain(master, "www");
 * domain.mapSubDomain(dev);
 * </pre></blockquote>
 * <p>
 * <h2>Restricting access</h2>
 * <p>
 * Password protect the app with basic auth by specifying the <code>basicAuth</code> prop.
 * <p>
 * Use <code>BasicAuth.fromCredentials</code> when referencing an existing secret:
 * <p>
 * <blockquote><pre>
 * App amplifyApp = App.Builder.create(this, "MyApp")
 *         .sourceCodeProvider(GitHubSourceCodeProvider.Builder.create()
 *                 .owner("&lt;user&gt;")
 *                 .repository("&lt;repo&gt;")
 *                 .oauthToken(SecretValue.secretsManager("my-github-token"))
 *                 .build())
 *         .basicAuth(BasicAuth.fromCredentials("username", SecretValue.secretsManager("my-github-token")))
 *         .build();
 * </pre></blockquote>
 * <p>
 * Use <code>BasicAuth.fromGeneratedPassword</code> to generate a password in Secrets Manager:
 * <p>
 * <blockquote><pre>
 * App amplifyApp = App.Builder.create(this, "MyApp")
 *         .sourceCodeProvider(GitHubSourceCodeProvider.Builder.create()
 *                 .owner("&lt;user&gt;")
 *                 .repository("&lt;repo&gt;")
 *                 .oauthToken(SecretValue.secretsManager("my-github-token"))
 *                 .build())
 *         .basicAuth(BasicAuth.fromGeneratedPassword("username"))
 *         .build();
 * </pre></blockquote>
 * <p>
 * Basic auth can be added to specific branches:
 * <p>
 * <blockquote><pre>
 * App amplifyApp;
 * 
 * amplifyApp.addBranch("feature/next", BranchOptions.builder()
 *         .basicAuth(BasicAuth.fromGeneratedPassword("username"))
 *         .build());
 * </pre></blockquote>
 * <p>
 * <h2>Automatically creating and deleting branches</h2>
 * <p>
 * Use the <code>autoBranchCreation</code> and <code>autoBranchDeletion</code> props to control creation/deletion
 * of branches:
 * <p>
 * <blockquote><pre>
 * App amplifyApp = App.Builder.create(this, "MyApp")
 *         .sourceCodeProvider(GitHubSourceCodeProvider.Builder.create()
 *                 .owner("&lt;user&gt;")
 *                 .repository("&lt;repo&gt;")
 *                 .oauthToken(SecretValue.secretsManager("my-github-token"))
 *                 .build())
 *         .autoBranchCreation(AutoBranchCreation.builder() // Automatically connect branches that match a pattern set
 *                 .patterns(List.of("feature/*", "test/*")).build())
 *         .autoBranchDeletion(true)
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h2>Adding custom response headers</h2>
 * <p>
 * Use the <code>customResponseHeaders</code> prop to configure custom response headers for an Amplify app:
 * <p>
 * <blockquote><pre>
 * App amplifyApp = App.Builder.create(this, "App")
 *         .sourceCodeProvider(GitHubSourceCodeProvider.Builder.create()
 *                 .owner("&lt;user&gt;")
 *                 .repository("&lt;repo&gt;")
 *                 .oauthToken(SecretValue.secretsManager("my-github-token"))
 *                 .build())
 *         .customResponseHeaders(List.of(CustomResponseHeader.builder()
 *                 .pattern("*.json")
 *                 .headers(Map.of(
 *                         "custom-header-name-1", "custom-header-value-1",
 *                         "custom-header-name-2", "custom-header-value-2"))
 *                 .build(), CustomResponseHeader.builder()
 *                 .pattern("/path/*")
 *                 .headers(Map.of(
 *                         "custom-header-name-1", "custom-header-value-2"))
 *                 .build()))
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h2>Deploying Assets</h2>
 * <p>
 * <code>sourceCodeProvider</code> is optional; when this is not specified the Amplify app can be deployed to using <code>.zip</code> packages. The <code>asset</code> property can be used to deploy S3 assets to Amplify as part of the CDK:
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.core.*;
 * 
 * Asset asset;
 * App amplifyApp;
 * 
 * Branch branch = amplifyApp.addBranch("dev", BranchOptions.builder().asset(asset).build());
 * </pre></blockquote>
 */
package software.amazon.awscdk.services.amplify;
