/**
 * <h1>integ-tests</h1>
 * <p>
 * <h2>Overview</h2>
 * <p>
 * This library is meant to be used in combination with the <a href="">integ-runner</a> CLI
 * to enable users to write and execute integration tests for AWS CDK Constructs.
 * <p>
 * An integration test should be defined as a CDK application, and
 * there should be a 1:1 relationship between an integration test and a CDK application.
 * <p>
 * So for example, in order to create an integration test called <code>my-function</code>
 * we would need to create a file to contain our integration test application.
 * <p>
 * <em>test/integ.my-function.ts</em>
 * <p>
 * <blockquote><pre>
 * App app = new App();
 * Stack stack = new Stack();
 * Function.Builder.create(stack, "MyFunction")
 *         .runtime(Runtime.NODEJS_14_X)
 *         .handler("index.handler")
 *         .code(Code.fromAsset(join(__dirname, "lambda-handler")))
 *         .build();
 * </pre></blockquote>
 * <p>
 * This is a self contained CDK application which we could deploy by running
 * <p>
 * <blockquote><pre>
 * cdk deploy --app 'node test/integ.my-function.js'
 * </pre></blockquote>
 * <p>
 * In order to turn this into an integration test, all that is needed is to
 * use the <code>IntegTest</code> construct.
 * <p>
 * <blockquote><pre>
 * App app;
 * Stack stack;
 * 
 * IntegTest.Builder.create(app, "Integ").testCases(List.of(stack)).build();
 * </pre></blockquote>
 * <p>
 * You will notice that the <code>stack</code> is registered to the <code>IntegTest</code> as a test case.
 * Each integration test can contain multiple test cases, which are just instances
 * of a stack. See the <a href="#usage">Usage</a> section for more details.
 * <p>
 * <h2>Usage</h2>
 * <p>
 * <h3>IntegTest</h3>
 * <p>
 * Suppose you have a simple stack, that only encapsulates a Lambda function with a
 * certain handler:
 * <p>
 * <blockquote><pre>
 * public class StackUnderTestProps extends StackProps {
 *     private Architecture architecture;
 *     public Architecture getArchitecture() {
 *         return this.architecture;
 *     }
 *     public StackUnderTestProps architecture(Architecture architecture) {
 *         this.architecture = architecture;
 *         return this;
 *     }
 * }
 * 
 * public class StackUnderTest extends Stack {
 *     public StackUnderTest(Construct scope, String id, StackUnderTestProps props) {
 *         super(scope, id, props);
 * 
 *         Function.Builder.create(this, "Handler")
 *                 .runtime(Runtime.NODEJS_14_X)
 *                 .handler("index.handler")
 *                 .code(Code.fromAsset(join(__dirname, "lambda-handler")))
 *                 .architecture(props.getArchitecture())
 *                 .build();
 *     }
 * }
 * </pre></blockquote>
 * <p>
 * You may want to test this stack under different conditions. For example, we want
 * this stack to be deployed correctly, regardless of the architecture we choose
 * for the Lambda function. In particular, it should work for both <code>ARM_64</code> and
 * <code>X86_64</code>. So you can create an <code>IntegTestCase</code> that exercises both scenarios:
 * <p>
 * <blockquote><pre>
 * public class StackUnderTestProps extends StackProps {
 *     private Architecture architecture;
 *     public Architecture getArchitecture() {
 *         return this.architecture;
 *     }
 *     public StackUnderTestProps architecture(Architecture architecture) {
 *         this.architecture = architecture;
 *         return this;
 *     }
 * }
 * 
 * public class StackUnderTest extends Stack {
 *     public StackUnderTest(Construct scope, String id, StackUnderTestProps props) {
 *         super(scope, id, props);
 * 
 *         Function.Builder.create(this, "Handler")
 *                 .runtime(Runtime.NODEJS_14_X)
 *                 .handler("index.handler")
 *                 .code(Code.fromAsset(join(__dirname, "lambda-handler")))
 *                 .architecture(props.getArchitecture())
 *                 .build();
 *     }
 * }
 * 
 * // Beginning of the test suite
 * App app = new App();
 * 
 * IntegTest.Builder.create(app, "DifferentArchitectures")
 *         .testCases(List.of(
 *             new StackUnderTest(app, "Stack1", new StackUnderTestProps()
 *                     .architecture(Architecture.ARM_64)
 *                     ),
 *             new StackUnderTest(app, "Stack2", new StackUnderTestProps()
 *                     .architecture(Architecture.X86_64)
 *                     )))
 *         .build();
 * </pre></blockquote>
 * <p>
 * This is all the instruction you need for the integration test runner to know
 * which stacks to synthesize, deploy and destroy. But you may also need to
 * customize the behavior of the runner by changing its parameters. For example:
 * <p>
 * <blockquote><pre>
 * App app = new App();
 * 
 * Stack stackUnderTest = new Stack(app, "StackUnderTest");
 * 
 * Stack stack = new Stack(app, "stack");
 * 
 * IntegTest testCase = IntegTest.Builder.create(app, "CustomizedDeploymentWorkflow")
 *         .testCases(List.of(stackUnderTest))
 *         .diffAssets(true)
 *         .stackUpdateWorkflow(true)
 *         .cdkCommandOptions(CdkCommands.builder()
 *                 .deploy(DeployCommand.builder()
 *                         .args(DeployOptions.builder()
 *                                 .requireApproval(RequireApproval.NEVER)
 *                                 .json(true)
 *                                 .build())
 *                         .build())
 *                 .destroy(DestroyCommand.builder()
 *                         .args(DestroyOptions.builder()
 *                                 .force(true)
 *                                 .build())
 *                         .build())
 *                 .build())
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h3>IntegTestCaseStack</h3>
 * <p>
 * In the majority of cases an integration test will contain a single <code>IntegTestCase</code>.
 * By default when you create an <code>IntegTest</code> an <code>IntegTestCase</code> is created for you
 * and all of your test cases are registered to this <code>IntegTestCase</code>. The <code>IntegTestCase</code>
 * and <code>IntegTestCaseStack</code> constructs are only needed when it is necessary to
 * defined different options for individual test cases.
 * <p>
 * For example, you might want to have one test case where <code>diffAssets</code> is enabled.
 * <p>
 * <blockquote><pre>
 * App app;
 * Stack stackUnderTest;
 * 
 * IntegTestCaseStack testCaseWithAssets = IntegTestCaseStack.Builder.create(app, "TestCaseAssets")
 *         .diffAssets(true)
 *         .build();
 * 
 * IntegTest.Builder.create(app, "Integ").testCases(List.of(stackUnderTest, testCaseWithAssets)).build();
 * </pre></blockquote>
 * <p>
 * <h2>Assertions</h2>
 * <p>
 * This library also provides a utility to make assertions against the infrastructure that the integration test deploys.
 * <p>
 * There are two main scenarios in which assertions are created.
 * <p>
 * <ul>
 * <li>Part of an integration test using <code>integ-runner</code></li>
 * </ul>
 * <p>
 * In this case you would create an integration test using the <code>IntegTest</code> construct and then make assertions using the <code>assert</code> property.
 * You should <strong>not</strong> utilize the assertion constructs directly, but should instead use the <code>methods</code> on <code>IntegTest.assert</code>.
 * <p>
 * <blockquote><pre>
 * App app;
 * Stack stack;
 * 
 * 
 * IntegTest integ = IntegTest.Builder.create(app, "Integ").testCases(List.of(stack)).build();
 * integ.assertions.awsApiCall("S3", "getObject");
 * </pre></blockquote>
 * <p>
 * <ul>
 * <li>Part of a  normal CDK deployment</li>
 * </ul>
 * <p>
 * In this case you may be using assertions as part of a normal CDK deployment in order to make an assertion on the infrastructure
 * before the deployment is considered successful. In this case you can utilize the assertions constructs directly.
 * <p>
 * <blockquote><pre>
 * Stack myAppStack;
 * 
 * 
 * AwsApiCall.Builder.create(myAppStack, "GetObject")
 *         .service("S3")
 *         .api("getObject")
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h3>DeployAssert</h3>
 * <p>
 * Assertions are created by using the <code>DeployAssert</code> construct. This construct creates it's own <code>Stack</code> separate from
 * any stacks that you create as part of your integration tests. This <code>Stack</code> is treated differently from other stacks
 * by the <code>integ-runner</code> tool. For example, this stack will not be diffed by the <code>integ-runner</code>.
 * <p>
 * <code>DeployAssert</code> also provides utilities to register your own assertions.
 * <p>
 * <blockquote><pre>
 * CustomResource myCustomResource;
 * Stack stack;
 * App app;
 * 
 * 
 * IntegTest integ = IntegTest.Builder.create(app, "Integ").testCases(List.of(stack)).build();
 * integ.assertions.expect("CustomAssertion", ExpectedResult.objectLike(Map.of("foo", "bar")), ActualResult.fromCustomResource(myCustomResource, "data"));
 * </pre></blockquote>
 * <p>
 * In the above example an assertion is created that will trigger a user defined <code>CustomResource</code>
 * and assert that the <code>data</code> attribute is equal to <code>{ foo: 'bar' }</code>.
 * <p>
 * <h3>AwsApiCall</h3>
 * <p>
 * A common method to retrieve the "actual" results to compare with what is expected is to make an
 * AWS API call to receive some data. This library does this by utilizing CloudFormation custom resources
 * which means that CloudFormation will call out to a Lambda Function which will
 * use the AWS JavaScript SDK to make the API call.
 * <p>
 * This can be done by using the class directory (in the case of a normal deployment):
 * <p>
 * <blockquote><pre>
 * Stack stack;
 * 
 * 
 * AwsApiCall.Builder.create(stack, "MyAssertion")
 *         .service("SQS")
 *         .api("receiveMessage")
 *         .parameters(Map.of(
 *                 "QueueUrl", "url"))
 *         .build();
 * </pre></blockquote>
 * <p>
 * Or by using the <code>awsApiCall</code> method on <code>DeployAssert</code> (when writing integration tests):
 * <p>
 * <blockquote><pre>
 * App app;
 * Stack stack;
 * 
 * IntegTest integ = IntegTest.Builder.create(app, "Integ")
 *         .testCases(List.of(stack))
 *         .build();
 * integ.assertions.awsApiCall("SQS", "receiveMessage", Map.of(
 *         "QueueUrl", "url"));
 * </pre></blockquote>
 * <p>
 * <h3>EqualsAssertion</h3>
 * <p>
 * This library currently provides the ability to assert that two values are equal
 * to one another by utilizing the <code>EqualsAssertion</code> class. This utilizes a Lambda
 * backed <code>CustomResource</code> which in tern uses the <a href="https://docs.aws.amazon.com/cdk/api/v2/docs/aws-cdk-lib.assertions.Match.html">Match</a> utility from the
 * <a href="https://docs.aws.amazon.com/cdk/api/v2/docs/aws-cdk-lib.assertions-readme.html">&#64;aws-cdk/assertions</a> library.
 * <p>
 * <blockquote><pre>
 * App app;
 * Stack stack;
 * Queue queue;
 * IFunction fn;
 * 
 * 
 * IntegTest integ = IntegTest.Builder.create(app, "Integ")
 *         .testCases(List.of(stack))
 *         .build();
 * 
 * integ.assertions.invokeFunction(LambdaInvokeFunctionProps.builder()
 *         .functionName(fn.getFunctionName())
 *         .invocationType(InvocationType.EVENT)
 *         .payload(JSON.stringify(Map.of("status", "OK")))
 *         .build());
 * 
 * IAwsApiCall message = integ.assertions.awsApiCall("SQS", "receiveMessage", Map.of(
 *         "QueueUrl", queue.getQueueUrl(),
 *         "WaitTimeSeconds", 20));
 * 
 * message.assertAtPath("Messages.0.Body", ExpectedResult.objectLike(Map.of(
 *         "requestContext", Map.of(
 *                 "condition", "Success"),
 *         "requestPayload", Map.of(
 *                 "status", "OK"),
 *         "responseContext", Map.of(
 *                 "statusCode", 200),
 *         "responsePayload", "success")));
 * </pre></blockquote>
 * <p>
 * <h4>Match</h4>
 * <p>
 * <code>integ-tests</code> also provides a <code>Match</code> utility similar to the <code>&#64;aws-cdk/assertions</code> module. <code>Match</code>
 * can be used to construct the <code>ExpectedResult</code>.
 * <p>
 * <blockquote><pre>
 * AwsApiCall message;
 * 
 * 
 * message.expect(ExpectedResult.objectLike(Map.of(
 *         "Messages", Match.arrayWith(List.of(Map.of(
 *                 "Body", Map.of(
 *                         "Values", Match.arrayWith(List.of(Map.of("Asdf", 3))),
 *                         "Message", Match.stringLikeRegexp("message"))))))));
 * </pre></blockquote>
 * <p>
 * <h3>Examples</h3>
 * <p>
 * <h4>Invoke a Lambda Function</h4>
 * <p>
 * In this example there is a Lambda Function that is invoked and
 * we assert that the payload that is returned is equal to '200'.
 * <p>
 * <blockquote><pre>
 * IFunction lambdaFunction;
 * App app;
 * 
 * 
 * Stack stack = new Stack(app, "cdk-integ-lambda-bundling");
 * 
 * IntegTest integ = IntegTest.Builder.create(app, "IntegTest")
 *         .testCases(List.of(stack))
 *         .build();
 * 
 * IAwsApiCall invoke = integ.assertions.invokeFunction(LambdaInvokeFunctionProps.builder()
 *         .functionName(lambdaFunction.getFunctionName())
 *         .build());
 * invoke.expect(ExpectedResult.objectLike(Map.of(
 *         "Payload", "200")));
 * </pre></blockquote>
 * <p>
 * <h4>Make an AWS API Call</h4>
 * <p>
 * In this example there is a StepFunctions state machine that is executed
 * and then we assert that the result of the execution is successful.
 * <p>
 * <blockquote><pre>
 * App app;
 * Stack stack;
 * IStateMachine sm;
 * 
 * 
 * IntegTest testCase = IntegTest.Builder.create(app, "IntegTest")
 *         .testCases(List.of(stack))
 *         .build();
 * 
 * // Start an execution
 * IAwsApiCall start = testCase.assertions.awsApiCall("StepFunctions", "startExecution", Map.of(
 *         "stateMachineArn", sm.getStateMachineArn()));
 * 
 * // describe the results of the execution
 * IAwsApiCall describe = testCase.assertions.awsApiCall("StepFunctions", "describeExecution", Map.of(
 *         "executionArn", start.getAttString("executionArn")));
 * 
 * // assert the results
 * describe.expect(ExpectedResult.objectLike(Map.of(
 *         "status", "SUCCEEDED")));
 * </pre></blockquote>
 */
package software.amazon.awscdk.integtests;
