/*
 * Decompiled with CFR 0.152.
 */
package org.xerial.snappy;

import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.nio.ByteBuffer;
import java.nio.ByteOrder;
import java.nio.channels.Channels;
import java.nio.channels.ClosedChannelException;
import java.nio.channels.ReadableByteChannel;
import java.nio.channels.WritableByteChannel;
import org.xerial.snappy.Snappy;
import org.xerial.snappy.SnappyFramed;

public final class SnappyFramedOutputStream
extends OutputStream
implements WritableByteChannel {
    public static final int MAX_BLOCK_SIZE = 65536;
    public static final int DEFAULT_BLOCK_SIZE = 65536;
    public static final double DEFAULT_MIN_COMPRESSION_RATIO = 0.85;
    private final ByteBuffer headerBuffer = ByteBuffer.allocate(8).order(ByteOrder.LITTLE_ENDIAN);
    private final ByteBuffer buffer;
    private final ByteBuffer directInputBuffer;
    private final ByteBuffer outputBuffer;
    private final double minCompressionRatio;
    private final WritableByteChannel out;
    private boolean closed;

    public SnappyFramedOutputStream(OutputStream out) throws IOException {
        this(out, 65536, 0.85);
    }

    public SnappyFramedOutputStream(OutputStream out, int blockSize, double minCompressionRatio) throws IOException {
        this(Channels.newChannel(out), blockSize, minCompressionRatio);
    }

    public SnappyFramedOutputStream(WritableByteChannel out) throws IOException {
        this(out, 65536, 0.85);
    }

    public SnappyFramedOutputStream(WritableByteChannel out, int blockSize, double minCompressionRatio) throws IOException {
        if (out == null) {
            throw new NullPointerException();
        }
        if (minCompressionRatio <= 0.0 || minCompressionRatio > 1.0) {
            throw new IllegalArgumentException("minCompressionRatio " + minCompressionRatio + " must be in (0,1.0]");
        }
        if (blockSize <= 0 || blockSize > 65536) {
            throw new IllegalArgumentException("block size " + blockSize + " must be in (0, 65536]");
        }
        this.out = out;
        this.minCompressionRatio = minCompressionRatio;
        this.buffer = ByteBuffer.allocate(blockSize);
        this.directInputBuffer = ByteBuffer.allocateDirect(blockSize);
        this.outputBuffer = ByteBuffer.allocateDirect(Snappy.maxCompressedLength(blockSize));
        this.writeHeader(out);
    }

    private void writeHeader(WritableByteChannel out) throws IOException {
        out.write(ByteBuffer.wrap(SnappyFramed.HEADER_BYTES));
    }

    @Override
    public boolean isOpen() {
        return !this.closed;
    }

    @Override
    public void write(int b) throws IOException {
        if (this.closed) {
            throw new IOException("Stream is closed");
        }
        if (this.buffer.remaining() <= 0) {
            this.flushBuffer();
        }
        this.buffer.put((byte)b);
    }

    @Override
    public void write(byte[] input, int offset, int length) throws IOException {
        if (this.closed) {
            throw new IOException("Stream is closed");
        }
        if (input == null) {
            throw new NullPointerException();
        }
        if (offset < 0 || offset > input.length || length < 0 || offset + length > input.length || offset + length < 0) {
            throw new IndexOutOfBoundsException();
        }
        while (length > 0) {
            if (this.buffer.remaining() <= 0) {
                this.flushBuffer();
            }
            int toPut = Math.min(length, this.buffer.remaining());
            this.buffer.put(input, offset, toPut);
            offset += toPut;
            length -= toPut;
        }
    }

    @Override
    public int write(ByteBuffer src) throws IOException {
        if (this.closed) {
            throw new ClosedChannelException();
        }
        if (this.buffer.remaining() <= 0) {
            this.flushBuffer();
        }
        int srcLength = src.remaining();
        if (this.buffer.remaining() >= src.remaining()) {
            this.buffer.put(src);
            return srcLength;
        }
        int srcEnd = src.position() + src.remaining();
        while (src.position() + this.buffer.remaining() <= srcEnd) {
            src.limit(src.position() + this.buffer.remaining());
            this.buffer.put(src);
            this.flushBuffer();
        }
        src.limit(srcEnd);
        this.buffer.put(src);
        return srcLength;
    }

    public long transferFrom(InputStream is) throws IOException {
        int read;
        if (this.closed) {
            throw new ClosedChannelException();
        }
        if (is == null) {
            throw new NullPointerException();
        }
        if (this.buffer.remaining() == 0) {
            this.flushBuffer();
        }
        assert (this.buffer.hasArray());
        byte[] bytes = this.buffer.array();
        int arrayOffset = this.buffer.arrayOffset();
        long totTransfered = 0L;
        while ((read = is.read(bytes, arrayOffset + this.buffer.position(), this.buffer.remaining())) != -1) {
            this.buffer.position(this.buffer.position() + read);
            if (this.buffer.remaining() == 0) {
                this.flushBuffer();
            }
            totTransfered += (long)read;
        }
        return totTransfered;
    }

    public long transferFrom(ReadableByteChannel rbc) throws IOException {
        int read;
        if (this.closed) {
            throw new ClosedChannelException();
        }
        if (rbc == null) {
            throw new NullPointerException();
        }
        if (this.buffer.remaining() == 0) {
            this.flushBuffer();
        }
        long totTransfered = 0L;
        while ((read = rbc.read(this.buffer)) != -1) {
            if (this.buffer.remaining() == 0) {
                this.flushBuffer();
            }
            totTransfered += (long)read;
        }
        return totTransfered;
    }

    @Override
    public final void flush() throws IOException {
        if (this.closed) {
            throw new IOException("Stream is closed");
        }
        this.flushBuffer();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public final void close() throws IOException {
        if (this.closed) {
            return;
        }
        try {
            this.flush();
            this.out.close();
        }
        finally {
            this.closed = true;
            SnappyFramed.releaseDirectByteBuffer(this.directInputBuffer);
            SnappyFramed.releaseDirectByteBuffer(this.outputBuffer);
        }
    }

    private void flushBuffer() throws IOException {
        if (this.buffer.position() > 0) {
            this.buffer.flip();
            this.writeCompressed(this.buffer);
            this.buffer.clear();
        }
    }

    private void writeCompressed(ByteBuffer buffer) throws IOException {
        byte[] input = buffer.array();
        int length = buffer.remaining();
        int crc32c = SnappyFramed.maskedCrc32c(input, 0, length);
        this.directInputBuffer.clear();
        this.directInputBuffer.put(buffer);
        this.directInputBuffer.flip();
        this.outputBuffer.clear();
        Snappy.compress(this.directInputBuffer, this.outputBuffer);
        int compressedLength = this.outputBuffer.remaining();
        if ((double)compressedLength / (double)length <= this.minCompressionRatio) {
            this.writeBlock(this.out, this.outputBuffer, true, crc32c);
        } else {
            buffer.flip();
            this.writeBlock(this.out, buffer, false, crc32c);
        }
    }

    private void writeBlock(WritableByteChannel out, ByteBuffer data, boolean compressed, int crc32c) throws IOException {
        this.headerBuffer.clear();
        this.headerBuffer.put((byte)(!compressed ? 1 : 0));
        int headerLength = data.remaining() + 4;
        this.headerBuffer.put((byte)headerLength);
        this.headerBuffer.put((byte)(headerLength >>> 8));
        this.headerBuffer.put((byte)(headerLength >>> 16));
        this.headerBuffer.putInt(crc32c);
        this.headerBuffer.flip();
        out.write(this.headerBuffer);
        out.write(data);
    }
}

