package org.xbib.interlibrary.action.config;

import org.xbib.content.settings.Settings;
import org.xbib.interlibrary.api.action.AbstractResponse;
import org.xbib.interlibrary.api.action.Action;
import org.xbib.interlibrary.api.action.ActionContext;
import org.xbib.interlibrary.api.action.ActionExecutor;
import org.xbib.interlibrary.api.action.Response;
import org.xbib.interlibrary.api.service.Service;
import org.xbib.interlibrary.api.service.ServiceArguments;

import java.io.IOException;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.ServiceLoader;
import java.util.concurrent.ExecutorCompletionService;
import java.util.concurrent.Future;
import java.util.logging.Level;
import java.util.logging.Logger;
import java.util.stream.Collectors;
import java.util.stream.Stream;

/**
 *
 */
public class ConfigAction implements Action<ConfigRequest, ConfigResponse> {

    public static final ConfigAction INSTANCE = new ConfigAction();

    private static final Logger logger = Logger.getLogger(ConfigAction.class.getName());

    private List<Service<ConfigRequest, ConfigResponse>> services = new ArrayList<>();

    @Override
    public Action<ConfigRequest, ConfigResponse> getInstance() {
        return INSTANCE;
    }

    @Override
    public void init(Settings settings, ServiceArguments arguments, ClassLoader classLoader) {
        this.services = createServices(settings, arguments, classLoader);
    }

    @Override
    public Service<ConfigRequest, ConfigResponse> getService(String name) {
        Optional<Service<ConfigRequest, ConfigResponse>> optional =
                services.stream().filter(s -> s.getName().equalsIgnoreCase(name)).findFirst();
        return optional.orElse(null);
    }

    @Override
    public List<Service<ConfigRequest, ConfigResponse>> getServices() {
        return services;
    }

    @Override
    public ConfigResponse newErrorResponse(ConfigRequest request, String domain, Throwable throwable) {
        return new ConfigResponse(request, ConfigResponse.Status.ERROR, throwable.getMessage());
    }

    @Override
    public void execute(ActionContext<ConfigRequest, ConfigResponse> context) {
        try {
            ConfigResponse configResponse = createActionExecutor(context).execute(context.getRequest());
            context.setResponse(configResponse);
        } catch (Throwable t) {
            if (context.getListener() != null) {
                context.getListener().onFailure(t);
            }
        }
    }

    @Override
    public void submit(ActionContext<ConfigRequest, ConfigResponse> context) {
        try {
            Stream<Future<ConfigResponse>> futureResponses = createActionExecutor(context).submitToAllProviders(context.getRequest());
            context.setFutureResponses(futureResponses);
        } catch (Throwable t) {
            if (context.getListener() != null) {
                context.getListener().onFailure(t);
            }
        }
    }

    @Override
    public void close() {
        services.forEach(s -> {
            try {
                s.close();
            } catch (IOException e) {
                logger.log(Level.SEVERE, e.getMessage(), e);
            }
        });
    }

    private ActionExecutor<ConfigRequest, ConfigResponse> createActionExecutor(ActionContext<ConfigRequest, ConfigResponse> context) {
        return new ActionExecutor<>() {
            @Override
            protected Stream<Service<ConfigRequest, ConfigResponse>> getServices() {
                return services.stream();
            }

            @Override
            protected ExecutorCompletionService<ConfigResponse> getExecutorCompletionService() {
                return context.getExecutorCompletionService();
            }

            @Override
            protected ConfigResponse getFinalResponse(ConfigRequest request, List<ConfigResponse> list) {
                String messageList = list.stream().map(AbstractResponse::getMessage).collect(Collectors.joining());
                return new ConfigResponse(request, ConfigResponse.Status.EMPTY, messageList);
            }
        };
    }

    @SuppressWarnings("unchecked")
    private List<Service<ConfigRequest, ConfigResponse>> createServices(Settings settings,
                                                                        ServiceArguments arguments,
                                                                        ClassLoader classLoader) {
        List<ConfigServiceProvider> providers = new ArrayList<>();
        for (ConfigServiceProvider provider : ServiceLoader.load(ConfigServiceProvider.class, classLoader)) {
            providers.add(provider);
        }
        List<Service<ConfigRequest, ConfigResponse>> services = new ArrayList<>();
        Map<String, Settings> providerDefs = settings.getGroups("service.config");
        for (Map.Entry<String, Settings> providerDef  : providerDefs.entrySet()) {
            String providerName = providerDef.getKey();
            Settings providerSettings = providerDef.getValue();
            if (providerSettings.getAsBoolean("enabled", true)) {
                try {
                    String className = providerSettings.get("class");
                    Class<ConfigServiceProvider> clazz =
                            (Class<ConfigServiceProvider>) Class.forName(className, false, classLoader);
                    providers.stream().filter(clazz::isInstance).forEach(configServiceProvider -> {
                        try {
                            Service<ConfigRequest, ConfigResponse> service =
                                    configServiceProvider.createService(settings, providerName, providerSettings, arguments);
                            services.add(service);
                        } catch (IOException e) {
                            logger.log(Level.SEVERE, e.getMessage(), e);
                        }
                    });
                    if (services.isEmpty()) {
                        logger.log(Level.WARNING, "no provider found for instance of " + clazz);
                    }
                } catch (ClassNotFoundException e) {
                    if (logger.isLoggable(Level.WARNING)) {
                        logger.log(Level.WARNING, e.getMessage(), e);
                    }
                }
            } else {
                if (logger.isLoggable(Level.WARNING)) {
                    logger.log(Level.WARNING, "provider is disabled: " + providerName);
                }
            }
        }
        if (logger.isLoggable(Level.INFO)) {
            logger.log(Level.INFO, "services: " + services.size() + " " + services);
        }
        return services;
    }
}
