/*
 * Decompiled with CFR 0.152.
 */
package org.apache.activemq.artemis.utils.collections;

import java.util.Objects;
import java.util.concurrent.atomic.AtomicLongFieldUpdater;
import java.util.concurrent.atomic.AtomicReferenceArray;
import java.util.function.IntFunction;

public final class ConcurrentAppendOnlyChunkedList<T> {
    private static final AtomicLongFieldUpdater<ConcurrentAppendOnlyChunkedList> LAST_INDEX_UPDATER = AtomicLongFieldUpdater.newUpdater(ConcurrentAppendOnlyChunkedList.class, "lastIndex");
    private static final AtomicLongFieldUpdater<ConcurrentAppendOnlyChunkedList> CACHED_LAST_INDEX_UPDATER = AtomicLongFieldUpdater.newUpdater(ConcurrentAppendOnlyChunkedList.class, "cachedLastIndex");
    private final int chunkSize;
    private final int chunkMask;
    private final int chunkSizeLog2;
    private static final long RESIZING = -1L;
    private AtomicChunk<T> firstBuffer = null;
    private AtomicChunk<T> lastBuffer = null;
    private volatile long lastIndex = 0L;
    private volatile long cachedLastIndex = 0L;

    public ConcurrentAppendOnlyChunkedList(int chunkSize) {
        if (chunkSize <= 0) {
            throw new IllegalArgumentException("chunkSize must be >0");
        }
        if (Integer.bitCount(chunkSize) != 1) {
            throw new IllegalArgumentException("chunkSize must be a power of 2");
        }
        this.chunkSize = chunkSize;
        this.chunkMask = chunkSize - 1;
        this.chunkSizeLog2 = Integer.numberOfTrailingZeros(chunkSize);
    }

    private long getValidLastIndex() {
        long lastIndex;
        while ((lastIndex = this.lastIndex) == -1L) {
            Thread.yield();
        }
        return lastIndex;
    }

    public int size() {
        return (int)this.getValidLastIndex();
    }

    public void addAll(T[] elements) {
        for (T e : elements) {
            this.add(e);
        }
    }

    public T get(int index) {
        AtomicChunk<T> buffer;
        int offset;
        if (index < 0) {
            return null;
        }
        long lastIndex = this.cachedLastIndex;
        if ((long)index >= lastIndex) {
            lastIndex = this.getValidLastIndex();
            if ((long)index >= lastIndex) {
                return null;
            }
            CACHED_LAST_INDEX_UPDATER.lazySet(this, lastIndex);
        }
        if (index >= this.chunkSize) {
            offset = index & this.chunkMask;
            buffer = this.getChunkOf(index, lastIndex);
        } else {
            offset = index;
            buffer = this.firstBuffer;
        }
        return ConcurrentAppendOnlyChunkedList.pollElement(buffer, offset);
    }

    private AtomicChunk<T> getChunkOf(int index, long lastIndex) {
        int chunkSizeLog2 = this.chunkSizeLog2;
        int chunkIndex = index >> chunkSizeLog2;
        int lastChunkIndex = (int)lastIndex >> chunkSizeLog2;
        int chunkIndexes = chunkIndex;
        AtomicChunk<T> buffer = null;
        boolean forward = true;
        int distanceFromLastChunkIndex = lastChunkIndex - chunkIndex;
        if (distanceFromLastChunkIndex < chunkIndex) {
            AtomicChunk<T> lastBuffer = this.lastBuffer;
            distanceFromLastChunkIndex = lastBuffer.index - chunkIndex;
            if (distanceFromLastChunkIndex < chunkIndex) {
                buffer = lastBuffer;
                chunkIndexes = distanceFromLastChunkIndex;
                forward = false;
            }
        }
        if (buffer == null) {
            buffer = this.firstBuffer;
        }
        for (int i = 0; i < chunkIndexes; ++i) {
            buffer = forward ? buffer.next : buffer.prev;
        }
        return buffer;
    }

    public void add(T e) {
        Objects.requireNonNull(e);
        while (true) {
            long lastIndex;
            if ((lastIndex = this.lastIndex) != -1L) {
                if (lastIndex == Integer.MAX_VALUE) {
                    throw new IllegalStateException("can't add more then 2147483647 elements");
                }
                AtomicChunk<T> lastBuffer = this.lastBuffer;
                int offset = (int)(lastIndex & (long)this.chunkMask);
                if (offset == 0) {
                    if (this.addChunkAndElement(lastBuffer, lastIndex, e)) {
                        return;
                    }
                } else if (LAST_INDEX_UPDATER.compareAndSet(this, lastIndex, lastIndex + 1L)) {
                    lastBuffer.lazySet(offset, e);
                    return;
                }
            }
            Thread.yield();
        }
    }

    private boolean addChunkAndElement(AtomicChunk<T> lastBuffer, long lastIndex, T element) {
        AtomicChunk<T> newChunk;
        if (!LAST_INDEX_UPDATER.compareAndSet(this, lastIndex, -1L)) {
            return false;
        }
        try {
            int index = (int)(lastIndex >> this.chunkSizeLog2);
            newChunk = new AtomicChunk<T>(index, lastBuffer, this.chunkSize);
        }
        catch (OutOfMemoryError oom) {
            LAST_INDEX_UPDATER.lazySet(this, lastIndex);
            throw oom;
        }
        newChunk.lazySet(0, element);
        if (lastBuffer != null) {
            lastBuffer.next = newChunk;
        } else {
            this.firstBuffer = newChunk;
        }
        this.lastBuffer = newChunk;
        LAST_INDEX_UPDATER.lazySet(this, lastIndex + 1L);
        return true;
    }

    public T[] toArray(IntFunction<T[]> arrayAllocator) {
        long lastIndex = this.getValidLastIndex();
        assert (lastIndex <= Integer.MAX_VALUE);
        int size = (int)lastIndex;
        T[] elements = arrayAllocator.apply(size);
        int chunkSize = this.chunkSize;
        int chunks = size > chunkSize ? size >> this.chunkSizeLog2 : 0;
        AtomicChunk<T> buffer = this.firstBuffer;
        int elementIndex = 0;
        for (int i = 0; i < chunks; ++i) {
            ConcurrentAppendOnlyChunkedList.drain(buffer, elements, elementIndex, chunkSize);
            elementIndex += chunkSize;
            buffer = buffer.next;
        }
        int remaining = chunks > 0 ? size & this.chunkMask : size;
        ConcurrentAppendOnlyChunkedList.drain(buffer, elements, elementIndex, remaining);
        return elements;
    }

    private static <T> T pollElement(AtomicChunk<T> buffer, int i) {
        Object e;
        while ((e = buffer.get(i)) == null) {
            Thread.yield();
        }
        return (T)e;
    }

    private static <T> void drain(AtomicChunk<T> buffer, T[] elements, int elementNumber, int length) {
        for (int j = 0; j < length; ++j) {
            T e = ConcurrentAppendOnlyChunkedList.pollElement(buffer, j);
            assert (e != null);
            elements[elementNumber] = e;
            ++elementNumber;
        }
    }

    private static final class AtomicChunk<T>
    extends AtomicReferenceArray<T> {
        AtomicChunk<T> next = null;
        final AtomicChunk<T> prev;
        final int index;

        AtomicChunk(int index, AtomicChunk<T> prev, int length) {
            super(length);
            this.index = index;
            this.prev = prev;
        }
    }
}

