/*!
 * Copyright (c) 2015, Salesforce.com, Inc.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * 1. Redistributions of source code must retain the above copyright notice,
 * this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 * this list of conditions and the following disclaimer in the documentation
 * and/or other materials provided with the distribution.
 *
 * 3. Neither the name of Salesforce.com nor the names of its contributors may
 * be used to endorse or promote products derived from this software without
 * specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */
'use strict';
Object.defineProperty(exports, "__esModule", { value: true });
exports.inspectFallback = exports.MemoryCookieStore = void 0;
const pathMatch_1 = require("./pathMatch");
const permuteDomain_1 = require("./permuteDomain");
const store_1 = require("./store");
const utilHelper_1 = require("./utilHelper");
const utils_1 = require("./utils");
class MemoryCookieStore extends store_1.Store {
    constructor() {
        super();
        this.synchronous = true;
        this.idx = Object.create(null);
        const customInspectSymbol = (0, utilHelper_1.getCustomInspectSymbol)();
        if (customInspectSymbol) {
            Object.defineProperty(this, customInspectSymbol, {
                value: this.inspect.bind(this),
                enumerable: false,
                writable: false,
                configurable: false,
            });
        }
    }
    inspect() {
        const util = { inspect: (0, utilHelper_1.getUtilInspect)(inspectFallback) };
        return `{ idx: ${util.inspect(this.idx, false, 2)} }`;
    }
    findCookie(domain, path, key, 
    // eslint-disable-next-line @typescript-eslint/no-unused-vars
    _callback) {
        const promiseCallback = (0, utils_1.createPromiseCallback)(arguments);
        const cb = promiseCallback.callback;
        if (domain == null || path == null) {
            return promiseCallback.resolve(undefined);
        }
        const domainEntry = this.idx[domain];
        if (!domainEntry) {
            return promiseCallback.resolve(undefined);
        }
        const pathEntry = domainEntry[path];
        if (!pathEntry) {
            return promiseCallback.resolve(undefined);
        }
        if (key == null) {
            return promiseCallback.resolve(null);
        }
        cb(null, pathEntry[key] || null);
        return promiseCallback.promise;
    }
    findCookies(domain, path, allowSpecialUseDomain = false, 
    // eslint-disable-next-line @typescript-eslint/no-unused-vars
    _callback) {
        if (typeof allowSpecialUseDomain === 'function') {
            allowSpecialUseDomain = true;
        }
        const results = [];
        const promiseCallback = (0, utils_1.createPromiseCallback)(arguments);
        const cb = promiseCallback.callback;
        if (!domain) {
            return promiseCallback.resolve([]);
        }
        let pathMatcher;
        if (!path) {
            // null means "all paths"
            pathMatcher = function matchAll(domainIndex) {
                for (const curPath in domainIndex) {
                    const pathIndex = domainIndex[curPath];
                    for (const key in pathIndex) {
                        const value = pathIndex[key];
                        if (value) {
                            results.push(value);
                        }
                    }
                }
            };
        }
        else {
            pathMatcher = function matchRFC(domainIndex) {
                //NOTE: we should use path-match algorithm from S5.1.4 here
                //(see : https://github.com/ChromiumWebApps/chromium/blob/b3d3b4da8bb94c1b2e061600df106d590fda3620/net/cookies/canonical_cookie.cc#L299)
                for (const cookiePath in domainIndex) {
                    if ((0, pathMatch_1.pathMatch)(path, cookiePath)) {
                        const pathIndex = domainIndex[cookiePath];
                        for (const key in pathIndex) {
                            const value = pathIndex[key];
                            if (value) {
                                results.push(value);
                            }
                        }
                    }
                }
            };
        }
        const domains = (0, permuteDomain_1.permuteDomain)(domain, allowSpecialUseDomain) || [domain];
        const idx = this.idx;
        domains.forEach((curDomain) => {
            const domainIndex = idx[curDomain];
            if (!domainIndex) {
                return;
            }
            pathMatcher(domainIndex);
        });
        cb(null, results);
        return promiseCallback.promise;
    }
    // eslint-disable-next-line @typescript-eslint/no-unused-vars
    putCookie(cookie, _callback) {
        const promiseCallback = (0, utils_1.createPromiseCallback)(arguments);
        const cb = promiseCallback.callback;
        const { domain, path, key } = cookie;
        if (domain == null || path == null || key == null) {
            cb(null, undefined);
            return promiseCallback.promise;
        }
        const domainEntry = this.idx[domain] ??
            Object.create(null);
        this.idx[domain] = domainEntry;
        const pathEntry = domainEntry[path] ??
            Object.create(null);
        domainEntry[path] = pathEntry;
        pathEntry[key] = cookie;
        cb(null, undefined);
        return promiseCallback.promise;
    }
    updateCookie(_oldCookie, newCookie, callback) {
        // this seems wrong but it stops typescript from complaining and all the test pass...
        // eslint-disable-next-line @typescript-eslint/no-empty-function
        callback = callback ?? function () { };
        // updateCookie() may avoid updating cookies that are identical.  For example,
        // lastAccessed may not be important to some stores and an equality
        // comparison could exclude that field.
        return this.putCookie(newCookie, callback);
    }
    removeCookie(domain, path, key, 
    // eslint-disable-next-line @typescript-eslint/no-unused-vars
    _callback) {
        const promiseCallback = (0, utils_1.createPromiseCallback)(arguments);
        const cb = promiseCallback.callback;
        const domainEntry = this.idx[domain];
        if (domainEntry) {
            const pathEntry = domainEntry[path];
            if (pathEntry) {
                const keyEntry = pathEntry[key];
                if (keyEntry) {
                    delete pathEntry[key];
                }
            }
        }
        cb(null, undefined);
        return promiseCallback.promise;
    }
    removeCookies(domain, path, 
    // eslint-disable-next-line @typescript-eslint/no-unused-vars
    _callback) {
        const promiseCallback = (0, utils_1.createPromiseCallback)(arguments);
        const cb = promiseCallback.callback;
        const domainEntry = this.idx[domain];
        if (domainEntry) {
            if (path) {
                delete domainEntry[path];
            }
            else {
                delete this.idx[domain];
            }
        }
        cb(null);
        return promiseCallback.promise;
    }
    // eslint-disable-next-line @typescript-eslint/no-unused-vars
    removeAllCookies(_callback) {
        const promiseCallback = (0, utils_1.createPromiseCallback)(arguments);
        const cb = promiseCallback.callback;
        this.idx = Object.create(null);
        cb(null);
        return promiseCallback.promise;
    }
    // eslint-disable-next-line @typescript-eslint/no-unused-vars
    getAllCookies(_callback) {
        const promiseCallback = (0, utils_1.createPromiseCallback)(arguments);
        const cb = promiseCallback.callback;
        const cookies = [];
        const idx = this.idx;
        const domains = Object.keys(idx);
        domains.forEach((domain) => {
            const domainEntry = idx[domain] ?? {};
            const paths = Object.keys(domainEntry);
            paths.forEach((path) => {
                const pathEntry = domainEntry[path] ?? {};
                const keys = Object.keys(pathEntry);
                keys.forEach((key) => {
                    const keyEntry = pathEntry[key];
                    if (keyEntry != null) {
                        cookies.push(keyEntry);
                    }
                });
            });
        });
        // Sort by creationIndex so deserializing retains the creation order.
        // When implementing your own store, this SHOULD retain the order too
        cookies.sort((a, b) => {
            return (a.creationIndex || 0) - (b.creationIndex || 0);
        });
        cb(null, cookies);
        return promiseCallback.promise;
    }
}
exports.MemoryCookieStore = MemoryCookieStore;
function inspectFallback(val) {
    if (typeof val === 'string') {
        return `'${val}'`;
    }
    if (val && typeof val === 'object') {
        const domains = Object.keys(val);
        if (domains.length === 0) {
            return '[Object: null prototype] {}';
        }
        let result = '[Object: null prototype] {\n';
        Object.keys(val).forEach((domain, i) => {
            if ((0, utils_1.inOperator)(domain, val)) {
                result += formatDomain(domain, val[domain]);
                if (i < domains.length - 1) {
                    result += ',';
                }
                result += '\n';
            }
        });
        result += '}';
        return result;
    }
    return String(val);
}
exports.inspectFallback = inspectFallback;
function formatDomain(domainName, domainValue) {
    if (typeof domainValue === 'string') {
        return `'${domainValue}'`;
    }
    if (domainValue && typeof domainValue === 'object') {
        const indent = '  ';
        let result = `${indent}'${domainName}': [Object: null prototype] {\n`;
        Object.keys(domainValue).forEach((path, i, paths) => {
            if ((0, utils_1.inOperator)(path, domainValue)) {
                result += formatPath(path, domainValue[path]);
                if (i < paths.length - 1) {
                    result += ',';
                }
                result += '\n';
            }
        });
        result += `${indent}}`;
        return result;
    }
    return String(domainValue);
}
function formatPath(pathName, pathValue) {
    if (typeof pathValue === 'string') {
        return `'${pathValue}'`;
    }
    if (pathValue && typeof pathValue === 'object') {
        const indent = '    ';
        let result = `${indent}'${pathName}': [Object: null prototype] {\n`;
        Object.keys(pathValue).forEach((cookieName, i, cookieNames) => {
            if ((0, utils_1.inOperator)(cookieName, pathValue)) {
                const cookie = pathValue[cookieName];
                if (cookie != null &&
                    typeof cookie === 'object' &&
                    (0, utils_1.inOperator)('inspect', cookie) &&
                    typeof cookie.inspect === 'function') {
                    const inspectedValue = cookie.inspect();
                    if (typeof inspectedValue === 'string') {
                        result += `      ${cookieName}: ${inspectedValue}`;
                        if (i < cookieNames.length - 1) {
                            result += ',';
                        }
                        result += '\n';
                    }
                }
            }
        });
        result += `${indent}}`;
        return result;
    }
    return String(pathValue);
}
