/*
 * Decompiled with CFR 0.152.
 */
package org.web3j.crypto;

import java.math.BigInteger;
import java.nio.charset.StandardCharsets;
import java.security.SignatureException;
import java.util.Arrays;
import org.bouncycastle.asn1.x9.X9ECParameters;
import org.bouncycastle.asn1.x9.X9IntegerConverter;
import org.bouncycastle.crypto.ec.CustomNamedCurves;
import org.bouncycastle.crypto.params.ECDomainParameters;
import org.bouncycastle.math.ec.ECAlgorithms;
import org.bouncycastle.math.ec.ECPoint;
import org.bouncycastle.math.ec.FixedPointCombMultiplier;
import org.bouncycastle.math.ec.custom.sec.SecP256K1Curve;
import org.bouncycastle.util.BigIntegers;
import org.web3j.crypto.ECDSASignature;
import org.web3j.crypto.ECKeyPair;
import org.web3j.crypto.Hash;
import org.web3j.utils.Assertions;
import org.web3j.utils.Numeric;

public class Sign {
    public static final X9ECParameters CURVE_PARAMS = CustomNamedCurves.getByName((String)"secp256k1");
    public static final int CHAIN_ID_INC = 35;
    public static final int LOWER_REAL_V = 27;
    public static final int REPLAY_PROTECTED_V_MIN = 37;
    static final ECDomainParameters CURVE = new ECDomainParameters(CURVE_PARAMS.getCurve(), CURVE_PARAMS.getG(), CURVE_PARAMS.getN(), CURVE_PARAMS.getH());
    static final BigInteger HALF_CURVE_ORDER = CURVE_PARAMS.getN().shiftRight(1);
    static final String MESSAGE_PREFIX = "\u0019Ethereum Signed Message:\n";

    static byte[] getEthereumMessagePrefix(int messageLength) {
        return MESSAGE_PREFIX.concat(String.valueOf(messageLength)).getBytes(StandardCharsets.UTF_8);
    }

    public static byte[] getEthereumMessageHash(byte[] message) {
        byte[] prefix = Sign.getEthereumMessagePrefix(message.length);
        byte[] result = new byte[prefix.length + message.length];
        System.arraycopy(prefix, 0, result, 0, prefix.length);
        System.arraycopy(message, 0, result, prefix.length, message.length);
        return Hash.sha3((byte[])result);
    }

    public static SignatureData signPrefixedMessage(byte[] message, ECKeyPair keyPair) {
        return Sign.signMessage(Sign.getEthereumMessageHash(message), keyPair, false);
    }

    public static SignatureData signMessage(byte[] message, ECKeyPair keyPair) {
        return Sign.signMessage(message, keyPair, true);
    }

    public static SignatureData signMessage(byte[] message, ECKeyPair keyPair, boolean needToHash) {
        BigInteger publicKey = keyPair.getPublicKey();
        byte[] messageHash = needToHash ? Hash.sha3((byte[])message) : message;
        ECDSASignature sig = keyPair.sign(messageHash);
        return Sign.createSignatureData(sig, publicKey, messageHash);
    }

    public static SignatureData createSignatureData(ECDSASignature sig, BigInteger publicKey, byte[] messageHash) {
        int recId = -1;
        for (int i = 0; i < 4; ++i) {
            BigInteger k = Sign.recoverFromSignature(i, sig, messageHash);
            if (k == null || !k.equals(publicKey)) continue;
            recId = i;
            break;
        }
        if (recId == -1) {
            throw new RuntimeException("Could not construct a recoverable key. Are your credentials valid?");
        }
        int headerByte = recId + 27;
        byte[] v = new byte[]{(byte)headerByte};
        byte[] r = Numeric.toBytesPadded((BigInteger)sig.r, (int)32);
        byte[] s = Numeric.toBytesPadded((BigInteger)sig.s, (int)32);
        return new SignatureData(v, r, s);
    }

    public static BigInteger recoverFromSignature(int recId, ECDSASignature sig, byte[] message) {
        Assertions.verifyPrecondition((recId >= 0 && recId <= 3 ? 1 : 0) != 0, (String)"recId must be in the range of [0, 3]");
        Assertions.verifyPrecondition((sig.r.signum() >= 0 ? 1 : 0) != 0, (String)"r must be positive");
        Assertions.verifyPrecondition((sig.s.signum() >= 0 ? 1 : 0) != 0, (String)"s must be positive");
        Assertions.verifyPrecondition((message != null ? 1 : 0) != 0, (String)"message cannot be null");
        BigInteger n = CURVE.getN();
        BigInteger i = BigInteger.valueOf((long)recId / 2L);
        BigInteger x = sig.r.add(i.multiply(n));
        BigInteger prime = SecP256K1Curve.q;
        if (x.compareTo(prime) >= 0) {
            return null;
        }
        ECPoint R = Sign.decompressKey(x, (recId & 1) == 1);
        if (!R.multiply(n).isInfinity()) {
            return null;
        }
        BigInteger e = new BigInteger(1, message);
        BigInteger eInv = BigInteger.ZERO.subtract(e).mod(n);
        BigInteger rInv = sig.r.modInverse(n);
        BigInteger srInv = rInv.multiply(sig.s).mod(n);
        BigInteger eInvrInv = rInv.multiply(eInv).mod(n);
        ECPoint q = ECAlgorithms.sumOfTwoMultiplies((ECPoint)CURVE.getG(), (BigInteger)eInvrInv, (ECPoint)R, (BigInteger)srInv);
        byte[] qBytes = q.getEncoded(false);
        return new BigInteger(1, Arrays.copyOfRange(qBytes, 1, qBytes.length));
    }

    private static ECPoint decompressKey(BigInteger xBN, boolean yBit) {
        X9IntegerConverter x9 = new X9IntegerConverter();
        byte[] compEnc = x9.integerToBytes(xBN, 1 + x9.getByteLength(CURVE.getCurve()));
        compEnc[0] = (byte)(yBit ? 3 : 2);
        return CURVE.getCurve().decodePoint(compEnc);
    }

    public static BigInteger signedMessageToKey(byte[] message, SignatureData signatureData) throws SignatureException {
        return Sign.signedMessageHashToKey(Hash.sha3((byte[])message), signatureData);
    }

    public static BigInteger signedPrefixedMessageToKey(byte[] message, SignatureData signatureData) throws SignatureException {
        return Sign.signedMessageHashToKey(Sign.getEthereumMessageHash(message), signatureData);
    }

    public static BigInteger signedMessageHashToKey(byte[] messageHash, SignatureData signatureData) throws SignatureException {
        byte[] r = signatureData.getR();
        byte[] s = signatureData.getS();
        Assertions.verifyPrecondition((r != null && r.length == 32 ? 1 : 0) != 0, (String)"r must be 32 bytes");
        Assertions.verifyPrecondition((s != null && s.length == 32 ? 1 : 0) != 0, (String)"s must be 32 bytes");
        int header = signatureData.getV()[0] & 0xFF;
        if (header < 27 || header > 34) {
            throw new SignatureException("Header byte out of range: " + header);
        }
        int recId = header - 27;
        ECDSASignature sig = new ECDSASignature(new BigInteger(1, signatureData.getR()), new BigInteger(1, signatureData.getS()));
        BigInteger key = Sign.recoverFromSignature(recId, sig, messageHash);
        if (key == null) {
            throw new SignatureException("Could not recover public key from signature");
        }
        return key;
    }

    public static int getRecId(SignatureData signatureData, long chainId) {
        BigInteger v = Numeric.toBigInt((byte[])signatureData.getV());
        BigInteger lowerRealV = BigInteger.valueOf(27L);
        BigInteger lowerRealVPlus1 = BigInteger.valueOf(28L);
        BigInteger lowerRealVReplayProtected = BigInteger.valueOf(37L);
        BigInteger chainIdInc = BigInteger.valueOf(35L);
        if (v.equals(lowerRealV) || v.equals(lowerRealVPlus1)) {
            return v.subtract(lowerRealV).intValue();
        }
        if (v.compareTo(lowerRealVReplayProtected) >= 0) {
            return v.subtract(BigInteger.valueOf(chainId).multiply(BigIntegers.TWO)).subtract(chainIdInc).intValue();
        }
        throw new IllegalArgumentException(String.format("Unsupported v parameter: %s", v));
    }

    public static byte[] getVFromRecId(int recId) {
        return new byte[]{(byte)(27 + recId)};
    }

    public static BigInteger publicKeyFromPrivate(BigInteger privKey) {
        ECPoint point = Sign.publicPointFromPrivate(privKey);
        byte[] encoded = point.getEncoded(false);
        return new BigInteger(1, Arrays.copyOfRange(encoded, 1, encoded.length));
    }

    public static ECPoint publicPointFromPrivate(BigInteger privKey) {
        if (privKey.bitLength() > CURVE.getN().bitLength()) {
            privKey = privKey.mod(CURVE.getN());
        }
        return new FixedPointCombMultiplier().multiply(CURVE.getG(), privKey);
    }

    public static BigInteger publicFromPoint(byte[] bits) {
        return new BigInteger(1, Arrays.copyOfRange(bits, 1, bits.length));
    }

    public static class SignatureData {
        private final byte[] v;
        private final byte[] r;
        private final byte[] s;

        public SignatureData(byte v, byte[] r, byte[] s) {
            this(new byte[]{v}, r, s);
        }

        public SignatureData(byte[] v, byte[] r, byte[] s) {
            this.v = v;
            this.r = r;
            this.s = s;
        }

        public byte[] getV() {
            return this.v;
        }

        public byte[] getR() {
            return this.r;
        }

        public byte[] getS() {
            return this.s;
        }

        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || this.getClass() != o.getClass()) {
                return false;
            }
            SignatureData that = (SignatureData)o;
            if (!Arrays.equals(this.v, that.v)) {
                return false;
            }
            if (!Arrays.equals(this.r, that.r)) {
                return false;
            }
            return Arrays.equals(this.s, that.s);
        }

        public int hashCode() {
            int result = Arrays.hashCode(this.v);
            result = 31 * result + Arrays.hashCode(this.r);
            result = 31 * result + Arrays.hashCode(this.s);
            return result;
        }
    }
}

