/*
 * Decompiled with CFR 0.152.
 */
package org.voltdb.client;

import io.netty.handler.ssl.ClientAuth;
import io.netty.handler.ssl.SslContext;
import io.netty.handler.ssl.SslContextBuilder;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileReader;
import java.io.IOException;
import java.io.InputStream;
import java.nio.file.Files;
import java.nio.file.OpenOption;
import java.nio.file.Paths;
import java.security.GeneralSecurityException;
import java.security.KeyStore;
import java.util.Properties;
import javax.net.ssl.KeyManagerFactory;
import javax.net.ssl.TrustManagerFactory;
import org.voltcore.network.util.ssl.SSLConfiguration;
import org.voltdb.client.PemTruststore;
import org.voltdb.common.Constants;

public class ClientSslSetup {
    public static final String KEYSTORE_CONFIG_PROP = "keyStore";
    public static final String KEYSTORE_PASSWORD_CONFIG_PROP = "keyStorePassword";
    public static final String TRUSTSTORE_CONFIG_PROP = "trustStore";
    public static final String TRUSTSTORE_PASSWORD_CONFIG_PROP = "trustStorePassword";
    private static final String STORE_TYPE = "JKS";
    private static final String USER_HOME = System.getProperty("user.home");

    public static SslContext createClientSslContext(SslConfig sslConfig) {
        if (sslConfig == null) {
            throw new IllegalArgumentException("sslConfig is null");
        }
        try {
            SslContextBuilder builder = SslContextBuilder.forClient().clientAuth(ClientAuth.NONE).protocols(SSLConfiguration.ENABLED_PROTOCOLS).ciphers(SSLConfiguration.PREFERRED_CIPHERS, SSLConfiguration.CIPHER_FILTER);
            if (sslConfig.keyStorePath != null) {
                if (sslConfig.keyStorePassword == null || sslConfig.keyStorePassword.isEmpty()) {
                    throw new SslSetupException(String.format("No password was provided for keystore '%s', but is required", sslConfig.keyStorePath));
                }
                builder.keyManager(ClientSslSetup.keyManagerFactory(sslConfig.keyStorePath, sslConfig.keyStorePassword, sslConfig.keyStorePassword));
            }
            if (sslConfig.trustStorePath != null) {
                TrustManagerFactory fac = sslConfig.trustStoreIsPem ? ClientSslSetup.trustManagerFactory(PemTruststore.loadPemTruststore(sslConfig.trustStorePath, STORE_TYPE)) : ClientSslSetup.trustManagerFactory(sslConfig.trustStorePath, sslConfig.trustStorePassword);
                builder.trustManager(fac);
            }
            return builder.build();
        }
        catch (IOException | GeneralSecurityException | PemTruststore.PemException ex) {
            String exmsg = ex.getMessage();
            if (exmsg != null && exmsg.startsWith("Invalid keystore")) {
                throw new SslSetupException(String.format("%s using %s", exmsg, sslConfig));
            }
            throw new SslSetupException(String.format("Failed to initialize TLS/SSL using %s: %s", sslConfig, exmsg));
        }
    }

    private static KeyManagerFactory keyManagerFactory(String filepath, String keystorePassword, String keyPassword) throws IOException, GeneralSecurityException {
        ClientSslSetup.checkAccess("key", filepath);
        char[] storePassChars = null;
        if (keystorePassword != null && !keystorePassword.isEmpty()) {
            storePassChars = keystorePassword.toCharArray();
        }
        char[] keyPassChars = null;
        if (keyPassword != null && !keyPassword.isEmpty()) {
            keyPassChars = keyPassword.toCharArray();
        }
        KeyStore keyStore = KeyStore.getInstance(STORE_TYPE);
        try (FileInputStream keyStoreIS = new FileInputStream(filepath);){
            keyStore.load(keyStoreIS, storePassChars);
        }
        KeyManagerFactory kmf = KeyManagerFactory.getInstance(KeyManagerFactory.getDefaultAlgorithm());
        kmf.init(keyStore, keyPassChars);
        return kmf;
    }

    private static TrustManagerFactory trustManagerFactory(String filepath, String keystorePassword) throws IOException, GeneralSecurityException {
        ClientSslSetup.checkAccess("trust", filepath);
        char[] storePassChars = null;
        if (keystorePassword != null && !keystorePassword.isEmpty()) {
            storePassChars = keystorePassword.toCharArray();
        }
        KeyStore trustStore = KeyStore.getInstance(STORE_TYPE);
        try (FileInputStream trustStoreIS = new FileInputStream(filepath);){
            trustStore.load(trustStoreIS, storePassChars);
        }
        TrustManagerFactory trustFactory = TrustManagerFactory.getInstance(TrustManagerFactory.getDefaultAlgorithm());
        trustFactory.init(trustStore);
        return trustFactory;
    }

    private static TrustManagerFactory trustManagerFactory(KeyStore trustStore) throws GeneralSecurityException {
        TrustManagerFactory trustFactory = TrustManagerFactory.getInstance(TrustManagerFactory.getDefaultAlgorithm());
        trustFactory.init(trustStore);
        return trustFactory;
    }

    private static void checkAccess(String what, String path) {
        File f = new File(path);
        if (!f.exists()) {
            throw new SslSetupException(String.format("The %sstore file '%s' does not exist", what, path));
        }
        if (!f.canRead() || !f.isFile()) {
            throw new SslSetupException(String.format("The %sstore file '%s' cannot be read", what, path));
        }
    }

    public static SslConfig createSslConfig(String trustStorePath, String trustStorePassword) {
        boolean isPem = PemTruststore.looksLikePem(trustStorePath);
        return SslConfig.createWithTrustStoreOnly(trustStorePath, trustStorePassword, isPem);
    }

    public static SslConfig createSslConfig(String keyStorePath, String keyStorePassword, String trustStorePath, String trustStorePassword) {
        return SslConfig.createWithKeyStoreAndTrustStore(keyStorePath, keyStorePassword, trustStorePath, trustStorePassword);
    }

    public static SslConfig sslConfigFromPropertyFile(String path) {
        return ClientSslSetup.sslConfigFromPropertyFile(path, false);
    }

    public static SslConfig sslConfigFromPropertyFile(String path, boolean incKeyStore) {
        String string = path = path != null ? path.trim() : "";
        if (path.isEmpty()) {
            throw new IllegalArgumentException("TLS/SSL properties file name required");
        }
        File pf = new File(path = ClientSslSetup.expandTilde(path));
        if (!pf.exists()) {
            throw new SslConfigException(String.format("Properties file '%s' does not exist", path));
        }
        if (!pf.canRead() || !pf.isFile()) {
            throw new SslConfigException(String.format("Properties file '%s' cannot be read", path));
        }
        Properties props = new Properties();
        try (FileReader fr = new FileReader(pf);){
            props.load(fr);
        }
        catch (IOException ex) {
            throw new SslConfigException(String.format("I/O error on properties file '%s': %s", path, ex.getMessage()));
        }
        String keyStore = props.getProperty(KEYSTORE_CONFIG_PROP);
        String keyPswd = props.getProperty(KEYSTORE_PASSWORD_CONFIG_PROP);
        String trustStore = props.getProperty(TRUSTSTORE_CONFIG_PROP);
        String trustPswd = props.getProperty(TRUSTSTORE_PASSWORD_CONFIG_PROP);
        if (incKeyStore) {
            ClientSslSetup.requireProperty(keyStore, KEYSTORE_CONFIG_PROP, path);
            ClientSslSetup.requireProperty(trustStore, TRUSTSTORE_CONFIG_PROP, path);
            return SslConfig.createWithKeyStoreAndTrustStore(keyStore, keyPswd, trustStore, trustPswd);
        }
        ClientSslSetup.requireProperty(trustStore, TRUSTSTORE_CONFIG_PROP, path);
        return SslConfig.createWithTrustStoreOnly(trustStore, trustPswd, false);
    }

    private static void requireProperty(String store, String prop, String path) {
        if (store == null || store.isEmpty()) {
            throw new SslConfigException(String.format("Required property '%s' not present in property file '%s'", prop, path));
        }
    }

    public static SslConfig sslConfigFromDefaults() {
        return SslConfig.createWithDefaultTrustStoreOnly();
    }

    public static SslConfig sslConfigFromDefaults(boolean incKeyStore) {
        if (incKeyStore) {
            return SslConfig.createWithDefaultKeyStoreAndTrustStore();
        }
        return SslConfig.createWithDefaultTrustStoreOnly();
    }

    public static SslConfig sslConfigTrustStoreGeneric(String path) {
        String string = path = path != null ? path.trim() : "";
        if (path.isEmpty()) {
            throw new IllegalArgumentException("TLS/SSL file name required");
        }
        if (ClientSslSetup.looksLikeJavaKeystore(path = ClientSslSetup.expandTilde(path))) {
            return SslConfig.createWithTrustStoreOnly(path, null, false);
        }
        if (PemTruststore.looksLikePem(path)) {
            return SslConfig.createWithTrustStoreOnly(path, null, true);
        }
        return ClientSslSetup.sslConfigFromPropertyFile(path, false);
    }

    private static boolean looksLikeJavaKeystore(String path) {
        int sampleSize = 64;
        byte[] jksMagic = new byte[]{-2, -19, -2, -19};
        boolean binary = false;
        try (InputStream in = Files.newInputStream(Paths.get(path, new String[0]), new OpenOption[0]);){
            byte[] buff = new byte[64];
            int len = in.read(buff);
            if (len == 64) {
                byte b;
                int i;
                binary = true;
                for (i = 0; i < jksMagic.length; ++i) {
                    binary &= buff[i] == jksMagic[i];
                }
                for (i = 0; !binary && i < len; binary |= ((b = buff[i]) & 0x7F) < 32 && b != 13 && b != 10 && b != 9, ++i) {
                }
            }
        }
        catch (FileNotFoundException ex) {
            throw new SslConfigException(String.format("TLS/SSL file '%s' does not exist", path));
        }
        catch (Exception ex) {
            throw new SslConfigException(String.format("TLS/SSL file '%s' cannot be read", path));
        }
        return binary;
    }

    private static String expandTilde(String s) {
        return s.startsWith("~/") ? s.replace("~", USER_HOME) : s;
    }

    public static class SslConfig {
        public final String keyStorePath;
        public final String keyStorePassword;
        public final String trustStorePath;
        public final String trustStorePassword;
        public final boolean trustStoreIsPem;

        private SslConfig(boolean initKeyStore, String keyStorePath, String keyStorePassword, boolean initTrustStore, String trustStorePath, String trustStorePassword, boolean pemTrust) {
            String keyPath = null;
            String keyPswd = null;
            if (initKeyStore) {
                keyPath = SslConfig.normalize(keyStorePath, null);
                keyPswd = SslConfig.normalize(keyStorePassword, "");
                if (keyPath == null) {
                    keyPath = SslConfig.normalize(System.getProperty("javax.net.ssl.keyStore"), null);
                    keyPswd = SslConfig.normalize(System.getProperty("javax.net.ssl.keyStorePassword"), "");
                }
                if (keyPath == null) {
                    keyPath = "keystore";
                    keyPswd = "password";
                }
            }
            String trustPath = null;
            String trustPswd = null;
            if (initTrustStore) {
                trustPath = SslConfig.normalize(trustStorePath, null);
                trustPswd = SslConfig.normalize(trustStorePassword, "");
                if (trustPath == null) {
                    trustPath = SslConfig.normalize(System.getProperty("javax.net.ssl.trustStore"), null);
                    trustPswd = SslConfig.normalize(System.getProperty("javax.net.ssl.trustStorePassword"), "");
                    pemTrust = false;
                }
                if (trustPath == null) {
                    trustPath = Constants.DEFAULT_TRUSTSTORE_RESOURCE;
                    trustPswd = "changeit";
                    pemTrust = false;
                }
            }
            this.keyStorePath = keyPath;
            this.keyStorePassword = keyPswd;
            this.trustStorePath = trustPath;
            this.trustStorePassword = trustPswd;
            this.trustStoreIsPem = pemTrust;
        }

        private static String normalize(String str, String def) {
            String out = def;
            if (str != null && (out = str.trim()).isEmpty()) {
                out = def;
            }
            return out;
        }

        static SslConfig createWithTrustStoreOnly(String trustStorePath, String trustStorePassword, boolean isPemFile) {
            return new SslConfig(false, null, null, true, trustStorePath, trustStorePassword, isPemFile);
        }

        static SslConfig createWithKeyStoreAndTrustStore(String keyStorePath, String keyStorePassword, String trustStorePath, String trustStorePassword) {
            return new SslConfig(true, keyStorePath, keyStorePassword, true, trustStorePath, trustStorePassword, false);
        }

        static SslConfig createWithDefaultTrustStoreOnly() {
            return new SslConfig(false, null, null, true, null, null, false);
        }

        static SslConfig createWithDefaultKeyStoreAndTrustStore() {
            return new SslConfig(true, null, null, true, null, null, false);
        }

        public String toString() {
            if (this.keyStorePath == null) {
                return String.format("SslConfig [trustStorePath=%s]", this.trustStorePath);
            }
            return String.format("SslConfig [keyStorePath=%s, trustStorePath=%s]", this.keyStorePath, this.trustStorePath);
        }
    }

    public static class SslSetupException
    extends RuntimeException {
        SslSetupException(String msg) {
            super(String.format("TLS/SSL setup failed: %s", msg));
        }
    }

    public static class SslConfigException
    extends RuntimeException {
        SslConfigException(String msg) {
            super(msg);
        }
    }
}

