/*
 * Decompiled with CFR 0.152.
 */
package org.voltcore.memory;

import java.io.PrintWriter;
import java.io.StringWriter;
import java.lang.reflect.Method;
import java.nio.ByteBuffer;
import java.util.Map;
import java.util.Objects;
import java.util.Queue;
import java.util.concurrent.ConcurrentLinkedQueue;
import java.util.concurrent.atomic.AtomicLong;
import org.cliffc_voltpatches.high_scale_lib.NonBlockingHashMap;
import org.voltcore.logging.VoltLogger;
import org.voltcore.memory.DBBPoolNative;
import org.voltcore.memory.MemcheckConfigurator;
import org.voltcore.memory.MemoryCheckExtension;
import org.voltcore.memory.SkipMemoryCheckExtension;
import org.voltcore.memory.direct.VoltUnsafe;
import sun.nio.ch.DirectBuffer;

public final class DBBPool {
    public static final boolean SHOULD_CHECK_MEMORY = MemcheckConfigurator.shouldCheckMemory();
    private static final VoltLogger LOGGER = new VoltLogger("DBBPOOL");
    private static final Map<Integer, Queue<BBContainer>> m_pooledBuffers = new NonBlockingHashMap<Integer, Queue<BBContainer>>();
    private static final AtomicLong bytesAllocatedGlobally = new AtomicLong(0L);

    public static void cleanup() {
        if (SHOULD_CHECK_MEMORY) {
            System.gc();
            System.runFinalization();
        }
    }

    static Map<Integer, Queue<BBContainer>> getPool() {
        return m_pooledBuffers;
    }

    public static BBContainer dummyWrapBB(ByteBuffer b) {
        return new BBWrapperContainer(b);
    }

    public static BBContainer wrapBB(ByteBuffer b) {
        if (b.isDirect()) {
            return new DBBWrapperContainer(b);
        }
        return new BBWrapperContainer(b);
    }

    static long getBytesAllocatedGlobally() {
        return bytesAllocatedGlobally.get();
    }

    static int roundToClosestPowerOf2(int capacity) {
        if (capacity == 0) {
            return 0;
        }
        if (capacity == 1) {
            return 2;
        }
        int result = Integer.highestOneBit(capacity - 1) << 1;
        return result < 0 ? capacity : result;
    }

    public static BBContainer allocateDirectAndPool(Integer capacity) {
        int bucket = DBBPool.roundToClosestPowerOf2(capacity);
        Queue pooledBuffers = m_pooledBuffers.computeIfAbsent(bucket, key -> new ConcurrentLinkedQueue());
        BBContainer container = (BBContainer)pooledBuffers.poll();
        if (container == null) {
            container = DBBPool.allocateDirect(bucket);
        }
        container = new BBCachingContainer(container, c -> {
            pooledBuffers.offer(c);
            return false;
        });
        container.b().clear();
        container.b().limit(capacity);
        return container;
    }

    public static void clear() {
        long startingBytes = bytesAllocatedGlobally.get();
        for (Queue<BBContainer> pool : m_pooledBuffers.values()) {
            BBContainer container;
            while ((container = pool.poll()) != null) {
                container.discard();
            }
        }
        LOGGER.warn("Attempted to resolve DirectByteBuffer OOM by freeing pooled buffers. Starting bytes was " + startingBytes + " after clearing " + bytesAllocatedGlobally.get() + " change " + (startingBytes - bytesAllocatedGlobally.get()));
    }

    public static BBContainer allocateDirect(int capacity) {
        ByteBuffer byteBuffer;
        try {
            byteBuffer = ByteBuffer.allocateDirect(capacity);
        }
        catch (OutOfMemoryError e) {
            if (e.getMessage().contains("Direct buffer memory")) {
                DBBPool.clear();
                byteBuffer = ByteBuffer.allocateDirect(capacity);
            }
            throw new Error(e);
        }
        bytesAllocatedGlobally.getAndAdd(capacity);
        DBBPool.logAllocation(capacity);
        return new DeallocatingContainer(byteBuffer, bytesAllocatedGlobally);
    }

    public static BBContainer allocateUnsafeByteBuffer(long size) {
        return new NDBBWrapperContainer(DBBPoolNative.nativeAllocateUnsafeByteBuffer(size));
    }

    private static void logAllocation(int capacity) {
        if (LOGGER.isTraceEnabled()) {
            String message = "Allocated DBB capacity " + capacity + " total allocated " + bytesAllocatedGlobally.get() + " from " + DBBPool.throwableToString(new Throwable());
            LOGGER.trace(message);
        }
    }

    public static BBContainer allocateAlignedUnsafeByteBuffer(int alignment, int size) {
        return new NDBBWrapperContainer(DBBPoolNative.nativeAllocateAlignedUnsafeByteBuffer(alignment, size));
    }

    private static void cleanByteBuffer(ByteBuffer buf) {
        if (buf == null || !buf.isDirect()) {
            return;
        }
        VoltUnsafe.cleanDirectBuffer(buf);
    }

    public static String throwableToString(Throwable t) {
        StringWriter sw = new StringWriter();
        PrintWriter pw = new PrintWriter(sw);
        t.printStackTrace(pw);
        pw.flush();
        return sw.toString();
    }

    private static void checkArgument(boolean result, String exceptionMessage) {
        if (!result) {
            throw new IllegalStateException(exceptionMessage);
        }
    }

    public static final class BBWrapperContainer
    extends BBContainer {
        private BBWrapperContainer(ByteBuffer b) {
            super(b);
        }
    }

    public static final class DBBWrapperContainer
    extends BBContainer {
        private DBBWrapperContainer(ByteBuffer b) {
            super(b);
        }

        @Override
        public void discard() {
            ByteBuffer buf = this.checkDoubleFree();
            DBBPool.cleanByteBuffer(buf);
        }
    }

    public static abstract class BBContainer {
        private static final MemoryCheckExtension NOOP = new SkipMemoryCheckExtension();
        private final MemoryCheckExtension memoryChecker;
        private final ByteBuffer buffer;
        private volatile boolean m_previouslyFreed = false;

        static void crash(String msg, boolean stackTrace, Throwable crashReason) {
            try {
                Class<?> vdbClz = Class.forName("org.voltdb.VoltDB");
                Method m = vdbClz.getMethod("crashLocalVoltDB", String.class, Boolean.TYPE, Throwable.class);
                m.invoke(null, msg, stackTrace, crashReason);
            }
            catch (Exception ex) {
                LOGGER.fatal(msg, ex);
                System.err.println(msg);
                crashReason.printStackTrace();
                System.exit(-1);
            }
        }

        public BBContainer(ByteBuffer buffer) {
            Objects.requireNonNull(buffer, "Given buffer cannot be null");
            this.buffer = buffer;
            this.memoryChecker = SHOULD_CHECK_MEMORY ? new MemoryCheckExtension() : NOOP;
        }

        public final long address() {
            this.checkUseAfterFree();
            return ((DirectBuffer)((Object)this.buffer)).address();
        }

        public void discard() {
            this.checkDoubleFree();
        }

        public ByteBuffer b() {
            this.checkUseAfterFree();
            return this.buffer;
        }

        public final ByteBuffer bD() {
            return this.b().duplicate();
        }

        public final ByteBuffer bDR() {
            return this.b().asReadOnlyBuffer();
        }

        protected final ByteBuffer checkDoubleFree() {
            this.memoryChecker.checkDoubleFree();
            if (this.m_previouslyFreed) {
                BBContainer.crash("Double free in DBBPool", true, null);
            }
            this.m_previouslyFreed = true;
            return this.buffer;
        }

        public final void tag(String tag) {
            this.memoryChecker.tag(tag);
        }

        public final void addToTagTrail(String tag) {
            this.memoryChecker.addToTagTrail(tag);
        }

        public final boolean isTagged() {
            return this.memoryChecker.isTagged();
        }

        public String toString() {
            return this.getClass().getSimpleName() + ": " + this.buffer;
        }

        private void checkUseAfterFree() {
            if (this.m_previouslyFreed) {
                BBContainer.crash("Used after free in DBBPool", true, null);
            }
        }
    }

    public static class BBCachingContainer
    extends BBContainer {
        private final BBContainer m_delegate;
        private final CachingPredicate predicate;

        public BBCachingContainer(BBContainer delegate, CachingPredicate predicate) {
            super(delegate.b());
            this.m_delegate = delegate;
            this.predicate = predicate;
        }

        @Override
        public void discard() {
            this.checkDoubleFree();
            if (this.predicate.shouldDiscardDelegate(this.m_delegate)) {
                this.m_delegate.discard();
            }
        }

        @FunctionalInterface
        public static interface CachingPredicate {
            public boolean shouldDiscardDelegate(BBContainer var1);
        }
    }

    private static class DeallocatingContainer
    extends BBContainer {
        private final AtomicLong bytesAllocatedGlobally;

        private DeallocatingContainer(ByteBuffer buf, AtomicLong bytesAllocatedGlobally) {
            super(buf);
            this.bytesAllocatedGlobally = bytesAllocatedGlobally;
        }

        @Override
        public void discard() {
            ByteBuffer buf = this.checkDoubleFree();
            try {
                this.bytesAllocatedGlobally.getAndAdd(-buf.capacity());
                this.logDeallocation(buf.capacity());
                DBBPool.cleanByteBuffer(buf);
            }
            catch (Throwable e) {
                LOGGER.fatal("Failed to deallocate direct byte buffer", e);
                DeallocatingContainer.crash("Failed to deallocate direct byte buffer", false, e);
            }
        }

        private void logDeallocation(int capacity) {
            if (LOGGER.isTraceEnabled()) {
                String message = "Deallocated DBB capacity " + capacity + " total allocated " + this.bytesAllocatedGlobally.get() + " from " + DBBPool.throwableToString(new Throwable());
                LOGGER.trace(message);
            }
        }
    }

    public static final class NDBBWrapperContainer
    extends BBContainer {
        private NDBBWrapperContainer(ByteBuffer b) {
            super(Objects.requireNonNull(b, "Given buffer cannot be null!"));
            DBBPool.checkArgument(b.isDirect(), "Given buffer must be direct!");
        }

        @Override
        public void discard() {
            DirectBuffer dbuf = (DirectBuffer)((Object)this.checkDoubleFree());
            DBBPoolNative.nativeFreeMemory(dbuf.address());
        }
    }

    public static class DBBDelegateContainer
    extends BBContainer {
        protected final BBContainer m_delegate;

        protected DBBDelegateContainer(BBContainer delegate) {
            super(delegate.b());
            this.m_delegate = delegate;
        }

        @Override
        public void discard() {
            this.checkDoubleFree();
            this.m_delegate.discard();
        }
    }
}

