/*
 * Copyright (c) 2015-2020, Oracle and/or its affiliates. All rights reserved.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.tribuo.classification.explanations.lime;

import org.tribuo.Model;
import org.tribuo.Prediction;
import org.tribuo.SparseModel;
import org.tribuo.classification.Label;
import org.tribuo.classification.explanations.Explanation;
import org.tribuo.regression.Regressor;
import org.tribuo.regression.evaluation.RegressionEvaluation;

import java.util.List;
import java.util.Map;

/**
 * An {@link Explanation} using LIME.
 * <p>
 * Wraps a {@link SparseModel} {@link Regressor} which is trained to predict the probabilities
 * generated by the true {@link Model}.
 * <p>
 * See:
 * <pre>
 * Ribeiro MT, Singh S, Guestrin C.
 * "Why should I trust you?: Explaining the predictions of any classifier"
 * Proceedings of the 22nd ACM SIGKDD International Conference on Knowledge Discovery and Data Mining 2016.
 * </pre>
 */
public class LIMEExplanation implements Explanation<Regressor> {
    private static final long serialVersionUID = 1L;

    private final SparseModel<Regressor> model;

    private final Prediction<Label> prediction;

    private final RegressionEvaluation evaluation;

    /**
     * Constructs a LIME explanation.
     * @param model The explanation model.
     * @param prediction The prediction being explained.
     * @param evaluation The evaluation of the explanation model on the sampled data.
     */
    public LIMEExplanation(SparseModel<Regressor> model, Prediction<Label> prediction, RegressionEvaluation evaluation) {
        this.model = model;
        this.prediction = prediction;
        this.evaluation = evaluation;
    }

    @Override
    public List<String> getActiveFeatures() {
        Map<String,List<String>> features = model.getActiveFeatures();
        if (features.containsKey(Model.ALL_OUTPUTS)) {
            return features.get(Model.ALL_OUTPUTS);
        } else {
            return features.get(prediction.getOutput().getLabel());
        }
    }

    @Override
    public SparseModel<Regressor> getModel() {
        return model;
    }

    @Override
    public Prediction<Label> getPrediction() {
        return prediction;
    }

    /**
     * Gets the evaluator which scores how close the sparse model's
     * predictions are to the complex model's predictions.
     * @return The evaluation.
     */
    public RegressionEvaluation getEvaluation() {
        return evaluation;
    }

    /**
     * Get the RMSE of a specific dimension of the explanation model.
     * @param name The dimension to look at.
     * @return The RMSE of the explanation model.
     */
    public double getRMSE(String name) {
        return evaluation.rmse().get(new Regressor.DimensionTuple(name,Double.NaN));
    }

    @Override
    public String toString() {
        return "LIMEExplanation(linearRMSE="+evaluation.rmse()+",modelPrediction="+prediction+",activeFeatures="+getActiveFeatures().toString()+")";
    }
}
