/*
 *  Copyright 2016 Alexey Andreev.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *       http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.teavm.classlib.java.lang.reflect;

public class TModifier {
    private static String[] modifierNames;
    public static final int PUBLIC = 1;
    public static final int PRIVATE = 2;
    public static final int PROTECTED = 4;
    public static final int STATIC = 8;
    public static final int FINAL = 16;
    public static final int SYNCHRONIZED = 32;
    public static final int VOLATILE = 64;
    public static final int TRANSIENT = 128;
    public static final int NATIVE = 256;
    public static final int INTERFACE = 512;
    public static final int ABSTRACT = 1024;
    public static final int STRICT = 2048;
    private static final int[] canonicalOrder = { PUBLIC, PROTECTED, PRIVATE, ABSTRACT, STATIC, FINAL, TRANSIENT,
            VOLATILE, SYNCHRONIZED, NATIVE, STRICT, INTERFACE };

    private TModifier() {
    }

    public static boolean isPublic(int mod) {
        return (mod & PUBLIC) != 0;
    }

    public static boolean isPrivate(int mod) {
        return (mod & PRIVATE) != 0;
    }

    public static boolean isProtected(int mod) {
        return (mod & PROTECTED) != 0;
    }

    public static boolean isStatic(int mod) {
        return (mod & STATIC) != 0;
    }

    public static boolean isFinal(int mod) {
        return (mod & FINAL) != 0;
    }

    public static boolean isSynchronized(int mod) {
        return (mod & SYNCHRONIZED) != 0;
    }

    public static boolean isVolatile(int mod) {
        return (mod & VOLATILE) != 0;
    }

    public static boolean isTransient(int mod) {
        return (mod & TRANSIENT) != 0;
    }

    public static boolean isNative(int mod) {
        return (mod & NATIVE) != 0;
    }

    public static boolean isInterface(int mod) {
        return (mod & INTERFACE) != 0;
    }

    public static boolean isAbstract(int mod) {
        return (mod & ABSTRACT) != 0;
    }

    public static boolean isStrict(int mod) {
        return (mod & STRICT) != 0;
    }

    public static String toString(int mod) {
       return toString(mod, false);
    }

    public static String toString(int mod, boolean abstractExpected) {
        StringBuilder sb = new StringBuilder();
        String[] modifierNames = getModifierNames();
        int index = 0;
        for (int modifier : canonicalOrder) {
            if (modifier == ABSTRACT && abstractExpected) {
                if ((mod & ABSTRACT) == 0) {
                    if (sb.length() > 0) {
                        sb.append(' ');
                    }
                    sb.append("default");
                }
            } else if ((mod & modifier) != 0) {
                if (sb.length() > 0) {
                    sb.append(' ');
                }
                sb.append(modifierNames[index]);
            }
            ++index;
        }
        return sb.toString();
    }

    private static String[] getModifierNames() {
        if (modifierNames == null) {
            modifierNames = new String[] { "public", "protected", "private", "abstract", "static", "final",
                    "transient", "volatile", "synchronized", "native", "strictfp", "interface" };
        }
        return modifierNames;
    }
}
