/*
 *  Copyright 2016 Alexey Andreev.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *       http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.teavm.classlib.java.lang;

import java.util.Set;
import org.teavm.backend.javascript.codegen.SourceWriter;
import org.teavm.backend.javascript.rendering.Precedence;
import org.teavm.backend.javascript.rendering.RenderingUtil;
import org.teavm.backend.javascript.spi.Generator;
import org.teavm.backend.javascript.spi.GeneratorContext;
import org.teavm.backend.javascript.spi.Injector;
import org.teavm.backend.javascript.spi.InjectorContext;
import org.teavm.classlib.impl.ReflectionDependencyListener;
import org.teavm.dependency.DependencyAgent;
import org.teavm.dependency.DependencyPlugin;
import org.teavm.dependency.MethodDependency;
import org.teavm.model.AccessLevel;
import org.teavm.model.ClassReader;
import org.teavm.model.ElementModifier;
import org.teavm.model.FieldReader;
import org.teavm.model.FieldReference;
import org.teavm.model.MemberReader;
import org.teavm.model.MethodDescriptor;
import org.teavm.model.MethodReader;
import org.teavm.model.MethodReference;
import org.teavm.model.ValueType;

public class ClassGenerator implements Generator, Injector, DependencyPlugin {
    private static final FieldReference platformClassField =
            new FieldReference(Class.class.getName(), "platformClass");

    @Override
    public void generate(GeneratorContext context, SourceWriter writer, MethodReference methodRef) {
        switch (methodRef.getName()) {
            case "createMetadata":
                generateCreateMetadata(context, writer);
                break;
        }
    }

    @Override
    public void generate(InjectorContext context, MethodReference methodRef) {
        switch (methodRef.getName()) {
            case "newEmptyInstance":
                context.getWriter().append("new").ws().append("(");
                context.writeExpr(context.getArgument(0), Precedence.MEMBER_ACCESS);
                context.getWriter().append('.').appendField(platformClassField);
                context.getWriter().append(")");
                break;
        }
    }

    @Override
    public void methodReached(DependencyAgent agent, MethodDependency method) {
        switch (method.getReference().getName()) {
            case "newEmptyInstance":
                method.getVariable(0).getClassValueNode().addConsumer(type -> {
                    String className = type.getName();
                    if (!className.startsWith("[") && !className.startsWith("~")) {
                        ClassReader cls = agent.getClassSource().get(className);
                        if (cls != null && !cls.hasModifier(ElementModifier.ABSTRACT)
                                && !cls.hasModifier(ElementModifier.INTERFACE)) {
                            method.getResult().propagate(type);
                        }
                    }
                });
                break;
            case "getSuperclass":
                reachGetSuperclass(agent, method);
                break;
            case "getInterfaces":
                reachGetInterfaces(agent, method);
                break;
            case "getComponentType":
                reachGetComponentType(agent, method);
                break;
        }
    }

    private void reachGetSuperclass(DependencyAgent agent, MethodDependency method) {
        method.getVariable(0).getClassValueNode().addConsumer(type -> {
            String className = type.getName();
            if (className.startsWith("[")) {
                return;
            }

            ClassReader cls = agent.getClassSource().get(className);
            if (cls != null && cls.getParent() != null) {
                method.getResult().getClassValueNode().propagate(agent.getType(cls.getParent()));
            }
        });
    }

    private void reachGetInterfaces(DependencyAgent agent, MethodDependency method) {
        method.getVariable(0).getClassValueNode().addConsumer(type -> {
            String className = type.getName();
            if (className.startsWith("[")) {
                return;
            }

            ClassReader cls = agent.getClassSource().get(className);
            method.getResult().propagate(agent.getType("[java/lang/Class;"));
            method.getResult().getArrayItem().propagate(agent.getType("java/lang/Class;"));
            if (cls != null) {
                for (String iface : cls.getInterfaces()) {
                    method.getResult().getArrayItem().getClassValueNode().propagate(agent.getType(iface));
                }
            }
        });
    }

    private void reachGetComponentType(DependencyAgent agent, MethodDependency method) {
        method.getVariable(0).getClassValueNode().addConsumer(t -> {
            if (!t.getName().startsWith("[")) {
                return;
            }
            String typeName = t.getName().substring(1);
            if (typeName.charAt(0) == 'L') {
                typeName = ((ValueType.Object) ValueType.parse(typeName)).getClassName();
            }
            method.getResult().getClassValueNode().propagate(agent.getType(typeName));
        });
    }

    private void generateCreateMetadata(GeneratorContext context, SourceWriter writer) {
        ReflectionDependencyListener reflection = context.getService(ReflectionDependencyListener.class);
        for (String className : reflection.getClassesWithReflectableFields()) {
            generateCreateFieldsForClass(context, writer, className);
        }
        for (String className : reflection.getClassesWithReflectableMethods()) {
            generateCreateMethodsForClass(context, writer, className);
        }
    }

    private void generateCreateFieldsForClass(GeneratorContext context, SourceWriter writer, String className) {
        ReflectionDependencyListener reflection = context.getService(ReflectionDependencyListener.class);
        Set<String> accessibleFields = reflection.getAccessibleFields(className);

        ClassReader cls = context.getClassSource().get(className);
        if (cls == null) {
            return;
        }

        writer.appendClass(className).append(".$meta.fields").ws().append('=').ws().append('[').indent();

        var skipPrivates = ReflectionDependencyListener.shouldSkipPrivates(cls);
        generateCreateMembers(writer, skipPrivates, cls.getFields(), field -> {
            appendProperty(writer, "type", false, () -> context.typeToClassString(writer, field.getType()));

            appendProperty(writer, "getter", false, () -> {
                if (accessibleFields != null && accessibleFields.contains(field.getName())
                        && reflection.isRead(field.getReference())) {
                    renderGetter(context, writer, field);
                } else {
                    writer.append("null");
                }
            });

            appendProperty(writer, "setter", false, () -> {
                if (accessibleFields != null && accessibleFields.contains(field.getName())
                        && reflection.isWritten(field.getReference())) {
                    renderSetter(context, writer, field);
                } else {
                    writer.append("null");
                }
            });
        });

        writer.outdent().append("];").softNewLine();
    }

    private void generateCreateMethodsForClass(GeneratorContext context, SourceWriter writer, String className) {
        ReflectionDependencyListener reflection = context.getService(ReflectionDependencyListener.class);
        Set<MethodDescriptor> accessibleMethods = reflection.getAccessibleMethods(className);

        ClassReader cls = context.getInitialClassSource().get(className);
        if (cls == null) {
            return;
        }

        writer.appendClass(className).append(".$meta.methods").ws().append('=').ws().append('[').indent();

        var skipPrivates = ReflectionDependencyListener.shouldSkipPrivates(cls);
        generateCreateMembers(writer, skipPrivates, cls.getMethods(), method -> {
            appendProperty(writer, "parameterTypes", false, () -> {
                writer.append('[');
                for (int i = 0; i < method.parameterCount(); ++i) {
                    if (i > 0) {
                        writer.append(',').ws();
                    }
                    context.typeToClassString(writer, method.parameterType(i));
                }
                writer.append(']');
            });

            appendProperty(writer, "returnType", false, () -> {
                context.typeToClassString(writer, method.getResultType());
            });

            appendProperty(writer, "callable", false, () -> {
                if (accessibleMethods != null && accessibleMethods.contains(method.getDescriptor())
                        && reflection.isCalled(method.getReference())) {
                    renderCallable(context, writer, method);
                } else {
                    writer.append("null");
                }
            });
        });

        writer.outdent().append("];").softNewLine();
    }

    private <T extends MemberReader> void generateCreateMembers(SourceWriter writer, boolean skipPrivates,
            Iterable<T> members, MemberRenderer<T> renderer) {
        boolean first = true;
        for (T member : members) {
            if (skipPrivates) {
                if (member.getLevel() == AccessLevel.PRIVATE || member.getLevel() == AccessLevel.PACKAGE_PRIVATE) {
                    continue;
                }
            }
            if (!first) {
                writer.append(",").ws();
            } else {
                writer.softNewLine();
            }
            first = false;
            writer.append("{").indent().softNewLine();

            appendProperty(writer, "name", true, () ->  writer.append('"')
                    .append(RenderingUtil.escapeString(member.getName())).append('"'));
            appendProperty(writer, "modifiers", false, () -> writer.append(
                    ElementModifier.pack(member.readModifiers())));
            appendProperty(writer, "accessLevel", false, () -> writer.append(member.getLevel().ordinal()));
            renderer.render(member);
            writer.outdent().softNewLine().append("}");
        }
    }

    private void appendProperty(SourceWriter writer, String name, boolean first, Fragment value) {
        if (!first) {
            writer.append(",").softNewLine();
        }
        writer.append(name).ws().append(':').ws();
        value.render();
    }

    private void renderGetter(GeneratorContext context, SourceWriter writer, FieldReader field) {
        writer.append("function(obj)").ws().append("{").indent().softNewLine();
        initClass(context, writer, field);
        writer.append("return ");
        boxIfNecessary(writer, field.getType(), () -> fieldAccess(writer, field));
        writer.append(";").softNewLine();
        writer.outdent().append("}");
    }

    private void renderSetter(GeneratorContext context, SourceWriter writer, FieldReader field) {
        writer.append("function(obj,").ws().append("val)").ws().append("{").indent().softNewLine();
        initClass(context, writer, field);
        fieldAccess(writer, field);
        writer.ws().append('=').ws();
        unboxIfNecessary(writer, field.getType(), () -> writer.append("val"));
        writer.append(";").softNewLine();
        writer.outdent().append("}");
    }

    private void renderCallable(GeneratorContext context, SourceWriter writer, MethodReader method) {
        writer.append("function(obj,").ws().append("args)").ws().append("{").indent().softNewLine();

        initClass(context, writer, method);

        if (method.getResultType() != ValueType.VOID) {
            writer.append("return ");
        }
        var receiverWritten = false;
        if (!method.hasModifier(ElementModifier.STATIC) && !method.hasModifier(ElementModifier.FINAL)
                && method.getLevel() != AccessLevel.PRIVATE && !method.getName().equals("<init>")) {
            writer.append("obj.").appendVirtualMethod(method.getDescriptor());
            receiverWritten = true;
        } else {
            writer.appendMethod(method.getReference());
        }

        writer.append('(');
        boolean first = true;
        if (!receiverWritten && !method.hasModifier(ElementModifier.STATIC)) {
            writer.append("obj").ws();
            first = false;
        }
        for (int i = 0; i < method.parameterCount(); ++i) {
            if (!first) {
                writer.append(',').ws();
            }
            first = false;
            int index = i;
            unboxIfNecessary(writer, method.parameterType(i), () -> writer.append("args[" + index + "]"));
        }
        writer.append(");").softNewLine();

        if (method.getResultType() == ValueType.VOID) {
            writer.append("return null;").softNewLine();
        }
        writer.outdent().append("}");
    }

    private void initClass(GeneratorContext context, SourceWriter writer, MemberReader member) {
        if (member.hasModifier(ElementModifier.STATIC) && context.isDynamicInitializer(member.getOwnerName())) {
            writer.appendClassInit(member.getOwnerName()).append("();").softNewLine();
        }
    }

    private void fieldAccess(SourceWriter writer, FieldReader field) {
        if (field.hasModifier(ElementModifier.STATIC)) {
            writer.appendStaticField(field.getReference());
        } else {
            writer.append("obj.").appendField(field.getReference());
        }
    }

    private void boxIfNecessary(SourceWriter writer, ValueType type, Fragment fragment) {
        boolean boxed = false;
        if (type instanceof ValueType.Primitive) {
            switch (((ValueType.Primitive) type).getKind()) {
                case BOOLEAN:
                    writer.appendMethod(new MethodReference(Boolean.class, "valueOf", boolean.class,
                            Boolean.class));
                    break;
                case BYTE:
                    writer.appendMethod(new MethodReference(Byte.class, "valueOf", byte.class, Byte.class));
                    break;
                case SHORT:
                    writer.appendMethod(new MethodReference(Short.class, "valueOf", short.class, Short.class));
                    break;
                case CHARACTER:
                    writer.appendMethod(new MethodReference(Character.class, "valueOf", char.class,
                            Character.class));
                    break;
                case INTEGER:
                    writer.appendMethod(new MethodReference(Integer.class, "valueOf", int.class, Integer.class));
                    break;
                case LONG:
                    writer.appendMethod(new MethodReference(Long.class, "valueOf", long.class, Long.class));
                    break;
                case FLOAT:
                    writer.appendMethod(new MethodReference(Float.class, "valueOf", float.class, Float.class));
                    break;
                case DOUBLE:
                    writer.appendMethod(new MethodReference(Double.class, "valueOf", double.class, Double.class));
                    break;
            }
            writer.append('(');
            boxed = true;
        }
        fragment.render();
        if (boxed) {
            writer.append(')');
        }
    }

    private void unboxIfNecessary(SourceWriter writer, ValueType type, Fragment fragment) {
        boolean boxed = false;
        if (type instanceof ValueType.Primitive) {
            switch (((ValueType.Primitive) type).getKind()) {
                case BOOLEAN:
                    writer.appendMethod(new MethodReference(Boolean.class, "booleanValue", boolean.class));
                    break;
                case BYTE:
                    writer.appendMethod(new MethodReference(Byte.class, "byteValue", byte.class));
                    break;
                case SHORT:
                    writer.appendMethod(new MethodReference(Short.class, "shortValue", short.class));
                    break;
                case CHARACTER:
                    writer.appendMethod(new MethodReference(Character.class, "charValue", char.class));
                    break;
                case INTEGER:
                    writer.appendMethod(new MethodReference(Integer.class, "intValue", int.class));
                    break;
                case LONG:
                    writer.appendMethod(new MethodReference(Long.class, "longValue", long.class));
                    break;
                case FLOAT:
                    writer.appendMethod(new MethodReference(Float.class, "floatValue", float.class));
                    break;
                case DOUBLE:
                    writer.appendMethod(new MethodReference(Double.class, "doubleValue", double.class));
                    break;
            }
            writer.append('(');
            boxed = true;
        }
        fragment.render();
        if (boxed) {
            writer.append(')');
        }
    }

    private interface Fragment {
        void render();
    }

    private interface MemberRenderer<T extends MemberReader> {
        void render(T member);
    }
}
