/*
 * Decompiled with CFR 0.152.
 */
package org.springframework.integration.aws.lock;

import java.io.Closeable;
import java.time.Duration;
import java.time.Instant;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;
import java.util.UUID;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.CompletionException;
import java.util.concurrent.CompletionStage;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.TimeUnit;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.springframework.beans.factory.DisposableBean;
import org.springframework.beans.factory.InitializingBean;
import org.springframework.integration.aws.lock.DynamoDbLockRegistry;
import org.springframework.util.Assert;
import software.amazon.awssdk.core.retry.backoff.BackoffStrategy;
import software.amazon.awssdk.core.retry.backoff.FixedDelayBackoffStrategy;
import software.amazon.awssdk.services.dynamodb.DynamoDbAsyncClient;
import software.amazon.awssdk.services.dynamodb.model.AttributeDefinition;
import software.amazon.awssdk.services.dynamodb.model.AttributeValue;
import software.amazon.awssdk.services.dynamodb.model.BillingMode;
import software.amazon.awssdk.services.dynamodb.model.ConditionalCheckFailedException;
import software.amazon.awssdk.services.dynamodb.model.CreateTableRequest;
import software.amazon.awssdk.services.dynamodb.model.DeleteItemRequest;
import software.amazon.awssdk.services.dynamodb.model.KeySchemaElement;
import software.amazon.awssdk.services.dynamodb.model.KeyType;
import software.amazon.awssdk.services.dynamodb.model.ProvisionedThroughput;
import software.amazon.awssdk.services.dynamodb.model.PutItemRequest;
import software.amazon.awssdk.services.dynamodb.model.QueryRequest;
import software.amazon.awssdk.services.dynamodb.model.QueryResponse;
import software.amazon.awssdk.services.dynamodb.model.ResourceNotFoundException;
import software.amazon.awssdk.services.dynamodb.model.ScalarAttributeType;
import software.amazon.awssdk.services.dynamodb.model.Select;
import software.amazon.awssdk.services.dynamodb.model.UpdateItemRequest;

public class DynamoDbLockRepository
implements InitializingBean,
DisposableBean,
Closeable {
    public static final String DEFAULT_TABLE_NAME = "SpringIntegrationLockRegistry";
    public static final String KEY_ATTR = "lockKey";
    public static final String OWNER_ATTR = "lockOwner";
    public static final String CREATED_ATTR = "createdAt";
    public static final String TTL_ATTR = "expireAt";
    private static final String LOCK_EXISTS_EXPRESSION = String.format("attribute_exists(%s) AND %s = :owner", "lockKey", "lockOwner");
    private static final String LOCK_NOT_EXISTS_EXPRESSION = String.format("attribute_not_exists(%s) OR %s = :owner OR %s < :ttl", "lockKey", "lockOwner", "expireAt");
    public static final Duration DEFAULT_LEASE_DURATION = Duration.ofSeconds(60L);
    private static final Log LOGGER = LogFactory.getLog(DynamoDbLockRegistry.class);
    private final CountDownLatch createTableLatch = new CountDownLatch(1);
    private final Set<String> heldLocks = Collections.synchronizedSet(new HashSet());
    private final DynamoDbAsyncClient dynamoDB;
    private final String tableName;
    private BillingMode billingMode = BillingMode.PAY_PER_REQUEST;
    private long readCapacity = 1L;
    private long writeCapacity = 1L;
    private String owner = UUID.randomUUID().toString();
    private Duration leaseDuration = DEFAULT_LEASE_DURATION;
    private Map<String, AttributeValue> ownerAttribute;
    private volatile boolean initialized;

    public DynamoDbLockRepository(DynamoDbAsyncClient dynamoDB) {
        this(dynamoDB, DEFAULT_TABLE_NAME);
    }

    public DynamoDbLockRepository(DynamoDbAsyncClient dynamoDB, String tableName) {
        this.dynamoDB = dynamoDB;
        this.tableName = tableName;
    }

    public void setBillingMode(BillingMode billingMode) {
        Assert.notNull((Object)billingMode, (String)"'billingMode' must not be null");
        this.billingMode = billingMode;
    }

    public void setReadCapacity(long readCapacity) {
        this.readCapacity = readCapacity;
    }

    public void setWriteCapacity(long writeCapacity) {
        this.writeCapacity = writeCapacity;
    }

    public void setOwner(String owner) {
        this.owner = owner;
    }

    public void setLeaseDuration(Duration leaseDuration) {
        this.leaseDuration = leaseDuration;
    }

    public String getTableName() {
        return this.tableName;
    }

    public String getOwner() {
        return this.owner;
    }

    public void afterPropertiesSet() {
        ((CompletableFuture)((CompletableFuture)((CompletableFuture)this.dynamoDB.describeTable(request -> request.tableName(this.tableName)).thenRun(() -> {})).exceptionallyCompose(ex -> {
            Throwable cause = ex.getCause();
            if (cause instanceof ResourceNotFoundException) {
                if (LOGGER.isInfoEnabled()) {
                    LOGGER.info((Object)("No table '" + this.getTableName() + "'. Creating one..."));
                }
                return this.createTable();
            }
            return (CompletionStage)DynamoDbLockRepository.rethrowAsRuntimeException(cause);
        })).exceptionally(ex -> {
            LOGGER.error((Object)("Cannot create DynamoDb table: " + this.tableName), ex.getCause());
            return null;
        })).thenRun(this.createTableLatch::countDown);
        this.ownerAttribute = Map.of(":owner", AttributeValue.fromS((String)this.owner));
        this.initialized = true;
    }

    private CompletableFuture<Void> createTable() {
        CreateTableRequest.Builder createTableRequest = CreateTableRequest.builder().tableName(this.tableName).keySchema(new KeySchemaElement[]{(KeySchemaElement)KeySchemaElement.builder().attributeName(KEY_ATTR).keyType(KeyType.HASH).build()}).attributeDefinitions(new AttributeDefinition[]{(AttributeDefinition)AttributeDefinition.builder().attributeName(KEY_ATTR).attributeType(ScalarAttributeType.S).build()}).billingMode(this.billingMode);
        if (BillingMode.PROVISIONED.equals((Object)this.billingMode)) {
            createTableRequest.provisionedThroughput((ProvisionedThroughput)ProvisionedThroughput.builder().readCapacityUnits(Long.valueOf(this.readCapacity)).writeCapacityUnits(Long.valueOf(this.writeCapacity)).build());
        }
        return ((CompletableFuture)((CompletableFuture)this.dynamoDB.createTable((CreateTableRequest)createTableRequest.build()).thenCompose(result -> this.dynamoDB.waiter().waitUntilTableExists(request -> request.tableName(this.tableName), waiter -> waiter.maxAttempts(Integer.valueOf(60)).backoffStrategy((BackoffStrategy)FixedDelayBackoffStrategy.create((Duration)Duration.ofSeconds(1L)))))).thenCompose(response -> this.updateTimeToLive())).thenRun(() -> {});
    }

    private CompletableFuture<?> updateTimeToLive() {
        return this.dynamoDB.updateTimeToLive(ttlRequest -> ttlRequest.tableName(this.tableName).timeToLiveSpecification(ttlSpec -> ttlSpec.enabled(Boolean.valueOf(true)).attributeName(TTL_ATTR)));
    }

    private void awaitForActive() {
        Assert.state((boolean)this.initialized, () -> "The component has not been initialized: " + this + ".\n Is it declared as a bean?");
        try {
            if (!this.createTableLatch.await(60L, TimeUnit.SECONDS)) {
                throw new IllegalStateException("The DynamoDb table " + this.getTableName() + " has not been created during 60 seconds");
            }
        }
        catch (InterruptedException e) {
            Thread.currentThread().interrupt();
            throw new IllegalStateException("The DynamoDb table " + this.getTableName() + " has not been created and waiting thread is interrupted");
        }
    }

    public boolean isAcquired(String lock) {
        this.awaitForActive();
        if (this.heldLocks.contains(lock)) {
            Map<String, AttributeValue> values = this.ownerWithTtlValues(DynamoDbLockRepository.currentEpochSeconds());
            values.put(":lock", AttributeValue.fromS((String)lock));
            QueryRequest.Builder queryRequest = QueryRequest.builder().tableName(this.tableName).select(Select.COUNT).limit(Integer.valueOf(1)).keyConditionExpression("lockKey = :lock").filterExpression("lockOwner = :owner AND expireAt >= :ttl").expressionAttributeValues(values);
            try {
                return ((QueryResponse)this.dynamoDB.query((QueryRequest)queryRequest.build()).get()).count() > 0;
            }
            catch (CompletionException | ExecutionException ex) {
                DynamoDbLockRepository.rethrowAsRuntimeException(ex.getCause());
            }
            catch (InterruptedException ex) {
                Thread.currentThread().interrupt();
                return (Boolean)DynamoDbLockRepository.rethrowAsRuntimeException(ex);
            }
        }
        return false;
    }

    public void delete(String lock) {
        this.awaitForActive();
        if (this.heldLocks.remove(lock)) {
            this.deleteFromDb(lock);
        }
    }

    private void deleteFromDb(String lock) {
        this.doDelete(DeleteItemRequest.builder().key(Map.of(KEY_ATTR, AttributeValue.fromS((String)lock))).conditionExpression("lockOwner = :owner").expressionAttributeValues(this.ownerAttribute));
    }

    private void doDelete(DeleteItemRequest.Builder deleteItemRequest) {
        try {
            this.dynamoDB.deleteItem((DeleteItemRequest)deleteItemRequest.tableName(this.tableName).build()).get();
        }
        catch (CompletionException | ExecutionException ex) {
            Throwable cause = ex.getCause();
            if (!(cause instanceof ConditionalCheckFailedException)) {
                DynamoDbLockRepository.rethrowAsRuntimeException(cause);
            }
        }
        catch (InterruptedException ex) {
            Thread.currentThread().interrupt();
            DynamoDbLockRepository.rethrowAsRuntimeException(ex);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void deleteExpired() {
        this.awaitForActive();
        Set<String> set = this.heldLocks;
        synchronized (set) {
            this.heldLocks.forEach(lock -> this.doDelete(DeleteItemRequest.builder().key(Map.of(KEY_ATTR, AttributeValue.fromS((String)lock))).conditionExpression("lockOwner = :owner AND expireAt < :ttl").expressionAttributeValues(this.ownerWithTtlValues(DynamoDbLockRepository.currentEpochSeconds()))));
            this.heldLocks.clear();
        }
    }

    private Map<String, AttributeValue> ownerWithTtlValues(long epochSeconds) {
        HashMap<String, AttributeValue> valueMap = new HashMap<String, AttributeValue>();
        valueMap.put(":ttl", AttributeValue.fromN((String)("" + epochSeconds)));
        valueMap.putAll(this.ownerAttribute);
        return valueMap;
    }

    public boolean acquire(String lock) throws InterruptedException {
        this.awaitForActive();
        if (Thread.currentThread().isInterrupted()) {
            throw new InterruptedException();
        }
        long currentTime = DynamoDbLockRepository.currentEpochSeconds();
        HashMap<String, AttributeValue> item = new HashMap<String, AttributeValue>();
        item.put(KEY_ATTR, AttributeValue.fromS((String)lock));
        item.put(OWNER_ATTR, AttributeValue.fromS((String)this.owner));
        item.put(CREATED_ATTR, AttributeValue.fromN((String)("" + currentTime)));
        item.put(TTL_ATTR, AttributeValue.fromN((String)("" + this.ttlEpochSeconds())));
        PutItemRequest.Builder putItemRequest = PutItemRequest.builder().tableName(this.tableName).item(item).conditionExpression(LOCK_NOT_EXISTS_EXPRESSION).expressionAttributeValues(this.ownerWithTtlValues(currentTime));
        try {
            ((CompletableFuture)this.dynamoDB.putItem((PutItemRequest)putItemRequest.build()).thenRun(() -> this.heldLocks.add(lock))).get();
            return true;
        }
        catch (CompletionException | ExecutionException ex) {
            Throwable cause = ex.getCause();
            if (!(cause instanceof ConditionalCheckFailedException)) {
                DynamoDbLockRepository.rethrowAsRuntimeException(cause);
            }
            return false;
        }
        catch (InterruptedException ex) {
            Thread.currentThread().interrupt();
            throw ex;
        }
    }

    public boolean renew(String lock) {
        this.awaitForActive();
        if (this.heldLocks.contains(lock)) {
            UpdateItemRequest.Builder updateItemRequest = UpdateItemRequest.builder().tableName(this.tableName).key(Map.of(KEY_ATTR, AttributeValue.fromS((String)lock))).updateExpression("SET expireAt = :ttl").conditionExpression(LOCK_EXISTS_EXPRESSION).expressionAttributeValues(this.ownerWithTtlValues(this.ttlEpochSeconds()));
            try {
                this.dynamoDB.updateItem((UpdateItemRequest)updateItemRequest.build()).get();
                return true;
            }
            catch (CompletionException | ExecutionException ex) {
                Throwable cause = ex.getCause();
                if (!(cause instanceof ConditionalCheckFailedException)) {
                    DynamoDbLockRepository.rethrowAsRuntimeException(cause);
                }
                return false;
            }
            catch (InterruptedException ex) {
                Thread.currentThread().interrupt();
                return (Boolean)DynamoDbLockRepository.rethrowAsRuntimeException(ex.getCause());
            }
        }
        return false;
    }

    public void destroy() {
        this.close();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public void close() {
        Set<String> set = this.heldLocks;
        synchronized (set) {
            this.heldLocks.forEach(this::deleteFromDb);
            this.heldLocks.clear();
        }
    }

    private long ttlEpochSeconds() {
        return Instant.now().plus(this.leaseDuration).getEpochSecond();
    }

    private static long currentEpochSeconds() {
        return Instant.now().getEpochSecond();
    }

    private static <T> T rethrowAsRuntimeException(Throwable cause) {
        if (cause instanceof RuntimeException) {
            RuntimeException runtimeException = (RuntimeException)cause;
            throw runtimeException;
        }
        throw new IllegalStateException(cause);
    }
}

