/*
 * Copyright 2020 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      https://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.springframework.hateoas;

import java.util.Arrays;
import java.util.Collection;
import java.util.HashSet;

/**
 * A list of well-defined URI schemes governed by IANA.
 *
 * @author Oliver Drotbohm
 * @see <a href="https://www.iana.org/assignments/uri-schemes/uri-schemes.xhtml>Official IANA URI scheme registry</a>
 * @since 1.2
 */
public class IanaUriSchemes {

	private static Collection<String> IANA_URI_SCHEMES = new HashSet<>(Arrays.asList( //
			"aaa", //
			"aaas", //
			"about", //
			"acap", //
			"acct", //
			"acd", //
			"acr", //
			"adiumxtra", //
			"adt", //
			"afp", //
			"afs", //
			"aim", //
			"amss", //
			"android", //
			"appdata", //
			"apt", //
			"ark", //
			"attachment", //
			"aw", //
			"barion", //
			"bb", //
			"beshare", //
			"bitcoin", //
			"bitcoincash", //
			"blob", //
			"bolo", //
			"browserext", //
			"cabal", //
			"calculator", //
			"callto", //
			"cap", //
			"cast", //
			"casts", //
			"chrome", //
			"chrome-extension", //
			"cid", //
			"coap", //
			"coap+tcp", //
			"coap+ws", //
			"coaps", //
			"coaps+tcp", //
			"coaps+ws", //
			"com-eventbrite-attendee", //
			"content", //
			"conti", //
			"crid", //
			"cvs", //
			"dab", //
			"dat", //
			"data", //
			"dav", //
			"diaspora", //
			"dict", //
			"did", //
			"dis", //
			"dlna-playcontainer", //
			"dlna-playsingle", //
			"dns", //
			"dntp", //
			"doi", //
			"dpp", //
			"drm", //
			"drop", //
			"dtmi", //
			"dtn", //
			"dvb", //
			"dweb", //
			"ed2k", //
			"elsi", //
			"ethereum", //
			"example", //
			"facetime", //
			"fax", //
			"feed", //
			"feedready", //
			"file", //
			"filesystem", //
			"finger", //
			"first-run-pen-experience", //
			"fish", //
			"fm", //
			"ftp", //
			"fuchsia-pkg", //
			"geo", //
			"gg", //
			"git", //
			"gizmoproject", //
			"go", //
			"gopher", //
			"graph", //
			"grd", //
			"gtalk", //
			"h323", //
			"ham", //
			"hcap", //
			"hcp", //
			"http", //
			"https", //
			"hxxp", //
			"hxxps", //
			"hydrazone", //
			"hyper", //
			"iax", //
			"icap", //
			"icon", //
			"im", //
			"imap", //
			"info", //
			"iotdisco", //
			"ipfs", //
			"ipn", //
			"ipns", //
			"ipp", //
			"ipps", //
			"irc", //
			"irc6", //
			"ircs", //
			"iris", //
			"iris.beep", //
			"iris.lwz", //
			"iris.xpc", //
			"iris.xpcs", //
			"isostore", //
			"itms", //
			"jabber", //
			"jar", //
			"jms", //
			"keyparc", //
			"lastfm", //
			"lbry", //
			"ldap", //
			"ldaps", //
			"leaptofrogans", //
			"lorawan", //
			"lvlt", //
			"magnet", //
			"mailserver", //
			"mailto", //
			"maps", //
			"market", //
			"message", //
			"microsoft.windows.camera", //
			"microsoft.windows.camera.multipicker", //
			"microsoft.windows.camera.picker", //
			"mid", //
			"mms", //
			"modem", //
			"mongodb", //
			"moz", //
			"ms-access", //
			"ms-browser-extension", //
			"ms-calculator", //
			"ms-drive-to", //
			"ms-enrollment", //
			"ms-excel", //
			"ms-eyecontrolspeech", //
			"ms-gamebarservices", //
			"ms-gamingoverlay", //
			"ms-getoffice", //
			"ms-help", //
			"ms-infopath", //
			"ms-inputapp", //
			"ms-lockscreencomponent-config", //
			"ms-media-stream-id", //
			"ms-mixedrealitycapture", //
			"ms-mobileplans", //
			"ms-officeapp", //
			"ms-people", //
			"ms-project", //
			"ms-powerpoint", //
			"ms-publisher", //
			"ms-restoretabcompanion", //
			"ms-screenclip", //
			"ms-screensketch", //
			"ms-search", //
			"ms-search-repair", //
			"ms-secondary-screen-controller", //
			"ms-secondary-screen-setup", //
			"ms-settings", //
			"ms-settings-airplanemode", //
			"ms-settings-bluetooth", //
			"ms-settings-camera", //
			"ms-settings-cellular", //
			"ms-settings-cloudstorage", //
			"ms-settings-connectabledevices", //
			"ms-settings-displays-topology", //
			"ms-settings-emailandaccounts", //
			"ms-settings-language", //
			"ms-settings-location", //
			"ms-settings-lock", //
			"ms-settings-nfctransactions", //
			"ms-settings-notifications", //
			"ms-settings-power", //
			"ms-settings-privacy", //
			"ms-settings-proximity", //
			"ms-settings-screenrotation", //
			"ms-settings-wifi", //
			"ms-settings-workplace", //
			"ms-spd", //
			"ms-sttoverlay", //
			"ms-transit-to", //
			"ms-useractivityset", //
			"ms-virtualtouchpad", //
			"ms-visio", //
			"ms-walk-to", //
			"ms-whiteboard", //
			"ms-whiteboard-cmd", //
			"ms-word", //
			"msnim", //
			"msrp", //
			"msrps", //
			"mss", //
			"mtqp", //
			"mumble", //
			"mupdate", //
			"mvn", //
			"news", //
			"nfs", //
			"ni", //
			"nih", //
			"nntp", //
			"notes", //
			"ocf", //
			"oid", //
			"onenote", //
			"onenote-cmd", //
			"opaquelocktoken", //
			"openpgp4fpr", //
			"otpauth", //
			"p1", //
			"pack", //
			"palm", //
			"paparazzi", //
			"payment", //
			"payto", //
			"pkcs11", //
			"platform", //
			"pop", //
			"pres", //
			"prospero", //
			"proxy", //
			"pwid", //
			"psyc", //
			"pttp", //
			"qb", //
			"query", //
			"quic-transport", //
			"redis", //
			"rediss", //
			"reload", //
			"res", //
			"resource", //
			"rmi", //
			"rsync", //
			"rtmfp", //
			"rtmp", //
			"rtsp", //
			"rtsps", //
			"rtspu", //
			"secondlife", //
			"service", //
			"session", //
			"sftp", //
			"sgn", //
			"shttp", //
			"sieve", //
			"simpleledger", //
			"sip", //
			"sips", //
			"skype", //
			"smb", //
			"sms", //
			"smtp", //
			"snews", //
			"snmp", //
			"soap.beep", //
			"soap.beeps", //
			"soldat", //
			"spiffe", //
			"spotify", //
			"ssb", //
			"ssh", //
			"steam", //
			"stun", //
			"stuns", //
			"submit", //
			"swh", //
			"svn", //
			"tag", //
			"teamspeak", //
			"tel", //
			"teliaeid", //
			"telnet", //
			"tftp", //
			"things", //
			"thismessage", //
			"tip", //
			"tn3270", //
			"tool", //
			"turn", //
			"turns", //
			"tv", //
			"udp", //
			"unreal", //
			"upt", //
			"urn", //
			"ut2004", //
			"v-event", //
			"vemmi", //
			"ventrilo", //
			"videotex", //
			"vnc", //
			"view-source", //
			"vscode", //
			"vscode-insiders", //
			"vsls", //
			"wais", //
			"webcal", //
			"wpid", //
			"ws", //
			"wss", //
			"wtai", //
			"wyciwyg", //
			"xcon", //
			"xcon-userid", //
			"xfire", //
			"xmlrpc.beep", //
			"xmlrpc.beeps", //
			"xmpp", //
			"xri", //
			"ymsgr", //
			"z39.50", //
			"z39.50r", //
			"z39.50s"));

	public static boolean isIanaUriScheme(String scheme) {
		return IANA_URI_SCHEMES.contains(scheme);
	}
}
