/*
 * Copyright 2024 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      https://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.springframework.data.cassandra.core.mapping;

import java.lang.annotation.Documented;
import java.lang.annotation.ElementType;
import java.lang.annotation.Retention;
import java.lang.annotation.RetentionPolicy;
import java.lang.annotation.Target;

/**
 * Identifies a Storage-Attached Indexing (SAI) index. You can create multiple secondary indexes on the same database
 * table, with each SAI index based on any column in the table. All column date types except the following are supported
 * for SAI indexes:
 * <ul>
 * <li>counter</li>
 * <li>non-frozen user-defined type (UDT)</li>
 * </ul>
 * <p>
 * The following columns of a {@link Table} type can be annotated with {@link SaiIndexed}:
 * <ul>
 * <li>Scalar data types</li>
 * <li>Frozen user-defined types</li>
 * <li>Collection types</li>
 * <li>Map type</li>
 * <li>Vector types</li>
 * </ul>
 * <p>
 * Map types distinguish between entry, key or value-level indexing.
 *
 * <pre class="code">
 * &#64;Table
 * class Person {
 *
 * 	Map&lt;@SAIIndexed String, String&gt; indexedKey; // allows CONTAINS KEY queries
 * 	Map&lt;String, @SAIIndexed String&gt; indexedValue; // allows CONTAINS queries
 * }
 * </pre>
 *
 * @author Mark Paluch
 * @since 4.5
 */
@Documented
@Retention(value = RetentionPolicy.RUNTIME)
@Target(value = { ElementType.FIELD, ElementType.METHOD, ElementType.ANNOTATION_TYPE, ElementType.TYPE_USE })
public @interface SaiIndexed {

	/**
	 * The name of the index. If {@literal null} or empty, then the index name will be generated by Cassandra and will be
	 * unknown unless column metadata is used to discover the generated index name.
	 */
	String value() default "";

	/**
	 * Enable case-sensitive matching.
	 */
	boolean caseSensitive() default true;

	/**
	 * When set to true, perform Unicode normalization on indexed strings. SAI supports Normalization Form C (NFC)
	 * Unicode. When set to true, SAI normalizes the different versions of a given Unicode character to a single version,
	 * retaining all the marks and symbols in the index. For example, SAI would change the character {@code Å (U+212B)} to
	 * {@code Å (U+00C5)}.
	 */
	boolean normalize() default false;

	/**
	 * When set to true, SAI converts alphabetic, numeric, and symbolic characters that are not in the Basic Latin Unicode
	 * block (the first 127 ASCII characters) to the ASCII equivalent, if one exists. For example, this option changes
	 * {@code à} to {@code a}.
	 */
	boolean ascii() default false;

	/**
	 * Vector search relies on computing the similarity or distance between vectors to identify relevant matches. The
	 * similarity function is used to compute the similarity between two vectors. When using SAI indexes with vector data
	 * types, then only the similarity function is used with index options.
	 */
	SimilarityFunction similarityFunction() default SimilarityFunction.COSINE;

}
