/*
 * Decompiled with CFR 0.152.
 */
package org.springframework.cloud.deployer.spi.local;

import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStreamReader;
import java.io.Reader;
import java.net.HttpURLConnection;
import java.net.Inet4Address;
import java.net.URL;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.attribute.FileAttribute;
import java.util.HashMap;
import java.util.Map;
import java.util.Optional;
import java.util.UUID;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.CopyOnWriteArrayList;
import javax.annotation.PreDestroy;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.cloud.deployer.spi.core.AppDeploymentRequest;
import org.springframework.cloud.deployer.spi.core.RuntimeEnvironmentInfo;
import org.springframework.cloud.deployer.spi.local.AbstractLocalDeployerSupport;
import org.springframework.cloud.deployer.spi.local.LocalDeployerProperties;
import org.springframework.cloud.deployer.spi.task.LaunchState;
import org.springframework.cloud.deployer.spi.task.TaskLauncher;
import org.springframework.cloud.deployer.spi.task.TaskStatus;
import org.springframework.util.FileCopyUtils;
import org.springframework.util.StringUtils;

public class LocalTaskLauncher
extends AbstractLocalDeployerSupport
implements TaskLauncher {
    private Path logPathRoot;
    private static final Logger logger = LoggerFactory.getLogger(LocalTaskLauncher.class);
    private static final String JMX_DEFAULT_DOMAIN_KEY = "spring.jmx.default-domain";
    private final Map<String, TaskInstance> running = new ConcurrentHashMap<String, TaskInstance>();
    private final Map<String, CopyOnWriteArrayList<String>> taskInstanceHistory = new ConcurrentHashMap<String, CopyOnWriteArrayList<String>>();

    public LocalTaskLauncher(LocalDeployerProperties properties) {
        super(properties);
    }

    public String launch(AppDeploymentRequest request) {
        if (this.maxConcurrentExecutionsReached()) {
            throw new IllegalStateException(String.format("Cannot launch task %s. The maximum concurrent task executions is at its limit [%d].", request.getDefinition().getName(), this.getMaximumConcurrentTasks()));
        }
        String taskLaunchId = request.getDefinition().getName() + "-" + UUID.randomUUID().toString();
        this.pruneTaskInstanceHistory(request.getDefinition().getName(), taskLaunchId);
        HashMap<String, String> args = new HashMap<String, String>();
        args.putAll(request.getDefinition().getProperties());
        args.put(JMX_DEFAULT_DOMAIN_KEY, taskLaunchId);
        args.put("endpoints.shutdown.enabled", "true");
        args.put("endpoints.jmx.unique-names", "true");
        try {
            Path dir = this.createLogDir(request);
            Path workDir = this.createWorkingDir(taskLaunchId, dir);
            boolean useDynamicPort = this.isDynamicPort(request);
            int port = this.calcServerPort(request, useDynamicPort, args);
            ProcessBuilder builder = this.buildProcessBuilder(request, args, Optional.empty(), taskLaunchId).inheritIO();
            TaskInstance instance = new TaskInstance(builder, workDir, port);
            if (this.shouldInheritLogging(request)) {
                instance.start(builder);
                logger.info("launching task {}\n    Logs will be inherited.", (Object)taskLaunchId);
            } else {
                instance.start(builder, this.getLocalDeployerProperties().isDeleteFilesOnExit());
                logger.info("launching task {}\n   Logs will be in {}", (Object)taskLaunchId, (Object)workDir);
            }
            this.running.put(taskLaunchId, instance);
        }
        catch (IOException e) {
            throw new RuntimeException("Exception trying to launch " + request, e);
        }
        return taskLaunchId;
    }

    private void pruneTaskInstanceHistory(String taskDefinitionName, String taskLaunchId) {
        CopyOnWriteArrayList<String> oldTaskInstanceIds = this.taskInstanceHistory.get(taskDefinitionName);
        if (oldTaskInstanceIds == null) {
            oldTaskInstanceIds = new CopyOnWriteArrayList();
            this.taskInstanceHistory.put(taskDefinitionName, oldTaskInstanceIds);
        }
        for (String oldTaskInstanceId : oldTaskInstanceIds) {
            TaskInstance oldTaskInstance = this.running.get(oldTaskInstanceId);
            if (oldTaskInstance != null && oldTaskInstance.getState() != LaunchState.running && oldTaskInstance.getState() != LaunchState.launching) {
                this.running.remove(oldTaskInstanceId);
                oldTaskInstanceIds.remove(oldTaskInstanceId);
                continue;
            }
            oldTaskInstanceIds.remove(oldTaskInstanceId);
        }
        oldTaskInstanceIds.add(taskLaunchId);
    }

    private boolean isDynamicPort(AppDeploymentRequest request) {
        boolean isServerPortKeyonArgs = this.isServerPortKeyPresentOnArgs(request) != null;
        return !request.getDefinition().getProperties().containsKey("server.port") && !isServerPortKeyonArgs;
    }

    public void cancel(String id) {
        TaskInstance instance = this.running.get(id);
        if (instance != null) {
            instance.cancelled = true;
            if (this.isAlive(instance.getProcess())) {
                this.shutdownAndWait(instance);
            }
        }
    }

    public TaskStatus status(String id) {
        TaskInstance instance = this.running.get(id);
        if (instance != null) {
            return new TaskStatus(id, instance.getState(), instance.getAttributes());
        }
        return new TaskStatus(id, LaunchState.unknown, null);
    }

    public String getLog(String id) {
        TaskInstance instance = this.running.get(id);
        if (instance != null) {
            String stderr = instance.getStdErr();
            return StringUtils.hasText((String)stderr) ? stderr : instance.getStdOut();
        }
        return "Log could not be retrieved as the task instance is not running.";
    }

    public void cleanup(String id) {
    }

    public void destroy(String appName) {
    }

    public RuntimeEnvironmentInfo environmentInfo() {
        return super.createRuntimeEnvironmentInfo(TaskLauncher.class, this.getClass());
    }

    public int getMaximumConcurrentTasks() {
        return this.getLocalDeployerProperties().getMaximumConcurrentTasks();
    }

    public int getRunningTaskExecutionCount() {
        int runningExecutionCount = 0;
        for (TaskInstance taskInstance : this.running.values()) {
            if (!taskInstance.getProcess().isAlive()) continue;
            ++runningExecutionCount;
        }
        return runningExecutionCount;
    }

    private boolean maxConcurrentExecutionsReached() {
        return this.getRunningTaskExecutionCount() >= this.getMaximumConcurrentTasks();
    }

    @PreDestroy
    public void shutdown() throws Exception {
        for (String taskLaunchId : this.running.keySet()) {
            this.cancel(taskLaunchId);
        }
        this.taskInstanceHistory.clear();
    }

    private Path createWorkingDir(String taskLaunchId, Path dir) throws IOException {
        Path workDir = Files.createDirectory(Paths.get(dir.toFile().getAbsolutePath(), taskLaunchId), new FileAttribute[0]);
        if (this.getLocalDeployerProperties().isDeleteFilesOnExit()) {
            workDir.toFile().deleteOnExit();
        }
        return workDir;
    }

    private Path createLogDir(AppDeploymentRequest request) throws IOException {
        if (this.logPathRoot == null) {
            this.logPathRoot = Files.createTempDirectory(this.getLocalDeployerProperties().getWorkingDirectoriesRoot(), request.getDefinition().getName(), new FileAttribute[0]);
        }
        String qualifiedName = Long.toString(System.nanoTime());
        Path dir = Paths.get(this.logPathRoot.toFile().getAbsolutePath(), qualifiedName);
        if (!Files.exists(dir, new LinkOption[0])) {
            Files.createDirectory(dir, new FileAttribute[0]);
            dir.toFile().deleteOnExit();
        }
        return dir;
    }

    private static Integer getProcessExitValue(Process process) {
        try {
            return process.exitValue();
        }
        catch (IllegalThreadStateException e) {
            return null;
        }
    }

    private static class TaskInstance
    implements AbstractLocalDeployerSupport.Instance {
        private Process process;
        private final Path workDir;
        private File stdout;
        private File stderr;
        private final URL baseUrl;
        private boolean cancelled;

        private TaskInstance(ProcessBuilder builder, Path workDir, int port) throws IOException {
            builder.directory(workDir.toFile());
            this.workDir = workDir;
            this.baseUrl = new URL("http", Inet4Address.getLocalHost().getHostAddress(), port, "");
            if (logger.isDebugEnabled()) {
                logger.debug("Local Task Launcher Commands: " + String.join((CharSequence)",", builder.command()) + ", Environment: " + builder.environment());
            }
        }

        @Override
        public URL getBaseUrl() {
            return this.baseUrl;
        }

        @Override
        public Process getProcess() {
            return this.process;
        }

        public LaunchState getState() {
            if (this.cancelled) {
                return LaunchState.cancelled;
            }
            Integer exit = LocalTaskLauncher.getProcessExitValue(this.process);
            if (exit != null) {
                if (exit == 0) {
                    return LaunchState.complete;
                }
                return LaunchState.failed;
            }
            try {
                HttpURLConnection urlConnection = (HttpURLConnection)this.baseUrl.openConnection();
                urlConnection.setConnectTimeout(100);
                urlConnection.connect();
                urlConnection.disconnect();
                return LaunchState.running;
            }
            catch (IOException e) {
                return LaunchState.launching;
            }
        }

        public String getStdOut() {
            try {
                return FileCopyUtils.copyToString((Reader)new InputStreamReader(new FileInputStream(this.stdout)));
            }
            catch (IOException e) {
                return "Log retrieval returned " + e.getMessage();
            }
        }

        public String getStdErr() {
            try {
                return FileCopyUtils.copyToString((Reader)new InputStreamReader(new FileInputStream(this.stderr)));
            }
            catch (IOException e) {
                return "Log retrieval returned " + e.getMessage();
            }
        }

        private void start(ProcessBuilder builder) throws IOException {
            if (logger.isDebugEnabled()) {
                logger.debug("Local Task Launcher Commands: " + String.join((CharSequence)",", builder.command()) + ", Environment: " + builder.environment());
            }
            this.process = builder.start();
        }

        private void start(ProcessBuilder builder, boolean deleteOnExit) throws IOException {
            String workDirPath = this.workDir.toFile().getAbsolutePath();
            this.stdout = Files.createFile(Paths.get(workDirPath, "stdout.log"), new FileAttribute[0]).toFile();
            this.stderr = Files.createFile(Paths.get(workDirPath, "stderr.log"), new FileAttribute[0]).toFile();
            builder.redirectOutput(this.stdout);
            builder.redirectError(this.stderr);
            this.process = builder.start();
            if (deleteOnExit) {
                this.stdout.deleteOnExit();
                this.stderr.deleteOnExit();
            }
        }

        private Map<String, String> getAttributes() {
            HashMap<String, String> result = new HashMap<String, String>();
            result.put("working.dir", this.workDir.toFile().getAbsolutePath());
            if (this.stdout != null) {
                result.put("stdout", this.stdout.getAbsolutePath());
            }
            if (this.stderr != null) {
                result.put("stderr", this.stderr.getAbsolutePath());
            }
            result.put("url", this.baseUrl.toString());
            return result;
        }
    }
}

