/**
 * Copyright (c) 2024 Source Auditor Inc.
 *
 * SPDX-License-Identifier: Apache-2.0
 * 
 *   Licensed under the Apache License, Version 2.0 (the "License");
 *   you may not use this file except in compliance with the License.
 *   You may obtain a copy of the License at
 *
 *       http://www.apache.org/licenses/LICENSE-2.0
 *
 *   Unless required by applicable law or agreed to in writing, software
 *   distributed under the License is distributed on an "AS IS" BASIS,
 *   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *   See the License for the specific language governing permissions and
 *   limitations under the License.
 */
 
package org.spdx.library.model.v3_0_1.core;

import javax.annotation.Nullable;

import java.util.ArrayList;
import java.util.List;
import java.util.Set;

import org.spdx.core.CoreModelObject;
import org.spdx.core.DefaultModelStore;
import org.spdx.core.InvalidSPDXAnalysisException;
import org.spdx.core.IModelCopyManager;
import org.spdx.core.IndividualUriValue;
import org.spdx.library.model.v3_0_1.ModelObjectV3;
import org.spdx.storage.IModelStore;
import org.spdx.storage.IModelStore.IdType;
import org.spdx.storage.IModelStore.IModelStoreLock;

import java.util.Arrays;
import java.util.Collections;
import java.util.Objects;
import java.util.Optional;
import org.spdx.library.model.v3_0_1.SpdxConstantsV3;

/**
 * DO NOT EDIT - this file is generated by the Owl to Java Utility 
 * See: https://github.com/spdx/tools-java 
 * 
 * A mathematically calculated representation of a grouping of data. 
 */
public  class Hash extends IntegrityMethod  {

	
	/**
	 * Create the Hash with default model store and generated anonymous ID
	 * @throws InvalidSPDXAnalysisException when unable to create the Hash
	 */
	public Hash() throws InvalidSPDXAnalysisException {
		this(DefaultModelStore.getDefaultModelStore().getNextId(IdType.Anonymous));
	}

	/**
	 * @param objectUri URI or anonymous ID for the Hash
	 * @throws InvalidSPDXAnalysisException when unable to create the Hash
	 */
	public Hash(String objectUri) throws InvalidSPDXAnalysisException {
		this(DefaultModelStore.getDefaultModelStore(), objectUri, DefaultModelStore.getDefaultCopyManager(), 
				true, DefaultModelStore.getDefaultDocumentUri() + "#");
	}

	/**
	 * @param modelStore Model store where the Hash is to be stored
	 * @param objectUri URI or anonymous ID for the Hash
	 * @param copyManager Copy manager for the Hash - can be null if copying is not required
	 * @param create true if Hash is to be created
	 * @param idPrefix - prefix to be used when generating new SPDX IDs
	 * @throws InvalidSPDXAnalysisException when unable to create the Hash
	 */
	public Hash(IModelStore modelStore, String objectUri, @Nullable IModelCopyManager copyManager,
			boolean create, String idPrefix) throws InvalidSPDXAnalysisException {
		super(modelStore, objectUri, copyManager, create, idPrefix);
	}

	/**
	 * Create the Hash from the builder - used in the builder class
	 * @param builder Builder to create the Hash from
	 * @throws InvalidSPDXAnalysisException when unable to create the Hash
	 */
	protected Hash(HashBuilder builder) throws InvalidSPDXAnalysisException {
		super(builder);
		if (Objects.nonNull(builder.algorithm)) {
			setAlgorithm(builder.algorithm);
		}
		if (Objects.nonNull(builder.hashValue)) {
			setHashValue(builder.hashValue);
		}
	}

	/* (non-Javadoc)
	 * @see org.spdx.library.core.CoreModelObject#getType()
	 */
	@Override
	public String getType() {
		return "Core.Hash";
	}
	
	// Getters and Setters
	
	/**
	 * @return the algorithm
	 */
	public @Nullable HashAlgorithm getAlgorithm() throws InvalidSPDXAnalysisException {
		Optional<Enum<?>> retval = getEnumPropertyValue(SpdxConstantsV3.PROP_ALGORITHM);
		if (retval.isPresent()) {
			if (!(retval.get() instanceof HashAlgorithm)) {
				throw new InvalidSPDXAnalysisException("Incorrect type stored for algorithm");
			}
			return (HashAlgorithm)(retval.get());
		} else {
			return null;
		}
	}
	/**
	 * @param algorithm the algorithm to set
	 * @return this to chain setters
	 * @throws InvalidSPDXAnalysisException 
	 */
	public Hash setAlgorithm(@Nullable HashAlgorithm algorithm) throws InvalidSPDXAnalysisException {
		if (isStrict() && Objects.isNull(algorithm)) {
			throw new InvalidSPDXAnalysisException("algorithm is a required property");
		}
		setPropertyValue(SpdxConstantsV3.PROP_ALGORITHM, algorithm);
		return this;
	}
	/**
	 * @return the hashValue
	 */
	public @Nullable String getHashValue() throws InvalidSPDXAnalysisException {
		Optional<String> retval = getStringPropertyValue(SpdxConstantsV3.PROP_HASH_VALUE);
		return retval.isPresent() ? retval.get() : null;
	}
		/**
	 * @param hashValue the hashValue to set
	 * @return this to chain setters
	 * @throws InvalidSPDXAnalysisException 
	 */
	public Hash setHashValue(@Nullable String hashValue) throws InvalidSPDXAnalysisException {
		if (isStrict() && Objects.isNull(hashValue)) {
			throw new InvalidSPDXAnalysisException("hashValue is a required property");
		}
		setPropertyValue(SpdxConstantsV3.PROP_HASH_VALUE, hashValue);
		return this;
	}
	/**
	 * @param comment the comment to set
	 * @return this to chain setters
	 * @throws InvalidSPDXAnalysisException 
	 */
	 @Override
	public Hash setComment(@Nullable String comment) throws InvalidSPDXAnalysisException {
		super.setComment(comment);
		return this;
	}	
	
	
	@Override
	public String toString() {
		return "Hash: "+getObjectUri();
		
	}
	
	/* (non-Javadoc)
	 * @see org.spdx.library.model.v3.ModelObjectV3#_verify(java.util.List)
	 */
	@Override
	public List<String> _verify(Set<String> verifiedIds, String specVersionForVerify, List<IndividualUriValue> profiles) {
		List<String> retval = new ArrayList<>();
		retval.addAll(super._verify(verifiedIds, specVersionForVerify, profiles));
		try {
			HashAlgorithm algorithm = getAlgorithm();
			if (Objects.isNull(algorithm) && 
						Collections.disjoint(profiles, Arrays.asList(new ProfileIdentifierType[] { ProfileIdentifierType.CORE }))) {
				retval.add("Missing algorithm in Hash");
			}
		} catch (InvalidSPDXAnalysisException e) {
			retval.add("Error getting algorithm for Hash: "+e.getMessage());
		}
		try {
			String hashValue = getHashValue();
			if (Objects.isNull(hashValue) &&
					Collections.disjoint(profiles, Arrays.asList(new ProfileIdentifierType[] { ProfileIdentifierType.CORE }))) {
				retval.add("Missing hashValue in Hash");
			}
		} catch (InvalidSPDXAnalysisException e) {
			retval.add("Error getting hashValue for Hash: "+e.getMessage());
		}
		return retval;
	}
	
	
	
	public static class HashBuilder extends IntegrityMethodBuilder {
	
		/**
		 * Create an HashBuilder from another model object copying the modelStore and copyManager and using an anonymous ID
		 * @param from model object to copy the model store and copyManager from
		 * @throws InvalidSPDXAnalysisException
		 */
		public HashBuilder(ModelObjectV3 from) throws InvalidSPDXAnalysisException {
			this(from, from.getModelStore().getNextId(IdType.Anonymous));
		}
	
		/**
		 * Create an HashBuilder from another model object copying the modelStore and copyManager
		 * @param from model object to copy the model store and copyManager from
		 * @param objectUri URI for the object
		 * @throws InvalidSPDXAnalysisException if creationInfo can not be fetched from the from
		 */
		public HashBuilder(ModelObjectV3 from, String objectUri) throws InvalidSPDXAnalysisException {
			this(from.getModelStore(), objectUri, from.getCopyManager());
			setStrict(from.isStrict());
			setIdPrefix(from.getIdPrefix());
		}
		
		/**
		 * Creates a HashBuilder
		 * @param modelStore model store for the built Hash
		 * @param objectUri objectUri for the built Hash
		 * @param copyManager optional copyManager for the built Hash
		 */
		public HashBuilder(IModelStore modelStore, String objectUri, @Nullable IModelCopyManager copyManager) {
			super(modelStore, objectUri, copyManager);
		}
		
		protected HashAlgorithm algorithm = null;
		protected String hashValue = null;
		
		
		/**
		 * Sets the initial value of algorithm
		 * @parameter algorithm value to set
		 * @return this for chaining
		**/
		public HashBuilder setAlgorithm(HashAlgorithm algorithm) {
			this.algorithm = algorithm;
			return this;
		}
		
		/**
		 * Sets the initial value of hashValue
		 * @parameter hashValue value to set
		 * @return this for chaining
		**/
		public HashBuilder setHashValue(String hashValue) {
			this.hashValue = hashValue;
			return this;
		}
		
		/**
		 * Sets the initial value of comment
		 * @parameter comment value to set
		 * @return this for chaining
		**/
		public HashBuilder setComment(String comment) {
			super.comment = comment;
			return this;
		}
	
		/**
		 * @return the Hash
		 * @throws InvalidSPDXAnalysisException on any errors during build
		 */
		public Hash build() throws InvalidSPDXAnalysisException {
			IModelStoreLock lock = getModelStore().enterCriticalSection(false);
			try {
				return new Hash(this);
			} finally {
				getModelStore().leaveCriticalSection(lock);
			}
		}
	}
}
