/*
 * Decompiled with CFR 0.152.
 */
package org.simplify4u.plugins.keyserver;

import com.google.common.util.concurrent.Uninterruptibles;
import io.vavr.CheckedFunction0;
import io.vavr.control.Try;
import java.io.BufferedOutputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.Serializable;
import java.net.InetAddress;
import java.net.URI;
import java.nio.file.FileSystemException;
import java.nio.file.Files;
import java.nio.file.StandardCopyOption;
import java.security.SecureRandom;
import java.time.Duration;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Optional;
import java.util.concurrent.TimeUnit;
import java.util.regex.Pattern;
import java.util.stream.Collectors;
import javax.inject.Inject;
import javax.inject.Named;
import org.bouncycastle.openpgp.PGPException;
import org.bouncycastle.openpgp.PGPPublicKeyRing;
import org.simplify4u.plugins.keyserver.PGPKeysServerClient;
import org.simplify4u.plugins.utils.ExceptionUtils;
import org.simplify4u.plugins.utils.MavenProxy;
import org.simplify4u.plugins.utils.PGPKeyId;
import org.simplify4u.plugins.utils.PublicKeyUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@Named
public class PGPKeysCache {
    private static final Logger LOGGER = LoggerFactory.getLogger(PGPKeysCache.class);
    private static final String NL = System.lineSeparator();
    private static final Pattern KEY_SERVERS_SPLIT_PATTERN = Pattern.compile("[;,\\s]");
    private final MavenProxy mavenProxy;
    private File cachePath;
    private KeyServerList keyServerList;
    private static final Object LOCK = new Object();

    @Inject
    PGPKeysCache(MavenProxy mavenProxy) {
        this.mavenProxy = mavenProxy;
    }

    public void init(File cachePath, String keyServers, boolean loadBalance, String proxyName) throws IOException {
        this.init(cachePath, this.prepareClients(keyServers, proxyName), loadBalance);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    void init(File cachePath, List<PGPKeysServerClient> pgpKeysServerClients, boolean loadBalance) throws IOException {
        this.cachePath = cachePath;
        this.keyServerList = PGPKeysCache.createKeyServerList(pgpKeysServerClients, loadBalance);
        LOGGER.info("Key server(s) - {}", (Object)this.keyServerList);
        Object object = LOCK;
        synchronized (object) {
            if (this.cachePath.exists()) {
                if (!this.cachePath.isDirectory()) {
                    throw new IOException("PGP keys cache path exist but is not a directory: " + this.cachePath);
                }
            } else if (this.cachePath.mkdirs()) {
                LOGGER.info("Create cache directory for PGP keys: {}", (Object)this.cachePath);
            } else {
                throw new IOException("Cache directory create error");
            }
        }
    }

    List<PGPKeysServerClient> prepareClients(String keyServers, String proxyName) {
        List keyServersList = Arrays.stream(KEY_SERVERS_SPLIT_PATTERN.split(keyServers)).map(String::trim).filter(s -> s.length() > 0).collect(Collectors.toList());
        return keyServersList.stream().map(keyserver -> (PGPKeysServerClient)Try.of((CheckedFunction0 & Serializable)() -> PGPKeysServerClient.getClient(keyserver, this.mavenProxy.getProxyByName(proxyName))).get()).collect(Collectors.toList());
    }

    static KeyServerList createKeyServerList(List<PGPKeysServerClient> pgpKeysServerClients, boolean loadBalance) {
        if (pgpKeysServerClients == null || pgpKeysServerClients.isEmpty()) {
            throw new IllegalArgumentException("Not allowed empty key server clients list ");
        }
        KeyServerList ret = pgpKeysServerClients.size() == 1 ? new KeyServerListOne() : (loadBalance ? new KeyServerListLoadBalance() : new KeyServerListFallback());
        return ret.withClients(pgpKeysServerClients);
    }

    public String getUrlForShowKey(PGPKeyId keyID) {
        return this.keyServerList.getUriForShowKey(keyID).toString();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public PGPPublicKeyRing getKeyRing(PGPKeyId keyID) throws IOException {
        String path = keyID.getHashPath();
        File keyFile = new File(this.cachePath, path);
        Object object = LOCK;
        synchronized (object) {
            Optional<PGPPublicKeyRing> keyRing;
            if (keyFile.exists() && (keyRing = PGPKeysCache.loadKeyFromFile(keyFile, keyID)).isPresent()) {
                return keyRing.get();
            }
            return this.keyServerList.execute(keysServerClient -> PGPKeysCache.receiveKey(keyFile, keyID, keysServerClient));
        }
    }

    private static Optional<PGPPublicKeyRing> loadKeyFromFile(File keyFile, PGPKeyId keyID) throws IOException {
        Optional<Object> keyRing = Optional.empty();
        try (FileInputStream keyFileStream = new FileInputStream(keyFile);){
            keyRing = PublicKeyUtils.loadPublicKeyRing(keyFileStream, keyID);
        }
        catch (PGPException e) {
            throw new IOException(e);
        }
        finally {
            if (!keyRing.isPresent()) {
                PGPKeysCache.deleteFile(keyFile);
            }
        }
        return keyRing;
    }

    private static PGPPublicKeyRing receiveKey(File keyFile, PGPKeyId keyId, PGPKeysServerClient keysServerClient) throws IOException {
        File dir = keyFile.getParentFile();
        if (dir == null) {
            throw new IOException("No parent dir for: " + keyFile);
        }
        if (dir.exists() && !dir.isDirectory()) {
            throw new IOException("Path exist but it isn't directory: " + dir);
        }
        dir.mkdirs();
        File partFile = File.createTempFile(String.valueOf(keyId), "pgp-public-key");
        try {
            try (BufferedOutputStream outputStream = new BufferedOutputStream(new FileOutputStream(partFile));){
                keysServerClient.copyKeyToOutputStream(keyId, outputStream, PGPKeysCache::onRetry);
            }
            PGPKeysCache.moveFile(partFile, keyFile);
        }
        catch (IOException e) {
            PGPKeysCache.deleteFile(keyFile);
            PGPKeysCache.deleteFile(partFile);
            throw e;
        }
        LOGGER.info("Receive key: {}{}\tto {}", new Object[]{keysServerClient.getUriForGetKey(keyId), NL, keyFile});
        return PGPKeysCache.loadKeyFromFile(keyFile, keyId).orElseThrow(() -> new IOException(String.format("Can't find public key %s in download file: %s", keyId, keyFile)));
    }

    private static void onRetry(InetAddress address, int numberOfRetryAttempts, Duration waitInterval, Throwable lastThrowable) {
        LOGGER.warn("[Retry #{} waiting: {}] Last address {} with problem: [{}] {}", new Object[]{numberOfRetryAttempts, waitInterval, address, lastThrowable.getClass().getName(), ExceptionUtils.getMessage(lastThrowable)});
    }

    private static void deleteFile(File file) {
        Optional.ofNullable(file).map(File::toPath).ifPresent(filePath -> Try.run(() -> Files.deleteIfExists(filePath)).onFailure(e -> LOGGER.warn("Can't delete: {} with exception: {}", filePath, (Object)e.getMessage())));
    }

    private static void moveFile(File source, File destination) throws IOException {
        try {
            Files.move(source.toPath(), destination.toPath(), StandardCopyOption.REPLACE_EXISTING);
        }
        catch (FileSystemException fse) {
            Uninterruptibles.sleepUninterruptibly((long)(250L + (long)new SecureRandom().nextInt(1000)), (TimeUnit)TimeUnit.MILLISECONDS);
            Files.move(source.toPath(), destination.toPath(), StandardCopyOption.REPLACE_EXISTING);
        }
    }

    static class KeyServerListLoadBalance
    extends KeyServerList {
        private int lastIndex = 0;

        KeyServerListLoadBalance() {
        }

        @Override
        String getName() {
            return "load balance";
        }

        @Override
        PGPPublicKeyRing execute(KeyServerExecutor executor) throws IOException {
            for (int i = 0; i < this.keysServerClients.size(); ++i) {
                PGPKeysServerClient client = (PGPKeysServerClient)this.keysServerClients.get(this.lastIndex);
                this.lastIndex = (this.lastIndex + 1) % this.keysServerClients.size();
                Optional<PGPPublicKeyRing> pgpPublicKeys = this.executeWithClient(executor, client);
                if (!pgpPublicKeys.isPresent()) continue;
                return pgpPublicKeys.get();
            }
            LOGGER.error("All servers from list was failed");
            throw this.lastException;
        }
    }

    static class KeyServerListFallback
    extends KeyServerList {
        KeyServerListFallback() {
        }

        @Override
        String getName() {
            return "fallback";
        }

        @Override
        PGPPublicKeyRing execute(KeyServerExecutor executor) throws IOException {
            for (PGPKeysServerClient client : this.keysServerClients) {
                Optional<PGPPublicKeyRing> pgpPublicKeys = this.executeWithClient(executor, client);
                if (!pgpPublicKeys.isPresent()) continue;
                return pgpPublicKeys.get();
            }
            LOGGER.error("All servers from list was failed");
            throw this.lastException;
        }
    }

    static class KeyServerListOne
    extends KeyServerList {
        KeyServerListOne() {
        }

        @Override
        String getName() {
            return "one item";
        }

        @Override
        PGPPublicKeyRing execute(KeyServerExecutor executor) throws IOException {
            return executor.run(this.lastClient);
        }

        @Override
        public String toString() {
            return this.lastClient.toString();
        }
    }

    static abstract class KeyServerList {
        protected List<PGPKeysServerClient> keysServerClients = new ArrayList<PGPKeysServerClient>();
        protected PGPKeysServerClient lastClient;
        protected IOException lastException;

        KeyServerList() {
        }

        KeyServerList withClients(List<PGPKeysServerClient> keysServerClients) {
            this.keysServerClients = keysServerClients;
            this.lastClient = keysServerClients.get(0);
            return this;
        }

        URI getUriForShowKey(PGPKeyId keyID) {
            return this.lastClient.getUriForShowKey(keyID);
        }

        protected Optional<PGPPublicKeyRing> executeWithClient(KeyServerExecutor executor, PGPKeysServerClient client) {
            try {
                Optional<PGPPublicKeyRing> ret = Optional.of(executor.run(client));
                this.lastClient = client;
                return ret;
            }
            catch (IOException e) {
                this.lastException = e;
                LOGGER.warn("{} throw exception: {} - {} try next client", new Object[]{client, ExceptionUtils.getMessage(e), this.getName()});
                return Optional.empty();
            }
        }

        public String toString() {
            return String.format("%s list: %s", this.getName(), this.keysServerClients);
        }

        abstract String getName();

        abstract PGPPublicKeyRing execute(KeyServerExecutor var1) throws IOException;
    }

    @FunctionalInterface
    static interface KeyServerExecutor {
        public PGPPublicKeyRing run(PGPKeysServerClient var1) throws IOException;
    }
}

