import {
    ComponentPropsWithRef,
    ReactNode,
    forwardRef,
    ForwardedRef,
} from 'react';
import {
    DialogTitle,
    DialogDescription,
    DialogClose,
} from '@radix-ui/react-dialog';

import { styled } from '../../stitches.config';

import { Icon } from '../Icon';

const StyledHeader = styled('div', {
    margin: '0 0 $4 0',
    padding: '0 $4 0 $4',
    width: '$space$full',
});

const StyledDescription = styled(DialogDescription, {
    color: '$grey-1',
    fontSize: '$sm',
    margin: '$2 0 0 0',
    width: '$space$full',
});

const StyledContainer = styled('div', {
    alignItems: 'center',
    display: 'flex',
    flexDirection: 'row',
    columnGap: '$2',
    lineHeight: '$space$8',
    width: '$space$full',
});

const StyledTitle = styled(DialogTitle, {
    color: '$grey-1',
    flex: '1 1 0',
    fontWeight: '$semibold',
    fontSize: '$xl',
    lineHeight: '$default',
    whiteSpace: 'nowrap',
    overflow: 'hidden',
    textOverflow: 'ellipsis',
});

const StyledClose = styled(DialogClose, {
    alignItems: 'center',
    backgroundColor: `transparent`,
    borderRadius: '$default',
    color: `$grey-1`,
    cursor: 'pointer',
    display: 'inline-flex',
    justifyContent: 'center',
    fontSize: '$space$6',
    height: '$space$6',
    width: '$space$6',
    margin: '0',
    outline: 'none',
    textAlign: 'center',
    userSelect: 'none',
    whiteSpace: 'nowrap',
    '&:hover': {
        color: `$grey-2`,
    },
    '&:focus': {
        outline: '2px solid $primary-1',
        outlineOffset: '2px',
    },
});

// TODO: Need to figure out why @mdi/js isn't tree shaking
const mdiClose =
    'M19,6.41L17.59,5L12,10.59L6.41,5L5,6.41L10.59,12L5,17.59L6.41,19L12,13.41L17.59,19L19,17.59L13.41,12L19,6.41Z';

export interface ModalHeaderProps extends ComponentPropsWithRef<'div'> {
    /** Content to act as the modal's header */
    children: ReactNode;

    /** Allow the modal to be closed */
    closeable?: boolean;

    /** Description to pass into the header */
    description?: ReactNode;
}

/**
 * ModalHeader Component
 */
const _ModalHeader = (
    props: ModalHeaderProps,
    ref: ForwardedRef<HTMLDivElement>,
): JSX.Element => {
    const {
        children,
        closeable = true,
        description = '',
        ...otherProps
    } = props;

    return (
        <StyledHeader ref={ref} {...otherProps}>
            <StyledContainer>
                <StyledTitle>{children}</StyledTitle>
                {closeable && (
                    <StyledClose tabIndex={-1}>
                        <Icon path={mdiClose} />
                    </StyledClose>
                )}
            </StyledContainer>
            {description && (
                <StyledDescription>{description}</StyledDescription>
            )}
        </StyledHeader>
    );
};

export const ModalHeader = forwardRef(_ModalHeader) as (
    props: ModalHeaderProps & {
        ref?: ForwardedRef<HTMLHeadingElement>;
    },
) => ReturnType<typeof _ModalHeader>;
