/*
 * Copyright 2009-10 www.scribble.org
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 */
package org.scribble.protocol.projection.impl;

import java.text.MessageFormat;

import org.scribble.common.logging.Journal;
import org.scribble.protocol.model.*;
import org.scribble.protocol.projection.ProtocolProjector;

/**
 * This class provides an implementation of the protocol projector.
 *
 */
public class ProtocolProjectorImpl implements ProtocolProjector {

	/**
	 * This method projects a 'global' protocol model to a specified
	 * role's 'local' protocol model.
	 * 
	 * @param model The 'global' protocol model
	 * @param role The role to project
	 * @param journal Journal for reporting issues
	 * @return The 'local' protocol model
	 */
	public ProtocolModel project(ProtocolModel model,
			Role role, Journal journal) {
		return(project(model, null, role, journal));
	}
	
	/**
	 * This method projects a 'global' protocol model to a specified
	 * role's 'local' protocol model.
	 * 
	 * @param model The 'global' protocol model
	 * @param subPath The subpath in the model to be projected, or null if all
	 * @param role The role to project
	 * @param journal Journal for reporting issues
	 * @return The 'local' protocol model
	 */
	public ProtocolModel project(ProtocolModel model, SubProtocolPath subPath,
			Role role, Journal journal) {
		ProtocolModel ret=null;
		
		DefaultProjectorContext context=new DefaultProjectorContext(subPath);
		ModelObject obj=context.project(model, role, journal);
		
		if (obj != null) {
			if (obj instanceof ProtocolModel) {
				ret = (ProtocolModel)obj;
			} else {
				String modelName=model.getProtocol().getName();
				
				if (model.getProtocol().getRole() != null) {
					modelName += ","+model.getProtocol().getRole();
				}
					
				journal.error(MessageFormat.format(
						java.util.PropertyResourceBundle.getBundle("org.scribble.protocol.projection.Messages").getString(
						"_NOT_PROJECTED_MODEL"), modelName), null);
			}
		}
		
		return(ret);
	}

}
